/*
 * Quick & dirty crypto testing module.
 *
 * This will only exist until we have a better testing mechanism
 * (e.g. a char device).
 *
 * Copyright (c) 2002 James Morris <jmorris@intercode.com.au>
 * Copyright (c) 2002 Jean-Francois Dive <jef@linuxbe.org>
 * Copyright (c) 2007 Nokia Siemens Networks
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * 2007-11-13 Added GCM tests
 * 2007-11-13 Added AEAD support
 * 2006-12-07 Added SHA384 HMAC and SHA512 HMAC tests
 * 2004-08-09 Cipher speed tests by Reyk Floeter <reyk@vantronix.net>
 * 2003-09-14 Changes by Kartikey Mahendra Bhatt
 *
 */
#ifndef _CRYPTO_TCRYPT_H
#define _CRYPTO_TCRYPT_H

#define MAX_DIGEST_SIZE		64
#define MAX_TAP			8

#define MAX_KEYLEN		56
#define MAX_IVLEN		32

struct hash_testvec {
	/* only used with keyed hash algorithms */
	char key[132] __attribute__ ((__aligned__(4)));
	char plaintext[240];
	char digest[MAX_DIGEST_SIZE];
	unsigned char tap[MAX_TAP];
	unsigned char psize;
	unsigned char np;
	unsigned char ksize;
};

struct cipher_testvec {
	char key[MAX_KEYLEN] __attribute__ ((__aligned__(4)));
	char iv[MAX_IVLEN];
	char input[4100];
	char result[4100];
	unsigned char tap[MAX_TAP];
	int np;
	unsigned char fail;
	unsigned char wk; /* weak key flag */
	unsigned char klen;
	unsigned short ilen;
	unsigned short rlen;
};

struct aead_testvec {
	char key[MAX_KEYLEN] __attribute__ ((__aligned__(4)));
	char iv[MAX_IVLEN];
	char input[512];
	char assoc[512];
	char result[512];
	unsigned char tap[MAX_TAP];
	unsigned char atap[MAX_TAP];
	int np;
	int anp;
	unsigned char fail;
	unsigned char wk; /* weak key flag */
	unsigned char klen;
	unsigned short ilen;
	unsigned short alen;
	unsigned short rlen;
};

struct cipher_speed {
	unsigned char klen;
	unsigned int blen;
};

struct hash_speed {
	unsigned int blen;	/* buffer length */
	unsigned int plen;	/* per-update length */
};

/*
 * MD4 test vectors from RFC1320
 */
#define MD4_TEST_VECTORS	7

static struct hash_testvec md4_tv_template [] = {
	{
		.plaintext = "",
		.digest	= { 0x31, 0xd6, 0xcf, 0xe0, 0xd1, 0x6a, 0xe9, 0x31,
			    0xb7, 0x3c, 0x59, 0xd7, 0xe0, 0xc0, 0x89, 0xc0 },
	}, {
		.plaintext = "a",
		.psize	= 1,
		.digest	= { 0xbd, 0xe5, 0x2c, 0xb3, 0x1d, 0xe3, 0x3e, 0x46,
			    0x24, 0x5e, 0x05, 0xfb, 0xdb, 0xd6, 0xfb, 0x24 },
	}, {
		.plaintext = "abc",
		.psize	= 3,
		.digest	= { 0xa4, 0x48, 0x01, 0x7a, 0xaf, 0x21, 0xd8, 0x52,
			    0x5f, 0xc1, 0x0a, 0xe8, 0x7a, 0xa6, 0x72, 0x9d },
	}, {
		.plaintext = "message digest",
		.psize	= 14,
		.digest	= { 0xd9, 0x13, 0x0a, 0x81, 0x64, 0x54, 0x9f, 0xe8,
			    0x18, 0x87, 0x48, 0x06, 0xe1, 0xc7, 0x01, 0x4b },
	}, {
		.plaintext = "abcdefghijklmnopqrstuvwxyz",
		.psize	= 26,
		.digest	= { 0xd7, 0x9e, 0x1c, 0x30, 0x8a, 0xa5, 0xbb, 0xcd,
			    0xee, 0xa8, 0xed, 0x63, 0xdf, 0x41, 0x2d, 0xa9 },
		.np	= 2,
		.tap	= { 13, 13 },
	}, {
		.plaintext = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789",
		.psize	= 62,
		.digest	= { 0x04, 0x3f, 0x85, 0x82, 0xf2, 0x41, 0xdb, 0x35,
			    0x1c, 0xe6, 0x27, 0xe1, 0x53, 0xe7, 0xf0, 0xe4 },
	}, {
		.plaintext = "123456789012345678901234567890123456789012345678901234567890123"
			     "45678901234567890",
		.psize	= 80,
		.digest	= { 0xe3, 0x3b, 0x4d, 0xdc, 0x9c, 0x38, 0xf2, 0x19,
			    0x9c, 0x3e, 0x7b, 0x16, 0x4f, 0xcc, 0x05, 0x36 },
	},
};

/*
 * MD5 test vectors from RFC1321
 */
#define MD5_TEST_VECTORS	7

static struct hash_testvec md5_tv_template[] = {
	{
		.digest	= { 0xd4, 0x1d, 0x8c, 0xd9, 0x8f, 0x00, 0xb2, 0x04,
			    0xe9, 0x80, 0x09, 0x98, 0xec, 0xf8, 0x42, 0x7e },
	}, {
		.plaintext = "a",
		.psize	= 1,
		.digest	= { 0x0c, 0xc1, 0x75, 0xb9, 0xc0, 0xf1, 0xb6, 0xa8,
			    0x31, 0xc3, 0x99, 0xe2, 0x69, 0x77, 0x26, 0x61 },
	}, {
		.plaintext = "abc",
		.psize	= 3,
		.digest	= { 0x90, 0x01, 0x50, 0x98, 0x3c, 0xd2, 0x4f, 0xb0,
			    0xd6, 0x96, 0x3f, 0x7d, 0x28, 0xe1, 0x7f, 0x72 },
	}, {
		.plaintext = "message digest",
		.psize	= 14,
		.digest	= { 0xf9, 0x6b, 0x69, 0x7d, 0x7c, 0xb7, 0x93, 0x8d,
			    0x52, 0x5a, 0x2f, 0x31, 0xaa, 0xf1, 0x61, 0xd0 },
	}, {
		.plaintext = "abcdefghijklmnopqrstuvwxyz",
		.psize	= 26,
		.digest	= { 0xc3, 0xfc, 0xd3, 0xd7, 0x61, 0x92, 0xe4, 0x00,
			    0x7d, 0xfb, 0x49, 0x6c, 0xca, 0x67, 0xe1, 0x3b },
		.np	= 2,
		.tap	= {13, 13}
	}, {
		.plaintext = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789",
		.psize	= 62,
		.digest	= { 0xd1, 0x74, 0xab, 0x98, 0xd2, 0x77, 0xd9, 0xf5,
			    0xa5, 0x61, 0x1c, 0x2c, 0x9f, 0x41, 0x9d, 0x9f },
	}, {
		.plaintext = "12345678901234567890123456789012345678901234567890123456789012"
			     "345678901234567890",
		.psize	= 80,
		.digest	= { 0x57, 0xed, 0xf4, 0xa2, 0x2b, 0xe3, 0xc9, 0x55,
			    0xac, 0x49, 0xda, 0x2e, 0x21, 0x07, 0xb6, 0x7a },
	}
};

/*
 * SHA1 test vectors  from from FIPS PUB 180-1
 */
#define SHA1_TEST_VECTORS	2

static struct hash_testvec sha1_tv_template[] = {
	{
		.plaintext = "abc",
		.psize	= 3,
		.digest	= { 0xa9, 0x99, 0x3e, 0x36, 0x47, 0x06, 0x81, 0x6a, 0xba, 0x3e,
			    0x25, 0x71, 0x78, 0x50, 0xc2, 0x6c, 0x9c, 0xd0, 0xd8, 0x9d },
	}, {
		.plaintext = "abcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmnomnopnopq",
		.psize	= 56,
		.digest	= { 0x84, 0x98, 0x3e, 0x44, 0x1c, 0x3b, 0xd2, 0x6e, 0xba, 0xae,
			    0x4a, 0xa1, 0xf9, 0x51, 0x29, 0xe5, 0xe5, 0x46, 0x70, 0xf1 },
		.np	= 2,
		.tap	= { 28, 28 }
	}
};


/*
 * SHA224 test vectors from from FIPS PUB 180-2
 */
#define SHA224_TEST_VECTORS     2

static struct hash_testvec sha224_tv_template[] = {
	{
		.plaintext = "abc",
		.psize  = 3,
		.digest = { 0x23, 0x09, 0x7D, 0x22, 0x34, 0x05, 0xD8, 0x22,
			0x86, 0x42, 0xA4, 0x77, 0xBD, 0xA2, 0x55, 0xB3,
			0x2A, 0xAD, 0xBC, 0xE4, 0xBD, 0xA0, 0xB3, 0xF7,
			0xE3, 0x6C, 0x9D, 0xA7},
	}, {
		.plaintext =
		"abcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmnomnopnopq",
		.psize  = 56,
		.digest = { 0x75, 0x38, 0x8B, 0x16, 0x51, 0x27, 0x76, 0xCC,
			0x5D, 0xBA, 0x5D, 0xA1, 0xFD, 0x89, 0x01, 0x50,
			0xB0, 0xC6, 0x45, 0x5C, 0xB4, 0xF5, 0x8B, 0x19,
			0x52, 0x52, 0x25, 0x25 },
		.np     = 2,
		.tap    = { 28, 28 }
	}
};

/*
 * SHA256 test vectors from from NIST
 */
#define SHA256_TEST_VECTORS	2

static struct hash_testvec sha256_tv_template[] = {
	{
		.plaintext = "abc",
		.psize	= 3,
		.digest	= { 0xba, 0x78, 0x16, 0xbf, 0x8f, 0x01, 0xcf, 0xea,
			    0x41, 0x41, 0x40, 0xde, 0x5d, 0xae, 0x22, 0x23,
			    0xb0, 0x03, 0x61, 0xa3, 0x96, 0x17, 0x7a, 0x9c,
			    0xb4, 0x10, 0xff, 0x61, 0xf2, 0x00, 0x15, 0xad },
	}, {
		.plaintext = "abcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmnomnopnopq",
		.psize	= 56,
		.digest	= { 0x24, 0x8d, 0x6a, 0x61, 0xd2, 0x06, 0x38, 0xb8,
			    0xe5, 0xc0, 0x26, 0x93, 0x0c, 0x3e, 0x60, 0x39,
			    0xa3, 0x3c, 0xe4, 0x59, 0x64, 0xff, 0x21, 0x67,
			    0xf6, 0xec, 0xed, 0xd4, 0x19, 0xdb, 0x06, 0xc1 },
		.np	= 2,
		.tap	= { 28, 28 }
	},
};

/*
 * SHA384 test vectors from from NIST and kerneli
 */
#define SHA384_TEST_VECTORS	4

static struct hash_testvec sha384_tv_template[] = {
	{
		.plaintext= "abc",
		.psize	= 3,
		.digest	= { 0xcb, 0x00, 0x75, 0x3f, 0x45, 0xa3, 0x5e, 0x8b,
			    0xb5, 0xa0, 0x3d, 0x69, 0x9a, 0xc6, 0x50, 0x07,
			    0x27, 0x2c, 0x32, 0xab, 0x0e, 0xde, 0xd1, 0x63,
			    0x1a, 0x8b, 0x60, 0x5a, 0x43, 0xff, 0x5b, 0xed,
			    0x80, 0x86, 0x07, 0x2b, 0xa1, 0xe7, 0xcc, 0x23,
			    0x58, 0xba, 0xec, 0xa1, 0x34, 0xc8, 0x25, 0xa7 },
	}, {
		.plaintext = "abcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmnomnopnopq",
		.psize	= 56,
		.digest	= { 0x33, 0x91, 0xfd, 0xdd, 0xfc, 0x8d, 0xc7, 0x39,
			    0x37, 0x07, 0xa6, 0x5b, 0x1b, 0x47, 0x09, 0x39,
			    0x7c, 0xf8, 0xb1, 0xd1, 0x62, 0xaf, 0x05, 0xab,
			    0xfe, 0x8f, 0x45, 0x0d, 0xe5, 0xf3, 0x6b, 0xc6,
			    0xb0, 0x45, 0x5a, 0x85, 0x20, 0xbc, 0x4e, 0x6f,
			    0x5f, 0xe9, 0x5b, 0x1f, 0xe3, 0xc8, 0x45, 0x2b},
	}, {
		.plaintext = "abcdefghbcdefghicdefghijdefghijkefghijklfghijklmghijklmn"
			     "hijklmnoijklmnopjklmnopqklmnopqrlmnopqrsmnopqrstnopqrstu",
		.psize	= 112,
		.digest	= { 0x09, 0x33, 0x0c, 0x33, 0xf7, 0x11, 0x47, 0xe8,
			    0x3d, 0x19, 0x2f, 0xc7, 0x82, 0xcd, 0x1b, 0x47,
			    0x53, 0x11, 0x1b, 0x17, 0x3b, 0x3b, 0x05, 0xd2,
			    0x2f, 0xa0, 0x80, 0x86, 0xe3, 0xb0, 0xf7, 0x12,
			    0xfc, 0xc7, 0xc7, 0x1a, 0x55, 0x7e, 0x2d, 0xb9,
			    0x66, 0xc3, 0xe9, 0xfa, 0x91, 0x74, 0x60, 0x39  },
	}, {
		.plaintext = "abcdefghijklmnopqrstuvwxyzabcdefghijklmnopqrstuvwxyzabcd"
			     "efghijklmnopqrstuvwxyzabcdefghijklmnopqrstuvwxyz",
		.psize	= 104,
		.digest	= { 0x3d, 0x20, 0x89, 0x73, 0xab, 0x35, 0x08, 0xdb,
			    0xbd, 0x7e, 0x2c, 0x28, 0x62, 0xba, 0x29, 0x0a,
			    0xd3, 0x01, 0x0e, 0x49, 0x78, 0xc1, 0x98, 0xdc,
			    0x4d, 0x8f, 0xd0, 0x14, 0xe5, 0x82, 0x82, 0x3a,
			    0x89, 0xe1, 0x6f, 0x9b, 0x2a, 0x7b, 0xbc, 0x1a,
			    0xc9, 0x38, 0xe2, 0xd1, 0x99, 0xe8, 0xbe, 0xa4 },
		.np	= 4,
		.tap	= { 26, 26, 26, 26 }
	},
};

/*
 * SHA512 test vectors from from NIST and kerneli
 */
#define SHA512_TEST_VECTORS	4

static struct hash_testvec sha512_tv_template[] = {
	{
		.plaintext = "abc",
		.psize	= 3,
		.digest	= { 0xdd, 0xaf, 0x35, 0xa1, 0x93, 0x61, 0x7a, 0xba,
			    0xcc, 0x41, 0x73, 0x49, 0xae, 0x20, 0x41, 0x31,
			    0x12, 0xe6, 0xfa, 0x4e, 0x89, 0xa9, 0x7e, 0xa2,
			    0x0a, 0x9e, 0xee, 0xe6, 0x4b, 0x55, 0xd3, 0x9a,
			    0x21, 0x92, 0x99, 0x2a, 0x27, 0x4f, 0xc1, 0xa8,
			    0x36, 0xba, 0x3c, 0x23, 0xa3, 0xfe, 0xeb, 0xbd,
			    0x45, 0x4d, 0x44, 0x23, 0x64, 0x3c, 0xe8, 0x0e,
			    0x2a, 0x9a, 0xc9, 0x4f, 0xa5, 0x4c, 0xa4, 0x9f },
	}, {
		.plaintext = "abcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmnomnopnopq",
		.psize	= 56,
		.digest	= { 0x20, 0x4a, 0x8f, 0xc6, 0xdd, 0xa8, 0x2f, 0x0a,
			    0x0c, 0xed, 0x7b, 0xeb, 0x8e, 0x08, 0xa4, 0x16,
			    0x57, 0xc1, 0x6e, 0xf4, 0x68, 0xb2, 0x28, 0xa8,
			    0x27, 0x9b, 0xe3, 0x31, 0xa7, 0x03, 0xc3, 0x35,
			    0x96, 0xfd, 0x15, 0xc1, 0x3b, 0x1b, 0x07, 0xf9,
			    0xaa, 0x1d, 0x3b, 0xea, 0x57, 0x78, 0x9c, 0xa0,
			    0x31, 0xad, 0x85, 0xc7, 0xa7, 0x1d, 0xd7, 0x03,
			    0x54, 0xec, 0x63, 0x12, 0x38, 0xca, 0x34, 0x45 },
	}, {
		.plaintext = "abcdefghbcdefghicdefghijdefghijkefghijklfghijklmghijklmn"
			     "hijklmnoijklmnopjklmnopqklmnopqrlmnopqrsmnopqrstnopqrstu",
		.psize	= 112,
		.digest	= { 0x8e, 0x95, 0x9b, 0x75, 0xda, 0xe3, 0x13, 0xda,
			    0x8c, 0xf4, 0xf7, 0x28, 0x14, 0xfc, 0x14, 0x3f,
			    0x8f, 0x77, 0x79, 0xc6, 0xeb, 0x9f, 0x7f, 0xa1,
			    0x72, 0x99, 0xae, 0xad, 0xb6, 0x88, 0x90, 0x18,
			    0x50, 0x1d, 0x28, 0x9e, 0x49, 0x00, 0xf7, 0xe4,
			    0x33, 0x1b, 0x99, 0xde, 0xc4, 0xb5, 0x43, 0x3a,
			    0xc7, 0xd3, 0x29, 0xee, 0xb6, 0xdd, 0x26, 0x54,
			    0x5e, 0x96, 0xe5, 0x5b, 0x87, 0x4b, 0xe9, 0x09 },
	}, {
		.plaintext = "abcdefghijklmnopqrstuvwxyzabcdefghijklmnopqrstuvwxyzabcd"
			     "efghijklmnopqrstuvwxyzabcdefghijklmnopqrstuvwxyz",
		.psize	= 104,
		.digest	= { 0x93, 0x0d, 0x0c, 0xef, 0xcb, 0x30, 0xff, 0x11,
			    0x33, 0xb6, 0x89, 0x81, 0x21, 0xf1, 0xcf, 0x3d,
			    0x27, 0x57, 0x8a, 0xfc, 0xaf, 0xe8, 0x67, 0x7c,
			    0x52, 0x57, 0xcf, 0x06, 0x99, 0x11, 0xf7, 0x5d,
			    0x8f, 0x58, 0x31, 0xb5, 0x6e, 0xbf, 0xda, 0x67,
			    0xb2, 0x78, 0xe6, 0x6d, 0xff, 0x8b, 0x84, 0xfe,
			    0x2b, 0x28, 0x70, 0xf7, 0x42, 0xa5, 0x80, 0xd8,
			    0xed, 0xb4, 0x19, 0x87, 0x23, 0x28, 0x50, 0xc9 },
		.np	= 4,
		.tap	= { 26, 26, 26, 26 }
	},
};


/*
 * WHIRLPOOL test vectors from Whirlpool package
 * by Vincent Rijmen and Paulo S. L. M. Barreto as part of the NESSIE
 * submission
 */
#define WP512_TEST_VECTORS	8

static struct hash_testvec wp512_tv_template[] = {
	{
		.plaintext = "",
		.psize	= 0,
		.digest	= { 0x19, 0xFA, 0x61, 0xD7, 0x55, 0x22, 0xA4, 0x66,
			    0x9B, 0x44, 0xE3, 0x9C, 0x1D, 0x2E, 0x17, 0x26,
			    0xC5, 0x30, 0x23, 0x21, 0x30, 0xD4, 0x07, 0xF8,
			    0x9A, 0xFE, 0xE0, 0x96, 0x49, 0x97, 0xF7, 0xA7,
			    0x3E, 0x83, 0xBE, 0x69, 0x8B, 0x28, 0x8F, 0xEB,
			    0xCF, 0x88, 0xE3, 0xE0, 0x3C, 0x4F, 0x07, 0x57,
			    0xEA, 0x89, 0x64, 0xE5, 0x9B, 0x63, 0xD9, 0x37,
			    0x08, 0xB1, 0x38, 0xCC, 0x42, 0xA6, 0x6E, 0xB3 },


	}, {
		.plaintext = "a",
		.psize	= 1,
		.digest	= { 0x8A, 0xCA, 0x26, 0x02, 0x79, 0x2A, 0xEC, 0x6F,
			    0x11, 0xA6, 0x72, 0x06, 0x53, 0x1F, 0xB7, 0xD7,
			    0xF0, 0xDF, 0xF5, 0x94, 0x13, 0x14, 0x5E, 0x69,
			    0x73, 0xC4, 0x50, 0x01, 0xD0, 0x08, 0x7B, 0x42,
			    0xD1, 0x1B, 0xC6, 0x45, 0x41, 0x3A, 0xEF, 0xF6,
			    0x3A, 0x42, 0x39, 0x1A, 0x39, 0x14, 0x5A, 0x59,
			    0x1A, 0x92, 0x20, 0x0D, 0x56, 0x01, 0x95, 0xE5,
			    0x3B, 0x47, 0x85, 0x84, 0xFD, 0xAE, 0x23, 0x1A },
	}, {
		.plaintext = "abc",
		.psize	= 3,
		.digest	= { 0x4E, 0x24, 0x48, 0xA4, 0xC6, 0xF4, 0x86, 0xBB,
			    0x16, 0xB6, 0x56, 0x2C, 0x73, 0xB4, 0x02, 0x0B,
			    0xF3, 0x04, 0x3E, 0x3A, 0x73, 0x1B, 0xCE, 0x72,
			    0x1A, 0xE1, 0xB3, 0x03, 0xD9, 0x7E, 0x6D, 0x4C,
			    0x71, 0x81, 0xEE, 0xBD, 0xB6, 0xC5, 0x7E, 0x27,
			    0x7D, 0x0E, 0x34, 0x95, 0x71, 0x14, 0xCB, 0xD6,
			    0xC7, 0x97, 0xFC, 0x9D, 0x95, 0xD8, 0xB5, 0x82,
			    0xD2, 0x25, 0x29, 0x20, 0x76, 0xD4, 0xEE, 0xF5 },
	}, {
		.plaintext = "message digest",
		.psize	= 14,
		.digest	= { 0x37, 0x8C, 0x84, 0xA4, 0x12, 0x6E, 0x2D, 0xC6,
			    0xE5, 0x6D, 0xCC, 0x74, 0x58, 0x37, 0x7A, 0xAC,
			    0x83, 0x8D, 0x00, 0x03, 0x22, 0x30, 0xF5, 0x3C,
			    0xE1, 0xF5, 0x70, 0x0C, 0x0F, 0xFB, 0x4D, 0x3B,
			    0x84, 0x21, 0x55, 0x76, 0x59, 0xEF, 0x55, 0xC1,
			    0x06, 0xB4, 0xB5, 0x2A, 0xC5, 0xA4, 0xAA, 0xA6,
			    0x92, 0xED, 0x92, 0x00, 0x52, 0x83, 0x8F, 0x33,
			    0x62, 0xE8, 0x6D, 0xBD, 0x37, 0xA8, 0x90, 0x3E },
	}, {
		.plaintext = "abcdefghijklmnopqrstuvwxyz",
		.psize	= 26,
		.digest	= { 0xF1, 0xD7, 0x54, 0x66, 0x26, 0x36, 0xFF, 0xE9,
			    0x2C, 0x82, 0xEB, 0xB9, 0x21, 0x2A, 0x48, 0x4A,
			    0x8D, 0x38, 0x63, 0x1E, 0xAD, 0x42, 0x38, 0xF5,
			    0x44, 0x2E, 0xE1, 0x3B, 0x80, 0x54, 0xE4, 0x1B,
			    0x08, 0xBF, 0x2A, 0x92, 0x51, 0xC3, 0x0B, 0x6A,
			    0x0B, 0x8A, 0xAE, 0x86, 0x17, 0x7A, 0xB4, 0xA6,
			    0xF6, 0x8F, 0x67, 0x3E, 0x72, 0x07, 0x86, 0x5D,
			    0x5D, 0x98, 0x19, 0xA3, 0xDB, 0xA4, 0xEB, 0x3B },
	}, {
		.plaintext = "ABCDEFGHIJKLMNOPQRSTUVWXYZ"
			     "abcdefghijklmnopqrstuvwxyz0123456789",
		.psize	= 62,
		.digest	= { 0xDC, 0x37, 0xE0, 0x08, 0xCF, 0x9E, 0xE6, 0x9B,
			    0xF1, 0x1F, 0x00, 0xED, 0x9A, 0xBA, 0x26, 0x90,
			    0x1D, 0xD7, 0xC2, 0x8C, 0xDE, 0xC0, 0x66, 0xCC,
			    0x6A, 0xF4, 0x2E, 0x40, 0xF8, 0x2F, 0x3A, 0x1E,
			    0x08, 0xEB, 0xA2, 0x66, 0x29, 0x12, 0x9D, 0x8F,
			    0xB7, 0xCB, 0x57, 0x21, 0x1B, 0x92, 0x81, 0xA6,
			    0x55, 0x17, 0xCC, 0x87, 0x9D, 0x7B, 0x96, 0x21,
			    0x42, 0xC6, 0x5F, 0x5A, 0x7A, 0xF0, 0x14, 0x67 },
	}, {
		.plaintext = "1234567890123456789012345678901234567890"
			     "1234567890123456789012345678901234567890",
		.psize	= 80,
		.digest	= { 0x46, 0x6E, 0xF1, 0x8B, 0xAB, 0xB0, 0x15, 0x4D,
			    0x25, 0xB9, 0xD3, 0x8A, 0x64, 0x14, 0xF5, 0xC0,
			    0x87, 0x84, 0x37, 0x2B, 0xCC, 0xB2, 0x04, 0xD6,
			    0x54, 0x9C, 0x4A, 0xFA, 0xDB, 0x60, 0x14, 0x29,
			    0x4D, 0x5B, 0xD8, 0xDF, 0x2A, 0x6C, 0x44, 0xE5,
			    0x38, 0xCD, 0x04, 0x7B, 0x26, 0x81, 0xA5, 0x1A,
			    0x2C, 0x60, 0x48, 0x1E, 0x88, 0xC5, 0xA2, 0x0B,
			    0x2C, 0x2A, 0x80, 0xCF, 0x3A, 0x9A, 0x08, 0x3B },
	}, {
		.plaintext = "abcdbcdecdefdefgefghfghighijhijk",
		.psize	= 32,
		.digest	= { 0x2A, 0x98, 0x7E, 0xA4, 0x0F, 0x91, 0x70, 0x61,
			    0xF5, 0xD6, 0xF0, 0xA0, 0xE4, 0x64, 0x4F, 0x48,
			    0x8A, 0x7A, 0x5A, 0x52, 0xDE, 0xEE, 0x65, 0x62,
			    0x07, 0xC5, 0x62, 0xF9, 0x88, 0xE9, 0x5C, 0x69,
			    0x16, 0xBD, 0xC8, 0x03, 0x1B, 0xC5, 0xBE, 0x1B,
			    0x7B, 0x94, 0x76, 0x39, 0xFE, 0x05, 0x0B, 0x56,
			    0x93, 0x9B, 0xAA, 0xA0, 0xAD, 0xFF, 0x9A, 0xE6,
			    0x74, 0x5B, 0x7B, 0x18, 0x1C, 0x3B, 0xE3, 0xFD },
	},
};

#define WP384_TEST_VECTORS	8

static struct hash_testvec wp384_tv_template[] = {
	{
		.plaintext = "",
		.psize	= 0,
		.digest	= { 0x19, 0xFA, 0x61, 0xD7, 0x55, 0x22, 0xA4, 0x66,
			    0x9B, 0x44, 0xE3, 0x9C, 0x1D, 0x2E, 0x17, 0x26,
			    0xC5, 0x30, 0x23, 0x21, 0x30, 0xD4, 0x07, 0xF8,
			    0x9A, 0xFE, 0xE0, 0x96, 0x49, 0x97, 0xF7, 0xA7,
			    0x3E, 0x83, 0xBE, 0x69, 0x8B, 0x28, 0x8F, 0xEB,
			    0xCF, 0x88, 0xE3, 0xE0, 0x3C, 0x4F, 0x07, 0x57 },


	}, {
		.plaintext = "a",
		.psize	= 1,
		.digest	= { 0x8A, 0xCA, 0x26, 0x02, 0x79, 0x2A, 0xEC, 0x6F,
			    0x11, 0xA6, 0x72, 0x06, 0x53, 0x1F, 0xB7, 0xD7,
			    0xF0, 0xDF, 0xF5, 0x94, 0x13, 0x14, 0x5E, 0x69,
			    0x73, 0xC4, 0x50, 0x01, 0xD0, 0x08, 0x7B, 0x42,
			    0xD1, 0x1B, 0xC6, 0x45, 0x41, 0x3A, 0xEF, 0xF6,
			    0x3A, 0x42, 0x39, 0x1A, 0x39, 0x14, 0x5A, 0x59 },
	}, {
		.plaintext = "abc",
		.psize	= 3,
		.digest	= { 0x4E, 0x24, 0x48, 0xA4, 0xC6, 0xF4, 0x86, 0xBB,
			    0x16, 0xB6, 0x56, 0x2C, 0x73, 0xB4, 0x02, 0x0B,
			    0xF3, 0x04, 0x3E, 0x3A, 0x73, 0x1B, 0xCE, 0x72,
			    0x1A, 0xE1, 0xB3, 0x03, 0xD9, 0x7E, 0x6D, 0x4C,
			    0x71, 0x81, 0xEE, 0xBD, 0xB6, 0xC5, 0x7E, 0x27,
			    0x7D, 0x0E, 0x34, 0x95, 0x71, 0x14, 0xCB, 0xD6 },
	}, {
		.plaintext = "message digest",
		.psize	= 14,
		.digest	= { 0x37, 0x8C, 0x84, 0xA4, 0x12, 0x6E, 0x2D, 0xC6,
			    0xE5, 0x6D, 0xCC, 0x74, 0x58, 0x37, 0x7A, 0xAC,
			    0x83, 0x8D, 0x00, 0x03, 0x22, 0x30, 0xF5, 0x3C,
			    0xE1, 0xF5, 0x70, 0x0C, 0x0F, 0xFB, 0x4D, 0x3B,
			    0x84, 0x21, 0x55, 0x76, 0x59, 0xEF, 0x55, 0xC1,
			    0x06, 0xB4, 0xB5, 0x2A, 0xC5, 0xA4, 0xAA, 0xA6 },
	}, {
		.plaintext = "abcdefghijklmnopqrstuvwxyz",
		.psize	= 26,
		.digest	= { 0xF1, 0xD7, 0x54, 0x66, 0x26, 0x36, 0xFF, 0xE9,
			    0x2C, 0x82, 0xEB, 0xB9, 0x21, 0x2A, 0x48, 0x4A,
			    0x8D, 0x38, 0x63, 0x1E, 0xAD, 0x42, 0x38, 0xF5,
			    0x44, 0x2E, 0xE1, 0x3B, 0x80, 0x54, 0xE4, 0x1B,
			    0x08, 0xBF, 0x2A, 0x92, 0x51, 0xC3, 0x0B, 0x6A,
			    0x0B, 0x8A, 0xAE, 0x86, 0x17, 0x7A, 0xB4, 0xA6 },
	}, {
		.plaintext = "ABCDEFGHIJKLMNOPQRSTUVWXYZ"
			     "abcdefghijklmnopqrstuvwxyz0123456789",
		.psize	= 62,
		.digest	= { 0xDC, 0x37, 0xE0, 0x08, 0xCF, 0x9E, 0xE6, 0x9B,
			    0xF1, 0x1F, 0x00, 0xED, 0x9A, 0xBA, 0x26, 0x90,
			    0x1D, 0xD7, 0xC2, 0x8C, 0xDE, 0xC0, 0x66, 0xCC,
			    0x6A, 0xF4, 0x2E, 0x40, 0xF8, 0x2F, 0x3A, 0x1E,
			    0x08, 0xEB, 0xA2, 0x66, 0x29, 0x12, 0x9D, 0x8F,
			    0xB7, 0xCB, 0x57, 0x21, 0x1B, 0x92, 0x81, 0xA6 },
	}, {
		.plaintext = "1234567890123456789012345678901234567890"
			     "1234567890123456789012345678901234567890",
		.psize	= 80,
		.digest	= { 0x46, 0x6E, 0xF1, 0x8B, 0xAB, 0xB0, 0x15, 0x4D,
			    0x25, 0xB9, 0xD3, 0x8A, 0x64, 0x14, 0xF5, 0xC0,
			    0x87, 0x84, 0x37, 0x2B, 0xCC, 0xB2, 0x04, 0xD6,
			    0x54, 0x9C, 0x4A, 0xFA, 0xDB, 0x60, 0x14, 0x29,
			    0x4D, 0x5B, 0xD8, 0xDF, 0x2A, 0x6C, 0x44, 0xE5,
			    0x38, 0xCD, 0x04, 0x7B, 0x26, 0x81, 0xA5, 0x1A },
	}, {
		.plaintext = "abcdbcdecdefdefgefghfghighijhijk",
		.psize	= 32,
		.digest	= { 0x2A, 0x98, 0x7E, 0xA4, 0x0F, 0x91, 0x70, 0x61,
			    0xF5, 0xD6, 0xF0, 0xA0, 0xE4, 0x64, 0x4F, 0x48,
			    0x8A, 0x7A, 0x5A, 0x52, 0xDE, 0xEE, 0x65, 0x62,
			    0x07, 0xC5, 0x62, 0xF9, 0x88, 0xE9, 0x5C, 0x69,
			    0x16, 0xBD, 0xC8, 0x03, 0x1B, 0xC5, 0xBE, 0x1B,
			    0x7B, 0x94, 0x76, 0x39, 0xFE, 0x05, 0x0B, 0x56 },
	},
};

#define WP256_TEST_VECTORS	8

static struct hash_testvec wp256_tv_template[] = {
	{
		.plaintext = "",
		.psize	= 0,
		.digest	= { 0x19, 0xFA, 0x61, 0xD7, 0x55, 0x22, 0xA4, 0x66,
			    0x9B, 0x44, 0xE3, 0x9C, 0x1D, 0x2E, 0x17, 0x26,
			    0xC5, 0x30, 0x23, 0x21, 0x30, 0xD4, 0x07, 0xF8,
			    0x9A, 0xFE, 0xE0, 0x96, 0x49, 0x97, 0xF7, 0xA7 },


	}, {
		.plaintext = "a",
		.psize	= 1,
		.digest	= { 0x8A, 0xCA, 0x26, 0x02, 0x79, 0x2A, 0xEC, 0x6F,
			    0x11, 0xA6, 0x72, 0x06, 0x53, 0x1F, 0xB7, 0xD7,
			    0xF0, 0xDF, 0xF5, 0x94, 0x13, 0x14, 0x5E, 0x69,
			    0x73, 0xC4, 0x50, 0x01, 0xD0, 0x08, 0x7B, 0x42 },
	}, {
		.plaintext = "abc",
		.psize	= 3,
		.digest	= { 0x4E, 0x24, 0x48, 0xA4, 0xC6, 0xF4, 0x86, 0xBB,
			    0x16, 0xB6, 0x56, 0x2C, 0x73, 0xB4, 0x02, 0x0B,
			    0xF3, 0x04, 0x3E, 0x3A, 0x73, 0x1B, 0xCE, 0x72,
			    0x1A, 0xE1, 0xB3, 0x03, 0xD9, 0x7E, 0x6D, 0x4C },
	}, {
		.plaintext = "message digest",
		.psize	= 14,
		.digest	= { 0x37, 0x8C, 0x84, 0xA4, 0x12, 0x6E, 0x2D, 0xC6,
			    0xE5, 0x6D, 0xCC, 0x74, 0x58, 0x37, 0x7A, 0xAC,
			    0x83, 0x8D, 0x00, 0x03, 0x22, 0x30, 0xF5, 0x3C,
			    0xE1, 0xF5, 0x70, 0x0C, 0x0F, 0xFB, 0x4D, 0x3B },
	}, {
		.plaintext = "abcdefghijklmnopqrstuvwxyz",
		.psize	= 26,
		.digest	= { 0xF1, 0xD7, 0x54, 0x66, 0x26, 0x36, 0xFF, 0xE9,
			    0x2C, 0x82, 0xEB, 0xB9, 0x21, 0x2A, 0x48, 0x4A,
			    0x8D, 0x38, 0x63, 0x1E, 0xAD, 0x42, 0x38, 0xF5,
			    0x44, 0x2E, 0xE1, 0x3B, 0x80, 0x54, 0xE4, 0x1B },
	}, {
		.plaintext = "ABCDEFGHIJKLMNOPQRSTUVWXYZ"
			     "abcdefghijklmnopqrstuvwxyz0123456789",
		.psize	= 62,
		.digest	= { 0xDC, 0x37, 0xE0, 0x08, 0xCF, 0x9E, 0xE6, 0x9B,
			    0xF1, 0x1F, 0x00, 0xED, 0x9A, 0xBA, 0x26, 0x90,
			    0x1D, 0xD7, 0xC2, 0x8C, 0xDE, 0xC0, 0x66, 0xCC,
			    0x6A, 0xF4, 0x2E, 0x40, 0xF8, 0x2F, 0x3A, 0x1E },
	}, {
		.plaintext = "1234567890123456789012345678901234567890"
			     "1234567890123456789012345678901234567890",
		.psize	= 80,
		.digest	= { 0x46, 0x6E, 0xF1, 0x8B, 0xAB, 0xB0, 0x15, 0x4D,
			    0x25, 0xB9, 0xD3, 0x8A, 0x64, 0x14, 0xF5, 0xC0,
			    0x87, 0x84, 0x37, 0x2B, 0xCC, 0xB2, 0x04, 0xD6,
			    0x54, 0x9C, 0x4A, 0xFA, 0xDB, 0x60, 0x14, 0x29 },
	}, {
		.plaintext = "abcdbcdecdefdefgefghfghighijhijk",
		.psize	= 32,
		.digest	= { 0x2A, 0x98, 0x7E, 0xA4, 0x0F, 0x91, 0x70, 0x61,
			    0xF5, 0xD6, 0xF0, 0xA0, 0xE4, 0x64, 0x4F, 0x48,
			    0x8A, 0x7A, 0x5A, 0x52, 0xDE, 0xEE, 0x65, 0x62,
			    0x07, 0xC5, 0x62, 0xF9, 0x88, 0xE9, 0x5C, 0x69 },
	},
};

/*
 * TIGER test vectors from Tiger website
 */
#define TGR192_TEST_VECTORS	6

static struct hash_testvec tgr192_tv_template[] = {
	{
		.plaintext = "",
		.psize	= 0,
		.digest = { 0x24, 0xf0, 0x13, 0x0c, 0x63, 0xac, 0x93, 0x32,
			    0x16, 0x16, 0x6e, 0x76, 0xb1, 0xbb, 0x92, 0x5f,
			    0xf3, 0x73, 0xde, 0x2d, 0x49, 0x58, 0x4e, 0x7a },
	}, {
		.plaintext = "abc",
		.psize	= 3,
		.digest = { 0xf2, 0x58, 0xc1, 0xe8, 0x84, 0x14, 0xab, 0x2a,
			    0x52, 0x7a, 0xb5, 0x41, 0xff, 0xc5, 0xb8, 0xbf,
			    0x93, 0x5f, 0x7b, 0x95, 0x1c, 0x13, 0x29, 0x51 },
	}, {
		.plaintext = "Tiger",
		.psize	= 5,
		.digest = { 0x9f, 0x00, 0xf5, 0x99, 0x07, 0x23, 0x00, 0xdd,
			    0x27, 0x6a, 0xbb, 0x38, 0xc8, 0xeb, 0x6d, 0xec,
			    0x37, 0x79, 0x0c, 0x11, 0x6f, 0x9d, 0x2b, 0xdf },
	}, {
		.plaintext = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+-",
		.psize	= 64,
		.digest = { 0x87, 0xfb, 0x2a, 0x90, 0x83, 0x85, 0x1c, 0xf7,
			    0x47, 0x0d, 0x2c, 0xf8, 0x10, 0xe6, 0xdf, 0x9e,
			    0xb5, 0x86, 0x44, 0x50, 0x34, 0xa5, 0xa3, 0x86 },
	}, {
		.plaintext = "ABCDEFGHIJKLMNOPQRSTUVWXYZ=abcdefghijklmnopqrstuvwxyz+0123456789",
		.psize	= 64,
		.digest = { 0x46, 0x7d, 0xb8, 0x08, 0x63, 0xeb, 0xce, 0x48,
			    0x8d, 0xf1, 0xcd, 0x12, 0x61, 0x65, 0x5d, 0xe9,
			    0x57, 0x89, 0x65, 0x65, 0x97, 0x5f, 0x91, 0x97 },
	}, {
		.plaintext = "Tiger - A Fast New Hash Function, "
			     "by Ross Anderson and Eli Biham, "
			     "proceedings of Fast Software Encryption 3, "
			     "Cambridge, 1996.",
		.psize  = 125,
		.digest = { 0x3d, 0x9a, 0xeb, 0x03, 0xd1, 0xbd, 0x1a, 0x63,
			    0x57, 0xb2, 0x77, 0x4d, 0xfd, 0x6d, 0x5b, 0x24,
			    0xdd, 0x68, 0x15, 0x1d, 0x50, 0x39, 0x74, 0xfc },
	},
};

#define TGR160_TEST_VECTORS	6

static struct hash_testvec tgr160_tv_template[] = {
	{
		.plaintext = "",
		.psize	= 0,
		.digest = { 0x24, 0xf0, 0x13, 0x0c, 0x63, 0xac, 0x93, 0x32,
			    0x16, 0x16, 0x6e, 0x76, 0xb1, 0xbb, 0x92, 0x5f,
			    0xf3, 0x73, 0xde, 0x2d },
	}, {
		.plaintext = "abc",
		.psize	= 3,
		.digest = { 0xf2, 0x58, 0xc1, 0xe8, 0x84, 0x14, 0xab, 0x2a,
			    0x52, 0x7a, 0xb5, 0x41, 0xff, 0xc5, 0xb8, 0xbf,
			    0x93, 0x5f, 0x7b, 0x95 },
	}, {
		.plaintext = "Tiger",
		.psize	= 5,
		.digest = { 0x9f, 0x00, 0xf5, 0x99, 0x07, 0x23, 0x00, 0xdd,
			    0x27, 0x6a, 0xbb, 0x38, 0xc8, 0xeb, 0x6d, 0xec,
			    0x37, 0x79, 0x0c, 0x11 },
	}, {
		.plaintext = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+-",
		.psize	= 64,
		.digest = { 0x87, 0xfb, 0x2a, 0x90, 0x83, 0x85, 0x1c, 0xf7,
			    0x47, 0x0d, 0x2c, 0xf8, 0x10, 0xe6, 0xdf, 0x9e,
			    0xb5, 0x86, 0x44, 0x50 },
	}, {
		.plaintext = "ABCDEFGHIJKLMNOPQRSTUVWXYZ=abcdefghijklmnopqrstuvwxyz+0123456789",
		.psize	= 64,
		.digest = { 0x46, 0x7d, 0xb8, 0x08, 0x63, 0xeb, 0xce, 0x48,
			    0x8d, 0xf1, 0xcd, 0x12, 0x61, 0x65, 0x5d, 0xe9,
			    0x57, 0x89, 0x65, 0x65 },
	}, {
		.plaintext = "Tiger - A Fast New Hash Function, "
			     "by Ross Anderson and Eli Biham, "
			     "proceedings of Fast Software Encryption 3, "
			     "Cambridge, 1996.",
		.psize  = 125,
		.digest = { 0x3d, 0x9a, 0xeb, 0x03, 0xd1, 0xbd, 0x1a, 0x63,
			    0x57, 0xb2, 0x77, 0x4d, 0xfd, 0x6d, 0x5b, 0x24,
			    0xdd, 0x68, 0x15, 0x1d },
	},
};

#define TGR128_TEST_VECTORS	6

static struct hash_testvec tgr128_tv_template[] = {
	{
		.plaintext = "",
		.psize	= 0,
		.digest = { 0x24, 0xf0, 0x13, 0x0c, 0x63, 0xac, 0x93, 0x32,
			    0x16, 0x16, 0x6e, 0x76, 0xb1, 0xbb, 0x92, 0x5f },
	}, {
		.plaintext = "abc",
		.psize	= 3,
		.digest = { 0xf2, 0x58, 0xc1, 0xe8, 0x84, 0x14, 0xab, 0x2a,
			    0x52, 0x7a, 0xb5, 0x41, 0xff, 0xc5, 0xb8, 0xbf },
	}, {
		.plaintext = "Tiger",
		.psize	= 5,
		.digest = { 0x9f, 0x00, 0xf5, 0x99, 0x07, 0x23, 0x00, 0xdd,
			    0x27, 0x6a, 0xbb, 0x38, 0xc8, 0xeb, 0x6d, 0xec },
	}, {
		.plaintext = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+-",
		.psize	= 64,
		.digest = { 0x87, 0xfb, 0x2a, 0x90, 0x83, 0x85, 0x1c, 0xf7,
			    0x47, 0x0d, 0x2c, 0xf8, 0x10, 0xe6, 0xdf, 0x9e },
	}, {
		.plaintext = "ABCDEFGHIJKLMNOPQRSTUVWXYZ=abcdefghijklmnopqrstuvwxyz+0123456789",
		.psize	= 64,
		.digest = { 0x46, 0x7d, 0xb8, 0x08, 0x63, 0xeb, 0xce, 0x48,
			    0x8d, 0xf1, 0xcd, 0x12, 0x61, 0x65, 0x5d, 0xe9 },
	}, {
		.plaintext = "Tiger - A Fast New Hash Function, "
			     "by Ross Anderson and Eli Biham, "
			     "proceedings of Fast Software Encryption 3, "
			     "Cambridge, 1996.",
		.psize  = 125,
		.digest = { 0x3d, 0x9a, 0xeb, 0x03, 0xd1, 0xbd, 0x1a, 0x63,
			    0x57, 0xb2, 0x77, 0x4d, 0xfd, 0x6d, 0x5b, 0x24 },
	},
};

/*
 * HMAC-MD5 test vectors from RFC2202
 * (These need to be fixed to not use strlen).
 */
#define HMAC_MD5_TEST_VECTORS	7

static struct hash_testvec hmac_md5_tv_template[] =
{
	{
		.key	= { [0 ... 15] =  0x0b },
		.ksize	= 16,
		.plaintext = "Hi There",
		.psize	= 8,
		.digest	= { 0x92, 0x94, 0x72, 0x7a, 0x36, 0x38, 0xbb, 0x1c,
			    0x13, 0xf4, 0x8e, 0xf8, 0x15, 0x8b, 0xfc, 0x9d },
	}, {
		.key	= { 'J', 'e', 'f', 'e' },
		.ksize	= 4,
		.plaintext = "what do ya want for nothing?",
		.psize	= 28,
		.digest	= { 0x75, 0x0c, 0x78, 0x3e, 0x6a, 0xb0, 0xb5, 0x03,
			    0xea, 0xa8, 0x6e, 0x31, 0x0a, 0x5d, 0xb7, 0x38 },
		.np	= 2,
		.tap	= {14, 14}
	}, {
		.key	= { [0 ... 15] = 0xaa },
		.ksize	= 16,
		.plaintext = { [0 ... 49] =  0xdd },
		.psize	= 50,
		.digest	= { 0x56, 0xbe, 0x34, 0x52, 0x1d, 0x14, 0x4c, 0x88,
			    0xdb, 0xb8, 0xc7, 0x33, 0xf0, 0xe8, 0xb3, 0xf6 },
	}, {
		.key	= { 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08,
			    0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f, 0x10,
			    0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19, },
		.ksize	= 25,
		.plaintext = { [0 ... 49] =  0xcd },
		.psize	= 50,
		.digest	= { 0x69, 0x7e, 0xaf, 0x0a, 0xca, 0x3a, 0x3a, 0xea,
			    0x3a, 0x75, 0x16, 0x47, 0x46, 0xff, 0xaa, 0x79 },
	}, {
		.key	= { [0 ... 15] = 0x0c },
		.ksize	= 16,
		.plaintext = "Test With Truncation",
		.psize	= 20,
		.digest	= { 0x56, 0x46, 0x1e, 0xf2, 0x34, 0x2e, 0xdc, 0x00,
			    0xf9, 0xba, 0xb9, 0x95, 0x69, 0x0e, 0xfd, 0x4c },
	}, {
		.key	= { [0 ... 79] =  0xaa },
		.ksize	= 80,
		.plaintext = "Test Using Larger Than Block-Size Key - Hash Key First",
		.psize	= 54,
		.digest	= { 0x6b, 0x1a, 0xb7, 0xfe, 0x4b, 0xd7, 0xbf, 0x8f,
			    0x0b, 0x62, 0xe6, 0xce, 0x61, 0xb9, 0xd0, 0xcd },
	}, {
		.key	= { [0 ... 79] =  0xaa },
		.ksize	= 80,
		.plaintext = "Test Using Larger Than Block-Size Key and Larger Than One "
			     "Block-Size Data",
		.psize	= 73,
		.digest	= { 0x6f, 0x63, 0x0f, 0xad, 0x67, 0xcd, 0xa0, 0xee,
			    0x1f, 0xb1, 0xf5, 0x62, 0xdb, 0x3a, 0xa5, 0x3e },
	},
};

/*
 * HMAC-SHA1 test vectors from RFC2202
 */
#define HMAC_SHA1_TEST_VECTORS	7

static struct hash_testvec hmac_sha1_tv_template[] = {
	{
		.key	= { [0 ... 19] = 0x0b },
		.ksize	= 20,
		.plaintext = "Hi There",
		.psize	= 8,
		.digest	= { 0xb6, 0x17, 0x31, 0x86, 0x55, 0x05, 0x72, 0x64,
			    0xe2, 0x8b, 0xc0, 0xb6, 0xfb, 0x37, 0x8c, 0x8e, 0xf1,
			    0x46, 0xbe },
	}, {
		.key	= { 'J', 'e', 'f', 'e' },
		.ksize	= 4,
		.plaintext = "what do ya want for nothing?",
		.psize	= 28,
		.digest	= { 0xef, 0xfc, 0xdf, 0x6a, 0xe5, 0xeb, 0x2f, 0xa2, 0xd2, 0x74,
			    0x16, 0xd5, 0xf1, 0x84, 0xdf, 0x9c, 0x25, 0x9a, 0x7c, 0x79 },
		.np	= 2,
		.tap	= { 14, 14 }
	}, {
		.key	= { [0 ... 19] = 0xaa },
		.ksize	= 20,
		.plaintext = { [0 ... 49] = 0xdd },
		.psize	= 50,
		.digest	= { 0x12, 0x5d, 0x73, 0x42, 0xb9, 0xac, 0x11, 0xcd, 0x91, 0xa3,
			    0x9a, 0xf4, 0x8a, 0xa1, 0x7b, 0x4f, 0x63, 0xf1, 0x75, 0xd3 },
	}, {
		.key	= { 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08,
			    0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f, 0x10,
			    0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19 },
		.ksize	= 25,
		.plaintext = { [0 ... 49] = 0xcd },
		.psize	= 50,
		.digest	= { 0x4c, 0x90, 0x07, 0xf4, 0x02, 0x62, 0x50, 0xc6, 0xbc, 0x84,
			    0x14, 0xf9, 0xbf, 0x50, 0xc8, 0x6c, 0x2d, 0x72, 0x35, 0xda },
	}, {
		.key	= { [0 ... 19] = 0x0c },
		.ksize	= 20,
		.plaintext = "Test With Truncation",
		.psize	= 20,
		.digest	= { 0x4c, 0x1a, 0x03, 0x42, 0x4b, 0x55, 0xe0, 0x7f, 0xe7, 0xf2,
			    0x7b, 0xe1, 0xd5, 0x8b, 0xb9, 0x32, 0x4a, 0x9a, 0x5a, 0x04 },
	}, {
		.key	= { [0 ... 79] = 0xaa },
		.ksize	= 80,
		.plaintext = "Test Using Larger Than Block-Size Key - Hash Key First",
		.psize	= 54,
		.digest	= { 0xaa, 0x4a, 0xe5, 0xe1, 0x52, 0x72, 0xd0, 0x0e, 0x95, 0x70,
			    0x56, 0x37, 0xce, 0x8a, 0x3b, 0x55, 0xed, 0x40, 0x21, 0x12 },
	}, {
		.key	= { [0 ... 79] = 0xaa },
		.ksize	= 80,
		.plaintext = "Test Using Larger Than Block-Size Key and Larger Than One "
			     "Block-Size Data",
		.psize	= 73,
		.digest	= { 0xe8, 0xe9, 0x9d, 0x0f, 0x45, 0x23, 0x7d, 0x78, 0x6d, 0x6b,
			    0xba, 0xa7, 0x96, 0x5c, 0x78, 0x08, 0xbb, 0xff, 0x1a, 0x91 },
	},
};


/*
 * SHA224 HMAC test vectors from RFC4231
 */
#define HMAC_SHA224_TEST_VECTORS    4

static struct hash_testvec hmac_sha224_tv_template[] = {
	{
		.key    = { 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b,
			0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b,
			0x0b, 0x0b, 0x0b, 0x0b },
		.ksize  = 20,
		/*  ("Hi There") */
		.plaintext = { 0x48, 0x69, 0x20, 0x54, 0x68, 0x65, 0x72, 0x65 },
		.psize  = 8,
		.digest = { 0x89, 0x6f, 0xb1, 0x12, 0x8a, 0xbb, 0xdf, 0x19,
			0x68, 0x32, 0x10, 0x7c, 0xd4, 0x9d, 0xf3, 0x3f,
			0x47, 0xb4, 0xb1, 0x16, 0x99, 0x12, 0xba, 0x4f,
			0x53, 0x68, 0x4b, 0x22},
	}, {
		.key    = { 0x4a, 0x65, 0x66, 0x65 }, /* ("Jefe") */
		.ksize  = 4,
		/* ("what do ya want for nothing?") */
		.plaintext = { 0x77, 0x68, 0x61, 0x74, 0x20, 0x64, 0x6f, 0x20,
			0x79, 0x61, 0x20, 0x77, 0x61, 0x6e, 0x74, 0x20,
			0x66, 0x6f, 0x72, 0x20, 0x6e, 0x6f, 0x74, 0x68,
			0x69, 0x6e, 0x67, 0x3f },
		.psize  = 28,
		.digest = { 0xa3, 0x0e, 0x01, 0x09, 0x8b, 0xc6, 0xdb, 0xbf,
			0x45, 0x69, 0x0f, 0x3a, 0x7e, 0x9e, 0x6d, 0x0f,
			0x8b, 0xbe, 0xa2, 0xa3, 0x9e, 0x61, 0x48, 0x00,
			0x8f, 0xd0, 0x5e, 0x44 },
		.np = 4,
		.tap    = { 7, 7, 7, 7 }
	}, {
		.key    = { 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			0xaa, 0xaa, 0xaa },
		.ksize  = 131,
		/* ("Test Using Larger Than Block-Size Key - Hash Key First") */
		.plaintext = { 0x54, 0x65, 0x73, 0x74, 0x20, 0x55, 0x73, 0x69,
			0x6e, 0x67, 0x20, 0x4c, 0x61, 0x72, 0x67, 0x65,
			0x72, 0x20, 0x54, 0x68, 0x61, 0x6e, 0x20, 0x42,
			0x6c, 0x6f, 0x63, 0x6b, 0x2d, 0x53, 0x69, 0x7a,
			0x65, 0x20, 0x4b, 0x65, 0x79, 0x20, 0x2d, 0x20,
			0x48, 0x61, 0x73, 0x68, 0x20, 0x4b, 0x65, 0x79,
			0x20, 0x46, 0x69, 0x72, 0x73, 0x74 },
		.psize  = 54,
		.digest = { 0x95, 0xe9, 0xa0, 0xdb, 0x96, 0x20, 0x95, 0xad,
			0xae, 0xbe, 0x9b, 0x2d, 0x6f, 0x0d, 0xbc, 0xe2,
			0xd4, 0x99, 0xf1, 0x12, 0xf2, 0xd2, 0xb7, 0x27,
			0x3f, 0xa6, 0x87, 0x0e },
	}, {
		.key    = { 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			0xaa, 0xaa, 0xaa },
		.ksize  = 131,
		/* ("This is a test using a larger than block-size key and a")
		(" larger than block-size data. The key needs to be")
			(" hashed before being used by the HMAC algorithm.") */
		.plaintext = { 0x54, 0x68, 0x69, 0x73, 0x20, 0x69, 0x73, 0x20,
			0x61, 0x20, 0x74, 0x65, 0x73, 0x74, 0x20, 0x75,
			0x73, 0x69, 0x6e, 0x67, 0x20, 0x61, 0x20, 0x6c,
			0x61, 0x72, 0x67, 0x65, 0x72, 0x20, 0x74, 0x68,
			0x61, 0x6e, 0x20, 0x62, 0x6c, 0x6f, 0x63, 0x6b,
			0x2d, 0x73, 0x69, 0x7a, 0x65, 0x20, 0x6b, 0x65,
			0x79, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x61, 0x20,
			0x6c, 0x61, 0x72, 0x67, 0x65, 0x72, 0x20, 0x74,
			0x68, 0x61, 0x6e, 0x20, 0x62, 0x6c, 0x6f, 0x63,
			0x6b, 0x2d, 0x73, 0x69, 0x7a, 0x65, 0x20, 0x64,
			0x61, 0x74, 0x61, 0x2e, 0x20, 0x54, 0x68, 0x65,
			0x20, 0x6b, 0x65, 0x79, 0x20, 0x6e, 0x65, 0x65,
			0x64, 0x73, 0x20, 0x74, 0x6f, 0x20, 0x62, 0x65,
			0x20, 0x68, 0x61, 0x73, 0x68, 0x65, 0x64, 0x20,
			0x62, 0x65, 0x66, 0x6f, 0x72, 0x65, 0x20, 0x62,
			0x65, 0x69, 0x6e, 0x67, 0x20, 0x75, 0x73, 0x65,
			0x64, 0x20, 0x62, 0x79, 0x20, 0x74, 0x68, 0x65,
			0x20, 0x48, 0x4d, 0x41, 0x43, 0x20, 0x61, 0x6c,
			0x67, 0x6f, 0x72, 0x69, 0x74, 0x68, 0x6d, 0x2e },
		.psize  = 152,
		.digest = { 0x3a, 0x85, 0x41, 0x66, 0xac, 0x5d, 0x9f, 0x02,
			0x3f, 0x54, 0xd5, 0x17, 0xd0, 0xb3, 0x9d, 0xbd,
			0x94, 0x67, 0x70, 0xdb, 0x9c, 0x2b, 0x95, 0xc9,
			0xf6, 0xf5, 0x65, 0xd1 },
	},
};

/*
 * HMAC-SHA256 test vectors from
 * draft-ietf-ipsec-ciph-sha-256-01.txt
 */
#define HMAC_SHA256_TEST_VECTORS	10

static struct hash_testvec hmac_sha256_tv_template[] = {
	{
		.key	= { 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08,
			    0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f, 0x10,
			    0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18,
			    0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f, 0x20},
		.ksize	= 32,
		.plaintext = "abc",
		.psize	= 3,
		.digest	= { 0xa2, 0x1b, 0x1f, 0x5d, 0x4c, 0xf4, 0xf7, 0x3a,
			    0x4d, 0xd9, 0x39, 0x75, 0x0f, 0x7a, 0x06, 0x6a,
			    0x7f, 0x98, 0xcc, 0x13, 0x1c, 0xb1, 0x6a, 0x66,
			    0x92, 0x75, 0x90, 0x21, 0xcf, 0xab, 0x81, 0x81 },
	}, {
		.key	= { 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08,
			    0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f, 0x10,
			    0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18,
			    0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f, 0x20 },
		.ksize	= 32,
		.plaintext = "abcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmnomnopnopq",
		.psize	= 56,
		.digest	= { 0x10, 0x4f, 0xdc, 0x12, 0x57, 0x32, 0x8f, 0x08,
			    0x18, 0x4b, 0xa7, 0x31, 0x31, 0xc5, 0x3c, 0xae,
			    0xe6, 0x98, 0xe3, 0x61, 0x19, 0x42, 0x11, 0x49,
			    0xea, 0x8c, 0x71, 0x24, 0x56, 0x69, 0x7d, 0x30 },
	}, {
		.key	= { 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08,
			    0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f, 0x10,
			    0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18,
			    0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f, 0x20 },
		.ksize	= 32,
		.plaintext = "abcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmnomnopnopq"
			     "abcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmnomnopnopq",
		.psize	= 112,
		.digest	= { 0x47, 0x03, 0x05, 0xfc, 0x7e, 0x40, 0xfe, 0x34,
			    0xd3, 0xee, 0xb3, 0xe7, 0x73, 0xd9, 0x5a, 0xab,
			    0x73, 0xac, 0xf0, 0xfd, 0x06, 0x04, 0x47, 0xa5,
			    0xeb, 0x45, 0x95, 0xbf, 0x33, 0xa9, 0xd1, 0xa3 },
	}, {
		.key	= { [0 ... 31] = 0x0b },
		.ksize	= 32,
		.plaintext = "Hi There",
		.psize	= 8,
		.digest	= { 0x19, 0x8a, 0x60, 0x7e, 0xb4, 0x4b, 0xfb, 0xc6,
			    0x99, 0x03, 0xa0, 0xf1, 0xcf, 0x2b, 0xbd, 0xc5,
			    0xba, 0x0a, 0xa3, 0xf3, 0xd9, 0xae, 0x3c, 0x1c,
			    0x7a, 0x3b, 0x16, 0x96, 0xa0, 0xb6, 0x8c, 0xf7 },
	}, {
		.key	= "Jefe",
		.ksize	= 4,
		.plaintext = "what do ya want for nothing?",
		.psize	= 28,
		.digest	= { 0x5b, 0xdc, 0xc1, 0x46, 0xbf, 0x60, 0x75, 0x4e,
			    0x6a, 0x04, 0x24, 0x26, 0x08, 0x95, 0x75, 0xc7,
			    0x5a, 0x00, 0x3f, 0x08, 0x9d, 0x27, 0x39, 0x83,
			    0x9d, 0xec, 0x58, 0xb9, 0x64, 0xec, 0x38, 0x43 },
		.np	= 2,
		.tap	= { 14, 14 }
	}, {
		.key	= { [0 ... 31] = 0xaa },
		.ksize	= 32,
		.plaintext = { [0 ... 49] = 0xdd },
		.psize	= 50,
		.digest	= { 0xcd, 0xcb, 0x12, 0x20, 0xd1, 0xec, 0xcc, 0xea,
			    0x91, 0xe5, 0x3a, 0xba, 0x30, 0x92, 0xf9, 0x62,
			    0xe5, 0x49, 0xfe, 0x6c, 0xe9, 0xed, 0x7f, 0xdc,
			    0x43, 0x19, 0x1f, 0xbd, 0xe4, 0x5c, 0x30, 0xb0 },
	}, {
		.key	= { 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08,
			    0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f, 0x10,
			    0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18,
			    0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f, 0x20,
			    0x21, 0x22, 0x23, 0x24, 0x25 },
		.ksize	= 37,
		.plaintext = { [0 ... 49] = 0xcd },
		.psize	= 50,
		.digest	= { 0xd4, 0x63, 0x3c, 0x17, 0xf6, 0xfb, 0x8d, 0x74,
			    0x4c, 0x66, 0xde, 0xe0, 0xf8, 0xf0, 0x74, 0x55,
			    0x6e, 0xc4, 0xaf, 0x55, 0xef, 0x07, 0x99, 0x85,
			    0x41, 0x46, 0x8e, 0xb4, 0x9b, 0xd2, 0xe9, 0x17 },
	}, {
		.key	= { [0 ... 31] = 0x0c },
		.ksize	= 32,
		.plaintext = "Test With Truncation",
		.psize	= 20,
		.digest	= { 0x75, 0x46, 0xaf, 0x01, 0x84, 0x1f, 0xc0, 0x9b,
			    0x1a, 0xb9, 0xc3, 0x74, 0x9a, 0x5f, 0x1c, 0x17,
			    0xd4, 0xf5, 0x89, 0x66, 0x8a, 0x58, 0x7b, 0x27,
			    0x00, 0xa9, 0xc9, 0x7c, 0x11, 0x93, 0xcf, 0x42 },
	}, {
		.key	= { [0 ... 79] = 0xaa },
		.ksize	= 80,
		.plaintext = "Test Using Larger Than Block-Size Key - Hash Key First",
		.psize	= 54,
		.digest	= { 0x69, 0x53, 0x02, 0x5e, 0xd9, 0x6f, 0x0c, 0x09,
			    0xf8, 0x0a, 0x96, 0xf7, 0x8e, 0x65, 0x38, 0xdb,
			    0xe2, 0xe7, 0xb8, 0x20, 0xe3, 0xdd, 0x97, 0x0e,
			    0x7d, 0xdd, 0x39, 0x09, 0x1b, 0x32, 0x35, 0x2f },
	}, {
		.key	= { [0 ... 79] = 0xaa },
		.ksize	= 80,
		.plaintext = "Test Using Larger Than Block-Size Key and Larger Than "
			     "One Block-Size Data",
		.psize	= 73,
		.digest	= { 0x63, 0x55, 0xac, 0x22, 0xe8, 0x90, 0xd0, 0xa3,
			    0xc8, 0x48, 0x1a, 0x5c, 0xa4, 0x82, 0x5b, 0xc8,
			    0x84, 0xd3, 0xe7, 0xa1, 0xff, 0x98, 0xa2, 0xfc,
			    0x2a, 0xc7, 0xd8, 0xe0, 0x64, 0xc3, 0xb2, 0xe6 },
	},
};

#define XCBC_AES_TEST_VECTORS 6

static struct hash_testvec aes_xcbc128_tv_template[] = {
	{
		.key	= { 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
			    0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f },
		.plaintext = { [0 ... 15] = 0 },
		.digest = { 0x75, 0xf0, 0x25, 0x1d, 0x52, 0x8a, 0xc0, 0x1c,
			    0x45, 0x73, 0xdf, 0xd5, 0x84, 0xd7, 0x9f, 0x29 },
		.psize	= 0,
		.ksize	= 16,
	}, {
		.key	= { 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
			    0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f },
		.plaintext = { 0x00, 0x01, 0x02 },
		.digest	= { 0x5b, 0x37, 0x65, 0x80, 0xae, 0x2f, 0x19, 0xaf,
			    0xe7, 0x21, 0x9c, 0xee, 0xf1, 0x72, 0x75, 0x6f },
		.psize	= 3,
		.ksize	= 16,
	} , {
		.key	= { 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
			    0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f },
		.plaintext = { 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
			       0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f },
		.digest = { 0xd2, 0xa2, 0x46, 0xfa, 0x34, 0x9b, 0x68, 0xa7,
			    0x99, 0x98, 0xa4, 0x39, 0x4f, 0xf7, 0xa2, 0x63 },
		.psize	= 16,
		.ksize	= 16,
	}, {
		.key	= { 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
			    0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f },
		.plaintext = { 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
			       0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
			       0x10, 0x11, 0x12, 0x13 },
		.digest = { 0x47, 0xf5, 0x1b, 0x45, 0x64, 0x96, 0x62, 0x15,
			    0xb8, 0x98, 0x5c, 0x63, 0x05, 0x5e, 0xd3, 0x08 },
		.tap	= { 10, 10 },
		.psize	= 20,
		.np	= 2,
		.ksize	= 16,
	}, {
		.key	= { 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
			    0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f },
		.plaintext = { 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
			       0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
			       0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
			       0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f },
		.digest = { 0xf5, 0x4f, 0x0e, 0xc8, 0xd2, 0xb9, 0xf3, 0xd3,
			    0x68, 0x07, 0x73, 0x4b, 0xd5, 0x28, 0x3f, 0xd4 },
		.psize	= 32,
		.ksize	= 16,
	}, {
		.key	= { 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
			    0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f },
		.plaintext = { 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
			       0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
			       0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
			       0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,
			       0x20, 0x21 },
		.digest = { 0xbe, 0xcb, 0xb3, 0xbc, 0xcd, 0xb5, 0x18, 0xa3,
			    0x06, 0x77, 0xd5, 0x48, 0x1f, 0xb6, 0xb4, 0xd8 },
		.tap	= { 17, 17 },
		.psize	= 34,
		.np	= 2,
		.ksize	= 16,
	}
};

/*
 * SHA384 HMAC test vectors from RFC4231
 */

#define HMAC_SHA384_TEST_VECTORS	4

static struct hash_testvec hmac_sha384_tv_template[] = {
	{
		.key	= { 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b,
			    0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b,
			    0x0b, 0x0b, 0x0b, 0x0b }, // (20 bytes)
		.ksize	= 20,
		.plaintext = { 0x48, 0x69, 0x20, 0x54, 0x68, 0x65, 0x72, 0x65 }, // ("Hi There")
		.psize	= 8,
		.digest	= { 0xaf, 0xd0, 0x39, 0x44, 0xd8, 0x48, 0x95, 0x62,
			    0x6b, 0x08, 0x25, 0xf4, 0xab, 0x46, 0x90, 0x7f,
			    0x15, 0xf9, 0xda, 0xdb, 0xe4, 0x10, 0x1e, 0xc6,
			    0x82, 0xaa, 0x03, 0x4c, 0x7c, 0xeb, 0xc5, 0x9c,
			    0xfa, 0xea, 0x9e, 0xa9, 0x07, 0x6e, 0xde, 0x7f,
			    0x4a, 0xf1, 0x52, 0xe8, 0xb2, 0xfa, 0x9c, 0xb6 },
	}, {
		.key	= { 0x4a, 0x65, 0x66, 0x65 }, // ("Jefe")
		.ksize	= 4,
		.plaintext = { 0x77, 0x68, 0x61, 0x74, 0x20, 0x64, 0x6f, 0x20,
			       0x79, 0x61, 0x20, 0x77, 0x61, 0x6e, 0x74, 0x20, // ("what do ya want ")
			       0x66, 0x6f, 0x72, 0x20, 0x6e, 0x6f, 0x74, 0x68,
			       0x69, 0x6e, 0x67, 0x3f }, // ("for nothing?")
		.psize	= 28,
		.digest	= { 0xaf, 0x45, 0xd2, 0xe3, 0x76, 0x48, 0x40, 0x31,
			    0x61, 0x7f, 0x78, 0xd2, 0xb5, 0x8a, 0x6b, 0x1b,
			    0x9c, 0x7e, 0xf4, 0x64, 0xf5, 0xa0, 0x1b, 0x47,
			    0xe4, 0x2e, 0xc3, 0x73, 0x63, 0x22, 0x44, 0x5e,
			    0x8e, 0x22, 0x40, 0xca, 0x5e, 0x69, 0xe2, 0xc7,
			    0x8b, 0x32, 0x39, 0xec, 0xfa, 0xb2, 0x16, 0x49 },
		.np	= 4,
		.tap	= { 7, 7, 7, 7 }
	}, {
		.key	= { 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			    0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			    0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			    0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			    0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			    0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			    0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			    0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			    0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			    0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			    0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			    0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			    0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			    0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			    0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			    0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			    0xaa, 0xaa, 0xaa }, // (131 bytes)
		.ksize	= 131,
		.plaintext = { 0x54, 0x65, 0x73, 0x74, 0x20, 0x55, 0x73, 0x69,
			       0x6e, 0x67, 0x20, 0x4c, 0x61, 0x72, 0x67, 0x65, // ("Test Using Large")
			       0x72, 0x20, 0x54, 0x68, 0x61, 0x6e, 0x20, 0x42,
			       0x6c, 0x6f, 0x63, 0x6b, 0x2d, 0x53, 0x69, 0x7a, // ("r Than Block-Siz")
			       0x65, 0x20, 0x4b, 0x65, 0x79, 0x20, 0x2d, 0x20,
			       0x48, 0x61, 0x73, 0x68, 0x20, 0x4b, 0x65, 0x79, // ("e Key - Hash Key")
			       0x20, 0x46, 0x69, 0x72, 0x73, 0x74 }, // (" First")
		.psize	= 54,
		.digest	= { 0x4e, 0xce, 0x08, 0x44, 0x85, 0x81, 0x3e, 0x90,
			    0x88, 0xd2, 0xc6, 0x3a, 0x04, 0x1b, 0xc5, 0xb4,
			    0x4f, 0x9e, 0xf1, 0x01, 0x2a, 0x2b, 0x58, 0x8f,
			    0x3c, 0xd1, 0x1f, 0x05, 0x03, 0x3a, 0xc4, 0xc6,
			    0x0c, 0x2e, 0xf6, 0xab, 0x40, 0x30, 0xfe, 0x82,
			    0x96, 0x24, 0x8d, 0xf1, 0x63, 0xf4, 0x49, 0x52 },
	}, {
		.key	= { 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			    0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			    0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			    0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			    0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			    0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			    0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			    0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			    0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			    0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			    0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			    0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			    0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			    0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			    0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			    0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			    0xaa, 0xaa, 0xaa }, // (131 bytes)
		.ksize	= 131,
		.plaintext = { 0x54, 0x68, 0x69, 0x73, 0x20, 0x69, 0x73, 0x20,
			       0x61, 0x20, 0x74, 0x65, 0x73, 0x74, 0x20, 0x75, // ("This is a test u")
			       0x73, 0x69, 0x6e, 0x67, 0x20, 0x61, 0x20, 0x6c,
			       0x61, 0x72, 0x67, 0x65, 0x72, 0x20, 0x74, 0x68, // ("sing a larger th")
			       0x61, 0x6e, 0x20, 0x62, 0x6c, 0x6f, 0x63, 0x6b,
			       0x2d, 0x73, 0x69, 0x7a, 0x65, 0x20, 0x6b, 0x65, // ("an block-size ke")
			       0x79, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x61, 0x20,
			       0x6c, 0x61, 0x72, 0x67, 0x65, 0x72, 0x20, 0x74, // ("y and a larger t")
			       0x68, 0x61, 0x6e, 0x20, 0x62, 0x6c, 0x6f, 0x63,
			       0x6b, 0x2d, 0x73, 0x69, 0x7a, 0x65, 0x20, 0x64, // ("han block-size d")
			       0x61, 0x74, 0x61, 0x2e, 0x20, 0x54, 0x68, 0x65,
			       0x20, 0x6b, 0x65, 0x79, 0x20, 0x6e, 0x65, 0x65, // ("ata. The key nee")
			       0x64, 0x73, 0x20, 0x74, 0x6f, 0x20, 0x62, 0x65,
			       0x20, 0x68, 0x61, 0x73, 0x68, 0x65, 0x64, 0x20, // ("ds to be hashed ")
			       0x62, 0x65, 0x66, 0x6f, 0x72, 0x65, 0x20, 0x62,
			       0x65, 0x69, 0x6e, 0x67, 0x20, 0x75, 0x73, 0x65, // ("before being use")
			       0x64, 0x20, 0x62, 0x79, 0x20, 0x74, 0x68, 0x65,
			       0x20, 0x48, 0x4d, 0x41, 0x43, 0x20, 0x61, 0x6c, // ("d by the HMAC al")
			       0x67, 0x6f, 0x72, 0x69, 0x74, 0x68, 0x6d, 0x2e }, // ("gorithm.")
		.psize	= 152,
		.digest	= { 0x66, 0x17, 0x17, 0x8e, 0x94, 0x1f, 0x02, 0x0d,
			    0x35, 0x1e, 0x2f, 0x25, 0x4e, 0x8f, 0xd3, 0x2c,
			    0x60, 0x24, 0x20, 0xfe, 0xb0, 0xb8, 0xfb, 0x9a,
			    0xdc, 0xce, 0xbb, 0x82, 0x46, 0x1e, 0x99, 0xc5,
			    0xa6, 0x78, 0xcc, 0x31, 0xe7, 0x99, 0x17, 0x6d,
			    0x38, 0x60, 0xe6, 0x11, 0x0c, 0x46, 0x52, 0x3e },
	},
};

/*
 * SHA512 HMAC test vectors from RFC4231
 */

#define HMAC_SHA512_TEST_VECTORS	4

static struct hash_testvec hmac_sha512_tv_template[] = {
	{
		.key	= { 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b,
			    0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b,
			    0x0b, 0x0b, 0x0b, 0x0b }, // (20 bytes)
		.ksize	= 20,
		.plaintext = { 0x48, 0x69, 0x20, 0x54, 0x68, 0x65, 0x72, 0x65 }, // ("Hi There")
		.psize	= 8,
		.digest	= { 0x87, 0xaa, 0x7c, 0xde, 0xa5, 0xef, 0x61, 0x9d,
			    0x4f, 0xf0, 0xb4, 0x24, 0x1a, 0x1d, 0x6c, 0xb0,
			    0x23, 0x79, 0xf4, 0xe2, 0xce, 0x4e, 0xc2, 0x78,
			    0x7a, 0xd0, 0xb3, 0x05, 0x45, 0xe1, 0x7c, 0xde,
			    0xda, 0xa8, 0x33, 0xb7, 0xd6, 0xb8, 0xa7, 0x02,
			    0x03, 0x8b, 0x27, 0x4e, 0xae, 0xa3, 0xf4, 0xe4,
			    0xbe, 0x9d, 0x91, 0x4e, 0xeb, 0x61, 0xf1, 0x70,
			    0x2e, 0x69, 0x6c, 0x20, 0x3a, 0x12, 0x68, 0x54 },
	}, {
		.key	= { 0x4a, 0x65, 0x66, 0x65 }, // ("Jefe")
		.ksize	= 4,
		.plaintext = { 0x77, 0x68, 0x61, 0x74, 0x20, 0x64, 0x6f, 0x20,
			       0x79, 0x61, 0x20, 0x77, 0x61, 0x6e, 0x74, 0x20, // ("what do ya want ")
			       0x66, 0x6f, 0x72, 0x20, 0x6e, 0x6f, 0x74, 0x68,
			       0x69, 0x6e, 0x67, 0x3f }, // ("for nothing?")
		.psize	= 28,
		.digest	= { 0x16, 0x4b, 0x7a, 0x7b, 0xfc, 0xf8, 0x19, 0xe2,
			    0xe3, 0x95, 0xfb, 0xe7, 0x3b, 0x56, 0xe0, 0xa3,
			    0x87, 0xbd, 0x64, 0x22, 0x2e, 0x83, 0x1f, 0xd6,
			    0x10, 0x27, 0x0c, 0xd7, 0xea, 0x25, 0x05, 0x54,
			    0x97, 0x58, 0xbf, 0x75, 0xc0, 0x5a, 0x99, 0x4a,
			    0x6d, 0x03, 0x4f, 0x65, 0xf8, 0xf0, 0xe6, 0xfd,
			    0xca, 0xea, 0xb1, 0xa3, 0x4d, 0x4a, 0x6b, 0x4b,
			    0x63, 0x6e, 0x07, 0x0a, 0x38, 0xbc, 0xe7, 0x37 },
		.np	= 4,
		.tap	= { 7, 7, 7, 7 }
	}, {
		.key	= { 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			    0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			    0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			    0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			    0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			    0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			    0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			    0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			    0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			    0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			    0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			    0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			    0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			    0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			    0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			    0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			    0xaa, 0xaa, 0xaa }, // (131 bytes)
		.ksize	= 131,
		.plaintext = { 0x54, 0x65, 0x73, 0x74, 0x20, 0x55, 0x73, 0x69,
			       0x6e, 0x67, 0x20, 0x4c, 0x61, 0x72, 0x67, 0x65, // ("Test Using Large")
			       0x72, 0x20, 0x54, 0x68, 0x61, 0x6e, 0x20, 0x42,
			       0x6c, 0x6f, 0x63, 0x6b, 0x2d, 0x53, 0x69, 0x7a, // ("r Than Block-Siz")
			       0x65, 0x20, 0x4b, 0x65, 0x79, 0x20, 0x2d, 0x20,
			       0x48, 0x61, 0x73, 0x68, 0x20, 0x4b, 0x65, 0x79, // ("e Key - Hash Key")
			       0x20, 0x46, 0x69, 0x72, 0x73, 0x74 }, // (" First")
		.psize	= 54,
		.digest	= { 0x80, 0xb2, 0x42, 0x63, 0xc7, 0xc1, 0xa3, 0xeb,
			    0xb7, 0x14, 0x93, 0xc1, 0xdd, 0x7b, 0xe8, 0xb4,
			    0x9b, 0x46, 0xd1, 0xf4, 0x1b, 0x4a, 0xee, 0xc1,
			    0x12, 0x1b, 0x01, 0x37, 0x83, 0xf8, 0xf3, 0x52,
			    0x6b, 0x56, 0xd0, 0x37, 0xe0, 0x5f, 0x25, 0x98,
			    0xbd, 0x0f, 0xd2, 0x21, 0x5d, 0x6a, 0x1e, 0x52,
			    0x95, 0xe6, 0x4f, 0x73, 0xf6, 0x3f, 0x0a, 0xec,
			    0x8b, 0x91, 0x5a, 0x98, 0x5d, 0x78, 0x65, 0x98 },
	}, {
		.key	= { 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			    0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			    0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			    0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			    0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			    0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			    0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			    0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			    0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			    0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			    0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			    0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			    0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			    0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			    0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			    0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
			    0xaa, 0xaa, 0xaa }, // (131 bytes)
		.ksize	= 131,
		.plaintext = { 0x54, 0x68, 0x69, 0x73, 0x20, 0x69, 0x73, 0x20,
			       0x61, 0x20, 0x74, 0x65, 0x73, 0x74, 0x20, 0x75, // ("This is a test u")
			       0x73, 0x69, 0x6e, 0x67, 0x20, 0x61, 0x20, 0x6c,
			       0x61, 0x72, 0x67, 0x65, 0x72, 0x20, 0x74, 0x68, // ("sing a larger th")
			       0x61, 0x6e, 0x20, 0x62, 0x6c, 0x6f, 0x63, 0x6b,
			       0x2d, 0x73, 0x69, 0x7a, 0x65, 0x20, 0x6b, 0x65, // ("an block-size ke")
			       0x79, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x61, 0x20,
			       0x6c, 0x61, 0x72, 0x67, 0x65, 0x72, 0x20, 0x74, // ("y and a larger t")
			       0x68, 0x61, 0x6e, 0x20, 0x62, 0x6c, 0x6f, 0x63,
			       0x6b, 0x2d, 0x73, 0x69, 0x7a, 0x65, 0x20, 0x64, // ("han block-size d")
			       0x61, 0x74, 0x61, 0x2e, 0x20, 0x54, 0x68, 0x65,
			       0x20, 0x6b, 0x65, 0x79, 0x20, 0x6e, 0x65, 0x65, // ("ata. The key nee")
			       0x64, 0x73, 0x20, 0x74, 0x6f, 0x20, 0x62, 0x65,
			       0x20, 0x68, 0x61, 0x73, 0x68, 0x65, 0x64, 0x20, // ("ds to be hashed ")
			       0x62, 0x65, 0x66, 0x6f, 0x72, 0x65, 0x20, 0x62,
			       0x65, 0x69, 0x6e, 0x67, 0x20, 0x75, 0x73, 0x65, // ("before being use")
			       0x64, 0x20, 0x62, 0x79, 0x20, 0x74, 0x68, 0x65,
			       0x20, 0x48, 0x4d, 0x41, 0x43, 0x20, 0x61, 0x6c, // ("d by the HMAC al")
			       0x67, 0x6f, 0x72, 0x69, 0x74, 0x68, 0x6d, 0x2e }, // ("gorithm.")
		.psize	= 152,
		.digest	= { 0xe3, 0x7b, 0x6a, 0x77, 0x5d, 0xc8, 0x7d, 0xba,
			    0xa4, 0xdf, 0xa9, 0xf9, 0x6e, 0x5e, 0x3f, 0xfd,
			    0xde, 0xbd, 0x71, 0xf8, 0x86, 0x72, 0x89, 0x86,
			    0x5d, 0xf5, 0xa3, 0x2d, 0x20, 0xcd, 0xc9, 0x44,
			    0xb6, 0x02, 0x2c, 0xac, 0x3c, 0x49, 0x82, 0xb1,
			    0x0d, 0x5e, 0xeb, 0x55, 0xc3, 0xe4, 0xde, 0x15,
			    0x13, 0x46, 0x76, 0xfb, 0x6d, 0xe0, 0x44, 0x60,
			    0x65, 0xc9, 0x74, 0x40, 0xfa, 0x8c, 0x6a, 0x58 },
	},
};

/*
 * DES test vectors.
 */
#define DES_ENC_TEST_VECTORS		10
#define DES_DEC_TEST_VECTORS		4
#define DES_CBC_ENC_TEST_VECTORS	5
#define DES_CBC_DEC_TEST_VECTORS	4
#define DES3_EDE_ENC_TEST_VECTORS	3
#define DES3_EDE_DEC_TEST_VECTORS	3

static struct cipher_testvec des_enc_tv_template[] = {
	{ /* From Applied Cryptography */
		.key	= { 0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef },
		.klen	= 8,
		.input	= { 0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xe7 },
		.ilen	= 8,
		.result	= { 0xc9, 0x57, 0x44, 0x25, 0x6a, 0x5e, 0xd3, 0x1d },
		.rlen	= 8,
	}, { /* Same key, different plaintext block */
		.key	= { 0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef },
		.klen	= 8,
		.input	= { 0x22, 0x33, 0x44, 0x55, 0x66, 0x77, 0x88, 0x99 },
		.ilen	= 8,
		.result	= { 0xf7, 0x9c, 0x89, 0x2a, 0x33, 0x8f, 0x4a, 0x8b },
		.rlen	= 8,
	}, { /* Sbox test from NBS */
		.key	= { 0x7c, 0xa1, 0x10, 0x45, 0x4a, 0x1a, 0x6e, 0x57 },
		.klen	= 8,
		.input	= { 0x01, 0xa1, 0xd6, 0xd0, 0x39, 0x77, 0x67, 0x42 },
		.ilen	= 8,
		.result	= { 0x69, 0x0f, 0x5b, 0x0d, 0x9a, 0x26, 0x93, 0x9b },
		.rlen	= 8,
	}, { /* Three blocks */
		.key	= { 0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef },
		.klen	= 8,
		.input	= { 0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xe7,
			    0x22, 0x33, 0x44, 0x55, 0x66, 0x77, 0x88, 0x99,
			    0xca, 0xfe, 0xba, 0xbe, 0xfe, 0xed, 0xbe, 0xef },
		.ilen	= 24,
		.result	= { 0xc9, 0x57, 0x44, 0x25, 0x6a, 0x5e, 0xd3, 0x1d,
			    0xf7, 0x9c, 0x89, 0x2a, 0x33, 0x8f, 0x4a, 0x8b,
			    0xb4, 0x99, 0x26, 0xf7, 0x1f, 0xe1, 0xd4, 0x90 },
		.rlen	= 24,
	}, { /* Weak key */
		.fail	= 1,
		.wk	= 1,
		.key	= { 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01 },
		.klen	= 8,
		.input	= { 0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xe7 },
		.ilen	= 8,
		.result	= { 0xc9, 0x57, 0x44, 0x25, 0x6a, 0x5e, 0xd3, 0x1d },
		.rlen	= 8,
	}, { /* Two blocks -- for testing encryption across pages */
		.key	= { 0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef },
		.klen	= 8,
		.input	= { 0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xe7,
			    0x22, 0x33, 0x44, 0x55, 0x66, 0x77, 0x88, 0x99 },
		.ilen	= 16,
		.result	= { 0xc9, 0x57, 0x44, 0x25, 0x6a, 0x5e, 0xd3, 0x1d,
			    0xf7, 0x9c, 0x89, 0x2a, 0x33, 0x8f, 0x4a, 0x8b },
		.rlen	= 16,
		.np	= 2,
		.tap	= { 8, 8 }
	}, { /* Four blocks -- for testing encryption with chunking */
		.key	= { 0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef },
		.klen	= 8,
		.input	= { 0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xe7,
			    0x22, 0x33, 0x44, 0x55, 0x66, 0x77, 0x88, 0x99,
			    0xca, 0xfe, 0xba, 0xbe, 0xfe, 0xed, 0xbe, 0xef,
			    0x22, 0x33, 0x44, 0x55, 0x66, 0x77, 0x88, 0x99 },
		.ilen	= 32,
		.result	= { 0xc9, 0x57, 0x44, 0x25, 0x6a, 0x5e, 0xd3, 0x1d,
			    0xf7, 0x9c, 0x89, 0x2a, 0x33, 0x8f, 0x4a, 0x8b,
			    0xb4, 0x99, 0x26, 0xf7, 0x1f, 0xe1, 0xd4, 0x90,
			    0xf7, 0x9c, 0x89, 0x2a, 0x33, 0x8f, 0x4a, 0x8b },
		.rlen	= 32,
		.np	= 3,
		.tap	= { 14, 10, 8 }
	}, {
		.key	= { 0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef },
		.klen	= 8,
		.input	= { 0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xe7,
			    0x22, 0x33, 0x44, 0x55, 0x66, 0x77, 0x88, 0x99,
			    0xca, 0xfe, 0xba, 0xbe, 0xfe, 0xed, 0xbe, 0xef },
		.ilen	= 24,
		.result	= { 0xc9, 0x57, 0x44, 0x25, 0x6a, 0x5e, 0xd3, 0x1d,
			    0xf7, 0x9c, 0x89, 0x2a, 0x33, 0x8f, 0x4a, 0x8b,
			    0xb4, 0x99, 0x26, 0xf7, 0x1f, 0xe1, 0xd4, 0x90 },
		.rlen	= 24,
		.np	= 4,
		.tap	= { 2, 1, 3, 18 }
	}, {
		.key	= { 0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef },
		.klen	= 8,
		.input	= { 0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xe7,
			    0x22, 0x33, 0x44, 0x55, 0x66, 0x77, 0x88, 0x99 },
		.ilen	= 16,
		.result	= { 0xc9, 0x57, 0x44, 0x25, 0x6a, 0x5e, 0xd3, 0x1d,
			    0xf7, 0x9c, 0x89, 0x2a, 0x33, 0x8f, 0x4a, 0x8b },
		.rlen	= 16,
		.np	= 5,
		.tap	= { 2, 2, 2, 2, 8 }
	}, {
		.key	= { 0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef },
		.klen	= 8,
		.input	= { 0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xe7 },
		.ilen	= 8,
		.result	= { 0xc9, 0x57, 0x44, 0x25, 0x6a, 0x5e, 0xd3, 0x1d },
		.rlen	= 8,
		.np	= 8,
		.tap	= { 1, 1, 1, 1, 1, 1, 1, 1 }
	},
};

static struct cipher_testvec des_dec_tv_template[] = {
	{ /* From Applied Cryptography */
		.key	= { 0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef },
		.klen	= 8,
		.input	= { 0xc9, 0x57, 0x44, 0x25, 0x6a, 0x5e, 0xd3, 0x1d },
		.ilen	= 8,
		.result	= { 0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xe7 },
		.rlen	= 8,
	}, { /* Sbox test from NBS */
		.key	= { 0x7c, 0xa1, 0x10, 0x45, 0x4a, 0x1a, 0x6e, 0x57 },
		.klen	= 8,
		.input	= { 0x69, 0x0f, 0x5b, 0x0d, 0x9a, 0x26, 0x93, 0x9b },
		.ilen	= 8,
		.result	= { 0x01, 0xa1, 0xd6, 0xd0, 0x39, 0x77, 0x67, 0x42 },
		.rlen	= 8,
	}, { /* Two blocks, for chunking test */
		.key	= { 0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef },
		.klen	= 8,
		.input	= { 0xc9, 0x57, 0x44, 0x25, 0x6a, 0x5e, 0xd3, 0x1d,
			    0x69, 0x0f, 0x5b, 0x0d, 0x9a, 0x26, 0x93, 0x9b },
		.ilen	= 16,
		.result	= { 0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xe7,
			    0xa3, 0x99, 0x7b, 0xca, 0xaf, 0x69, 0xa0, 0xf5 },
		.rlen	= 16,
		.np	= 2,
		.tap	= { 8, 8 }
	}, {
		.key	= { 0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef },
		.klen	= 8,
		.input	= { 0xc9, 0x57, 0x44, 0x25, 0x6a, 0x5e, 0xd3, 0x1d,
			    0x69, 0x0f, 0x5b, 0x0d, 0x9a, 0x26, 0x93, 0x9b },
		.ilen	= 16,
		.result	= { 0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xe7,
			    0xa3, 0x99, 0x7b, 0xca, 0xaf, 0x69, 0xa0, 0xf5 },
		.rlen	= 16,
		.np	= 3,
		.tap	= { 3, 12, 1 }
	},
};

static struct cipher_testvec des_cbc_enc_tv_template[] = {
	{ /* From OpenSSL */
		.key	= { 0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef},
		.klen	= 8,
		.iv	= { 0xfe, 0xdc, 0xba, 0x98, 0x76, 0x54, 0x32, 0x10},
		.input	= { 0x37, 0x36, 0x35, 0x34, 0x33, 0x32, 0x31, 0x20,
			    0x4e, 0x6f, 0x77, 0x20, 0x69, 0x73, 0x20, 0x74,
			    0x68, 0x65, 0x20, 0x74, 0x69, 0x6d, 0x65, 0x20 },
		.ilen	= 24,
		.result	= { 0xcc, 0xd1, 0x73, 0xff, 0xab, 0x20, 0x39, 0xf4,
			    0xac, 0xd8, 0xae, 0xfd, 0xdf, 0xd8, 0xa1, 0xeb,
			    0x46, 0x8e, 0x91, 0x15, 0x78, 0x88, 0xba, 0x68 },
		.rlen	= 24,
	}, { /* FIPS Pub 81 */
		.key	= { 0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef },
		.klen	= 8,
		.iv	= { 0x12, 0x34, 0x56, 0x78, 0x90, 0xab, 0xcd, 0xef },
		.input	= { 0x4e, 0x6f, 0x77, 0x20, 0x69, 0x73, 0x20, 0x74 },
		.ilen	= 8,
		.result	= { 0xe5, 0xc7, 0xcd, 0xde, 0x87, 0x2b, 0xf2, 0x7c },
		.rlen	= 8,
	}, {
		.key	= { 0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef },
		.klen	= 8,
		.iv	= { 0xe5, 0xc7, 0xcd, 0xde, 0x87, 0x2b, 0xf2, 0x7c },
		.input	= { 0x68, 0x65, 0x20, 0x74, 0x69, 0x6d, 0x65, 0x20 },
		.ilen	= 8,
		.result	= { 0x43, 0xe9, 0x34, 0x00, 0x8c, 0x38, 0x9c, 0x0f },
		.rlen	= 8,
	}, {
		.key	= { 0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef },
		.klen	= 8,
		.iv	= { 0x43, 0xe9, 0x34, 0x00, 0x8c, 0x38, 0x9c, 0x0f },
		.input	= { 0x66, 0x6f, 0x72, 0x20, 0x61, 0x6c, 0x6c, 0x20 },
		.ilen	= 8,
		.result	= { 0x68, 0x37, 0x88, 0x49, 0x9a, 0x7c, 0x05, 0xf6 },
		.rlen	= 8,
	}, { /* Copy of openssl vector for chunk testing */
	     /* From OpenSSL */
		.key	= { 0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef},
		.klen	= 8,
		.iv	= { 0xfe, 0xdc, 0xba, 0x98, 0x76, 0x54, 0x32, 0x10},
		.input	= { 0x37, 0x36, 0x35, 0x34, 0x33, 0x32, 0x31, 0x20,
			    0x4e, 0x6f, 0x77, 0x20, 0x69, 0x73, 0x20, 0x74,
			    0x68, 0x65, 0x20, 0x74, 0x69, 0x6d, 0x65, 0x20 },
		.ilen	= 24,
		.result	= { 0xcc, 0xd1, 0x73, 0xff, 0xab, 0x20, 0x39, 0xf4,
			    0xac, 0xd8, 0xae, 0xfd, 0xdf, 0xd8, 0xa1, 0xeb,
			    0x46, 0x8e, 0x91, 0x15, 0x78, 0x88, 0xba, 0x68 },
		.rlen	= 24,
		.np	= 2,
		.tap	= { 13, 11 }
	},
};

static struct cipher_testvec des_cbc_dec_tv_template[] = {
	{ /* FIPS Pub 81 */
		.key	= { 0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef },
		.klen	= 8,
		.iv	= { 0x12, 0x34, 0x56, 0x78, 0x90, 0xab, 0xcd, 0xef },
		.input	= { 0xe5, 0xc7, 0xcd, 0xde, 0x87, 0x2b, 0xf2, 0x7c },
		.ilen	= 8,
		.result	= { 0x4e, 0x6f, 0x77, 0x20, 0x69, 0x73, 0x20, 0x74 },
		.rlen	= 8,
	}, {
		.key	= { 0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef },
		.klen	= 8,
		.iv	= { 0xe5, 0xc7, 0xcd, 0xde, 0x87, 0x2b, 0xf2, 0x7c },
		.input	= { 0x43, 0xe9, 0x34, 0x00, 0x8c, 0x38, 0x9c, 0x0f },
		.ilen	= 8,
		.result	= { 0x68, 0x65, 0x20, 0x74, 0x69, 0x6d, 0x65, 0x20 },
		.rlen	= 8,
	}, {
		.key	= { 0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef },
		.klen	= 8,
		.iv	= { 0x43, 0xe9, 0x34, 0x00, 0x8c, 0x38, 0x9c, 0x0f },
		.input	= { 0x68, 0x37, 0x88, 0x49, 0x9a, 0x7c, 0x05, 0xf6 },
		.ilen	= 8,
		.result	= { 0x66, 0x6f, 0x72, 0x20, 0x61, 0x6c, 0x6c, 0x20 },
		.rlen	= 8,
	}, { /* Copy of above, for chunk testing */
		.key	= { 0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef },
		.klen	= 8,
		.iv	= { 0x43, 0xe9, 0x34, 0x00, 0x8c, 0x38, 0x9c, 0x0f },
		.input	= { 0x68, 0x37, 0x88, 0x49, 0x9a, 0x7c, 0x05, 0xf6 },
		.ilen	= 8,
		.result	= { 0x66, 0x6f, 0x72, 0x20, 0x61, 0x6c, 0x6c, 0x20 },
		.rlen	= 8,
		.np	= 2,
		.tap	= { 4, 4 }
	},
};

/*
 * We really need some more test vectors, especially for DES3 CBC.
 */
static struct cipher_testvec des3_ede_enc_tv_template[] = {
	{ /* These are from openssl */
		.key	= { 0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef,
			    0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
			    0xfe, 0xdc, 0xba, 0x98, 0x76, 0x54, 0x32, 0x10},
		.klen	= 24,
		.input	= { 0x73, 0x6f, 0x6d, 0x65, 0x64, 0x61, 0x74, 0x61 },
		.ilen	= 8,
		.result	= { 0x18, 0xd7, 0x48, 0xe5, 0x63, 0x62, 0x05, 0x72 },
		.rlen	= 8,
	}, {
		.key	= { 0x03, 0x52, 0x02, 0x07, 0x67, 0x20, 0x82, 0x17,
			    0x86, 0x02, 0x87, 0x66, 0x59, 0x08, 0x21, 0x98,
			    0x64, 0x05, 0x6a, 0xbd, 0xfe, 0xa9, 0x34, 0x57 },
		.klen	= 24,
		.input	= { 0x73, 0x71, 0x75, 0x69, 0x67, 0x67, 0x6c, 0x65 },
		.ilen	= 8,
		.result	= { 0xc0, 0x7d, 0x2a, 0x0f, 0xa5, 0x66, 0xfa, 0x30 },
		.rlen	= 8,
	}, {
		.key	= { 0x10, 0x46, 0x10, 0x34, 0x89, 0x98, 0x80, 0x20,
			    0x91, 0x07, 0xd0, 0x15, 0x89, 0x19, 0x01, 0x01,
			    0x19, 0x07, 0x92, 0x10, 0x98, 0x1a, 0x01, 0x01 },
		.klen	= 24,
		.input	= { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
		.ilen	= 8,
		.result	= { 0xe1, 0xef, 0x62, 0xc3, 0x32, 0xfe, 0x82, 0x5b },
		.rlen	= 8,
	},
};

static struct cipher_testvec des3_ede_dec_tv_template[] = {
	{ /* These are from openssl */
		.key	= { 0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef,
			    0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55, 0x55,
			    0xfe, 0xdc, 0xba, 0x98, 0x76, 0x54, 0x32, 0x10},
		.klen	= 24,
		.input	= { 0x18, 0xd7, 0x48, 0xe5, 0x63, 0x62, 0x05, 0x72 },
		.ilen	= 8,
		.result	= { 0x73, 0x6f, 0x6d, 0x65, 0x64, 0x61, 0x74, 0x61 },
		.rlen	= 8,
	}, {
		.key	= { 0x03, 0x52, 0x02, 0x07, 0x67, 0x20, 0x82, 0x17,
			    0x86, 0x02, 0x87, 0x66, 0x59, 0x08, 0x21, 0x98,
			    0x64, 0x05, 0x6a, 0xbd, 0xfe, 0xa9, 0x34, 0x57 },
		.klen	= 24,
		.input	= { 0xc0, 0x7d, 0x2a, 0x0f, 0xa5, 0x66, 0xfa, 0x30 },
		.ilen	= 8,
		.result	= { 0x73, 0x71, 0x75, 0x69, 0x67, 0x67, 0x6c, 0x65 },
		.rlen	= 8,
	}, {
		.key	= { 0x10, 0x46, 0x10, 0x34, 0x89, 0x98, 0x80, 0x20,
			    0x91, 0x07, 0xd0, 0x15, 0x89, 0x19, 0x01, 0x01,
			    0x19, 0x07, 0x92, 0x10, 0x98, 0x1a, 0x01, 0x01 },
		.klen	= 24,
		.input	= { 0xe1, 0xef, 0x62, 0xc3, 0x32, 0xfe, 0x82, 0x5b },
		.ilen	= 8,
		.result	= { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
		.rlen	= 8,
	},
};

/*
 * Blowfish test vectors.
 */
#define BF_ENC_TEST_VECTORS	6
#define BF_DEC_TEST_VECTORS	6
#define BF_CBC_ENC_TEST_VECTORS	1
#define BF_CBC_DEC_TEST_VECTORS	1

static struct cipher_testvec bf_enc_tv_template[] = {
	{ /* DES test vectors from OpenSSL */
		.key	= { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, },
		.klen	= 8,
		.input	= { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
		.ilen	= 8,
		.result	= { 0x4e, 0xf9, 0x97, 0x45, 0x61, 0x98, 0xdd, 0x78 },
		.rlen	= 8,
	}, {
		.key	= { 0x1f, 0x1f, 0x1f, 0x1f, 0x0e, 0x0e, 0x0e, 0x0e },
		.klen	= 8,
		.input	= { 0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef },
		.ilen	= 8,
		.result	= { 0xa7, 0x90, 0x79, 0x51, 0x08, 0xea, 0x3c, 0xae },
		.rlen	= 8,
	}, {
		.key	= { 0xf0, 0xe1, 0xd2, 0xc3, 0xb4, 0xa5, 0x96, 0x87 },
		.klen	= 8,
		.input	= { 0xfe, 0xdc, 0xba, 0x98, 0x76, 0x54, 0x32, 0x10 },
		.ilen	= 8,
		.result	= { 0xe8, 0x7a, 0x24, 0x4e, 0x2c, 0xc8, 0x5e, 0x82 },
		.rlen	= 8,
	}, { /* Vary the keylength... */
		.key	= { 0xf0, 0xe1, 0xd2, 0xc3, 0xb4, 0xa5, 0x96, 0x87,
			    0x78, 0x69, 0x5a, 0x4b, 0x3c, 0x2d, 0x1e, 0x0f },
		.klen	= 16,
		.input	= { 0xfe, 0xdc, 0xba, 0x98, 0x76, 0x54, 0x32, 0x10 },
		.ilen	= 8,
		.result	= { 0x93, 0x14, 0x28, 0x87, 0xee, 0x3b, 0xe1, 0x5c },
		.rlen	= 8,
	}, {
		.key	= { 0xf0, 0xe1, 0xd2, 0xc3, 0xb4, 0xa5, 0x96, 0x87,
			    0x78, 0x69, 0x5a, 0x4b, 0x3c, 0x2d, 0x1e, 0x0f,
			    0x00, 0x11, 0x22, 0x33, 0x44 },
		.klen	= 21,
		.input	= { 0xfe, 0xdc, 0xba, 0x98, 0x76, 0x54, 0x32, 0x10 },
		.ilen	= 8,
		.result	= { 0xe6, 0xf5, 0x1e, 0xd7, 0x9b, 0x9d, 0xb2, 0x1f },
		.rlen	= 8,
	}, { /* Generated with bf488 */
		.key	= { 0xf0, 0xe1, 0xd2, 0xc3, 0xb4, 0xa5, 0x96, 0x87,
			    0x78, 0x69, 0x5a, 0x4b, 0x3c, 0x2d, 0x1e, 0x0f,
			    0x00, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77,
			    0x04, 0x68, 0x91, 0x04, 0xc2, 0xfd, 0x3b, 0x2f,
			    0x58, 0x40, 0x23, 0x64, 0x1a, 0xba, 0x61, 0x76,
			    0x1f, 0x1f, 0x1f, 0x1f, 0x0e, 0x0e, 0x0e, 0x0e,
			    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff },
		.klen	= 56,
		.input	= { 0xfe, 0xdc, 0xba, 0x98, 0x76, 0x54, 0x32, 0x10 },
		.ilen	= 8,
		.result	= { 0xc0, 0x45, 0x04, 0x01, 0x2e, 0x4e, 0x1f, 0x53 },
		.rlen	= 8,
	},
};

static struct cipher_testvec bf_dec_tv_template[] = {
	{ /* DES test vectors from OpenSSL */
		.key	= { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
		.klen	= 8,
		.input	= { 0x4e, 0xf9, 0x97, 0x45, 0x61, 0x98, 0xdd, 0x78 },
		.ilen	= 8,
		.result	= { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
		.rlen	= 8,
	}, {
		.key	= { 0x1f, 0x1f, 0x1f, 0x1f, 0x0e, 0x0e, 0x0e, 0x0e },
		.klen	= 8,
		.input	= { 0xa7, 0x90, 0x79, 0x51, 0x08, 0xea, 0x3c, 0xae },
		.ilen	= 8,
		.result	= { 0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef },
		.rlen	= 8,
	}, {
		.key	= { 0xf0, 0xe1, 0xd2, 0xc3, 0xb4, 0xa5, 0x96, 0x87 },
		.klen	= 8,
		.input	= { 0xe8, 0x7a, 0x24, 0x4e, 0x2c, 0xc8, 0x5e, 0x82 },
		.ilen	= 8,
		.result	= { 0xfe, 0xdc, 0xba, 0x98, 0x76, 0x54, 0x32, 0x10 },
		.rlen	= 8,
	}, { /* Vary the keylength... */
		.key	= { 0xf0, 0xe1, 0xd2, 0xc3, 0xb4, 0xa5, 0x96, 0x87,
			    0x78, 0x69, 0x5a, 0x4b, 0x3c, 0x2d, 0x1e, 0x0f },
		.klen	= 16,
		.input	= { 0x93, 0x14, 0x28, 0x87, 0xee, 0x3b, 0xe1, 0x5c },
		.ilen	= 8,
		.result	= { 0xfe, 0xdc, 0xba, 0x98, 0x76, 0x54, 0x32, 0x10 },
		.rlen	= 8,
	}, {
		.key	= { 0xf0, 0xe1, 0xd2, 0xc3, 0xb4, 0xa5, 0x96, 0x87,
			    0x78, 0x69, 0x5a, 0x4b, 0x3c, 0x2d, 0x1e, 0x0f,
			    0x00, 0x11, 0x22, 0x33, 0x44 },
		.klen	= 21,
		.input	= { 0xe6, 0xf5, 0x1e, 0xd7, 0x9b, 0x9d, 0xb2, 0x1f },
		.ilen	= 8,
		.result	= { 0xfe, 0xdc, 0xba, 0x98, 0x76, 0x54, 0x32, 0x10 },
		.rlen	= 8,
	}, { /* Generated with bf488, using OpenSSL, Libgcrypt and Nettle */
		.key	= { 0xf0, 0xe1, 0xd2, 0xc3, 0xb4, 0xa5, 0x96, 0x87,
			    0x78, 0x69, 0x5a, 0x4b, 0x3c, 0x2d, 0x1e, 0x0f,
			    0x00, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77,
			    0x04, 0x68, 0x91, 0x04, 0xc2, 0xfd, 0x3b, 0x2f,
			    0x58, 0x40, 0x23, 0x64, 0x1a, 0xba, 0x61, 0x76,
			    0x1f, 0x1f, 0x1f, 0x1f, 0x0e, 0x0e, 0x0e, 0x0e,
			    0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff },
		.klen	= 56,
		.input	= { 0xc0, 0x45, 0x04, 0x01, 0x2e, 0x4e, 0x1f, 0x53 },
		.ilen	= 8,
		.result	= { 0xfe, 0xdc, 0xba, 0x98, 0x76, 0x54, 0x32, 0x10 },
		.rlen	= 8,
	},
};

static struct cipher_testvec bf_cbc_enc_tv_template[] = {
	{ /* From OpenSSL */
		.key	= { 0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef,
			    0xf0, 0xe1, 0xd2, 0xc3, 0xb4, 0xa5, 0x96, 0x87 },
		.klen	= 16,
		.iv	= { 0xfe, 0xdc, 0xba, 0x98, 0x76, 0x54, 0x32, 0x10 },
		.input	= { 0x37, 0x36, 0x35, 0x34, 0x33, 0x32, 0x31, 0x20,
			    0x4e, 0x6f, 0x77, 0x20, 0x69, 0x73, 0x20, 0x74,
			    0x68, 0x65, 0x20, 0x74, 0x69, 0x6d, 0x65, 0x20,
			    0x66, 0x6f, 0x72, 0x20, 0x00, 0x00, 0x00, 0x00 },
		.ilen	= 32,
		.result	= { 0x6b, 0x77, 0xb4, 0xd6, 0x30, 0x06, 0xde, 0xe6,
			    0x05, 0xb1, 0x56, 0xe2, 0x74, 0x03, 0x97, 0x93,
			    0x58, 0xde, 0xb9, 0xe7, 0x15, 0x46, 0x16, 0xd9,
			    0x59, 0xf1, 0x65, 0x2b, 0xd5, 0xff, 0x92, 0xcc },
		.rlen	= 32,
	},
};

static struct cipher_testvec bf_cbc_dec_tv_template[] = {
	{ /* From OpenSSL */
		.key	= { 0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef,
			    0xf0, 0xe1, 0xd2, 0xc3, 0xb4, 0xa5, 0x96, 0x87 },
		.klen	= 16,
		.iv	= { 0xfe, 0xdc, 0xba, 0x98, 0x76, 0x54, 0x32, 0x10 },
		.input	= { 0x6b, 0x77, 0xb4, 0xd6, 0x30, 0x06, 0xde, 0xe6,
			    0x05, 0xb1, 0x56, 0xe2, 0x74, 0x03, 0x97, 0x93,
			    0x58, 0xde, 0xb9, 0xe7, 0x15, 0x46, 0x16, 0xd9,
			    0x59, 0xf1, 0x65, 0x2b, 0xd5, 0xff, 0x92, 0xcc },
		.ilen	= 32,
		.result	= { 0x37, 0x36, 0x35, 0x34, 0x33, 0x32, 0x31, 0x20,
			    0x4e, 0x6f, 0x77, 0x20, 0x69, 0x73, 0x20, 0x74,
			    0x68, 0x65, 0x20, 0x74, 0x69, 0x6d, 0x65, 0x20,
			    0x66, 0x6f, 0x72, 0x20, 0x00, 0x00, 0x00, 0x00 },
		.rlen	= 32,
	},
};

/*
 * Twofish test vectors.
 */
#define TF_ENC_TEST_VECTORS		3
#define TF_DEC_TEST_VECTORS		3
#define TF_CBC_ENC_TEST_VECTORS		4
#define TF_CBC_DEC_TEST_VECTORS		4

static struct cipher_testvec tf_enc_tv_template[] = {
	{
		.key	= { [0 ... 15] = 0x00 },
		.klen	= 16,
		.input	= { [0 ... 15] = 0x00 },
		.ilen	= 16,
		.result	= { 0x9f, 0x58, 0x9f, 0x5c, 0xf6, 0x12, 0x2c, 0x32,
			    0xb6, 0xbf, 0xec, 0x2f, 0x2a, 0xe8, 0xc3, 0x5a },
		.rlen	= 16,
	}, {
		.key	= { 0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef,
			    0xfe, 0xdc, 0xba, 0x98, 0x76, 0x54, 0x32, 0x10,
			    0x00, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77 },
		.klen	= 24,
		.input	= { [0 ... 15] = 0x00 },
		.ilen	= 16,
		.result	= { 0xcf, 0xd1, 0xd2, 0xe5, 0xa9, 0xbe, 0x9c, 0xdf,
			    0x50, 0x1f, 0x13, 0xb8, 0x92, 0xbd, 0x22, 0x48 },
		.rlen	= 16,
	}, {
		.key	= { 0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef,
			    0xfe, 0xdc, 0xba, 0x98, 0x76, 0x54, 0x32, 0x10,
			    0x00, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77,
			    0x88, 0x99, 0xaa, 0xbb, 0xcc, 0xdd, 0xee, 0xff },
		.klen	= 32,
		.input	= { [0 ... 15] = 0x00 },
		.ilen	= 16,
		.result	= { 0x37, 0x52, 0x7b, 0xe0, 0x05, 0x23, 0x34, 0xb8,
			    0x9f, 0x0c, 0xfc, 0xca, 0xe8, 0x7c, 0xfa, 0x20 },
		.rlen	= 16,
	},
};

static struct cipher_testvec tf_dec_tv_template[] = {
	{
		.key	= { [0 ... 15] = 0x00 },
		.klen	= 16,
		.input	= { 0x9f, 0x58, 0x9f, 0x5c, 0xf6, 0x12, 0x2c, 0x32,
			    0xb6, 0xbf, 0xec, 0x2f, 0x2a, 0xe8, 0xc3, 0x5a },
		.ilen	= 16,
		.result	= { [0 ... 15] = 0x00 },
		.rlen	= 16,
	}, {
		.key	= { 0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef,
			    0xfe, 0xdc, 0xba, 0x98, 0x76, 0x54, 0x32, 0x10,
			    0x00, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77 },
		.klen	= 24,
		.input	= { 0xcf, 0xd1, 0xd2, 0xe5, 0xa9, 0xbe, 0x9c, 0xdf,
			    0x50, 0x1f, 0x13, 0xb8, 0x92, 0xbd, 0x22, 0x48 },
		.ilen	= 16,
		.result	= { [0 ... 15] = 0x00 },
		.rlen	= 16,
	}, {
		.key	= { 0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef,
			    0xfe, 0xdc, 0xba, 0x98, 0x76, 0x54, 0x32, 0x10,
			    0x00, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77,
			    0x88, 0x99, 0xaa, 0xbb, 0xcc, 0xdd, 0xee, 0xff },
		.klen	= 32,
		.input	= { 0x37, 0x52, 0x7b, 0xe0, 0x05, 0x23, 0x34, 0xb8,
			    0x9f, 0x0c, 0xfc, 0xca, 0xe8, 0x7c, 0xfa, 0x20 },
		.ilen	= 16,
		.result	= { [0 ... 15] = 0x00 },
		.rlen	= 16,
	},
};

static struct cipher_testvec tf_cbc_enc_tv_template[] = {
	{ /* Generated with Nettle */
		.key	= { [0 ... 15] = 0x00 },
		.klen	= 16,
		.iv	= { [0 ... 15] = 0x00 },
		.input	= { [0 ... 15] = 0x00 },
		.ilen	= 16,
		.result	= { 0x9f, 0x58, 0x9f, 0x5c, 0xf6, 0x12, 0x2c, 0x32,
			    0xb6, 0xbf, 0xec, 0x2f, 0x2a, 0xe8, 0xc3, 0x5a },
		.rlen	= 16,
	}, {
		.key	= { [0 ... 15] = 0x00 },
		.klen	= 16,
		.iv	= { 0x9f, 0x58, 0x9f, 0x5c, 0xf6, 0x12, 0x2c, 0x32,
			    0xb6, 0xbf, 0xec, 0x2f, 0x2a, 0xe8, 0xc3, 0x5a },
		.input	= { [0 ... 15] = 0x00 },
		.ilen	= 16,
		.result	= { 0xd4, 0x91, 0xdb, 0x16, 0xe7, 0xb1, 0xc3, 0x9e,
			    0x86, 0xcb, 0x08, 0x6b, 0x78, 0x9f, 0x54, 0x19 },
		.rlen	= 16,
	}, {
		.key	= { [0 ... 15] = 0x00 },
		.klen	= 16,
		.iv	= { 0xd4, 0x91, 0xdb, 0x16, 0xe7, 0xb1, 0xc3, 0x9e,
			    0x86, 0xcb, 0x08, 0x6b, 0x78, 0x9f, 0x54, 0x19 },
		.input	= { [0 ... 15] = 0x00 },
		.ilen	= 16,
		.result	= { 0x05, 0xef, 0x8c, 0x61, 0xa8, 0x11, 0x58, 0x26,
			    0x34, 0xba, 0x5c, 0xb7, 0x10, 0x6a, 0xa6, 0x41 },
		.rlen	= 16,
	}, {
		.key	= { [0 ... 15] = 0x00 },
		.klen	= 16,
		.iv	= { [0 ... 15] = 0x00 },
		.input	= { [0 ... 47] = 0x00 },
		.ilen	= 48,
		.result	= { 0x9f, 0x58, 0x9f, 0x5c, 0xf6, 0x12, 0x2c, 0x32,
			    0xb6, 0xbf, 0xec, 0x2f, 0x2a, 0xe8, 0xc3, 0x5a,
			    0xd4, 0x91, 0xdb, 0x16, 0xe7, 0xb1, 0xc3, 0x9e,
			    0x86, 0xcb, 0x08, 0x6b, 0x78, 0x9f, 0x54, 0x19,
			    0x05, 0xef, 0x8c, 0x61, 0xa8, 0x11, 0x58, 0x26,
			    0x34, 0xba, 0x5c, 0xb7, 0x10, 0x6a, 0xa6, 0x41 },
		.rlen	= 48,
	},
};

static struct cipher_testvec tf_cbc_dec_tv_template[] = {
	{ /* Reverse of the first four above */
		.key	= { [0 ... 15] = 0x00 },
		.klen	= 16,
		.iv	= { [0 ... 15] = 0x00 },
		.input	= { 0x9f, 0x58, 0x9f, 0x5c, 0xf6, 0x12, 0x2c, 0x32,
			    0xb6, 0xbf, 0xec, 0x2f, 0x2a, 0xe8, 0xc3, 0x5a },
		.ilen	= 16,
		.result	= { [0 ... 15] = 0x00 },
		.rlen	= 16,
	}, {
		.key	= { [0 ... 15] = 0x00 },
		.klen	= 16,
		.iv	= { 0x9f, 0x58, 0x9f, 0x5c, 0xf6, 0x12, 0x2c, 0x32,
			    0xb6, 0xbf, 0xec, 0x2f, 0x2a, 0xe8, 0xc3, 0x5a },
		.input	= { 0xd4, 0x91, 0xdb, 0x16, 0xe7, 0xb1, 0xc3, 0x9e,
			    0x86, 0xcb, 0x08, 0x6b, 0x78, 0x9f, 0x54, 0x19 },
		.ilen	= 16,
		.result	= { [0 ... 15] = 0x00 },
		.rlen	= 16,
	}, {
		.key	= { [0 ... 15] = 0x00 },
		.klen	= 16,
		.iv	= { 0xd4, 0x91, 0xdb, 0x16, 0xe7, 0xb1, 0xc3, 0x9e,
			    0x86, 0xcb, 0x08, 0x6b, 0x78, 0x9f, 0x54, 0x19 },
		.input	= { 0x05, 0xef, 0x8c, 0x61, 0xa8, 0x11, 0x58, 0x26,
			    0x34, 0xba, 0x5c, 0xb7, 0x10, 0x6a, 0xa6, 0x41 },
		.ilen	= 16,
		.result	= { [0 ... 15] = 0x00 },
		.rlen	= 16,
	}, {
		.key	= { [0 ... 15] = 0x00 },
		.klen	= 16,
		.iv	= { [0 ... 15] = 0x00 },
		.input	= { 0x9f, 0x58, 0x9f, 0x5c, 0xf6, 0x12, 0x2c, 0x32,
			    0xb6, 0xbf, 0xec, 0x2f, 0x2a, 0xe8, 0xc3, 0x5a,
			    0xd4, 0x91, 0xdb, 0x16, 0xe7, 0xb1, 0xc3, 0x9e,
			    0x86, 0xcb, 0x08, 0x6b, 0x78, 0x9f, 0x54, 0x19,
			    0x05, 0xef, 0x8c, 0x61, 0xa8, 0x11, 0x58, 0x26,
			    0x34, 0xba, 0x5c, 0xb7, 0x10, 0x6a, 0xa6, 0x41 },
		.ilen	= 48,
		.result	= { [0 ... 47] = 0x00 },
		.rlen	= 48,
	},
};

/*
 * Serpent test vectors.  These are backwards because Serpent writes
 * octet sequences in right-to-left mode.
 */
#define SERPENT_ENC_TEST_VECTORS	4
#define SERPENT_DEC_TEST_VECTORS	4

#define TNEPRES_ENC_TEST_VECTORS	4
#define TNEPRES_DEC_TEST_VECTORS	4

static struct cipher_testvec serpent_enc_tv_template[] = {
	{
		.input	= { 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
			    0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f },
		.ilen	= 16,
		.result	= { 0x12, 0x07, 0xfc, 0xce, 0x9b, 0xd0, 0xd6, 0x47,
			    0x6a, 0xe9, 0x8f, 0xbe, 0xd1, 0x43, 0xa0, 0xe2 },
		.rlen	= 16,
	}, {
		.key	= { 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
			    0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f },
		.klen	= 16,
		.input	= { 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
			    0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f },
		.ilen	= 16,
		.result	= { 0x4c, 0x7d, 0x8a, 0x32, 0x80, 0x72, 0xa2, 0x2c,
			    0x82, 0x3e, 0x4a, 0x1f, 0x3a, 0xcd, 0xa1, 0x6d },
		.rlen	= 16,
	}, {
		.key	= { 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
			    0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
			    0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
			    0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f },
		.klen	= 32,
		.input	= { 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
			    0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f },
		.ilen	= 16,
		.result	= { 0xde, 0x26, 0x9f, 0xf8, 0x33, 0xe4, 0x32, 0xb8,
			    0x5b, 0x2e, 0x88, 0xd2, 0x70, 0x1c, 0xe7, 0x5c },
		.rlen	= 16,
	}, {
		.key	= { [15] = 0x80 },
		.klen	= 16,
		.input	= { [0 ... 15] = 0x00 },
		.ilen	= 16,
		.result	= { 0xdd, 0xd2, 0x6b, 0x98, 0xa5, 0xff, 0xd8, 0x2c,
			    0x05, 0x34, 0x5a, 0x9d, 0xad, 0xbf, 0xaf, 0x49},
		.rlen	= 16,
	},
};

static struct cipher_testvec tnepres_enc_tv_template[] = {
	{ /* KeySize=128, PT=0, I=1 */
		.input	= { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
		.key    = { 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
		.klen   = 16,
		.ilen	= 16,
		.result	= { 0x49, 0xaf, 0xbf, 0xad, 0x9d, 0x5a, 0x34, 0x05,
			    0x2c, 0xd8, 0xff, 0xa5, 0x98, 0x6b, 0xd2, 0xdd },
		.rlen	= 16,
	}, { /* KeySize=192, PT=0, I=1 */
		.key	= { 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
		.klen	= 24,
		.input	= { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
		.ilen	= 16,
		.result	= { 0xe7, 0x8e, 0x54, 0x02, 0xc7, 0x19, 0x55, 0x68,
			    0xac, 0x36, 0x78, 0xf7, 0xa3, 0xf6, 0x0c, 0x66 },
		.rlen	= 16,
	}, { /* KeySize=256, PT=0, I=1 */
		.key	= { 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
		.klen	= 32,
		.input	= { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
		.ilen	= 16,
		.result	= { 0xab, 0xed, 0x96, 0xe7, 0x66, 0xbf, 0x28, 0xcb,
			    0xc0, 0xeb, 0xd2, 0x1a, 0x82, 0xef, 0x08, 0x19 },
		.rlen	= 16,
	}, { /* KeySize=256, I=257 */
	        .key	= { 0x1f, 0x1e, 0x1d, 0x1c, 0x1b, 0x1a, 0x19, 0x18,
			    0x17, 0x16, 0x15, 0x14, 0x13, 0x12, 0x11, 0x10,
			    0x0f, 0x0e, 0x0d, 0x0c, 0x0b, 0x0a, 0x09, 0x08,
			    0x07, 0x06, 0x05, 0x04, 0x03, 0x02, 0x01, 0x00 },
		.klen	= 32,
		.input	= { 0x0f, 0x0e, 0x0d, 0x0c, 0x0b, 0x0a, 0x09, 0x08,
			    0x07, 0x06, 0x05, 0x04, 0x03, 0x02, 0x01, 0x00 },
		.ilen	= 16,
		.result	= { 0x5c, 0xe7, 0x1c, 0x70, 0xd2, 0x88, 0x2e, 0x5b,
			    0xb8, 0x32, 0xe4, 0x33, 0xf8, 0x9f, 0x26, 0xde },
		.rlen	= 16,
	},
};


static struct cipher_testvec serpent_dec_tv_template[] = {
	{
		.input	= { 0x12, 0x07, 0xfc, 0xce, 0x9b, 0xd0, 0xd6, 0x47,
			    0x6a, 0xe9, 0x8f, 0xbe, 0xd1, 0x43, 0xa0, 0xe2 },
		.ilen	= 16,
		.result	= { 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
			    0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f },
		.rlen	= 16,
	}, {
		.key	= { 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
			    0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f },
		.klen	= 16,
		.input	= { 0x4c, 0x7d, 0x8a, 0x32, 0x80, 0x72, 0xa2, 0x2c,
			    0x82, 0x3e, 0x4a, 0x1f, 0x3a, 0xcd, 0xa1, 0x6d },
		.ilen	= 16,
		.result	= { 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
			    0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f },
		.rlen	= 16,
	}, {
		.key	= { 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
			    0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
			    0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
			    0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f },
		.klen	= 32,
		.input	= { 0xde, 0x26, 0x9f, 0xf8, 0x33, 0xe4, 0x32, 0xb8,
			    0x5b, 0x2e, 0x88, 0xd2, 0x70, 0x1c, 0xe7, 0x5c },
		.ilen	= 16,
		.result	= { 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
			    0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f },
		.rlen	= 16,
	}, {
		.key	= { [15] = 0x80 },
		.klen	= 16,
		.input	= { 0xdd, 0xd2, 0x6b, 0x98, 0xa5, 0xff, 0xd8, 0x2c,
			    0x05, 0x34, 0x5a, 0x9d, 0xad, 0xbf, 0xaf, 0x49},
		.ilen	= 16,
		.result	= { [0 ... 15] = 0x00 },
		.rlen	= 16,
	},
};

static struct cipher_testvec tnepres_dec_tv_template[] = {
	{
		.input	= { 0x41, 0xcc, 0x6b, 0x31, 0x59, 0x31, 0x45, 0x97,
			    0x6d, 0x6f, 0xbb, 0x38, 0x4b, 0x37, 0x21, 0x28 },
		.ilen	= 16,
		.result	= { 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
			    0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f },
		.rlen	= 16,
	}, {
		.key	= { 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
			    0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f },
		.klen	= 16,
		.input	= { 0xea, 0xf4, 0xd7, 0xfc, 0xd8, 0x01, 0x34, 0x47,
			    0x81, 0x45, 0x0b, 0xfa, 0x0c, 0xd6, 0xad, 0x6e },
		.ilen	= 16,
		.result	= { 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
			    0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f },
		.rlen	= 16,
	}, {
		.key	= { 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
			    0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
			    0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
			    0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f },
		.klen	= 32,
		.input	= { 0x64, 0xa9, 0x1a, 0x37, 0xed, 0x9f, 0xe7, 0x49,
			    0xa8, 0x4e, 0x76, 0xd6, 0xf5, 0x0d, 0x78, 0xee },
		.ilen	= 16,
		.result	= { 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
			    0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f },
		.rlen	= 16,
	}, { /* KeySize=128, I=121 */
		.key	= { [15] = 0x80 },
		.klen	= 16,
		.input	= { 0x3d, 0xda, 0xbf, 0xc0, 0x06, 0xda, 0xab, 0x06,
			    0x46, 0x2a, 0xf4, 0xef, 0x81, 0x54, 0x4e, 0x26 },
		.ilen	= 16,
		.result	= { [0 ... 15] = 0x00 },
		.rlen	= 16,
	},
};


/* Cast6 test vectors from RFC 2612 */
#define CAST6_ENC_TEST_VECTORS	3
#define CAST6_DEC_TEST_VECTORS  3

static struct cipher_testvec cast6_enc_tv_template[] = {
	{
		.key	= { 0x23, 0x42, 0xbb, 0x9e, 0xfa, 0x38, 0x54, 0x2c,
			    0x0a, 0xf7, 0x56, 0x47, 0xf2, 0x9f, 0x61, 0x5d },
		.klen	= 16,
		.input	= { [0 ... 15] = 0x00 },
		.ilen	= 16,
		.result	= { 0xc8, 0x42, 0xa0, 0x89, 0x72, 0xb4, 0x3d, 0x20,
			    0x83, 0x6c, 0x91, 0xd1, 0xb7, 0x53, 0x0f, 0x6b },
		.rlen	= 16,
	}, {
		.key	= { 0x23, 0x42, 0xbb, 0x9e, 0xfa, 0x38, 0x54, 0x2c,
			    0xbe, 0xd0, 0xac, 0x83, 0x94, 0x0a, 0xc2, 0x98,
			    0xba, 0xc7, 0x7a, 0x77, 0x17, 0x94, 0x28, 0x63 },
		.klen	= 24,
		.input	= { [0 ... 15] = 0x00 },
		.ilen	= 16,
		.result	= { 0x1b, 0x38, 0x6c, 0x02, 0x10, 0xdc, 0xad, 0xcb,
			    0xdd, 0x0e, 0x41, 0xaa, 0x08, 0xa7, 0xa7, 0xe8 },
		.rlen	= 16,
	}, {
		.key	= { 0x23, 0x42, 0xbb, 0x9e, 0xfa, 0x38, 0x54, 0x2c,
			    0xbe, 0xd0, 0xac, 0x83, 0x94, 0x0a, 0xc2, 0x98,
			    0x8d, 0x7c, 0x47, 0xce, 0x26, 0x49, 0x08, 0x46,
			    0x1c, 0xc1, 0xb5, 0x13, 0x7a, 0xe6, 0xb6, 0x04 },
		.klen	= 32,
		.input	= { [0 ... 15] = 0x00 },
		.ilen	= 16,
		.result	= { 0x4f, 0x6a, 0x20, 0x38, 0x28, 0x68, 0x97, 0xb9,
			    0xc9, 0x87, 0x01, 0x36, 0x55, 0x33, 0x17, 0xfa },
		.rlen	= 16,
	},
};

static struct cipher_testvec cast6_dec_tv_template[] = {
	{
		.key	= { 0x23, 0x42, 0xbb, 0x9e, 0xfa, 0x38, 0x54, 0x2c,
			    0x0a, 0xf7, 0x56, 0x47, 0xf2, 0x9f, 0x61, 0x5d },
		.klen	= 16,
		.input	= { 0xc8, 0x42, 0xa0, 0x89, 0x72, 0xb4, 0x3d, 0x20,
			    0x83, 0x6c, 0x91, 0xd1, 0xb7, 0x53, 0x0f, 0x6b },
		.ilen	= 16,
		.result	= { [0 ... 15] = 0x00 },
		.rlen	= 16,
	}, {
		.key	= { 0x23, 0x42, 0xbb, 0x9e, 0xfa, 0x38, 0x54, 0x2c,
			    0xbe, 0xd0, 0xac, 0x83, 0x94, 0x0a, 0xc2, 0x98,
			    0xba, 0xc7, 0x7a, 0x77, 0x17, 0x94, 0x28, 0x63 },
		.klen	= 24,
		.input	= { 0x1b, 0x38, 0x6c, 0x02, 0x10, 0xdc, 0xad, 0xcb,
			    0xdd, 0x0e, 0x41, 0xaa, 0x08, 0xa7, 0xa7, 0xe8 },
		.ilen	= 16,
		.result	= { [0 ... 15] = 0x00 },
		.rlen	= 16,
	}, {
		.key	= { 0x23, 0x42, 0xbb, 0x9e, 0xfa, 0x38, 0x54, 0x2c,
			    0xbe, 0xd0, 0xac, 0x83, 0x94, 0x0a, 0xc2, 0x98,
			    0x8d, 0x7c, 0x47, 0xce, 0x26, 0x49, 0x08, 0x46,
			    0x1c, 0xc1, 0xb5, 0x13, 0x7a, 0xe6, 0xb6, 0x04 },
		.klen	= 32,
		.input	= { 0x4f, 0x6a, 0x20, 0x38, 0x28, 0x68, 0x97, 0xb9,
			    0xc9, 0x87, 0x01, 0x36, 0x55, 0x33, 0x17, 0xfa },
		.ilen	= 16,
		.result	= { [0 ... 15] = 0x00 },
		.rlen	= 16,
	},
};


/*
 * AES test vectors.
 */
#define AES_ENC_TEST_VECTORS 3
#define AES_DEC_TEST_VECTORS 3
#define AES_CBC_ENC_TEST_VECTORS 4
#define AES_CBC_DEC_TEST_VECTORS 4
#define AES_LRW_ENC_TEST_VECTORS 8
#define AES_LRW_DEC_TEST_VECTORS 8
#define AES_XTS_ENC_TEST_VECTORS 4
#define AES_XTS_DEC_TEST_VECTORS 4
#define AES_CTR_ENC_TEST_VECTORS 7
#define AES_CTR_DEC_TEST_VECTORS 6
#define AES_GCM_ENC_TEST_VECTORS 9
#define AES_GCM_DEC_TEST_VECTORS 8
#define AES_CCM_ENC_TEST_VECTORS 7
#define AES_CCM_DEC_TEST_VECTORS 7

static struct cipher_testvec aes_enc_tv_template[] = {
	{ /* From FIPS-197 */
		.key	= { 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
			    0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f },
		.klen	= 16,
		.input	= { 0x00, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77,
			    0x88, 0x99, 0xaa, 0xbb, 0xcc, 0xdd, 0xee, 0xff },
		.ilen	= 16,
		.result	= { 0x69, 0xc4, 0xe0, 0xd8, 0x6a, 0x7b, 0x04, 0x30,
			    0xd8, 0xcd, 0xb7, 0x80, 0x70, 0xb4, 0xc5, 0x5a },
		.rlen	= 16,
	}, {
		.key	= { 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
			    0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
			    0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17 },
		.klen	= 24,
		.input	= { 0x00, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77,
			    0x88, 0x99, 0xaa, 0xbb, 0xcc, 0xdd, 0xee, 0xff },
		.ilen	= 16,
		.result	= { 0xdd, 0xa9, 0x7c, 0xa4, 0x86, 0x4c, 0xdf, 0xe0,
			    0x6e, 0xaf, 0x70, 0xa0, 0xec, 0x0d, 0x71, 0x91 },
		.rlen	= 16,
	}, {
		.key	= { 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
			    0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
			    0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
			    0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f },
		.klen	= 32,
		.input	= { 0x00, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77,
			    0x88, 0x99, 0xaa, 0xbb, 0xcc, 0xdd, 0xee, 0xff },
		.ilen	= 16,
		.result	= { 0x8e, 0xa2, 0xb7, 0xca, 0x51, 0x67, 0x45, 0xbf,
			    0xea, 0xfc, 0x49, 0x90, 0x4b, 0x49, 0x60, 0x89 },
		.rlen	= 16,
	},
};

static struct cipher_testvec aes_dec_tv_template[] = {
	{ /* From FIPS-197 */
		.key	= { 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
			    0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f },
		.klen	= 16,
		.input	= { 0x69, 0xc4, 0xe0, 0xd8, 0x6a, 0x7b, 0x04, 0x30,
			    0xd8, 0xcd, 0xb7, 0x80, 0x70, 0xb4, 0xc5, 0x5a },
		.ilen	= 16,
		.result	= { 0x00, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77,
			    0x88, 0x99, 0xaa, 0xbb, 0xcc, 0xdd, 0xee, 0xff },
		.rlen	= 16,
	}, {
		.key	= { 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
			    0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
			    0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17 },
		.klen	= 24,
		.input	= { 0xdd, 0xa9, 0x7c, 0xa4, 0x86, 0x4c, 0xdf, 0xe0,
			    0x6e, 0xaf, 0x70, 0xa0, 0xec, 0x0d, 0x71, 0x91 },
		.ilen	= 16,
		.result	= { 0x00, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77,
			    0x88, 0x99, 0xaa, 0xbb, 0xcc, 0xdd, 0xee, 0xff },
		.rlen	= 16,
	}, {
		.key	= { 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
			    0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
			    0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
			    0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f },
		.klen	= 32,
		.input	= { 0x8e, 0xa2, 0xb7, 0xca, 0x51, 0x67, 0x45, 0xbf,
			    0xea, 0xfc, 0x49, 0x90, 0x4b, 0x49, 0x60, 0x89 },
		.ilen	= 16,
		.result	= { 0x00, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77,
			    0x88, 0x99, 0xaa, 0xbb, 0xcc, 0xdd, 0xee, 0xff },
		.rlen	= 16,
	},
};

static struct cipher_testvec aes_cbc_enc_tv_template[] = {
	{ /* From RFC 3602 */
		.key    = { 0x06, 0xa9, 0x21, 0x40, 0x36, 0xb8, 0xa1, 0x5b,
			    0x51, 0x2e, 0x03, 0xd5, 0x34, 0x12, 0x00, 0x06 },
		.klen   = 16,
		.iv	= { 0x3d, 0xaf, 0xba, 0x42, 0x9d, 0x9e, 0xb4, 0x30,
			    0xb4, 0x22, 0xda, 0x80, 0x2c, 0x9f, 0xac, 0x41 },
		.input	= { "Single block msg" },
		.ilen   = 16,
		.result = { 0xe3, 0x53, 0x77, 0x9c, 0x10, 0x79, 0xae, 0xb8,
			    0x27, 0x08, 0x94, 0x2d, 0xbe, 0x77, 0x18, 0x1a },
		.rlen   = 16,
	}, {
		.key    = { 0xc2, 0x86, 0x69, 0x6d, 0x88, 0x7c, 0x9a, 0xa0,
			    0x61, 0x1b, 0xbb, 0x3e, 0x20, 0x25, 0xa4, 0x5a },
		.klen   = 16,
		.iv     = { 0x56, 0x2e, 0x17, 0x99, 0x6d, 0x09, 0x3d, 0x28,
			    0xdd, 0xb3, 0xba, 0x69, 0x5a, 0x2e, 0x6f, 0x58 },
		.input  = { 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
			    0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
			    0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
			    0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f },
		.ilen   = 32,
		.result = { 0xd2, 0x96, 0xcd, 0x94, 0xc2, 0xcc, 0xcf, 0x8a,
			    0x3a, 0x86, 0x30, 0x28, 0xb5, 0xe1, 0xdc, 0x0a,
			    0x75, 0x86, 0x60, 0x2d, 0x25, 0x3c, 0xff, 0xf9,
			    0x1b, 0x82, 0x66, 0xbe, 0xa6, 0xd6, 0x1a, 0xb1 },
		.rlen   = 32,
	}, { /* From NIST SP800-38A */
		.key	= { 0x8e, 0x73, 0xb0, 0xf7, 0xda, 0x0e, 0x64, 0x52,
			    0xc8, 0x10, 0xf3, 0x2b, 0x80, 0x90, 0x79, 0xe5,
			    0x62, 0xf8, 0xea, 0xd2, 0x52, 0x2c, 0x6b, 0x7b },
		.klen	= 24,
		.iv	= { 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
			    0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f },
		.input	= { 0x6b, 0xc1, 0xbe, 0xe2, 0x2e, 0x40, 0x9f, 0x96,
			    0xe9, 0x3d, 0x7e, 0x11, 0x73, 0x93, 0x17, 0x2a,
			    0xae, 0x2d, 0x8a, 0x57, 0x1e, 0x03, 0xac, 0x9c,
			    0x9e, 0xb7, 0x6f, 0xac, 0x45, 0xaf, 0x8e, 0x51,
			    0x30, 0xc8, 0x1c, 0x46, 0xa3, 0x5c, 0xe4, 0x11,
			    0xe5, 0xfb, 0xc1, 0x19, 0x1a, 0x0a, 0x52, 0xef,
			    0xf6, 0x9f, 0x24, 0x45, 0xdf, 0x4f, 0x9b, 0x17,
			    0xad, 0x2b, 0x41, 0x7b, 0xe6, 0x6c, 0x37, 0x10 },
		.ilen	= 64,
		.result	= { 0x4f, 0x02, 0x1d, 0xb2, 0x43, 0xbc, 0x63, 0x3d,
			    0x71, 0x78, 0x18, 0x3a, 0x9f, 0xa0, 0x71, 0xe8,
			    0xb4, 0xd9, 0xad, 0xa9, 0xad, 0x7d, 0xed, 0xf4,
			    0xe5, 0xe7, 0x38, 0x76, 0x3f, 0x69, 0x14, 0x5a,
			    0x57, 0x1b, 0x24, 0x20, 0x12, 0xfb, 0x7a, 0xe0,
			    0x7f, 0xa9, 0xba, 0xac, 0x3d, 0xf1, 0x02, 0xe0,
			    0x08, 0xb0, 0xe2, 0x79, 0x88, 0x59, 0x88, 0x81,
			    0xd9, 0x20, 0xa9, 0xe6, 0x4f, 0x56, 0x15, 0xcd },
		.rlen	= 64,
	}, {
		.key	= { 0x60, 0x3d, 0xeb, 0x10, 0x15, 0xca, 0x71, 0xbe,
			    0x2b, 0x73, 0xae, 0xf0, 0x85, 0x7d, 0x77, 0x81,
			    0x1f, 0x35, 0x2c, 0x07, 0x3b, 0x61, 0x08, 0xd7,
			    0x2d, 0x98, 0x10, 0xa3, 0x09, 0x14, 0xdf, 0xf4 },
		.klen	= 32,
		.iv	= { 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
			    0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f },
		.input	= { 0x6b, 0xc1, 0xbe, 0xe2, 0x2e, 0x40, 0x9f, 0x96,
			    0xe9, 0x3d, 0x7e, 0x11, 0x73, 0x93, 0x17, 0x2a,
			    0xae, 0x2d, 0x8a, 0x57, 0x1e, 0x03, 0xac, 0x9c,
			    0x9e, 0xb7, 0x6f, 0xac, 0x45, 0xaf, 0x8e, 0x51,
			    0x30, 0xc8, 0x1c, 0x46, 0xa3, 0x5c, 0xe4, 0x11,
			    0xe5, 0xfb, 0xc1, 0x19, 0x1a, 0x0a, 0x52, 0xef,
			    0xf6, 0x9f, 0x24, 0x45, 0xdf, 0x4f, 0x9b, 0x17,
			    0xad, 0x2b, 0x41, 0x7b, 0xe6, 0x6c, 0x37, 0x10 },
		.ilen	= 64,
		.result	= { 0xf5, 0x8c, 0x4c, 0x04, 0xd6, 0xe5, 0xf1, 0xba,
			    0x77, 0x9e, 0xab, 0xfb, 0x5f, 0x7b, 0xfb, 0xd6,
			    0x9c, 0xfc, 0x4e, 0x96, 0x7e, 0xdb, 0x80, 0x8d,
			    0x67, 0x9f, 0x77, 0x7b, 0xc6, 0x70, 0x2c, 0x7d,
			    0x39, 0xf2, 0x33, 0x69, 0xa9, 0xd9, 0xba, 0xcf,
			    0xa5, 0x30, 0xe2, 0x63, 0x04, 0x23, 0x14, 0x61,
			    0xb2, 0xeb, 0x05, 0xe2, 0xc3, 0x9b, 0xe9, 0xfc,
			    0xda, 0x6c, 0x19, 0x07, 0x8c, 0x6a, 0x9d, 0x1b },
		.rlen	= 64,
	},
};

static struct cipher_testvec aes_cbc_dec_tv_template[] = {
	{ /* From RFC 3602 */
		.key    = { 0x06, 0xa9, 0x21, 0x40, 0x36, 0xb8, 0xa1, 0x5b,
			    0x51, 0x2e, 0x03, 0xd5, 0x34, 0x12, 0x00, 0x06 },
		.klen   = 16,
		.iv     = { 0x3d, 0xaf, 0xba, 0x42, 0x9d, 0x9e, 0xb4, 0x30,
			    0xb4, 0x22, 0xda, 0x80, 0x2c, 0x9f, 0xac, 0x41 },
		.input  = { 0xe3, 0x53, 0x77, 0x9c, 0x10, 0x79, 0xae, 0xb8,
			    0x27, 0x08, 0x94, 0x2d, 0xbe, 0x77, 0x18, 0x1a },
		.ilen   = 16,
		.result = { "Single block msg" },
		.rlen   = 16,
	}, {
		.key    = { 0xc2, 0x86, 0x69, 0x6d, 0x88, 0x7c, 0x9a, 0xa0,
			    0x61, 0x1b, 0xbb, 0x3e, 0x20, 0x25, 0xa4, 0x5a },
		.klen   = 16,
		.iv     = { 0x56, 0x2e, 0x17, 0x99, 0x6d, 0x09, 0x3d, 0x28,
			    0xdd, 0xb3, 0xba, 0x69, 0x5a, 0x2e, 0x6f, 0x58 },
		.input  = { 0xd2, 0x96, 0xcd, 0x94, 0xc2, 0xcc, 0xcf, 0x8a,
			    0x3a, 0x86, 0x30, 0x28, 0xb5, 0xe1, 0xdc, 0x0a,
			    0x75, 0x86, 0x60, 0x2d, 0x25, 0x3c, 0xff, 0xf9,
			    0x1b, 0x82, 0x66, 0xbe, 0xa6, 0xd6, 0x1a, 0xb1 },
		.ilen   = 32,
		.result = { 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
			    0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
			    0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
			    0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f },
		.rlen   = 32,
	}, { /* From NIST SP800-38A */
		.key	= { 0x8e, 0x73, 0xb0, 0xf7, 0xda, 0x0e, 0x64, 0x52,
			    0xc8, 0x10, 0xf3, 0x2b, 0x80, 0x90, 0x79, 0xe5,
			    0x62, 0xf8, 0xea, 0xd2, 0x52, 0x2c, 0x6b, 0x7b },
		.klen	= 24,
		.iv	= { 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
			    0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f },
		.input	= { 0x4f, 0x02, 0x1d, 0xb2, 0x43, 0xbc, 0x63, 0x3d,
			    0x71, 0x78, 0x18, 0x3a, 0x9f, 0xa0, 0x71, 0xe8,
			    0xb4, 0xd9, 0xad, 0xa9, 0xad, 0x7d, 0xed, 0xf4,
			    0xe5, 0xe7, 0x38, 0x76, 0x3f, 0x69, 0x14, 0x5a,
			    0x57, 0x1b, 0x24, 0x20, 0x12, 0xfb, 0x7a, 0xe0,
			    0x7f, 0xa9, 0xba, 0xac, 0x3d, 0xf1, 0x02, 0xe0,
			    0x08, 0xb0, 0xe2, 0x79, 0x88, 0x59, 0x88, 0x81,
			    0xd9, 0x20, 0xa9, 0xe6, 0x4f, 0x56, 0x15, 0xcd },
		.ilen	= 64,
		.result	= { 0x6b, 0xc1, 0xbe, 0xe2, 0x2e, 0x40, 0x9f, 0x96,
			    0xe9, 0x3d, 0x7e, 0x11, 0x73, 0x93, 0x17, 0x2a,
			    0xae, 0x2d, 0x8a, 0x57, 0x1e, 0x03, 0xac, 0x9c,
			    0x9e, 0xb7, 0x6f, 0xac, 0x45, 0xaf, 0x8e, 0x51,
			    0x30, 0xc8, 0x1c, 0x46, 0xa3, 0x5c, 0xe4, 0x11,
			    0xe5, 0xfb, 0xc1, 0x19, 0x1a, 0x0a, 0x52, 0xef,
			    0xf6, 0x9f, 0x24, 0x45, 0xdf, 0x4f, 0x9b, 0x17,
			    0xad, 0x2b, 0x41, 0x7b, 0xe6, 0x6c, 0x37, 0x10 },
		.rlen	= 64,
	}, {
		.key	= { 0x60, 0x3d, 0xeb, 0x10, 0x15, 0xca, 0x71, 0xbe,
			    0x2b, 0x73, 0xae, 0xf0, 0x85, 0x7d, 0x77, 0x81,
			    0x1f, 0x35, 0x2c, 0x07, 0x3b, 0x61, 0x08, 0xd7,
			    0x2d, 0x98, 0x10, 0xa3, 0x09, 0x14, 0xdf, 0xf4 },
		.klen	= 32,
		.iv	= { 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
			    0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f },
		.input	= { 0xf5, 0x8c, 0x4c, 0x04, 0xd6, 0xe5, 0xf1, 0xba,
			    0x77, 0x9e, 0xab, 0xfb, 0x5f, 0x7b, 0xfb, 0xd6,
			    0x9c, 0xfc, 0x4e, 0x96, 0x7e, 0xdb, 0x80, 0x8d,
			    0x67, 0x9f, 0x77, 0x7b, 0xc6, 0x70, 0x2c, 0x7d,
			    0x39, 0xf2, 0x33, 0x69, 0xa9, 0xd9, 0xba, 0xcf,
			    0xa5, 0x30, 0xe2, 0x63, 0x04, 0x23, 0x14, 0x61,
			    0xb2, 0xeb, 0x05, 0xe2, 0xc3, 0x9b, 0xe9, 0xfc,
			    0xda, 0x6c, 0x19, 0x07, 0x8c, 0x6a, 0x9d, 0x1b },
		.ilen	= 64,
		.result	= { 0x6b, 0xc1, 0xbe, 0xe2, 0x2e, 0x40, 0x9f, 0x96,
			    0xe9, 0x3d, 0x7e, 0x11, 0x73, 0x93, 0x17, 0x2a,
			    0xae, 0x2d, 0x8a, 0x57, 0x1e, 0x03, 0xac, 0x9c,
			    0x9e, 0xb7, 0x6f, 0xac, 0x45, 0xaf, 0x8e, 0x51,
			    0x30, 0xc8, 0x1c, 0x46, 0xa3, 0x5c, 0xe4, 0x11,
			    0xe5, 0xfb, 0xc1, 0x19, 0x1a, 0x0a, 0x52, 0xef,
			    0xf6, 0x9f, 0x24, 0x45, 0xdf, 0x4f, 0x9b, 0x17,
			    0xad, 0x2b, 0x41, 0x7b, 0xe6, 0x6c, 0x37, 0x10 },
		.rlen	= 64,
	},
};

static struct cipher_testvec aes_lrw_enc_tv_template[] = {
	/* from http://grouper.ieee.org/groups/1619/email/pdf00017.pdf */
	{ /* LRW-32-AES 1 */
		.key    = { 0x45, 0x62, 0xac, 0x25, 0xf8, 0x28, 0x17, 0x6d,
			    0x4c, 0x26, 0x84, 0x14, 0xb5, 0x68, 0x01, 0x85,
			    0x25, 0x8e, 0x2a, 0x05, 0xe7, 0x3e, 0x9d, 0x03,
			    0xee, 0x5a, 0x83, 0x0c, 0xcc, 0x09, 0x4c, 0x87 },
		.klen   = 32,
		.iv     = { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01 },
		.input  = { 0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
			    0x38, 0x39, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46 },
		.ilen   = 16,
		.result = { 0xf1, 0xb2, 0x73, 0xcd, 0x65, 0xa3, 0xdf, 0x5f,
			    0xe9, 0x5d, 0x48, 0x92, 0x54, 0x63, 0x4e, 0xb8 },
		.rlen   = 16,
	}, { /* LRW-32-AES 2 */
		.key    = { 0x59, 0x70, 0x47, 0x14, 0xf5, 0x57, 0x47, 0x8c,
		  	    0xd7, 0x79, 0xe8, 0x0f, 0x54, 0x88, 0x79, 0x44,
			    0x0d, 0x48, 0xf0, 0xb7, 0xb1, 0x5a, 0x53, 0xea,
			    0x1c, 0xaa, 0x6b, 0x29, 0xc2, 0xca, 0xfb, 0xaf
		},
		.klen   = 32,
		.iv     = { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02 },
		.input  = { 0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
			    0x38, 0x39, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46 },
		.ilen   = 16,
		.result = { 0x00, 0xc8, 0x2b, 0xae, 0x95, 0xbb, 0xcd, 0xe5,
			    0x27, 0x4f, 0x07, 0x69, 0xb2, 0x60, 0xe1, 0x36 },
		.rlen   = 16,
	}, { /* LRW-32-AES 3 */
		.key    = { 0xd8, 0x2a, 0x91, 0x34, 0xb2, 0x6a, 0x56, 0x50,
		  	    0x30, 0xfe, 0x69, 0xe2, 0x37, 0x7f, 0x98, 0x47,
			    0xcd, 0xf9, 0x0b, 0x16, 0x0c, 0x64, 0x8f, 0xb6,
			    0xb0, 0x0d, 0x0d, 0x1b, 0xae, 0x85, 0x87, 0x1f },
		.klen   = 32,
		.iv     = { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			    0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00 },
		.input  = { 0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
			    0x38, 0x39, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46 },
		.ilen   = 16,
		.result = { 0x76, 0x32, 0x21, 0x83, 0xed, 0x8f, 0xf1, 0x82,
			    0xf9, 0x59, 0x62, 0x03, 0x69, 0x0e, 0x5e, 0x01 },
		.rlen   = 16,
	}, { /* LRW-32-AES 4 */
		.key    = { 0x0f, 0x6a, 0xef, 0xf8, 0xd3, 0xd2, 0xbb, 0x15,
		  	    0x25, 0x83, 0xf7, 0x3c, 0x1f, 0x01, 0x28, 0x74,
			    0xca, 0xc6, 0xbc, 0x35, 0x4d, 0x4a, 0x65, 0x54,
			    0x90, 0xae, 0x61, 0xcf, 0x7b, 0xae, 0xbd, 0xcc,
			    0xad, 0xe4, 0x94, 0xc5, 0x4a, 0x29, 0xae, 0x70 },
		.klen   = 40,
		.iv     = { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01 },
		.input  = { 0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
			    0x38, 0x39, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46 },
		.ilen   = 16,
		.result = { 0x9c, 0x0f, 0x15, 0x2f, 0x55, 0xa2, 0xd8, 0xf0,
			    0xd6, 0x7b, 0x8f, 0x9e, 0x28, 0x22, 0xbc, 0x41 },
		.rlen   = 16,
	}, { /* LRW-32-AES 5 */
		.key    = { 0x8a, 0xd4, 0xee, 0x10, 0x2f, 0xbd, 0x81, 0xff,
		  	    0xf8, 0x86, 0xce, 0xac, 0x93, 0xc5, 0xad, 0xc6,
			    0xa0, 0x19, 0x07, 0xc0, 0x9d, 0xf7, 0xbb, 0xdd,
			    0x52, 0x13, 0xb2, 0xb7, 0xf0, 0xff, 0x11, 0xd8,
			    0xd6, 0x08, 0xd0, 0xcd, 0x2e, 0xb1, 0x17, 0x6f },
		.klen   = 40,
		.iv     = { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			    0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00 },
		.input  = { 0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
			    0x38, 0x39, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46 },
		.ilen   = 16,
		.result = { 0xd4, 0x27, 0x6a, 0x7f, 0x14, 0x91, 0x3d, 0x65,
			    0xc8, 0x60, 0x48, 0x02, 0x87, 0xe3, 0x34, 0x06 },
		.rlen   = 16,
	}, { /* LRW-32-AES 6 */
		.key    = { 0xf8, 0xd4, 0x76, 0xff, 0xd6, 0x46, 0xee, 0x6c,
		  	    0x23, 0x84, 0xcb, 0x1c, 0x77, 0xd6, 0x19, 0x5d,
			    0xfe, 0xf1, 0xa9, 0xf3, 0x7b, 0xbc, 0x8d, 0x21,
			    0xa7, 0x9c, 0x21, 0xf8, 0xcb, 0x90, 0x02, 0x89,
			    0xa8, 0x45, 0x34, 0x8e, 0xc8, 0xc5, 0xb5, 0xf1,
			    0x26, 0xf5, 0x0e, 0x76, 0xfe, 0xfd, 0x1b, 0x1e },
		.klen   = 48,
		.iv     = { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01 },
		.input  = { 0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
			    0x38, 0x39, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46 },
		.ilen   = 16,
		.result = { 0xbd, 0x06, 0xb8, 0xe1, 0xdb, 0x98, 0x89, 0x9e,
			    0xc4, 0x98, 0xe4, 0x91, 0xcf, 0x1c, 0x70, 0x2b },
		.rlen   = 16,
	}, { /* LRW-32-AES 7 */
		.key    = { 0xfb, 0x76, 0x15, 0xb2, 0x3d, 0x80, 0x89, 0x1d,
		  	    0xd4, 0x70, 0x98, 0x0b, 0xc7, 0x95, 0x84, 0xc8,
			    0xb2, 0xfb, 0x64, 0xce, 0x60, 0x97, 0x87, 0x8d,
			    0x17, 0xfc, 0xe4, 0x5a, 0x49, 0xe8, 0x30, 0xb7,
			    0x6e, 0x78, 0x17, 0xe7, 0x2d, 0x5e, 0x12, 0xd4,
			    0x60, 0x64, 0x04, 0x7a, 0xf1, 0x2f, 0x9e, 0x0c },
		.klen   = 48,
		.iv     = { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			    0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00 },
		.input  = { 0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
			    0x38, 0x39, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46 },
		.ilen   = 16,
		.result = { 0x5b, 0x90, 0x8e, 0xc1, 0xab, 0xdd, 0x67, 0x5f,
			    0x3d, 0x69, 0x8a, 0x95, 0x53, 0xc8, 0x9c, 0xe5 },
		.rlen   = 16,
	}, {
/* http://www.mail-archive.com/stds-p1619@listserv.ieee.org/msg00173.html */
		.key    = { 0xf8, 0xd4, 0x76, 0xff, 0xd6, 0x46, 0xee, 0x6c,
			    0x23, 0x84, 0xcb, 0x1c, 0x77, 0xd6, 0x19, 0x5d,
			    0xfe, 0xf1, 0xa9, 0xf3, 0x7b, 0xbc, 0x8d, 0x21,
			    0xa7, 0x9c, 0x21, 0xf8, 0xcb, 0x90, 0x02, 0x89,
			    0xa8, 0x45, 0x34, 0x8e, 0xc8, 0xc5, 0xb5, 0xf1,
			    0x26, 0xf5, 0x0e, 0x76, 0xfe, 0xfd, 0x1b, 0x1e },
		.klen   = 48,
		.iv     = { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01 },
		.input  = { 0x05, 0x11, 0xb7, 0x18, 0xab, 0xc6, 0x2d, 0xac,
			    0x70, 0x5d, 0xf6, 0x22, 0x94, 0xcd, 0xe5, 0x6c,
			    0x17, 0x6b, 0xf6, 0x1c, 0xf0, 0xf3, 0x6e, 0xf8,
			    0x50, 0x38, 0x1f, 0x71, 0x49, 0xb6, 0x57, 0xd6,
			    0x8f, 0xcb, 0x8d, 0x6b, 0xe3, 0xa6, 0x29, 0x90,
			    0xfe, 0x2a, 0x62, 0x82, 0xae, 0x6d, 0x8b, 0xf6,
			    0xad, 0x1e, 0x9e, 0x20, 0x5f, 0x38, 0xbe, 0x04,
			    0xda, 0x10, 0x8e, 0xed, 0xa2, 0xa4, 0x87, 0xab,
			    0xda, 0x6b, 0xb4, 0x0c, 0x75, 0xba, 0xd3, 0x7c,
			    0xc9, 0xac, 0x42, 0x31, 0x95, 0x7c, 0xc9, 0x04,
			    0xeb, 0xd5, 0x6e, 0x32, 0x69, 0x8a, 0xdb, 0xa6,
			    0x15, 0xd7, 0x3f, 0x4f, 0x2f, 0x66, 0x69, 0x03,
			    0x9c, 0x1f, 0x54, 0x0f, 0xde, 0x1f, 0xf3, 0x65,
			    0x4c, 0x96, 0x12, 0xed, 0x7c, 0x92, 0x03, 0x01,
			    0x6f, 0xbc, 0x35, 0x93, 0xac, 0xf1, 0x27, 0xf1,
			    0xb4, 0x96, 0x82, 0x5a, 0x5f, 0xb0, 0xa0, 0x50,
			    0x89, 0xa4, 0x8e, 0x66, 0x44, 0x85, 0xcc, 0xfd,
			    0x33, 0x14, 0x70, 0xe3, 0x96, 0xb2, 0xc3, 0xd3,
			    0xbb, 0x54, 0x5a, 0x1a, 0xf9, 0x74, 0xa2, 0xc5,
			    0x2d, 0x64, 0x75, 0xdd, 0xb4, 0x54, 0xe6, 0x74,
			    0x8c, 0xd3, 0x9d, 0x9e, 0x86, 0xab, 0x51, 0x53,
			    0xb7, 0x93, 0x3e, 0x6f, 0xd0, 0x4e, 0x2c, 0x40,
			    0xf6, 0xa8, 0x2e, 0x3e, 0x9d, 0xf4, 0x66, 0xa5,
			    0x76, 0x12, 0x73, 0x44, 0x1a, 0x56, 0xd7, 0x72,
			    0x88, 0xcd, 0x21, 0x8c, 0x4c, 0x0f, 0xfe, 0xda,
			    0x95, 0xe0, 0x3a, 0xa6, 0xa5, 0x84, 0x46, 0xcd,
			    0xd5, 0x3e, 0x9d, 0x3a, 0xe2, 0x67, 0xe6, 0x60,
			    0x1a, 0xe2, 0x70, 0x85, 0x58, 0xc2, 0x1b, 0x09,
			    0xe1, 0xd7, 0x2c, 0xca, 0xad, 0xa8, 0x8f, 0xf9,
			    0xac, 0xb3, 0x0e, 0xdb, 0xca, 0x2e, 0xe2, 0xb8,
			    0x51, 0x71, 0xd9, 0x3c, 0x6c, 0xf1, 0x56, 0xf8,
			    0xea, 0x9c, 0xf1, 0xfb, 0x0c, 0xe6, 0xb7, 0x10,
			    0x1c, 0xf8, 0xa9, 0x7c, 0xe8, 0x53, 0x35, 0xc1,
			    0x90, 0x3e, 0x76, 0x4a, 0x74, 0xa4, 0x21, 0x2c,
			    0xf6, 0x2c, 0x4e, 0x0f, 0x94, 0x3a, 0x88, 0x2e,
			    0x41, 0x09, 0x6a, 0x33, 0x7d, 0xf6, 0xdd, 0x3f,
			    0x8d, 0x23, 0x31, 0x74, 0x84, 0xeb, 0x88, 0x6e,
			    0xcc, 0xb9, 0xbc, 0x22, 0x83, 0x19, 0x07, 0x22,
			    0xa5, 0x2d, 0xdf, 0xa5, 0xf3, 0x80, 0x85, 0x78,
			    0x84, 0x39, 0x6a, 0x6d, 0x6a, 0x99, 0x4f, 0xa5,
			    0x15, 0xfe, 0x46, 0xb0, 0xe4, 0x6c, 0xa5, 0x41,
			    0x3c, 0xce, 0x8f, 0x42, 0x60, 0x71, 0xa7, 0x75,
			    0x08, 0x40, 0x65, 0x8a, 0x82, 0xbf, 0xf5, 0x43,
			    0x71, 0x96, 0xa9, 0x4d, 0x44, 0x8a, 0x20, 0xbe,
			    0xfa, 0x4d, 0xbb, 0xc0, 0x7d, 0x31, 0x96, 0x65,
			    0xe7, 0x75, 0xe5, 0x3e, 0xfd, 0x92, 0x3b, 0xc9,
			    0x55, 0xbb, 0x16, 0x7e, 0xf7, 0xc2, 0x8c, 0xa4,
			    0x40, 0x1d, 0xe5, 0xef, 0x0e, 0xdf, 0xe4, 0x9a,
			    0x62, 0x73, 0x65, 0xfd, 0x46, 0x63, 0x25, 0x3d,
			    0x2b, 0xaf, 0xe5, 0x64, 0xfe, 0xa5, 0x5c, 0xcf,
			    0x24, 0xf3, 0xb4, 0xac, 0x64, 0xba, 0xdf, 0x4b,
			    0xc6, 0x96, 0x7d, 0x81, 0x2d, 0x8d, 0x97, 0xf7,
			    0xc5, 0x68, 0x77, 0x84, 0x32, 0x2b, 0xcc, 0x85,
			    0x74, 0x96, 0xf0, 0x12, 0x77, 0x61, 0xb9, 0xeb,
			    0x71, 0xaa, 0x82, 0xcb, 0x1c, 0xdb, 0x89, 0xc8,
			    0xc6, 0xb5, 0xe3, 0x5c, 0x7d, 0x39, 0x07, 0x24,
			    0xda, 0x39, 0x87, 0x45, 0xc0, 0x2b, 0xbb, 0x01,
			    0xac, 0xbc, 0x2a, 0x5c, 0x7f, 0xfc, 0xe8, 0xce,
			    0x6d, 0x9c, 0x6f, 0xed, 0xd3, 0xc1, 0xa1, 0xd6,
			    0xc5, 0x55, 0xa9, 0x66, 0x2f, 0xe1, 0xc8, 0x32,
			    0xa6, 0x5d, 0xa4, 0x3a, 0x98, 0x73, 0xe8, 0x45,
			    0xa4, 0xc7, 0xa8, 0xb4, 0xf6, 0x13, 0x03, 0xf6,
			    0xe9, 0x2e, 0xc4, 0x29, 0x0f, 0x84, 0xdb, 0xc4,
			    0x21, 0xc4, 0xc2, 0x75, 0x67, 0x89, 0x37, 0x0a },
		.ilen   = 512,
		.result = { 0x1a, 0x1d, 0xa9, 0x30, 0xad, 0xf9, 0x2f, 0x9b,
			    0xb6, 0x1d, 0xae, 0xef, 0xf0, 0x2f, 0xf8, 0x5a,
			    0x39, 0x3c, 0xbf, 0x2a, 0xb2, 0x45, 0xb2, 0x23,
			    0x1b, 0x63, 0x3c, 0xcf, 0xaa, 0xbe, 0xcf, 0x4e,
			    0xfa, 0xe8, 0x29, 0xc2, 0x20, 0x68, 0x2b, 0x3c,
			    0x2e, 0x8b, 0xf7, 0x6e, 0x25, 0xbd, 0xe3, 0x3d,
			    0x66, 0x27, 0xd6, 0xaf, 0xd6, 0x64, 0x3e, 0xe3,
			    0xe8, 0x58, 0x46, 0x97, 0x39, 0x51, 0x07, 0xde,
			    0xcb, 0x37, 0xbc, 0xa9, 0xc0, 0x5f, 0x75, 0xc3,
			    0x0e, 0x84, 0x23, 0x1d, 0x16, 0xd4, 0x1c, 0x59,
			    0x9c, 0x1a, 0x02, 0x55, 0xab, 0x3a, 0x97, 0x1d,
			    0xdf, 0xdd, 0xc7, 0x06, 0x51, 0xd7, 0x70, 0xae,
			    0x23, 0xc6, 0x8c, 0xf5, 0x1e, 0xa0, 0xe5, 0x82,
			    0xb8, 0xb2, 0xbf, 0x04, 0xa0, 0x32, 0x8e, 0x68,
			    0xeb, 0xaf, 0x6e, 0x2d, 0x94, 0x22, 0x2f, 0xce,
			    0x4c, 0xb5, 0x59, 0xe2, 0xa2, 0x2f, 0xa0, 0x98,
			    0x1a, 0x97, 0xc6, 0xd4, 0xb5, 0x00, 0x59, 0xf2,
			    0x84, 0x14, 0x72, 0xb1, 0x9a, 0x6e, 0xa3, 0x7f,
			    0xea, 0x20, 0xe7, 0xcb, 0x65, 0x77, 0x3a, 0xdf,
			    0xc8, 0x97, 0x67, 0x15, 0xc2, 0x2a, 0x27, 0xcc,
			    0x18, 0x55, 0xa1, 0x24, 0x0b, 0x24, 0x24, 0xaf,
			    0x5b, 0xec, 0x68, 0xb8, 0xc8, 0xf5, 0xba, 0x63,
			    0xff, 0xed, 0x89, 0xce, 0xd5, 0x3d, 0x88, 0xf3,
			    0x25, 0xef, 0x05, 0x7c, 0x3a, 0xef, 0xeb, 0xd8,
			    0x7a, 0x32, 0x0d, 0xd1, 0x1e, 0x58, 0x59, 0x99,
			    0x90, 0x25, 0xb5, 0x26, 0xb0, 0xe3, 0x2b, 0x6c,
			    0x4c, 0xa9, 0x8b, 0x84, 0x4f, 0x5e, 0x01, 0x50,
			    0x41, 0x30, 0x58, 0xc5, 0x62, 0x74, 0x52, 0x1d,
			    0x45, 0x24, 0x6a, 0x42, 0x64, 0x4f, 0x97, 0x1c,
			    0xa8, 0x66, 0xb5, 0x6d, 0x79, 0xd4, 0x0d, 0x48,
			    0xc5, 0x5f, 0xf3, 0x90, 0x32, 0xdd, 0xdd, 0xe1,
			    0xe4, 0xa9, 0x9f, 0xfc, 0xc3, 0x52, 0x5a, 0x46,
			    0xe4, 0x81, 0x84, 0x95, 0x36, 0x59, 0x7a, 0x6b,
			    0xaa, 0xb3, 0x60, 0xad, 0xce, 0x9f, 0x9f, 0x28,
			    0xe0, 0x01, 0x75, 0x22, 0xc4, 0x4e, 0xa9, 0x62,
			    0x5c, 0x62, 0x0d, 0x00, 0xcb, 0x13, 0xe8, 0x43,
			    0x72, 0xd4, 0x2d, 0x53, 0x46, 0xb5, 0xd1, 0x16,
			    0x22, 0x18, 0xdf, 0x34, 0x33, 0xf5, 0xd6, 0x1c,
			    0xb8, 0x79, 0x78, 0x97, 0x94, 0xff, 0x72, 0x13,
			    0x4c, 0x27, 0xfc, 0xcb, 0xbf, 0x01, 0x53, 0xa6,
			    0xb4, 0x50, 0x6e, 0xde, 0xdf, 0xb5, 0x43, 0xa4,
			    0x59, 0xdf, 0x52, 0xf9, 0x7c, 0xe0, 0x11, 0x6f,
			    0x2d, 0x14, 0x8e, 0x24, 0x61, 0x2c, 0xe1, 0x17,
			    0xcc, 0xce, 0x51, 0x0c, 0x19, 0x8a, 0x82, 0x30,
			    0x94, 0xd5, 0x3d, 0x6a, 0x53, 0x06, 0x5e, 0xbd,
			    0xb7, 0xeb, 0xfa, 0xfd, 0x27, 0x51, 0xde, 0x85,
			    0x1e, 0x86, 0x53, 0x11, 0x53, 0x94, 0x00, 0xee,
			    0x2b, 0x8c, 0x08, 0x2a, 0xbf, 0xdd, 0xae, 0x11,
			    0xcb, 0x1e, 0xa2, 0x07, 0x9a, 0x80, 0xcf, 0x62,
			    0x9b, 0x09, 0xdc, 0x95, 0x3c, 0x96, 0x8e, 0xb1,
			    0x09, 0xbd, 0xe4, 0xeb, 0xdb, 0xca, 0x70, 0x7a,
			    0x9e, 0xfa, 0x31, 0x18, 0x45, 0x3c, 0x21, 0x33,
			    0xb0, 0xb3, 0x2b, 0xea, 0xf3, 0x71, 0x2d, 0xe1,
			    0x03, 0xad, 0x1b, 0x48, 0xd4, 0x67, 0x27, 0xf0,
			    0x62, 0xe4, 0x3d, 0xfb, 0x9b, 0x08, 0x76, 0xe7,
			    0xdd, 0x2b, 0x01, 0x39, 0x04, 0x5a, 0x58, 0x7a,
			    0xf7, 0x11, 0x90, 0xec, 0xbd, 0x51, 0x5c, 0x32,
			    0x6b, 0xd7, 0x35, 0x39, 0x02, 0x6b, 0xf2, 0xa6,
			    0xd0, 0x0d, 0x07, 0xe1, 0x06, 0xc4, 0x5b, 0x7d,
			    0xe4, 0x6a, 0xd7, 0xee, 0x15, 0x1f, 0x83, 0xb4,
			    0xa3, 0xa7, 0x5e, 0xc3, 0x90, 0xb7, 0xef, 0xd3,
			    0xb7, 0x4f, 0xf8, 0x92, 0x4c, 0xb7, 0x3c, 0x29,
			    0xcd, 0x7e, 0x2b, 0x5d, 0x43, 0xea, 0x42, 0xe7,
			    0x74, 0x3f, 0x7d, 0x58, 0x88, 0x75, 0xde, 0x3e },
		.rlen   = 512,
	}
};

static struct cipher_testvec aes_lrw_dec_tv_template[] = {
	/* from http://grouper.ieee.org/groups/1619/email/pdf00017.pdf */
	/* same as enc vectors with input and result reversed */
	{ /* LRW-32-AES 1 */
		.key    = { 0x45, 0x62, 0xac, 0x25, 0xf8, 0x28, 0x17, 0x6d,
			    0x4c, 0x26, 0x84, 0x14, 0xb5, 0x68, 0x01, 0x85,
			    0x25, 0x8e, 0x2a, 0x05, 0xe7, 0x3e, 0x9d, 0x03,
			    0xee, 0x5a, 0x83, 0x0c, 0xcc, 0x09, 0x4c, 0x87 },
		.klen   = 32,
		.iv     = { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01 },
		.input  = { 0xf1, 0xb2, 0x73, 0xcd, 0x65, 0xa3, 0xdf, 0x5f,
			    0xe9, 0x5d, 0x48, 0x92, 0x54, 0x63, 0x4e, 0xb8 },
		.ilen   = 16,
		.result = { 0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
			    0x38, 0x39, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46 },
		.rlen   = 16,
	}, { /* LRW-32-AES 2 */
		.key    = { 0x59, 0x70, 0x47, 0x14, 0xf5, 0x57, 0x47, 0x8c,
		  	    0xd7, 0x79, 0xe8, 0x0f, 0x54, 0x88, 0x79, 0x44,
			    0x0d, 0x48, 0xf0, 0xb7, 0xb1, 0x5a, 0x53, 0xea,
			    0x1c, 0xaa, 0x6b, 0x29, 0xc2, 0xca, 0xfb, 0xaf
		},
		.klen   = 32,
		.iv     = { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02 },
		.input  = { 0x00, 0xc8, 0x2b, 0xae, 0x95, 0xbb, 0xcd, 0xe5,
			    0x27, 0x4f, 0x07, 0x69, 0xb2, 0x60, 0xe1, 0x36 },
		.ilen   = 16,
		.result  = { 0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
			     0x38, 0x39, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46 },
		.rlen   = 16,
	}, { /* LRW-32-AES 3 */
		.key    = { 0xd8, 0x2a, 0x91, 0x34, 0xb2, 0x6a, 0x56, 0x50,
		  	    0x30, 0xfe, 0x69, 0xe2, 0x37, 0x7f, 0x98, 0x47,
			    0xcd, 0xf9, 0x0b, 0x16, 0x0c, 0x64, 0x8f, 0xb6,
			    0xb0, 0x0d, 0x0d, 0x1b, 0xae, 0x85, 0x87, 0x1f },
		.klen   = 32,
		.iv     = { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			    0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00 },
		.input  = { 0x76, 0x32, 0x21, 0x83, 0xed, 0x8f, 0xf1, 0x82,
			    0xf9, 0x59, 0x62, 0x03, 0x69, 0x0e, 0x5e, 0x01 },
		.ilen   = 16,
		.result = { 0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
			    0x38, 0x39, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46 },
		.rlen   = 16,
	}, { /* LRW-32-AES 4 */
		.key    = { 0x0f, 0x6a, 0xef, 0xf8, 0xd3, 0xd2, 0xbb, 0x15,
		  	    0x25, 0x83, 0xf7, 0x3c, 0x1f, 0x01, 0x28, 0x74,
			    0xca, 0xc6, 0xbc, 0x35, 0x4d, 0x4a, 0x65, 0x54,
			    0x90, 0xae, 0x61, 0xcf, 0x7b, 0xae, 0xbd, 0xcc,
			    0xad, 0xe4, 0x94, 0xc5, 0x4a, 0x29, 0xae, 0x70 },
		.klen   = 40,
		.iv     = { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01 },
		.input  = { 0x9c, 0x0f, 0x15, 0x2f, 0x55, 0xa2, 0xd8, 0xf0,
			    0xd6, 0x7b, 0x8f, 0x9e, 0x28, 0x22, 0xbc, 0x41 },
		.ilen   = 16,
		.result = { 0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
			    0x38, 0x39, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46 },
		.rlen   = 16,
	}, { /* LRW-32-AES 5 */
		.key    = { 0x8a, 0xd4, 0xee, 0x10, 0x2f, 0xbd, 0x81, 0xff,
		  	    0xf8, 0x86, 0xce, 0xac, 0x93, 0xc5, 0xad, 0xc6,
			    0xa0, 0x19, 0x07, 0xc0, 0x9d, 0xf7, 0xbb, 0xdd,
			    0x52, 0x13, 0xb2, 0xb7, 0xf0, 0xff, 0x11, 0xd8,
			    0xd6, 0x08, 0xd0, 0xcd, 0x2e, 0xb1, 0x17, 0x6f },
		.klen   = 40,
		.iv     = { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			    0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00 },
		.input  = { 0xd4, 0x27, 0x6a, 0x7f, 0x14, 0x91, 0x3d, 0x65,
			    0xc8, 0x60, 0x48, 0x02, 0x87, 0xe3, 0x34, 0x06 },
		.ilen   = 16,
		.result = { 0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
			    0x38, 0x39, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46 },
		.rlen   = 16,
	}, { /* LRW-32-AES 6 */
		.key    = { 0xf8, 0xd4, 0x76, 0xff, 0xd6, 0x46, 0xee, 0x6c,
		  	    0x23, 0x84, 0xcb, 0x1c, 0x77, 0xd6, 0x19, 0x5d,
			    0xfe, 0xf1, 0xa9, 0xf3, 0x7b, 0xbc, 0x8d, 0x21,
			    0xa7, 0x9c, 0x21, 0xf8, 0xcb, 0x90, 0x02, 0x89,
			    0xa8, 0x45, 0x34, 0x8e, 0xc8, 0xc5, 0xb5, 0xf1,
			    0x26, 0xf5, 0x0e, 0x76, 0xfe, 0xfd, 0x1b, 0x1e },
		.klen   = 48,
		.iv     = { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01 },
		.input  = { 0xbd, 0x06, 0xb8, 0xe1, 0xdb, 0x98, 0x89, 0x9e,
			    0xc4, 0x98, 0xe4, 0x91, 0xcf, 0x1c, 0x70, 0x2b },
		.ilen   = 16,
		.result = { 0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
			    0x38, 0x39, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46 },
		.rlen   = 16,
	}, { /* LRW-32-AES 7 */
		.key    = { 0xfb, 0x76, 0x15, 0xb2, 0x3d, 0x80, 0x89, 0x1d,
		  	    0xd4, 0x70, 0x98, 0x0b, 0xc7, 0x95, 0x84, 0xc8,
			    0xb2, 0xfb, 0x64, 0xce, 0x60, 0x97, 0x87, 0x8d,
			    0x17, 0xfc, 0xe4, 0x5a, 0x49, 0xe8, 0x30, 0xb7,
			    0x6e, 0x78, 0x17, 0xe7, 0x2d, 0x5e, 0x12, 0xd4,
			    0x60, 0x64, 0x04, 0x7a, 0xf1, 0x2f, 0x9e, 0x0c },
		.klen   = 48,
		.iv     = { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			    0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00 },
		.input  = { 0x5b, 0x90, 0x8e, 0xc1, 0xab, 0xdd, 0x67, 0x5f,
			    0x3d, 0x69, 0x8a, 0x95, 0x53, 0xc8, 0x9c, 0xe5 },
		.ilen   = 16,
		.result = { 0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
			    0x38, 0x39, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46 },
		.rlen   = 16,
	}, {
/* http://www.mail-archive.com/stds-p1619@listserv.ieee.org/msg00173.html */
		.key    = { 0xf8, 0xd4, 0x76, 0xff, 0xd6, 0x46, 0xee, 0x6c,
			    0x23, 0x84, 0xcb, 0x1c, 0x77, 0xd6, 0x19, 0x5d,
			    0xfe, 0xf1, 0xa9, 0xf3, 0x7b, 0xbc, 0x8d, 0x21,
			    0xa7, 0x9c, 0x21, 0xf8, 0xcb, 0x90, 0x02, 0x89,
			    0xa8, 0x45, 0x34, 0x8e, 0xc8, 0xc5, 0xb5, 0xf1,
			    0x26, 0xf5, 0x0e, 0x76, 0xfe, 0xfd, 0x1b, 0x1e },
		.klen   = 48,
		.iv     = { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01 },
		.input	= { 0x1a, 0x1d, 0xa9, 0x30, 0xad, 0xf9, 0x2f, 0x9b,
			    0xb6, 0x1d, 0xae, 0xef, 0xf0, 0x2f, 0xf8, 0x5a,
			    0x39, 0x3c, 0xbf, 0x2a, 0xb2, 0x45, 0xb2, 0x23,
			    0x1b, 0x63, 0x3c, 0xcf, 0xaa, 0xbe, 0xcf, 0x4e,
			    0xfa, 0xe8, 0x29, 0xc2, 0x20, 0x68, 0x2b, 0x3c,
			    0x2e, 0x8b, 0xf7, 0x6e, 0x25, 0xbd, 0xe3, 0x3d,
			    0x66, 0x27, 0xd6, 0xaf, 0xd6, 0x64, 0x3e, 0xe3,
			    0xe8, 0x58, 0x46, 0x97, 0x39, 0x51, 0x07, 0xde,
			    0xcb, 0x37, 0xbc, 0xa9, 0xc0, 0x5f, 0x75, 0xc3,
			    0x0e, 0x84, 0x23, 0x1d, 0x16, 0xd4, 0x1c, 0x59,
			    0x9c, 0x1a, 0x02, 0x55, 0xab, 0x3a, 0x97, 0x1d,
			    0xdf, 0xdd, 0xc7, 0x06, 0x51, 0xd7, 0x70, 0xae,
			    0x23, 0xc6, 0x8c, 0xf5, 0x1e, 0xa0, 0xe5, 0x82,
			    0xb8, 0xb2, 0xbf, 0x04, 0xa0, 0x32, 0x8e, 0x68,
			    0xeb, 0xaf, 0x6e, 0x2d, 0x94, 0x22, 0x2f, 0xce,
			    0x4c, 0xb5, 0x59, 0xe2, 0xa2, 0x2f, 0xa0, 0x98,
			    0x1a, 0x97, 0xc6, 0xd4, 0xb5, 0x00, 0x59, 0xf2,
			    0x84, 0x14, 0x72, 0xb1, 0x9a, 0x6e, 0xa3, 0x7f,
			    0xea, 0x20, 0xe7, 0xcb, 0x65, 0x77, 0x3a, 0xdf,
			    0xc8, 0x97, 0x67, 0x15, 0xc2, 0x2a, 0x27, 0xcc,
			    0x18, 0x55, 0xa1, 0x24, 0x0b, 0x24, 0x24, 0xaf,
			    0x5b, 0xec, 0x68, 0xb8, 0xc8, 0xf5, 0xba, 0x63,
			    0xff, 0xed, 0x89, 0xce, 0xd5, 0x3d, 0x88, 0xf3,
			    0x25, 0xef, 0x05, 0x7c, 0x3a, 0xef, 0xeb, 0xd8,
			    0x7a, 0x32, 0x0d, 0xd1, 0x1e, 0x58, 0x59, 0x99,
			    0x90, 0x25, 0xb5, 0x26, 0xb0, 0xe3, 0x2b, 0x6c,
			    0x4c, 0xa9, 0x8b, 0x84, 0x4f, 0x5e, 0x01, 0x50,
			    0x41, 0x30, 0x58, 0xc5, 0x62, 0x74, 0x52, 0x1d,
			    0x45, 0x24, 0x6a, 0x42, 0x64, 0x4f, 0x97, 0x1c,
			    0xa8, 0x66, 0xb5, 0x6d, 0x79, 0xd4, 0x0d, 0x48,
			    0xc5, 0x5f, 0xf3, 0x90, 0x32, 0xdd, 0xdd, 0xe1,
			    0xe4, 0xa9, 0x9f, 0xfc, 0xc3, 0x52, 0x5a, 0x46,
			    0xe4, 0x81, 0x84, 0x95, 0x36, 0x59, 0x7a, 0x6b,
			    0xaa, 0xb3, 0x60, 0xad, 0xce, 0x9f, 0x9f, 0x28,
			    0xe0, 0x01, 0x75, 0x22, 0xc4, 0x4e, 0xa9, 0x62,
			    0x5c, 0x62, 0x0d, 0x00, 0xcb, 0x13, 0xe8, 0x43,
			    0x72, 0xd4, 0x2d, 0x53, 0x46, 0xb5, 0xd1, 0x16,
			    0x22, 0x18, 0xdf, 0x34, 0x33, 0xf5, 0xd6, 0x1c,
			    0xb8, 0x79, 0x78, 0x97, 0x94, 0xff, 0x72, 0x13,
			    0x4c, 0x27, 0xfc, 0xcb, 0xbf, 0x01, 0x53, 0xa6,
			    0xb4, 0x50, 0x6e, 0xde, 0xdf, 0xb5, 0x43, 0xa4,
			    0x59, 0xdf, 0x52, 0xf9, 0x7c, 0xe0, 0x11, 0x6f,
			    0x2d, 0x14, 0x8e, 0x24, 0x61, 0x2c, 0xe1, 0x17,
			    0xcc, 0xce, 0x51, 0x0c, 0x19, 0x8a, 0x82, 0x30,
			    0x94, 0xd5, 0x3d, 0x6a, 0x53, 0x06, 0x5e, 0xbd,
			    0xb7, 0xeb, 0xfa, 0xfd, 0x27, 0x51, 0xde, 0x85,
			    0x1e, 0x86, 0x53, 0x11, 0x53, 0x94, 0x00, 0xee,
			    0x2b, 0x8c, 0x08, 0x2a, 0xbf, 0xdd, 0xae, 0x11,
			    0xcb, 0x1e, 0xa2, 0x07, 0x9a, 0x80, 0xcf, 0x62,
			    0x9b, 0x09, 0xdc, 0x95, 0x3c, 0x96, 0x8e, 0xb1,
			    0x09, 0xbd, 0xe4, 0xeb, 0xdb, 0xca, 0x70, 0x7a,
			    0x9e, 0xfa, 0x31, 0x18, 0x45, 0x3c, 0x21, 0x33,
			    0xb0, 0xb3, 0x2b, 0xea, 0xf3, 0x71, 0x2d, 0xe1,
			    0x03, 0xad, 0x1b, 0x48, 0xd4, 0x67, 0x27, 0xf0,
			    0x62, 0xe4, 0x3d, 0xfb, 0x9b, 0x08, 0x76, 0xe7,
			    0xdd, 0x2b, 0x01, 0x39, 0x04, 0x5a, 0x58, 0x7a,
			    0xf7, 0x11, 0x90, 0xec, 0xbd, 0x51, 0x5c, 0x32,
			    0x6b, 0xd7, 0x35, 0x39, 0x02, 0x6b, 0xf2, 0xa6,
			    0xd0, 0x0d, 0x07, 0xe1, 0x06, 0xc4, 0x5b, 0x7d,
			    0xe4, 0x6a, 0xd7, 0xee, 0x15, 0x1f, 0x83, 0xb4,
			    0xa3, 0xa7, 0x5e, 0xc3, 0x90, 0xb7, 0xef, 0xd3,
			    0xb7, 0x4f, 0xf8, 0x92, 0x4c, 0xb7, 0x3c, 0x29,
			    0xcd, 0x7e, 0x2b, 0x5d, 0x43, 0xea, 0x42, 0xe7,
			    0x74, 0x3f, 0x7d, 0x58, 0x88, 0x75, 0xde, 0x3e },
		.ilen   = 512,
		.result	= { 0x05, 0x11, 0xb7, 0x18, 0xab, 0xc6, 0x2d, 0xac,
			    0x70, 0x5d, 0xf6, 0x22, 0x94, 0xcd, 0xe5, 0x6c,
			    0x17, 0x6b, 0xf6, 0x1c, 0xf0, 0xf3, 0x6e, 0xf8,
			    0x50, 0x38, 0x1f, 0x71, 0x49, 0xb6, 0x57, 0xd6,
			    0x8f, 0xcb, 0x8d, 0x6b, 0xe3, 0xa6, 0x29, 0x90,
			    0xfe, 0x2a, 0x62, 0x82, 0xae, 0x6d, 0x8b, 0xf6,
			    0xad, 0x1e, 0x9e, 0x20, 0x5f, 0x38, 0xbe, 0x04,
			    0xda, 0x10, 0x8e, 0xed, 0xa2, 0xa4, 0x87, 0xab,
			    0xda, 0x6b, 0xb4, 0x0c, 0x75, 0xba, 0xd3, 0x7c,
			    0xc9, 0xac, 0x42, 0x31, 0x95, 0x7c, 0xc9, 0x04,
			    0xeb, 0xd5, 0x6e, 0x32, 0x69, 0x8a, 0xdb, 0xa6,
			    0x15, 0xd7, 0x3f, 0x4f, 0x2f, 0x66, 0x69, 0x03,
			    0x9c, 0x1f, 0x54, 0x0f, 0xde, 0x1f, 0xf3, 0x65,
			    0x4c, 0x96, 0x12, 0xed, 0x7c, 0x92, 0x03, 0x01,
			    0x6f, 0xbc, 0x35, 0x93, 0xac, 0xf1, 0x27, 0xf1,
			    0xb4, 0x96, 0x82, 0x5a, 0x5f, 0xb0, 0xa0, 0x50,
			    0x89, 0xa4, 0x8e, 0x66, 0x44, 0x85, 0xcc, 0xfd,
			    0x33, 0x14, 0x70, 0xe3, 0x96, 0xb2, 0xc3, 0xd3,
			    0xbb, 0x54, 0x5a, 0x1a, 0xf9, 0x74, 0xa2, 0xc5,
			    0x2d, 0x64, 0x75, 0xdd, 0xb4, 0x54, 0xe6, 0x74,
			    0x8c, 0xd3, 0x9d, 0x9e, 0x86, 0xab, 0x51, 0x53,
			    0xb7, 0x93, 0x3e, 0x6f, 0xd0, 0x4e, 0x2c, 0x40,
			    0xf6, 0xa8, 0x2e, 0x3e, 0x9d, 0xf4, 0x66, 0xa5,
			    0x76, 0x12, 0x73, 0x44, 0x1a, 0x56, 0xd7, 0x72,
			    0x88, 0xcd, 0x21, 0x8c, 0x4c, 0x0f, 0xfe, 0xda,
			    0x95, 0xe0, 0x3a, 0xa6, 0xa5, 0x84, 0x46, 0xcd,
			    0xd5, 0x3e, 0x9d, 0x3a, 0xe2, 0x67, 0xe6, 0x60,
			    0x1a, 0xe2, 0x70, 0x85, 0x58, 0xc2, 0x1b, 0x09,
			    0xe1, 0xd7, 0x2c, 0xca, 0xad, 0xa8, 0x8f, 0xf9,
			    0xac, 0xb3, 0x0e, 0xdb, 0xca, 0x2e, 0xe2, 0xb8,
			    0x51, 0x71, 0xd9, 0x3c, 0x6c, 0xf1, 0x56, 0xf8,
			    0xea, 0x9c, 0xf1, 0xfb, 0x0c, 0xe6, 0xb7, 0x10,
			    0x1c, 0xf8, 0xa9, 0x7c, 0xe8, 0x53, 0x35, 0xc1,
			    0x90, 0x3e, 0x76, 0x4a, 0x74, 0xa4, 0x21, 0x2c,
			    0xf6, 0x2c, 0x4e, 0x0f, 0x94, 0x3a, 0x88, 0x2e,
			    0x41, 0x09, 0x6a, 0x33, 0x7d, 0xf6, 0xdd, 0x3f,
			    0x8d, 0x23, 0x31, 0x74, 0x84, 0xeb, 0x88, 0x6e,
			    0xcc, 0xb9, 0xbc, 0x22, 0x83, 0x19, 0x07, 0x22,
			    0xa5, 0x2d, 0xdf, 0xa5, 0xf3, 0x80, 0x85, 0x78,
			    0x84, 0x39, 0x6a, 0x6d, 0x6a, 0x99, 0x4f, 0xa5,
			    0x15, 0xfe, 0x46, 0xb0, 0xe4, 0x6c, 0xa5, 0x41,
			    0x3c, 0xce, 0x8f, 0x42, 0x60, 0x71, 0xa7, 0x75,
			    0x08, 0x40, 0x65, 0x8a, 0x82, 0xbf, 0xf5, 0x43,
			    0x71, 0x96, 0xa9, 0x4d, 0x44, 0x8a, 0x20, 0xbe,
			    0xfa, 0x4d, 0xbb, 0xc0, 0x7d, 0x31, 0x96, 0x65,
			    0xe7, 0x75, 0xe5, 0x3e, 0xfd, 0x92, 0x3b, 0xc9,
			    0x55, 0xbb, 0x16, 0x7e, 0xf7, 0xc2, 0x8c, 0xa4,
			    0x40, 0x1d, 0xe5, 0xef, 0x0e, 0xdf, 0xe4, 0x9a,
			    0x62, 0x73, 0x65, 0xfd, 0x46, 0x63, 0x25, 0x3d,
			    0x2b, 0xaf, 0xe5, 0x64, 0xfe, 0xa5, 0x5c, 0xcf,
			    0x24, 0xf3, 0xb4, 0xac, 0x64, 0xba, 0xdf, 0x4b,
			    0xc6, 0x96, 0x7d, 0x81, 0x2d, 0x8d, 0x97, 0xf7,
			    0xc5, 0x68, 0x77, 0x84, 0x32, 0x2b, 0xcc, 0x85,
			    0x74, 0x96, 0xf0, 0x12, 0x77, 0x61, 0xb9, 0xeb,
			    0x71, 0xaa, 0x82, 0xcb, 0x1c, 0xdb, 0x89, 0xc8,
			    0xc6, 0xb5, 0xe3, 0x5c, 0x7d, 0x39, 0x07, 0x24,
			    0xda, 0x39, 0x87, 0x45, 0xc0, 0x2b, 0xbb, 0x01,
			    0xac, 0xbc, 0x2a, 0x5c, 0x7f, 0xfc, 0xe8, 0xce,
			    0x6d, 0x9c, 0x6f, 0xed, 0xd3, 0xc1, 0xa1, 0xd6,
			    0xc5, 0x55, 0xa9, 0x66, 0x2f, 0xe1, 0xc8, 0x32,
			    0xa6, 0x5d, 0xa4, 0x3a, 0x98, 0x73, 0xe8, 0x45,
			    0xa4, 0xc7, 0xa8, 0xb4, 0xf6, 0x13, 0x03, 0xf6,
			    0xe9, 0x2e, 0xc4, 0x29, 0x0f, 0x84, 0xdb, 0xc4,
			    0x21, 0xc4, 0xc2, 0x75, 0x67, 0x89, 0x37, 0x0a },
		.rlen   = 512,
	}
};

static struct cipher_testvec aes_xts_enc_tv_template[] = {
	/* http://grouper.ieee.org/groups/1619/email/pdf00086.pdf */
	{ /* XTS-AES 1 */
		.key    = { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
		.klen   = 32,
		.iv     = { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
		.input  = { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
		.ilen   = 32,
		.result = { 0x91, 0x7c, 0xf6, 0x9e, 0xbd, 0x68, 0xb2, 0xec,
			    0x9b, 0x9f, 0xe9, 0xa3, 0xea, 0xdd, 0xa6, 0x92,
			    0xcd, 0x43, 0xd2, 0xf5, 0x95, 0x98, 0xed, 0x85,
			    0x8c, 0x02, 0xc2, 0x65, 0x2f, 0xbf, 0x92, 0x2e },
		.rlen   = 32,
	}, { /* XTS-AES 2 */
		.key    = { 0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11,
			    0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11,
			    0x22, 0x22, 0x22, 0x22, 0x22, 0x22, 0x22, 0x22,
			    0x22, 0x22, 0x22, 0x22, 0x22, 0x22, 0x22, 0x22 },
		.klen   = 32,
		.iv     = { 0x33, 0x33, 0x33, 0x33, 0x33, 0x00, 0x00, 0x00,
			    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
		.input  = { 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44,
			    0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44,
			    0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44,
			    0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44 },
		.ilen   = 32,
		.result = { 0xc4, 0x54, 0x18, 0x5e, 0x6a, 0x16, 0x93, 0x6e,
			    0x39, 0x33, 0x40, 0x38, 0xac, 0xef, 0x83, 0x8b,
			    0xfb, 0x18, 0x6f, 0xff, 0x74, 0x80, 0xad, 0xc4,
			    0x28, 0x93, 0x82, 0xec, 0xd6, 0xd3, 0x94, 0xf0 },
		.rlen   = 32,
	}, { /* XTS-AES 3 */
		.key    = { 0xff, 0xfe, 0xfd, 0xfc, 0xfb, 0xfa, 0xf9, 0xf8,
			    0xf7, 0xf6, 0xf5, 0xf4, 0xf3, 0xf2, 0xf1, 0xf0,
			    0x22, 0x22, 0x22, 0x22, 0x22, 0x22, 0x22, 0x22,
			    0x22, 0x22, 0x22, 0x22, 0x22, 0x22, 0x22, 0x22 },
		.klen   = 32,
		.iv     = { 0x33, 0x33, 0x33, 0x33, 0x33, 0x00, 0x00, 0x00,
			    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
		.input  = { 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44,
			    0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44,
			    0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44,
			    0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44 },
		.ilen   = 32,
		.result = { 0xaf, 0x85, 0x33, 0x6b, 0x59, 0x7a, 0xfc, 0x1a,
			    0x90, 0x0b, 0x2e, 0xb2, 0x1e, 0xc9, 0x49, 0xd2,
			    0x92, 0xdf, 0x4c, 0x04, 0x7e, 0x0b, 0x21, 0x53,
			    0x21, 0x86, 0xa5, 0x97, 0x1a, 0x22, 0x7a, 0x89 },
		.rlen   = 32,
	}, { /* XTS-AES 4 */
		.key    = { 0x27, 0x18, 0x28, 0x18, 0x28, 0x45, 0x90, 0x45,
			    0x23, 0x53, 0x60, 0x28, 0x74, 0x71, 0x35, 0x26,
			    0x31, 0x41, 0x59, 0x26, 0x53, 0x58, 0x97, 0x93,
			    0x23, 0x84, 0x62, 0x64, 0x33, 0x83, 0x27, 0x95 },
		.klen   = 32,
		.iv     = { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
		.input  = { 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
			    0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
			    0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
			    0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,
			    0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27,
			    0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f,
			    0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
			    0x38, 0x39, 0x3a, 0x3b, 0x3c, 0x3d, 0x3e, 0x3f,
			    0x40, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47,
			    0x48, 0x49, 0x4a, 0x4b, 0x4c, 0x4d, 0x4e, 0x4f,
			    0x50, 0x51, 0x52, 0x53, 0x54, 0x55, 0x56, 0x57,
			    0x58, 0x59, 0x5a, 0x5b, 0x5c, 0x5d, 0x5e, 0x5f,
			    0x60, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67,
			    0x68, 0x69, 0x6a, 0x6b, 0x6c, 0x6d, 0x6e, 0x6f,
			    0x70, 0x71, 0x72, 0x73, 0x74, 0x75, 0x76, 0x77,
			    0x78, 0x79, 0x7a, 0x7b, 0x7c, 0x7d, 0x7e, 0x7f,
			    0x80, 0x81, 0x82, 0x83, 0x84, 0x85, 0x86, 0x87,
			    0x88, 0x89, 0x8a, 0x8b, 0x8c, 0x8d, 0x8e, 0x8f,
			    0x90, 0x91, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97,
			    0x98, 0x99, 0x9a, 0x9b, 0x9c, 0x9d, 0x9e, 0x9f,
			    0xa0, 0xa1, 0xa2, 0xa3, 0xa4, 0xa5, 0xa6, 0xa7,
			    0xa8, 0xa9, 0xaa, 0xab, 0xac, 0xad, 0xae, 0xaf,
			    0xb0, 0xb1, 0xb2, 0xb3, 0xb4, 0xb5, 0xb6, 0xb7,
			    0xb8, 0xb9, 0xba, 0xbb, 0xbc, 0xbd, 0xbe, 0xbf,
			    0xc0, 0xc1, 0xc2, 0xc3, 0xc4, 0xc5, 0xc6, 0xc7,
			    0xc8, 0xc9, 0xca, 0xcb, 0xcc, 0xcd, 0xce, 0xcf,
			    0xd0, 0xd1, 0xd2, 0xd3, 0xd4, 0xd5, 0xd6, 0xd7,
			    0xd8, 0xd9, 0xda, 0xdb, 0xdc, 0xdd, 0xde, 0xdf,
			    0xe0, 0xe1, 0xe2, 0xe3, 0xe4, 0xe5, 0xe6, 0xe7,
			    0xe8, 0xe9, 0xea, 0xeb, 0xec, 0xed, 0xee, 0xef,
			    0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7,
			    0xf8, 0xf9, 0xfa, 0xfb, 0xfc, 0xfd, 0xfe, 0xff,
			    0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
			    0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
			    0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
			    0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,
			    0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27,
			    0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f,
			    0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
			    0x38, 0x39, 0x3a, 0x3b, 0x3c, 0x3d, 0x3e, 0x3f,
			    0x40, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47,
			    0x48, 0x49, 0x4a, 0x4b, 0x4c, 0x4d, 0x4e, 0x4f,
			    0x50, 0x51, 0x52, 0x53, 0x54, 0x55, 0x56, 0x57,
			    0x58, 0x59, 0x5a, 0x5b, 0x5c, 0x5d, 0x5e, 0x5f,
			    0x60, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67,
			    0x68, 0x69, 0x6a, 0x6b, 0x6c, 0x6d, 0x6e, 0x6f,
			    0x70, 0x71, 0x72, 0x73, 0x74, 0x75, 0x76, 0x77,
			    0x78, 0x79, 0x7a, 0x7b, 0x7c, 0x7d, 0x7e, 0x7f,
			    0x80, 0x81, 0x82, 0x83, 0x84, 0x85, 0x86, 0x87,
			    0x88, 0x89, 0x8a, 0x8b, 0x8c, 0x8d, 0x8e, 0x8f,
			    0x90, 0x91, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97,
			    0x98, 0x99, 0x9a, 0x9b, 0x9c, 0x9d, 0x9e, 0x9f,
			    0xa0, 0xa1, 0xa2, 0xa3, 0xa4, 0xa5, 0xa6, 0xa7,
			    0xa8, 0xa9, 0xaa, 0xab, 0xac, 0xad, 0xae, 0xaf,
			    0xb0, 0xb1, 0xb2, 0xb3, 0xb4, 0xb5, 0xb6, 0xb7,
			    0xb8, 0xb9, 0xba, 0xbb, 0xbc, 0xbd, 0xbe, 0xbf,
			    0xc0, 0xc1, 0xc2, 0xc3, 0xc4, 0xc5, 0xc6, 0xc7,
			    0xc8, 0xc9, 0xca, 0xcb, 0xcc, 0xcd, 0xce, 0xcf,
			    0xd0, 0xd1, 0xd2, 0xd3, 0xd4, 0xd5, 0xd6, 0xd7,
			    0xd8, 0xd9, 0xda, 0xdb, 0xdc, 0xdd, 0xde, 0xdf,
			    0xe0, 0xe1, 0xe2, 0xe3, 0xe4, 0xe5, 0xe6, 0xe7,
			    0xe8, 0xe9, 0xea, 0xeb, 0xec, 0xed, 0xee, 0xef,
			    0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7,
			    0xf8, 0xf9, 0xfa, 0xfb, 0xfc, 0xfd, 0xfe, 0xff },
		.ilen   = 512,
		.result = { 0x27, 0xa7, 0x47, 0x9b, 0xef, 0xa1, 0xd4, 0x76,
			    0x48, 0x9f, 0x30, 0x8c, 0xd4, 0xcf, 0xa6, 0xe2,
			    0xa9, 0x6e, 0x4b, 0xbe, 0x32, 0x08, 0xff, 0x25,
			    0x28, 0x7d, 0xd3, 0x81, 0x96, 0x16, 0xe8, 0x9c,
			    0xc7, 0x8c, 0xf7, 0xf5, 0xe5, 0x43, 0x44, 0x5f,
			    0x83, 0x33, 0xd8, 0xfa, 0x7f, 0x56, 0x00, 0x00,
			    0x05, 0x27, 0x9f, 0xa5, 0xd8, 0xb5, 0xe4, 0xad,
			    0x40, 0xe7, 0x36, 0xdd, 0xb4, 0xd3, 0x54, 0x12,
			    0x32, 0x80, 0x63, 0xfd, 0x2a, 0xab, 0x53, 0xe5,
			    0xea, 0x1e, 0x0a, 0x9f, 0x33, 0x25, 0x00, 0xa5,
			    0xdf, 0x94, 0x87, 0xd0, 0x7a, 0x5c, 0x92, 0xcc,
			    0x51, 0x2c, 0x88, 0x66, 0xc7, 0xe8, 0x60, 0xce,
			    0x93, 0xfd, 0xf1, 0x66, 0xa2, 0x49, 0x12, 0xb4,
			    0x22, 0x97, 0x61, 0x46, 0xae, 0x20, 0xce, 0x84,
			    0x6b, 0xb7, 0xdc, 0x9b, 0xa9, 0x4a, 0x76, 0x7a,
			    0xae, 0xf2, 0x0c, 0x0d, 0x61, 0xad, 0x02, 0x65,
			    0x5e, 0xa9, 0x2d, 0xc4, 0xc4, 0xe4, 0x1a, 0x89,
			    0x52, 0xc6, 0x51, 0xd3, 0x31, 0x74, 0xbe, 0x51,
			    0xa1, 0x0c, 0x42, 0x11, 0x10, 0xe6, 0xd8, 0x15,
			    0x88, 0xed, 0xe8, 0x21, 0x03, 0xa2, 0x52, 0xd8,
			    0xa7, 0x50, 0xe8, 0x76, 0x8d, 0xef, 0xff, 0xed,
			    0x91, 0x22, 0x81, 0x0a, 0xae, 0xb9, 0x9f, 0x91,
			    0x72, 0xaf, 0x82, 0xb6, 0x04, 0xdc, 0x4b, 0x8e,
			    0x51, 0xbc, 0xb0, 0x82, 0x35, 0xa6, 0xf4, 0x34,
			    0x13, 0x32, 0xe4, 0xca, 0x60, 0x48, 0x2a, 0x4b,
			    0xa1, 0xa0, 0x3b, 0x3e, 0x65, 0x00, 0x8f, 0xc5,
			    0xda, 0x76, 0xb7, 0x0b, 0xf1, 0x69, 0x0d, 0xb4,
			    0xea, 0xe2, 0x9c, 0x5f, 0x1b, 0xad, 0xd0, 0x3c,
			    0x5c, 0xcf, 0x2a, 0x55, 0xd7, 0x05, 0xdd, 0xcd,
			    0x86, 0xd4, 0x49, 0x51, 0x1c, 0xeb, 0x7e, 0xc3,
			    0x0b, 0xf1, 0x2b, 0x1f, 0xa3, 0x5b, 0x91, 0x3f,
			    0x9f, 0x74, 0x7a, 0x8a, 0xfd, 0x1b, 0x13, 0x0e,
			    0x94, 0xbf, 0xf9, 0x4e, 0xff, 0xd0, 0x1a, 0x91,
			    0x73, 0x5c, 0xa1, 0x72, 0x6a, 0xcd, 0x0b, 0x19,
			    0x7c, 0x4e, 0x5b, 0x03, 0x39, 0x36, 0x97, 0xe1,
			    0x26, 0x82, 0x6f, 0xb6, 0xbb, 0xde, 0x8e, 0xcc,
			    0x1e, 0x08, 0x29, 0x85, 0x16, 0xe2, 0xc9, 0xed,
			    0x03, 0xff, 0x3c, 0x1b, 0x78, 0x60, 0xf6, 0xde,
			    0x76, 0xd4, 0xce, 0xcd, 0x94, 0xc8, 0x11, 0x98,
			    0x55, 0xef, 0x52, 0x97, 0xca, 0x67, 0xe9, 0xf3,
			    0xe7, 0xff, 0x72, 0xb1, 0xe9, 0x97, 0x85, 0xca,
			    0x0a, 0x7e, 0x77, 0x20, 0xc5, 0xb3, 0x6d, 0xc6,
			    0xd7, 0x2c, 0xac, 0x95, 0x74, 0xc8, 0xcb, 0xbc,
			    0x2f, 0x80, 0x1e, 0x23, 0xe5, 0x6f, 0xd3, 0x44,
			    0xb0, 0x7f, 0x22, 0x15, 0x4b, 0xeb, 0xa0, 0xf0,
			    0x8c, 0xe8, 0x89, 0x1e, 0x64, 0x3e, 0xd9, 0x95,
			    0xc9, 0x4d, 0x9a, 0x69, 0xc9, 0xf1, 0xb5, 0xf4,
			    0x99, 0x02, 0x7a, 0x78, 0x57, 0x2a, 0xee, 0xbd,
			    0x74, 0xd2, 0x0c, 0xc3, 0x98, 0x81, 0xc2, 0x13,
			    0xee, 0x77, 0x0b, 0x10, 0x10, 0xe4, 0xbe, 0xa7,
			    0x18, 0x84, 0x69, 0x77, 0xae, 0x11, 0x9f, 0x7a,
			    0x02, 0x3a, 0xb5, 0x8c, 0xca, 0x0a, 0xd7, 0x52,
			    0xaf, 0xe6, 0x56, 0xbb, 0x3c, 0x17, 0x25, 0x6a,
			    0x9f, 0x6e, 0x9b, 0xf1, 0x9f, 0xdd, 0x5a, 0x38,
			    0xfc, 0x82, 0xbb, 0xe8, 0x72, 0xc5, 0x53, 0x9e,
			    0xdb, 0x60, 0x9e, 0xf4, 0xf7, 0x9c, 0x20, 0x3e,
			    0xbb, 0x14, 0x0f, 0x2e, 0x58, 0x3c, 0xb2, 0xad,
			    0x15, 0xb4, 0xaa, 0x5b, 0x65, 0x50, 0x16, 0xa8,
			    0x44, 0x92, 0x77, 0xdb, 0xd4, 0x77, 0xef, 0x2c,
			    0x8d, 0x6c, 0x01, 0x7d, 0xb7, 0x38, 0xb1, 0x8d,
			    0xeb, 0x4a, 0x42, 0x7d, 0x19, 0x23, 0xce, 0x3f,
			    0xf2, 0x62, 0x73, 0x57, 0x79, 0xa4, 0x18, 0xf2,
			    0x0a, 0x28, 0x2d, 0xf9, 0x20, 0x14, 0x7b, 0xea,
			    0xbe, 0x42, 0x1e, 0xe5, 0x31, 0x9d, 0x05, 0x68 },
		.rlen   = 512,
	}
};

static struct cipher_testvec aes_xts_dec_tv_template[] = {
	/* http://grouper.ieee.org/groups/1619/email/pdf00086.pdf */
	{ /* XTS-AES 1 */
		.key    = { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
		.klen   = 32,
		.iv     = { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
		.input = { 0x91, 0x7c, 0xf6, 0x9e, 0xbd, 0x68, 0xb2, 0xec,
			   0x9b, 0x9f, 0xe9, 0xa3, 0xea, 0xdd, 0xa6, 0x92,
			   0xcd, 0x43, 0xd2, 0xf5, 0x95, 0x98, 0xed, 0x85,
			   0x8c, 0x02, 0xc2, 0x65, 0x2f, 0xbf, 0x92, 0x2e },
		.ilen   = 32,
		.result  = { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			     0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			     0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			     0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
		.rlen   = 32,
	}, { /* XTS-AES 2 */
		.key    = { 0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11,
			    0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11,
			    0x22, 0x22, 0x22, 0x22, 0x22, 0x22, 0x22, 0x22,
			    0x22, 0x22, 0x22, 0x22, 0x22, 0x22, 0x22, 0x22 },
		.klen   = 32,
		.iv     = { 0x33, 0x33, 0x33, 0x33, 0x33, 0x00, 0x00, 0x00,
			    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
		.input  = { 0xc4, 0x54, 0x18, 0x5e, 0x6a, 0x16, 0x93, 0x6e,
			    0x39, 0x33, 0x40, 0x38, 0xac, 0xef, 0x83, 0x8b,
			    0xfb, 0x18, 0x6f, 0xff, 0x74, 0x80, 0xad, 0xc4,
			    0x28, 0x93, 0x82, 0xec, 0xd6, 0xd3, 0x94, 0xf0 },
		.ilen   = 32,
		.result = { 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44,
			    0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44,
			    0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44,
			    0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44 },
		.rlen   = 32,
	}, { /* XTS-AES 3 */
		.key    = { 0xff, 0xfe, 0xfd, 0xfc, 0xfb, 0xfa, 0xf9, 0xf8,
			    0xf7, 0xf6, 0xf5, 0xf4, 0xf3, 0xf2, 0xf1, 0xf0,
			    0x22, 0x22, 0x22, 0x22, 0x22, 0x22, 0x22, 0x22,
			    0x22, 0x22, 0x22, 0x22, 0x22, 0x22, 0x22, 0x22 },
		.klen   = 32,
		.iv     = { 0x33, 0x33, 0x33, 0x33, 0x33, 0x00, 0x00, 0x00,
			    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
		.input = { 0xaf, 0x85, 0x33, 0x6b, 0x59, 0x7a, 0xfc, 0x1a,
			    0x90, 0x0b, 0x2e, 0xb2, 0x1e, 0xc9, 0x49, 0xd2,
			    0x92, 0xdf, 0x4c, 0x04, 0x7e, 0x0b, 0x21, 0x53,
			    0x21, 0x86, 0xa5, 0x97, 0x1a, 0x22, 0x7a, 0x89 },
		.ilen   = 32,
		.result  = { 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44,
			    0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44,
			    0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44,
			    0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44 },
		.rlen   = 32,
	}, { /* XTS-AES 4 */
		.key    = { 0x27, 0x18, 0x28, 0x18, 0x28, 0x45, 0x90, 0x45,
			    0x23, 0x53, 0x60, 0x28, 0x74, 0x71, 0x35, 0x26,
			    0x31, 0x41, 0x59, 0x26, 0x53, 0x58, 0x97, 0x93,
			    0x23, 0x84, 0x62, 0x64, 0x33, 0x83, 0x27, 0x95 },
		.klen   = 32,
		.iv     = { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
		.input  = { 0x27, 0xa7, 0x47, 0x9b, 0xef, 0xa1, 0xd4, 0x76,
			    0x48, 0x9f, 0x30, 0x8c, 0xd4, 0xcf, 0xa6, 0xe2,
			    0xa9, 0x6e, 0x4b, 0xbe, 0x32, 0x08, 0xff, 0x25,
			    0x28, 0x7d, 0xd3, 0x81, 0x96, 0x16, 0xe8, 0x9c,
			    0xc7, 0x8c, 0xf7, 0xf5, 0xe5, 0x43, 0x44, 0x5f,
			    0x83, 0x33, 0xd8, 0xfa, 0x7f, 0x56, 0x00, 0x00,
			    0x05, 0x27, 0x9f, 0xa5, 0xd8, 0xb5, 0xe4, 0xad,
			    0x40, 0xe7, 0x36, 0xdd, 0xb4, 0xd3, 0x54, 0x12,
			    0x32, 0x80, 0x63, 0xfd, 0x2a, 0xab, 0x53, 0xe5,
			    0xea, 0x1e, 0x0a, 0x9f, 0x33, 0x25, 0x00, 0xa5,
			    0xdf, 0x94, 0x87, 0xd0, 0x7a, 0x5c, 0x92, 0xcc,
			    0x51, 0x2c, 0x88, 0x66, 0xc7, 0xe8, 0x60, 0xce,
			    0x93, 0xfd, 0xf1, 0x66, 0xa2, 0x49, 0x12, 0xb4,
			    0x22, 0x97, 0x61, 0x46, 0xae, 0x20, 0xce, 0x84,
			    0x6b, 0xb7, 0xdc, 0x9b, 0xa9, 0x4a, 0x76, 0x7a,
			    0xae, 0xf2, 0x0c, 0x0d, 0x61, 0xad, 0x02, 0x65,
			    0x5e, 0xa9, 0x2d, 0xc4, 0xc4, 0xe4, 0x1a, 0x89,
			    0x52, 0xc6, 0x51, 0xd3, 0x31, 0x74, 0xbe, 0x51,
			    0xa1, 0x0c, 0x42, 0x11, 0x10, 0xe6, 0xd8, 0x15,
			    0x88, 0xed, 0xe8, 0x21, 0x03, 0xa2, 0x52, 0xd8,
			    0xa7, 0x50, 0xe8, 0x76, 0x8d, 0xef, 0xff, 0xed,
			    0x91, 0x22, 0x81, 0x0a, 0xae, 0xb9, 0x9f, 0x91,
			    0x72, 0xaf, 0x82, 0xb6, 0x04, 0xdc, 0x4b, 0x8e,
			    0x51, 0xbc, 0xb0, 0x82, 0x35, 0xa6, 0xf4, 0x34,
			    0x13, 0x32, 0xe4, 0xca, 0x60, 0x48, 0x2a, 0x4b,
			    0xa1, 0xa0, 0x3b, 0x3e, 0x65, 0x00, 0x8f, 0xc5,
			    0xda, 0x76, 0xb7, 0x0b, 0xf1, 0x69, 0x0d, 0xb4,
			    0xea, 0xe2, 0x9c, 0x5f, 0x1b, 0xad, 0xd0, 0x3c,
			    0x5c, 0xcf, 0x2a, 0x55, 0xd7, 0x05, 0xdd, 0xcd,
			    0x86, 0xd4, 0x49, 0x51, 0x1c, 0xeb, 0x7e, 0xc3,
			    0x0b, 0xf1, 0x2b, 0x1f, 0xa3, 0x5b, 0x91, 0x3f,
			    0x9f, 0x74, 0x7a, 0x8a, 0xfd, 0x1b, 0x13, 0x0e,
			    0x94, 0xbf, 0xf9, 0x4e, 0xff, 0xd0, 0x1a, 0x91,
			    0x73, 0x5c, 0xa1, 0x72, 0x6a, 0xcd, 0x0b, 0x19,
			    0x7c, 0x4e, 0x5b, 0x03, 0x39, 0x36, 0x97, 0xe1,
			    0x26, 0x82, 0x6f, 0xb6, 0xbb, 0xde, 0x8e, 0xcc,
			    0x1e, 0x08, 0x29, 0x85, 0x16, 0xe2, 0xc9, 0xed,
			    0x03, 0xff, 0x3c, 0x1b, 0x78, 0x60, 0xf6, 0xde,
			    0x76, 0xd4, 0xce, 0xcd, 0x94, 0xc8, 0x11, 0x98,
			    0x55, 0xef, 0x52, 0x97, 0xca, 0x67, 0xe9, 0xf3,
			    0xe7, 0xff, 0x72, 0xb1, 0xe9, 0x97, 0x85, 0xca,
			    0x0a, 0x7e, 0x77, 0x20, 0xc5, 0xb3, 0x6d, 0xc6,
			    0xd7, 0x2c, 0xac, 0x95, 0x74, 0xc8, 0xcb, 0xbc,
			    0x2f, 0x80, 0x1e, 0x23, 0xe5, 0x6f, 0xd3, 0x44,
			    0xb0, 0x7f, 0x22, 0x15, 0x4b, 0xeb, 0xa0, 0xf0,
			    0x8c, 0xe8, 0x89, 0x1e, 0x64, 0x3e, 0xd9, 0x95,
			    0xc9, 0x4d, 0x9a, 0x69, 0xc9, 0xf1, 0xb5, 0xf4,
			    0x99, 0x02, 0x7a, 0x78, 0x57, 0x2a, 0xee, 0xbd,
			    0x74, 0xd2, 0x0c, 0xc3, 0x98, 0x81, 0xc2, 0x13,
			    0xee, 0x77, 0x0b, 0x10, 0x10, 0xe4, 0xbe, 0xa7,
			    0x18, 0x84, 0x69, 0x77, 0xae, 0x11, 0x9f, 0x7a,
			    0x02, 0x3a, 0xb5, 0x8c, 0xca, 0x0a, 0xd7, 0x52,
			    0xaf, 0xe6, 0x56, 0xbb, 0x3c, 0x17, 0x25, 0x6a,
			    0x9f, 0x6e, 0x9b, 0xf1, 0x9f, 0xdd, 0x5a, 0x38,
			    0xfc, 0x82, 0xbb, 0xe8, 0x72, 0xc5, 0x53, 0x9e,
			    0xdb, 0x60, 0x9e, 0xf4, 0xf7, 0x9c, 0x20, 0x3e,
			    0xbb, 0x14, 0x0f, 0x2e, 0x58, 0x3c, 0xb2, 0xad,
			    0x15, 0xb4, 0xaa, 0x5b, 0x65, 0x50, 0x16, 0xa8,
			    0x44, 0x92, 0x77, 0xdb, 0xd4, 0x77, 0xef, 0x2c,
			    0x8d, 0x6c, 0x01, 0x7d, 0xb7, 0x38, 0xb1, 0x8d,
			    0xeb, 0x4a, 0x42, 0x7d, 0x19, 0x23, 0xce, 0x3f,
			    0xf2, 0x62, 0x73, 0x57, 0x79, 0xa4, 0x18, 0xf2,
			    0x0a, 0x28, 0x2d, 0xf9, 0x20, 0x14, 0x7b, 0xea,
			    0xbe, 0x42, 0x1e, 0xe5, 0x31, 0x9d, 0x05, 0x68 },
		.ilen   = 512,
		.result = { 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
			    0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
			    0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
			    0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,
			    0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27,
			    0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f,
			    0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
			    0x38, 0x39, 0x3a, 0x3b, 0x3c, 0x3d, 0x3e, 0x3f,
			    0x40, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47,
			    0x48, 0x49, 0x4a, 0x4b, 0x4c, 0x4d, 0x4e, 0x4f,
			    0x50, 0x51, 0x52, 0x53, 0x54, 0x55, 0x56, 0x57,
			    0x58, 0x59, 0x5a, 0x5b, 0x5c, 0x5d, 0x5e, 0x5f,
			    0x60, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67,
			    0x68, 0x69, 0x6a, 0x6b, 0x6c, 0x6d, 0x6e, 0x6f,
			    0x70, 0x71, 0x72, 0x73, 0x74, 0x75, 0x76, 0x77,
			    0x78, 0x79, 0x7a, 0x7b, 0x7c, 0x7d, 0x7e, 0x7f,
			    0x80, 0x81, 0x82, 0x83, 0x84, 0x85, 0x86, 0x87,
			    0x88, 0x89, 0x8a, 0x8b, 0x8c, 0x8d, 0x8e, 0x8f,
			    0x90, 0x91, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97,
			    0x98, 0x99, 0x9a, 0x9b, 0x9c, 0x9d, 0x9e, 0x9f,
			    0xa0, 0xa1, 0xa2, 0xa3, 0xa4, 0xa5, 0xa6, 0xa7,
			    0xa8, 0xa9, 0xaa, 0xab, 0xac, 0xad, 0xae, 0xaf,
			    0xb0, 0xb1, 0xb2, 0xb3, 0xb4, 0xb5, 0xb6, 0xb7,
			    0xb8, 0xb9, 0xba, 0xbb, 0xbc, 0xbd, 0xbe, 0xbf,
			    0xc0, 0xc1, 0xc2, 0xc3, 0xc4, 0xc5, 0xc6, 0xc7,
			    0xc8, 0xc9, 0xca, 0xcb, 0xcc, 0xcd, 0xce, 0xcf,
			    0xd0, 0xd1, 0xd2, 0xd3, 0xd4, 0xd5, 0xd6, 0xd7,
			    0xd8, 0xd9, 0xda, 0xdb, 0xdc, 0xdd, 0xde, 0xdf,
			    0xe0, 0xe1, 0xe2, 0xe3, 0xe4, 0xe5, 0xe6, 0xe7,
			    0xe8, 0xe9, 0xea, 0xeb, 0xec, 0xed, 0xee, 0xef,
			    0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7,
			    0xf8, 0xf9, 0xfa, 0xfb, 0xfc, 0xfd, 0xfe, 0xff,
			    0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
			    0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
			    0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
			    0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,
			    0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27,
			    0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f,
			    0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
			    0x38, 0x39, 0x3a, 0x3b, 0x3c, 0x3d, 0x3e, 0x3f,
			    0x40, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47,
			    0x48, 0x49, 0x4a, 0x4b, 0x4c, 0x4d, 0x4e, 0x4f,
			    0x50, 0x51, 0x52, 0x53, 0x54, 0x55, 0x56, 0x57,
			    0x58, 0x59, 0x5a, 0x5b, 0x5c, 0x5d, 0x5e, 0x5f,
			    0x60, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67,
			    0x68, 0x69, 0x6a, 0x6b, 0x6c, 0x6d, 0x6e, 0x6f,
			    0x70, 0x71, 0x72, 0x73, 0x74, 0x75, 0x76, 0x77,
			    0x78, 0x79, 0x7a, 0x7b, 0x7c, 0x7d, 0x7e, 0x7f,
			    0x80, 0x81, 0x82, 0x83, 0x84, 0x85, 0x86, 0x87,
			    0x88, 0x89, 0x8a, 0x8b, 0x8c, 0x8d, 0x8e, 0x8f,
			    0x90, 0x91, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97,
			    0x98, 0x99, 0x9a, 0x9b, 0x9c, 0x9d, 0x9e, 0x9f,
			    0xa0, 0xa1, 0xa2, 0xa3, 0xa4, 0xa5, 0xa6, 0xa7,
			    0xa8, 0xa9, 0xaa, 0xab, 0xac, 0xad, 0xae, 0xaf,
			    0xb0, 0xb1, 0xb2, 0xb3, 0xb4, 0xb5, 0xb6, 0xb7,
			    0xb8, 0xb9, 0xba, 0xbb, 0xbc, 0xbd, 0xbe, 0xbf,
			    0xc0, 0xc1, 0xc2, 0xc3, 0xc4, 0xc5, 0xc6, 0xc7,
			    0xc8, 0xc9, 0xca, 0xcb, 0xcc, 0xcd, 0xce, 0xcf,
			    0xd0, 0xd1, 0xd2, 0xd3, 0xd4, 0xd5, 0xd6, 0xd7,
			    0xd8, 0xd9, 0xda, 0xdb, 0xdc, 0xdd, 0xde, 0xdf,
			    0xe0, 0xe1, 0xe2, 0xe3, 0xe4, 0xe5, 0xe6, 0xe7,
			    0xe8, 0xe9, 0xea, 0xeb, 0xec, 0xed, 0xee, 0xef,
			    0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7,
			    0xf8, 0xf9, 0xfa, 0xfb, 0xfc, 0xfd, 0xfe, 0xff },
		.rlen   = 512,
	}
};


static struct cipher_testvec aes_ctr_enc_tv_template[] = {
	{ /* From RFC 3686 */
		.key	= { 0xae, 0x68, 0x52, 0xf8, 0x12, 0x10, 0x67, 0xcc,
			    0x4b, 0xf7, 0xa5, 0x76, 0x55, 0x77, 0xf3, 0x9e,
			    0x00, 0x00, 0x00, 0x30 },
		.klen	= 20,
		.iv 	= { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
		.input 	= { "Single block msg" },
		.ilen	= 16,
		.result = { 0xe4, 0x09, 0x5d, 0x4f, 0xb7, 0xa7, 0xb3, 0x79,
			    0x2d, 0x61, 0x75, 0xa3, 0x26, 0x13, 0x11, 0xb8 },
		.rlen	= 16,
	}, {
		.key	= { 0x7e, 0x24, 0x06, 0x78, 0x17, 0xfa, 0xe0, 0xd7,
			    0x43, 0xd6, 0xce, 0x1f, 0x32, 0x53, 0x91, 0x63,
			    0x00, 0x6c, 0xb6, 0xdb },
		.klen	= 20,
		.iv 	= { 0xc0, 0x54, 0x3b, 0x59, 0xda, 0x48, 0xd9, 0x0b },
		.input	= { 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
			    0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
			    0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
			    0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f },
		.ilen 	= 32,
		.result = { 0x51, 0x04, 0xa1, 0x06, 0x16, 0x8a, 0x72, 0xd9,
			    0x79, 0x0d, 0x41, 0xee, 0x8e, 0xda, 0xd3, 0x88,
			    0xeb, 0x2e, 0x1e, 0xfc, 0x46, 0xda, 0x57, 0xc8,
			    0xfc, 0xe6, 0x30, 0xdf, 0x91, 0x41, 0xbe, 0x28 },
		.rlen	= 32,
	}, {
		.key 	= { 0x16, 0xaf, 0x5b, 0x14, 0x5f, 0xc9, 0xf5, 0x79,
			    0xc1, 0x75, 0xf9, 0x3e, 0x3b, 0xfb, 0x0e, 0xed,
			    0x86, 0x3d, 0x06, 0xcc, 0xfd, 0xb7, 0x85, 0x15,
			    0x00, 0x00, 0x00, 0x48 },
		.klen 	= 28,
		.iv	= { 0x36, 0x73, 0x3c, 0x14, 0x7d, 0x6d, 0x93, 0xcb },
		.input	= { "Single block msg" },
		.ilen 	= 16,
		.result	= { 0x4b, 0x55, 0x38, 0x4f, 0xe2, 0x59, 0xc9, 0xc8,
			    0x4e, 0x79, 0x35, 0xa0, 0x03, 0xcb, 0xe9, 0x28 },
		.rlen	= 16,
	}, {
		.key	= { 0x7c, 0x5c, 0xb2, 0x40, 0x1b, 0x3d, 0xc3, 0x3c,
			    0x19, 0xe7, 0x34, 0x08, 0x19, 0xe0, 0xf6, 0x9c,
			    0x67, 0x8c, 0x3d, 0xb8, 0xe6, 0xf6, 0xa9, 0x1a,
			    0x00, 0x96, 0xb0, 0x3b },
		.klen	= 28,
		.iv 	= { 0x02, 0x0c, 0x6e, 0xad, 0xc2, 0xcb, 0x50, 0x0d },
		.input	= { 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
			    0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
			    0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
			    0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f },
		.ilen	= 32,
		.result	= { 0x45, 0x32, 0x43, 0xfc, 0x60, 0x9b, 0x23, 0x32,
			    0x7e, 0xdf, 0xaa, 0xfa, 0x71, 0x31, 0xcd, 0x9f,
			    0x84, 0x90, 0x70, 0x1c, 0x5a, 0xd4, 0xa7, 0x9c,
			    0xfc, 0x1f, 0xe0, 0xff, 0x42, 0xf4, 0xfb, 0x00 },
		.rlen 	= 32,
	}, {
		.key 	= { 0x77, 0x6b, 0xef, 0xf2, 0x85, 0x1d, 0xb0, 0x6f,
			    0x4c, 0x8a, 0x05, 0x42, 0xc8, 0x69, 0x6f, 0x6c,
			    0x6a, 0x81, 0xaf, 0x1e, 0xec, 0x96, 0xb4, 0xd3,
			    0x7f, 0xc1, 0xd6, 0x89, 0xe6, 0xc1, 0xc1, 0x04,
			    0x00, 0x00, 0x00, 0x60 },
		.klen	= 36,
		.iv 	= { 0xdb, 0x56, 0x72, 0xc9, 0x7a, 0xa8, 0xf0, 0xb2 },
		.input	= { "Single block msg" },
		.ilen	= 16,
		.result = { 0x14, 0x5a, 0xd0, 0x1d, 0xbf, 0x82, 0x4e, 0xc7,
			    0x56, 0x08, 0x63, 0xdc, 0x71, 0xe3, 0xe0, 0xc0 },
		.rlen 	= 16,
	}, {
		.key	= { 0xf6, 0xd6, 0x6d, 0x6b, 0xd5, 0x2d, 0x59, 0xbb,
			    0x07, 0x96, 0x36, 0x58, 0x79, 0xef, 0xf8, 0x86,
			    0xc6, 0x6d, 0xd5, 0x1a, 0x5b, 0x6a, 0x99, 0x74,
			    0x4b, 0x50, 0x59, 0x0c, 0x87, 0xa2, 0x38, 0x84,
			    0x00, 0xfa, 0xac, 0x24 },
		.klen 	= 36,
		.iv	= { 0xc1, 0x58, 0x5e, 0xf1, 0x5a, 0x43, 0xd8, 0x75 },
		.input	= { 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
			    0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
			    0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
			    0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f },
		.ilen	= 32,
		.result = { 0xf0, 0x5e, 0x23, 0x1b, 0x38, 0x94, 0x61, 0x2c,
			    0x49, 0xee, 0x00, 0x0b, 0x80, 0x4e, 0xb2, 0xa9,
			    0xb8, 0x30, 0x6b, 0x50, 0x8f, 0x83, 0x9d, 0x6a,
			    0x55, 0x30, 0x83, 0x1d, 0x93, 0x44, 0xaf, 0x1c },
		.rlen	= 32,
	}, {
	// generated using Crypto++
		.key = {
			0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
			0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
			0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
			0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,
			0x00, 0x00, 0x00, 0x00,
		},
		.klen = 32 + 4,
		.iv = {
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		.input = {
			0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
			0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
			0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
			0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,
			0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27,
			0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f,
			0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
			0x38, 0x39, 0x3a, 0x3b, 0x3c, 0x3d, 0x3e, 0x3f,
			0x40, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47,
			0x48, 0x49, 0x4a, 0x4b, 0x4c, 0x4d, 0x4e, 0x4f,
			0x50, 0x51, 0x52, 0x53, 0x54, 0x55, 0x56, 0x57,
			0x58, 0x59, 0x5a, 0x5b, 0x5c, 0x5d, 0x5e, 0x5f,
			0x60, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67,
			0x68, 0x69, 0x6a, 0x6b, 0x6c, 0x6d, 0x6e, 0x6f,
			0x70, 0x71, 0x72, 0x73, 0x74, 0x75, 0x76, 0x77,
			0x78, 0x79, 0x7a, 0x7b, 0x7c, 0x7d, 0x7e, 0x7f,
			0x80, 0x81, 0x82, 0x83, 0x84, 0x85, 0x86, 0x87,
			0x88, 0x89, 0x8a, 0x8b, 0x8c, 0x8d, 0x8e, 0x8f,
			0x90, 0x91, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97,
			0x98, 0x99, 0x9a, 0x9b, 0x9c, 0x9d, 0x9e, 0x9f,
			0xa0, 0xa1, 0xa2, 0xa3, 0xa4, 0xa5, 0xa6, 0xa7,
			0xa8, 0xa9, 0xaa, 0xab, 0xac, 0xad, 0xae, 0xaf,
			0xb0, 0xb1, 0xb2, 0xb3, 0xb4, 0xb5, 0xb6, 0xb7,
			0xb8, 0xb9, 0xba, 0xbb, 0xbc, 0xbd, 0xbe, 0xbf,
			0xc0, 0xc1, 0xc2, 0xc3, 0xc4, 0xc5, 0xc6, 0xc7,
			0xc8, 0xc9, 0xca, 0xcb, 0xcc, 0xcd, 0xce, 0xcf,
			0xd0, 0xd1, 0xd2, 0xd3, 0xd4, 0xd5, 0xd6, 0xd7,
			0xd8, 0xd9, 0xda, 0xdb, 0xdc, 0xdd, 0xde, 0xdf,
			0xe0, 0xe1, 0xe2, 0xe3, 0xe4, 0xe5, 0xe6, 0xe7,
			0xe8, 0xe9, 0xea, 0xeb, 0xec, 0xed, 0xee, 0xef,
			0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7,
			0xf8, 0xf9, 0xfa, 0xfb, 0xfc, 0xfd, 0xfe, 0xff,
			0x00, 0x03, 0x06, 0x09, 0x0c, 0x0f, 0x12, 0x15,
			0x18, 0x1b, 0x1e, 0x21, 0x24, 0x27, 0x2a, 0x2d,
			0x30, 0x33, 0x36, 0x39, 0x3c, 0x3f, 0x42, 0x45,
			0x48, 0x4b, 0x4e, 0x51, 0x54, 0x57, 0x5a, 0x5d,
			0x60, 0x63, 0x66, 0x69, 0x6c, 0x6f, 0x72, 0x75,
			0x78, 0x7b, 0x7e, 0x81, 0x84, 0x87, 0x8a, 0x8d,
			0x90, 0x93, 0x96, 0x99, 0x9c, 0x9f, 0xa2, 0xa5,
			0xa8, 0xab, 0xae, 0xb1, 0xb4, 0xb7, 0xba, 0xbd,
			0xc0, 0xc3, 0xc6, 0xc9, 0xcc, 0xcf, 0xd2, 0xd5,
			0xd8, 0xdb, 0xde, 0xe1, 0xe4, 0xe7, 0xea, 0xed,
			0xf0, 0xf3, 0xf6, 0xf9, 0xfc, 0xff, 0x02, 0x05,
			0x08, 0x0b, 0x0e, 0x11, 0x14, 0x17, 0x1a, 0x1d,
			0x20, 0x23, 0x26, 0x29, 0x2c, 0x2f, 0x32, 0x35,
			0x38, 0x3b, 0x3e, 0x41, 0x44, 0x47, 0x4a, 0x4d,
			0x50, 0x53, 0x56, 0x59, 0x5c, 0x5f, 0x62, 0x65,
			0x68, 0x6b, 0x6e, 0x71, 0x74, 0x77, 0x7a, 0x7d,
			0x80, 0x83, 0x86, 0x89, 0x8c, 0x8f, 0x92, 0x95,
			0x98, 0x9b, 0x9e, 0xa1, 0xa4, 0xa7, 0xaa, 0xad,
			0xb0, 0xb3, 0xb6, 0xb9, 0xbc, 0xbf, 0xc2, 0xc5,
			0xc8, 0xcb, 0xce, 0xd1, 0xd4, 0xd7, 0xda, 0xdd,
			0xe0, 0xe3, 0xe6, 0xe9, 0xec, 0xef, 0xf2, 0xf5,
			0xf8, 0xfb, 0xfe, 0x01, 0x04, 0x07, 0x0a, 0x0d,
			0x10, 0x13, 0x16, 0x19, 0x1c, 0x1f, 0x22, 0x25,
			0x28, 0x2b, 0x2e, 0x31, 0x34, 0x37, 0x3a, 0x3d,
			0x40, 0x43, 0x46, 0x49, 0x4c, 0x4f, 0x52, 0x55,
			0x58, 0x5b, 0x5e, 0x61, 0x64, 0x67, 0x6a, 0x6d,
			0x70, 0x73, 0x76, 0x79, 0x7c, 0x7f, 0x82, 0x85,
			0x88, 0x8b, 0x8e, 0x91, 0x94, 0x97, 0x9a, 0x9d,
			0xa0, 0xa3, 0xa6, 0xa9, 0xac, 0xaf, 0xb2, 0xb5,
			0xb8, 0xbb, 0xbe, 0xc1, 0xc4, 0xc7, 0xca, 0xcd,
			0xd0, 0xd3, 0xd6, 0xd9, 0xdc, 0xdf, 0xe2, 0xe5,
			0xe8, 0xeb, 0xee, 0xf1, 0xf4, 0xf7, 0xfa, 0xfd,
			0x00, 0x05, 0x0a, 0x0f, 0x14, 0x19, 0x1e, 0x23,
			0x28, 0x2d, 0x32, 0x37, 0x3c, 0x41, 0x46, 0x4b,
			0x50, 0x55, 0x5a, 0x5f, 0x64, 0x69, 0x6e, 0x73,
			0x78, 0x7d, 0x82, 0x87, 0x8c, 0x91, 0x96, 0x9b,
			0xa0, 0xa5, 0xaa, 0xaf, 0xb4, 0xb9, 0xbe, 0xc3,
			0xc8, 0xcd, 0xd2, 0xd7, 0xdc, 0xe1, 0xe6, 0xeb,
			0xf0, 0xf5, 0xfa, 0xff, 0x04, 0x09, 0x0e, 0x13,
			0x18, 0x1d, 0x22, 0x27, 0x2c, 0x31, 0x36, 0x3b,
			0x40, 0x45, 0x4a, 0x4f, 0x54, 0x59, 0x5e, 0x63,
			0x68, 0x6d, 0x72, 0x77, 0x7c, 0x81, 0x86, 0x8b,
			0x90, 0x95, 0x9a, 0x9f, 0xa4, 0xa9, 0xae, 0xb3,
			0xb8, 0xbd, 0xc2, 0xc7, 0xcc, 0xd1, 0xd6, 0xdb,
			0xe0, 0xe5, 0xea, 0xef, 0xf4, 0xf9, 0xfe, 0x03,
			0x08, 0x0d, 0x12, 0x17, 0x1c, 0x21, 0x26, 0x2b,
			0x30, 0x35, 0x3a, 0x3f, 0x44, 0x49, 0x4e, 0x53,
			0x58, 0x5d, 0x62, 0x67, 0x6c, 0x71, 0x76, 0x7b,
			0x80, 0x85, 0x8a, 0x8f, 0x94, 0x99, 0x9e, 0xa3,
			0xa8, 0xad, 0xb2, 0xb7, 0xbc, 0xc1, 0xc6, 0xcb,
			0xd0, 0xd5, 0xda, 0xdf, 0xe4, 0xe9, 0xee, 0xf3,
			0xf8, 0xfd, 0x02, 0x07, 0x0c, 0x11, 0x16, 0x1b,
			0x20, 0x25, 0x2a, 0x2f, 0x34, 0x39, 0x3e, 0x43,
			0x48, 0x4d, 0x52, 0x57, 0x5c, 0x61, 0x66, 0x6b,
			0x70, 0x75, 0x7a, 0x7f, 0x84, 0x89, 0x8e, 0x93,
			0x98, 0x9d, 0xa2, 0xa7, 0xac, 0xb1, 0xb6, 0xbb,
			0xc0, 0xc5, 0xca, 0xcf, 0xd4, 0xd9, 0xde, 0xe3,
			0xe8, 0xed, 0xf2, 0xf7, 0xfc, 0x01, 0x06, 0x0b,
			0x10, 0x15, 0x1a, 0x1f, 0x24, 0x29, 0x2e, 0x33,
			0x38, 0x3d, 0x42, 0x47, 0x4c, 0x51, 0x56, 0x5b,
			0x60, 0x65, 0x6a, 0x6f, 0x74, 0x79, 0x7e, 0x83,
			0x88, 0x8d, 0x92, 0x97, 0x9c, 0xa1, 0xa6, 0xab,
			0xb0, 0xb5, 0xba, 0xbf, 0xc4, 0xc9, 0xce, 0xd3,
			0xd8, 0xdd, 0xe2, 0xe7, 0xec, 0xf1, 0xf6, 0xfb,
			0x00, 0x07, 0x0e, 0x15, 0x1c, 0x23, 0x2a, 0x31,
			0x38, 0x3f, 0x46, 0x4d, 0x54, 0x5b, 0x62, 0x69,
			0x70, 0x77, 0x7e, 0x85, 0x8c, 0x93, 0x9a, 0xa1,
			0xa8, 0xaf, 0xb6, 0xbd, 0xc4, 0xcb, 0xd2, 0xd9,
			0xe0, 0xe7, 0xee, 0xf5, 0xfc, 0x03, 0x0a, 0x11,
			0x18, 0x1f, 0x26, 0x2d, 0x34, 0x3b, 0x42, 0x49,
			0x50, 0x57, 0x5e, 0x65, 0x6c, 0x73, 0x7a, 0x81,
			0x88, 0x8f, 0x96, 0x9d, 0xa4, 0xab, 0xb2, 0xb9,
			0xc0, 0xc7, 0xce, 0xd5, 0xdc, 0xe3, 0xea, 0xf1,
			0xf8, 0xff, 0x06, 0x0d, 0x14, 0x1b, 0x22, 0x29,
			0x30, 0x37, 0x3e, 0x45, 0x4c, 0x53, 0x5a, 0x61,
			0x68, 0x6f, 0x76, 0x7d, 0x84, 0x8b, 0x92, 0x99,
			0xa0, 0xa7, 0xae, 0xb5, 0xbc, 0xc3, 0xca, 0xd1,
			0xd8, 0xdf, 0xe6, 0xed, 0xf4, 0xfb, 0x02, 0x09,
			0x10, 0x17, 0x1e, 0x25, 0x2c, 0x33, 0x3a, 0x41,
			0x48, 0x4f, 0x56, 0x5d, 0x64, 0x6b, 0x72, 0x79,
			0x80, 0x87, 0x8e, 0x95, 0x9c, 0xa3, 0xaa, 0xb1,
			0xb8, 0xbf, 0xc6, 0xcd, 0xd4, 0xdb, 0xe2, 0xe9,
			0xf0, 0xf7, 0xfe, 0x05, 0x0c, 0x13, 0x1a, 0x21,
			0x28, 0x2f, 0x36, 0x3d, 0x44, 0x4b, 0x52, 0x59,
			0x60, 0x67, 0x6e, 0x75, 0x7c, 0x83, 0x8a, 0x91,
			0x98, 0x9f, 0xa6, 0xad, 0xb4, 0xbb, 0xc2, 0xc9,
			0xd0, 0xd7, 0xde, 0xe5, 0xec, 0xf3, 0xfa, 0x01,
			0x08, 0x0f, 0x16, 0x1d, 0x24, 0x2b, 0x32, 0x39,
			0x40, 0x47, 0x4e, 0x55, 0x5c, 0x63, 0x6a, 0x71,
			0x78, 0x7f, 0x86, 0x8d, 0x94, 0x9b, 0xa2, 0xa9,
			0xb0, 0xb7, 0xbe, 0xc5, 0xcc, 0xd3, 0xda, 0xe1,
			0xe8, 0xef, 0xf6, 0xfd, 0x04, 0x0b, 0x12, 0x19,
			0x20, 0x27, 0x2e, 0x35, 0x3c, 0x43, 0x4a, 0x51,
			0x58, 0x5f, 0x66, 0x6d, 0x74, 0x7b, 0x82, 0x89,
			0x90, 0x97, 0x9e, 0xa5, 0xac, 0xb3, 0xba, 0xc1,
			0xc8, 0xcf, 0xd6, 0xdd, 0xe4, 0xeb, 0xf2, 0xf9,
			0x00, 0x09, 0x12, 0x1b, 0x24, 0x2d, 0x36, 0x3f,
			0x48, 0x51, 0x5a, 0x63, 0x6c, 0x75, 0x7e, 0x87,
			0x90, 0x99, 0xa2, 0xab, 0xb4, 0xbd, 0xc6, 0xcf,
			0xd8, 0xe1, 0xea, 0xf3, 0xfc, 0x05, 0x0e, 0x17,
			0x20, 0x29, 0x32, 0x3b, 0x44, 0x4d, 0x56, 0x5f,
			0x68, 0x71, 0x7a, 0x83, 0x8c, 0x95, 0x9e, 0xa7,
			0xb0, 0xb9, 0xc2, 0xcb, 0xd4, 0xdd, 0xe6, 0xef,
			0xf8, 0x01, 0x0a, 0x13, 0x1c, 0x25, 0x2e, 0x37,
			0x40, 0x49, 0x52, 0x5b, 0x64, 0x6d, 0x76, 0x7f,
			0x88, 0x91, 0x9a, 0xa3, 0xac, 0xb5, 0xbe, 0xc7,
			0xd0, 0xd9, 0xe2, 0xeb, 0xf4, 0xfd, 0x06, 0x0f,
			0x18, 0x21, 0x2a, 0x33, 0x3c, 0x45, 0x4e, 0x57,
			0x60, 0x69, 0x72, 0x7b, 0x84, 0x8d, 0x96, 0x9f,
			0xa8, 0xb1, 0xba, 0xc3, 0xcc, 0xd5, 0xde, 0xe7,
			0xf0, 0xf9, 0x02, 0x0b, 0x14, 0x1d, 0x26, 0x2f,
			0x38, 0x41, 0x4a, 0x53, 0x5c, 0x65, 0x6e, 0x77,
			0x80, 0x89, 0x92, 0x9b, 0xa4, 0xad, 0xb6, 0xbf,
			0xc8, 0xd1, 0xda, 0xe3, 0xec, 0xf5, 0xfe, 0x07,
			0x10, 0x19, 0x22, 0x2b, 0x34, 0x3d, 0x46, 0x4f,
			0x58, 0x61, 0x6a, 0x73, 0x7c, 0x85, 0x8e, 0x97,
			0xa0, 0xa9, 0xb2, 0xbb, 0xc4, 0xcd, 0xd6, 0xdf,
			0xe8, 0xf1, 0xfa, 0x03, 0x0c, 0x15, 0x1e, 0x27,
			0x30, 0x39, 0x42, 0x4b, 0x54, 0x5d, 0x66, 0x6f,
			0x78, 0x81, 0x8a, 0x93, 0x9c, 0xa5, 0xae, 0xb7,
			0xc0, 0xc9, 0xd2, 0xdb, 0xe4, 0xed, 0xf6, 0xff,
			0x08, 0x11, 0x1a, 0x23, 0x2c, 0x35, 0x3e, 0x47,
			0x50, 0x59, 0x62, 0x6b, 0x74, 0x7d, 0x86, 0x8f,
			0x98, 0xa1, 0xaa, 0xb3, 0xbc, 0xc5, 0xce, 0xd7,
			0xe0, 0xe9, 0xf2, 0xfb, 0x04, 0x0d, 0x16, 0x1f,
			0x28, 0x31, 0x3a, 0x43, 0x4c, 0x55, 0x5e, 0x67,
			0x70, 0x79, 0x82, 0x8b, 0x94, 0x9d, 0xa6, 0xaf,
			0xb8, 0xc1, 0xca, 0xd3, 0xdc, 0xe5, 0xee, 0xf7,
			0x00, 0x0b, 0x16, 0x21, 0x2c, 0x37, 0x42, 0x4d,
			0x58, 0x63, 0x6e, 0x79, 0x84, 0x8f, 0x9a, 0xa5,
			0xb0, 0xbb, 0xc6, 0xd1, 0xdc, 0xe7, 0xf2, 0xfd,
			0x08, 0x13, 0x1e, 0x29, 0x34, 0x3f, 0x4a, 0x55,
			0x60, 0x6b, 0x76, 0x81, 0x8c, 0x97, 0xa2, 0xad,
			0xb8, 0xc3, 0xce, 0xd9, 0xe4, 0xef, 0xfa, 0x05,
			0x10, 0x1b, 0x26, 0x31, 0x3c, 0x47, 0x52, 0x5d,
			0x68, 0x73, 0x7e, 0x89, 0x94, 0x9f, 0xaa, 0xb5,
			0xc0, 0xcb, 0xd6, 0xe1, 0xec, 0xf7, 0x02, 0x0d,
			0x18, 0x23, 0x2e, 0x39, 0x44, 0x4f, 0x5a, 0x65,
			0x70, 0x7b, 0x86, 0x91, 0x9c, 0xa7, 0xb2, 0xbd,
			0xc8, 0xd3, 0xde, 0xe9, 0xf4, 0xff, 0x0a, 0x15,
			0x20, 0x2b, 0x36, 0x41, 0x4c, 0x57, 0x62, 0x6d,
			0x78, 0x83, 0x8e, 0x99, 0xa4, 0xaf, 0xba, 0xc5,
			0xd0, 0xdb, 0xe6, 0xf1, 0xfc, 0x07, 0x12, 0x1d,
			0x28, 0x33, 0x3e, 0x49, 0x54, 0x5f, 0x6a, 0x75,
			0x80, 0x8b, 0x96, 0xa1, 0xac, 0xb7, 0xc2, 0xcd,
			0xd8, 0xe3, 0xee, 0xf9, 0x04, 0x0f, 0x1a, 0x25,
			0x30, 0x3b, 0x46, 0x51, 0x5c, 0x67, 0x72, 0x7d,
			0x88, 0x93, 0x9e, 0xa9, 0xb4, 0xbf, 0xca, 0xd5,
			0xe0, 0xeb, 0xf6, 0x01, 0x0c, 0x17, 0x22, 0x2d,
			0x38, 0x43, 0x4e, 0x59, 0x64, 0x6f, 0x7a, 0x85,
			0x90, 0x9b, 0xa6, 0xb1, 0xbc, 0xc7, 0xd2, 0xdd,
			0xe8, 0xf3, 0xfe, 0x09, 0x14, 0x1f, 0x2a, 0x35,
			0x40, 0x4b, 0x56, 0x61, 0x6c, 0x77, 0x82, 0x8d,
			0x98, 0xa3, 0xae, 0xb9, 0xc4, 0xcf, 0xda, 0xe5,
			0xf0, 0xfb, 0x06, 0x11, 0x1c, 0x27, 0x32, 0x3d,
			0x48, 0x53, 0x5e, 0x69, 0x74, 0x7f, 0x8a, 0x95,
			0xa0, 0xab, 0xb6, 0xc1, 0xcc, 0xd7, 0xe2, 0xed,
			0xf8, 0x03, 0x0e, 0x19, 0x24, 0x2f, 0x3a, 0x45,
			0x50, 0x5b, 0x66, 0x71, 0x7c, 0x87, 0x92, 0x9d,
			0xa8, 0xb3, 0xbe, 0xc9, 0xd4, 0xdf, 0xea, 0xf5,
			0x00, 0x0d, 0x1a, 0x27, 0x34, 0x41, 0x4e, 0x5b,
			0x68, 0x75, 0x82, 0x8f, 0x9c, 0xa9, 0xb6, 0xc3,
			0xd0, 0xdd, 0xea, 0xf7, 0x04, 0x11, 0x1e, 0x2b,
			0x38, 0x45, 0x52, 0x5f, 0x6c, 0x79, 0x86, 0x93,
			0xa0, 0xad, 0xba, 0xc7, 0xd4, 0xe1, 0xee, 0xfb,
			0x08, 0x15, 0x22, 0x2f, 0x3c, 0x49, 0x56, 0x63,
			0x70, 0x7d, 0x8a, 0x97, 0xa4, 0xb1, 0xbe, 0xcb,
			0xd8, 0xe5, 0xf2, 0xff, 0x0c, 0x19, 0x26, 0x33,
			0x40, 0x4d, 0x5a, 0x67, 0x74, 0x81, 0x8e, 0x9b,
			0xa8, 0xb5, 0xc2, 0xcf, 0xdc, 0xe9, 0xf6, 0x03,
			0x10, 0x1d, 0x2a, 0x37, 0x44, 0x51, 0x5e, 0x6b,
			0x78, 0x85, 0x92, 0x9f, 0xac, 0xb9, 0xc6, 0xd3,
			0xe0, 0xed, 0xfa, 0x07, 0x14, 0x21, 0x2e, 0x3b,
			0x48, 0x55, 0x62, 0x6f, 0x7c, 0x89, 0x96, 0xa3,
			0xb0, 0xbd, 0xca, 0xd7, 0xe4, 0xf1, 0xfe, 0x0b,
			0x18, 0x25, 0x32, 0x3f, 0x4c, 0x59, 0x66, 0x73,
			0x80, 0x8d, 0x9a, 0xa7, 0xb4, 0xc1, 0xce, 0xdb,
			0xe8, 0xf5, 0x02, 0x0f, 0x1c, 0x29, 0x36, 0x43,
			0x50, 0x5d, 0x6a, 0x77, 0x84, 0x91, 0x9e, 0xab,
			0xb8, 0xc5, 0xd2, 0xdf, 0xec, 0xf9, 0x06, 0x13,
			0x20, 0x2d, 0x3a, 0x47, 0x54, 0x61, 0x6e, 0x7b,
			0x88, 0x95, 0xa2, 0xaf, 0xbc, 0xc9, 0xd6, 0xe3,
			0xf0, 0xfd, 0x0a, 0x17, 0x24, 0x31, 0x3e, 0x4b,
			0x58, 0x65, 0x72, 0x7f, 0x8c, 0x99, 0xa6, 0xb3,
			0xc0, 0xcd, 0xda, 0xe7, 0xf4, 0x01, 0x0e, 0x1b,
			0x28, 0x35, 0x42, 0x4f, 0x5c, 0x69, 0x76, 0x83,
			0x90, 0x9d, 0xaa, 0xb7, 0xc4, 0xd1, 0xde, 0xeb,
			0xf8, 0x05, 0x12, 0x1f, 0x2c, 0x39, 0x46, 0x53,
			0x60, 0x6d, 0x7a, 0x87, 0x94, 0xa1, 0xae, 0xbb,
			0xc8, 0xd5, 0xe2, 0xef, 0xfc, 0x09, 0x16, 0x23,
			0x30, 0x3d, 0x4a, 0x57, 0x64, 0x71, 0x7e, 0x8b,
			0x98, 0xa5, 0xb2, 0xbf, 0xcc, 0xd9, 0xe6, 0xf3,
			0x00, 0x0f, 0x1e, 0x2d, 0x3c, 0x4b, 0x5a, 0x69,
			0x78, 0x87, 0x96, 0xa5, 0xb4, 0xc3, 0xd2, 0xe1,
			0xf0, 0xff, 0x0e, 0x1d, 0x2c, 0x3b, 0x4a, 0x59,
			0x68, 0x77, 0x86, 0x95, 0xa4, 0xb3, 0xc2, 0xd1,
			0xe0, 0xef, 0xfe, 0x0d, 0x1c, 0x2b, 0x3a, 0x49,
			0x58, 0x67, 0x76, 0x85, 0x94, 0xa3, 0xb2, 0xc1,
			0xd0, 0xdf, 0xee, 0xfd, 0x0c, 0x1b, 0x2a, 0x39,
			0x48, 0x57, 0x66, 0x75, 0x84, 0x93, 0xa2, 0xb1,
			0xc0, 0xcf, 0xde, 0xed, 0xfc, 0x0b, 0x1a, 0x29,
			0x38, 0x47, 0x56, 0x65, 0x74, 0x83, 0x92, 0xa1,
			0xb0, 0xbf, 0xce, 0xdd, 0xec, 0xfb, 0x0a, 0x19,
			0x28, 0x37, 0x46, 0x55, 0x64, 0x73, 0x82, 0x91,
			0xa0, 0xaf, 0xbe, 0xcd, 0xdc, 0xeb, 0xfa, 0x09,
			0x18, 0x27, 0x36, 0x45, 0x54, 0x63, 0x72, 0x81,
			0x90, 0x9f, 0xae, 0xbd, 0xcc, 0xdb, 0xea, 0xf9,
			0x08, 0x17, 0x26, 0x35, 0x44, 0x53, 0x62, 0x71,
			0x80, 0x8f, 0x9e, 0xad, 0xbc, 0xcb, 0xda, 0xe9,
			0xf8, 0x07, 0x16, 0x25, 0x34, 0x43, 0x52, 0x61,
			0x70, 0x7f, 0x8e, 0x9d, 0xac, 0xbb, 0xca, 0xd9,
			0xe8, 0xf7, 0x06, 0x15, 0x24, 0x33, 0x42, 0x51,
			0x60, 0x6f, 0x7e, 0x8d, 0x9c, 0xab, 0xba, 0xc9,
			0xd8, 0xe7, 0xf6, 0x05, 0x14, 0x23, 0x32, 0x41,
			0x50, 0x5f, 0x6e, 0x7d, 0x8c, 0x9b, 0xaa, 0xb9,
			0xc8, 0xd7, 0xe6, 0xf5, 0x04, 0x13, 0x22, 0x31,
			0x40, 0x4f, 0x5e, 0x6d, 0x7c, 0x8b, 0x9a, 0xa9,
			0xb8, 0xc7, 0xd6, 0xe5, 0xf4, 0x03, 0x12, 0x21,
			0x30, 0x3f, 0x4e, 0x5d, 0x6c, 0x7b, 0x8a, 0x99,
			0xa8, 0xb7, 0xc6, 0xd5, 0xe4, 0xf3, 0x02, 0x11,
			0x20, 0x2f, 0x3e, 0x4d, 0x5c, 0x6b, 0x7a, 0x89,
			0x98, 0xa7, 0xb6, 0xc5, 0xd4, 0xe3, 0xf2, 0x01,
			0x10, 0x1f, 0x2e, 0x3d, 0x4c, 0x5b, 0x6a, 0x79,
			0x88, 0x97, 0xa6, 0xb5, 0xc4, 0xd3, 0xe2, 0xf1,
			0x00, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77,
			0x88, 0x99, 0xaa, 0xbb, 0xcc, 0xdd, 0xee, 0xff,
			0x10, 0x21, 0x32, 0x43, 0x54, 0x65, 0x76, 0x87,
			0x98, 0xa9, 0xba, 0xcb, 0xdc, 0xed, 0xfe, 0x0f,
			0x20, 0x31, 0x42, 0x53, 0x64, 0x75, 0x86, 0x97,
			0xa8, 0xb9, 0xca, 0xdb, 0xec, 0xfd, 0x0e, 0x1f,
			0x30, 0x41, 0x52, 0x63, 0x74, 0x85, 0x96, 0xa7,
			0xb8, 0xc9, 0xda, 0xeb, 0xfc, 0x0d, 0x1e, 0x2f,
			0x40, 0x51, 0x62, 0x73, 0x84, 0x95, 0xa6, 0xb7,
			0xc8, 0xd9, 0xea, 0xfb, 0x0c, 0x1d, 0x2e, 0x3f,
			0x50, 0x61, 0x72, 0x83, 0x94, 0xa5, 0xb6, 0xc7,
			0xd8, 0xe9, 0xfa, 0x0b, 0x1c, 0x2d, 0x3e, 0x4f,
			0x60, 0x71, 0x82, 0x93, 0xa4, 0xb5, 0xc6, 0xd7,
			0xe8, 0xf9, 0x0a, 0x1b, 0x2c, 0x3d, 0x4e, 0x5f,
			0x70, 0x81, 0x92, 0xa3, 0xb4, 0xc5, 0xd6, 0xe7,
			0xf8, 0x09, 0x1a, 0x2b, 0x3c, 0x4d, 0x5e, 0x6f,
			0x80, 0x91, 0xa2, 0xb3, 0xc4, 0xd5, 0xe6, 0xf7,
			0x08, 0x19, 0x2a, 0x3b, 0x4c, 0x5d, 0x6e, 0x7f,
			0x90, 0xa1, 0xb2, 0xc3, 0xd4, 0xe5, 0xf6, 0x07,
			0x18, 0x29, 0x3a, 0x4b, 0x5c, 0x6d, 0x7e, 0x8f,
			0xa0, 0xb1, 0xc2, 0xd3, 0xe4, 0xf5, 0x06, 0x17,
			0x28, 0x39, 0x4a, 0x5b, 0x6c, 0x7d, 0x8e, 0x9f,
			0xb0, 0xc1, 0xd2, 0xe3, 0xf4, 0x05, 0x16, 0x27,
			0x38, 0x49, 0x5a, 0x6b, 0x7c, 0x8d, 0x9e, 0xaf,
			0xc0, 0xd1, 0xe2, 0xf3, 0x04, 0x15, 0x26, 0x37,
			0x48, 0x59, 0x6a, 0x7b, 0x8c, 0x9d, 0xae, 0xbf,
			0xd0, 0xe1, 0xf2, 0x03, 0x14, 0x25, 0x36, 0x47,
			0x58, 0x69, 0x7a, 0x8b, 0x9c, 0xad, 0xbe, 0xcf,
			0xe0, 0xf1, 0x02, 0x13, 0x24, 0x35, 0x46, 0x57,
			0x68, 0x79, 0x8a, 0x9b, 0xac, 0xbd, 0xce, 0xdf,
			0xf0, 0x01, 0x12, 0x23, 0x34, 0x45, 0x56, 0x67,
			0x78, 0x89, 0x9a, 0xab, 0xbc, 0xcd, 0xde, 0xef,
			0x00, 0x13, 0x26, 0x39, 0x4c, 0x5f, 0x72, 0x85,
			0x98, 0xab, 0xbe, 0xd1, 0xe4, 0xf7, 0x0a, 0x1d,
			0x30, 0x43, 0x56, 0x69, 0x7c, 0x8f, 0xa2, 0xb5,
			0xc8, 0xdb, 0xee, 0x01, 0x14, 0x27, 0x3a, 0x4d,
			0x60, 0x73, 0x86, 0x99, 0xac, 0xbf, 0xd2, 0xe5,
			0xf8, 0x0b, 0x1e, 0x31, 0x44, 0x57, 0x6a, 0x7d,
			0x90, 0xa3, 0xb6, 0xc9, 0xdc, 0xef, 0x02, 0x15,
			0x28, 0x3b, 0x4e, 0x61, 0x74, 0x87, 0x9a, 0xad,
			0xc0, 0xd3, 0xe6, 0xf9, 0x0c, 0x1f, 0x32, 0x45,
			0x58, 0x6b, 0x7e, 0x91, 0xa4, 0xb7, 0xca, 0xdd,
			0xf0, 0x03, 0x16, 0x29, 0x3c, 0x4f, 0x62, 0x75,
			0x88, 0x9b, 0xae, 0xc1, 0xd4, 0xe7, 0xfa, 0x0d,
			0x20, 0x33, 0x46, 0x59, 0x6c, 0x7f, 0x92, 0xa5,
			0xb8, 0xcb, 0xde, 0xf1, 0x04, 0x17, 0x2a, 0x3d,
			0x50, 0x63, 0x76, 0x89, 0x9c, 0xaf, 0xc2, 0xd5,
			0xe8, 0xfb, 0x0e, 0x21, 0x34, 0x47, 0x5a, 0x6d,
			0x80, 0x93, 0xa6, 0xb9, 0xcc, 0xdf, 0xf2, 0x05,
			0x18, 0x2b, 0x3e, 0x51, 0x64, 0x77, 0x8a, 0x9d,
			0xb0, 0xc3, 0xd6, 0xe9, 0xfc, 0x0f, 0x22, 0x35,
			0x48, 0x5b, 0x6e, 0x81, 0x94, 0xa7, 0xba, 0xcd,
			0xe0, 0xf3, 0x06, 0x19, 0x2c, 0x3f, 0x52, 0x65,
			0x78, 0x8b, 0x9e, 0xb1, 0xc4, 0xd7, 0xea, 0xfd,
			0x10, 0x23, 0x36, 0x49, 0x5c, 0x6f, 0x82, 0x95,
			0xa8, 0xbb, 0xce, 0xe1, 0xf4, 0x07, 0x1a, 0x2d,
			0x40, 0x53, 0x66, 0x79, 0x8c, 0x9f, 0xb2, 0xc5,
			0xd8, 0xeb, 0xfe, 0x11, 0x24, 0x37, 0x4a, 0x5d,
			0x70, 0x83, 0x96, 0xa9, 0xbc, 0xcf, 0xe2, 0xf5,
			0x08, 0x1b, 0x2e, 0x41, 0x54, 0x67, 0x7a, 0x8d,
			0xa0, 0xb3, 0xc6, 0xd9, 0xec, 0xff, 0x12, 0x25,
			0x38, 0x4b, 0x5e, 0x71, 0x84, 0x97, 0xaa, 0xbd,
			0xd0, 0xe3, 0xf6, 0x09, 0x1c, 0x2f, 0x42, 0x55,
			0x68, 0x7b, 0x8e, 0xa1, 0xb4, 0xc7, 0xda, 0xed,
			0x00, 0x15, 0x2a, 0x3f, 0x54, 0x69, 0x7e, 0x93,
			0xa8, 0xbd, 0xd2, 0xe7, 0xfc, 0x11, 0x26, 0x3b,
			0x50, 0x65, 0x7a, 0x8f, 0xa4, 0xb9, 0xce, 0xe3,
			0xf8, 0x0d, 0x22, 0x37, 0x4c, 0x61, 0x76, 0x8b,
			0xa0, 0xb5, 0xca, 0xdf, 0xf4, 0x09, 0x1e, 0x33,
			0x48, 0x5d, 0x72, 0x87, 0x9c, 0xb1, 0xc6, 0xdb,
			0xf0, 0x05, 0x1a, 0x2f, 0x44, 0x59, 0x6e, 0x83,
			0x98, 0xad, 0xc2, 0xd7, 0xec, 0x01, 0x16, 0x2b,
			0x40, 0x55, 0x6a, 0x7f, 0x94, 0xa9, 0xbe, 0xd3,
			0xe8, 0xfd, 0x12, 0x27, 0x3c, 0x51, 0x66, 0x7b,
			0x90, 0xa5, 0xba, 0xcf, 0xe4, 0xf9, 0x0e, 0x23,
			0x38, 0x4d, 0x62, 0x77, 0x8c, 0xa1, 0xb6, 0xcb,
			0xe0, 0xf5, 0x0a, 0x1f, 0x34, 0x49, 0x5e, 0x73,
			0x88, 0x9d, 0xb2, 0xc7, 0xdc, 0xf1, 0x06, 0x1b,
			0x30, 0x45, 0x5a, 0x6f, 0x84, 0x99, 0xae, 0xc3,
			0xd8, 0xed, 0x02, 0x17, 0x2c, 0x41, 0x56, 0x6b,
			0x80, 0x95, 0xaa, 0xbf, 0xd4, 0xe9, 0xfe, 0x13,
			0x28, 0x3d, 0x52, 0x67, 0x7c, 0x91, 0xa6, 0xbb,
			0xd0, 0xe5, 0xfa, 0x0f, 0x24, 0x39, 0x4e, 0x63,
			0x78, 0x8d, 0xa2, 0xb7, 0xcc, 0xe1, 0xf6, 0x0b,
			0x20, 0x35, 0x4a, 0x5f, 0x74, 0x89, 0x9e, 0xb3,
			0xc8, 0xdd, 0xf2, 0x07, 0x1c, 0x31, 0x46, 0x5b,
			0x70, 0x85, 0x9a, 0xaf, 0xc4, 0xd9, 0xee, 0x03,
			0x18, 0x2d, 0x42, 0x57, 0x6c, 0x81, 0x96, 0xab,
			0xc0, 0xd5, 0xea, 0xff, 0x14, 0x29, 0x3e, 0x53,
			0x68, 0x7d, 0x92, 0xa7, 0xbc, 0xd1, 0xe6, 0xfb,
			0x10, 0x25, 0x3a, 0x4f, 0x64, 0x79, 0x8e, 0xa3,
			0xb8, 0xcd, 0xe2, 0xf7, 0x0c, 0x21, 0x36, 0x4b,
			0x60, 0x75, 0x8a, 0x9f, 0xb4, 0xc9, 0xde, 0xf3,
			0x08, 0x1d, 0x32, 0x47, 0x5c, 0x71, 0x86, 0x9b,
			0xb0, 0xc5, 0xda, 0xef, 0x04, 0x19, 0x2e, 0x43,
			0x58, 0x6d, 0x82, 0x97, 0xac, 0xc1, 0xd6, 0xeb,
			0x00, 0x17, 0x2e, 0x45, 0x5c, 0x73, 0x8a, 0xa1,
			0xb8, 0xcf, 0xe6, 0xfd, 0x14, 0x2b, 0x42, 0x59,
			0x70, 0x87, 0x9e, 0xb5, 0xcc, 0xe3, 0xfa, 0x11,
			0x28, 0x3f, 0x56, 0x6d, 0x84, 0x9b, 0xb2, 0xc9,
			0xe0, 0xf7, 0x0e, 0x25, 0x3c, 0x53, 0x6a, 0x81,
			0x98, 0xaf, 0xc6, 0xdd, 0xf4, 0x0b, 0x22, 0x39,
			0x50, 0x67, 0x7e, 0x95, 0xac, 0xc3, 0xda, 0xf1,
			0x08, 0x1f, 0x36, 0x4d, 0x64, 0x7b, 0x92, 0xa9,
			0xc0, 0xd7, 0xee, 0x05, 0x1c, 0x33, 0x4a, 0x61,
			0x78, 0x8f, 0xa6, 0xbd, 0xd4, 0xeb, 0x02, 0x19,
			0x30, 0x47, 0x5e, 0x75, 0x8c, 0xa3, 0xba, 0xd1,
			0xe8, 0xff, 0x16, 0x2d, 0x44, 0x5b, 0x72, 0x89,
			0xa0, 0xb7, 0xce, 0xe5, 0xfc, 0x13, 0x2a, 0x41,
			0x58, 0x6f, 0x86, 0x9d, 0xb4, 0xcb, 0xe2, 0xf9,
			0x10, 0x27, 0x3e, 0x55, 0x6c, 0x83, 0x9a, 0xb1,
			0xc8, 0xdf, 0xf6, 0x0d, 0x24, 0x3b, 0x52, 0x69,
			0x80, 0x97, 0xae, 0xc5, 0xdc, 0xf3, 0x0a, 0x21,
			0x38, 0x4f, 0x66, 0x7d, 0x94, 0xab, 0xc2, 0xd9,
			0xf0, 0x07, 0x1e, 0x35, 0x4c, 0x63, 0x7a, 0x91,
			0xa8, 0xbf, 0xd6, 0xed, 0x04, 0x1b, 0x32, 0x49,
			0x60, 0x77, 0x8e, 0xa5, 0xbc, 0xd3, 0xea, 0x01,
			0x18, 0x2f, 0x46, 0x5d, 0x74, 0x8b, 0xa2, 0xb9,
			0xd0, 0xe7, 0xfe, 0x15, 0x2c, 0x43, 0x5a, 0x71,
			0x88, 0x9f, 0xb6, 0xcd, 0xe4, 0xfb, 0x12, 0x29,
			0x40, 0x57, 0x6e, 0x85, 0x9c, 0xb3, 0xca, 0xe1,
			0xf8, 0x0f, 0x26, 0x3d, 0x54, 0x6b, 0x82, 0x99,
			0xb0, 0xc7, 0xde, 0xf5, 0x0c, 0x23, 0x3a, 0x51,
			0x68, 0x7f, 0x96, 0xad, 0xc4, 0xdb, 0xf2, 0x09,
			0x20, 0x37, 0x4e, 0x65, 0x7c, 0x93, 0xaa, 0xc1,
			0xd8, 0xef, 0x06, 0x1d, 0x34, 0x4b, 0x62, 0x79,
			0x90, 0xa7, 0xbe, 0xd5, 0xec, 0x03, 0x1a, 0x31,
			0x48, 0x5f, 0x76, 0x8d, 0xa4, 0xbb, 0xd2, 0xe9,
			0x00, 0x19, 0x32, 0x4b, 0x64, 0x7d, 0x96, 0xaf,
			0xc8, 0xe1, 0xfa, 0x13, 0x2c, 0x45, 0x5e, 0x77,
			0x90, 0xa9, 0xc2, 0xdb, 0xf4, 0x0d, 0x26, 0x3f,
			0x58, 0x71, 0x8a, 0xa3, 0xbc, 0xd5, 0xee, 0x07,
			0x20, 0x39, 0x52, 0x6b, 0x84, 0x9d, 0xb6, 0xcf,
			0xe8, 0x01, 0x1a, 0x33, 0x4c, 0x65, 0x7e, 0x97,
			0xb0, 0xc9, 0xe2, 0xfb, 0x14, 0x2d, 0x46, 0x5f,
			0x78, 0x91, 0xaa, 0xc3, 0xdc, 0xf5, 0x0e, 0x27,
			0x40, 0x59, 0x72, 0x8b, 0xa4, 0xbd, 0xd6, 0xef,
			0x08, 0x21, 0x3a, 0x53, 0x6c, 0x85, 0x9e, 0xb7,
			0xd0, 0xe9, 0x02, 0x1b, 0x34, 0x4d, 0x66, 0x7f,
			0x98, 0xb1, 0xca, 0xe3, 0xfc, 0x15, 0x2e, 0x47,
			0x60, 0x79, 0x92, 0xab, 0xc4, 0xdd, 0xf6, 0x0f,
			0x28, 0x41, 0x5a, 0x73, 0x8c, 0xa5, 0xbe, 0xd7,
			0xf0, 0x09, 0x22, 0x3b, 0x54, 0x6d, 0x86, 0x9f,
			0xb8, 0xd1, 0xea, 0x03, 0x1c, 0x35, 0x4e, 0x67,
			0x80, 0x99, 0xb2, 0xcb, 0xe4, 0xfd, 0x16, 0x2f,
			0x48, 0x61, 0x7a, 0x93, 0xac, 0xc5, 0xde, 0xf7,
			0x10, 0x29, 0x42, 0x5b, 0x74, 0x8d, 0xa6, 0xbf,
			0xd8, 0xf1, 0x0a, 0x23, 0x3c, 0x55, 0x6e, 0x87,
			0xa0, 0xb9, 0xd2, 0xeb, 0x04, 0x1d, 0x36, 0x4f,
			0x68, 0x81, 0x9a, 0xb3, 0xcc, 0xe5, 0xfe, 0x17,
			0x30, 0x49, 0x62, 0x7b, 0x94, 0xad, 0xc6, 0xdf,
			0xf8, 0x11, 0x2a, 0x43, 0x5c, 0x75, 0x8e, 0xa7,
			0xc0, 0xd9, 0xf2, 0x0b, 0x24, 0x3d, 0x56, 0x6f,
			0x88, 0xa1, 0xba, 0xd3, 0xec, 0x05, 0x1e, 0x37,
			0x50, 0x69, 0x82, 0x9b, 0xb4, 0xcd, 0xe6, 0xff,
			0x18, 0x31, 0x4a, 0x63, 0x7c, 0x95, 0xae, 0xc7,
			0xe0, 0xf9, 0x12, 0x2b, 0x44, 0x5d, 0x76, 0x8f,
			0xa8, 0xc1, 0xda, 0xf3, 0x0c, 0x25, 0x3e, 0x57,
			0x70, 0x89, 0xa2, 0xbb, 0xd4, 0xed, 0x06, 0x1f,
			0x38, 0x51, 0x6a, 0x83, 0x9c, 0xb5, 0xce, 0xe7,
			0x00, 0x1b, 0x36, 0x51, 0x6c, 0x87, 0xa2, 0xbd,
			0xd8, 0xf3, 0x0e, 0x29, 0x44, 0x5f, 0x7a, 0x95,
			0xb0, 0xcb, 0xe6, 0x01, 0x1c, 0x37, 0x52, 0x6d,
			0x88, 0xa3, 0xbe, 0xd9, 0xf4, 0x0f, 0x2a, 0x45,
			0x60, 0x7b, 0x96, 0xb1, 0xcc, 0xe7, 0x02, 0x1d,
			0x38, 0x53, 0x6e, 0x89, 0xa4, 0xbf, 0xda, 0xf5,
			0x10, 0x2b, 0x46, 0x61, 0x7c, 0x97, 0xb2, 0xcd,
			0xe8, 0x03, 0x1e, 0x39, 0x54, 0x6f, 0x8a, 0xa5,
			0xc0, 0xdb, 0xf6, 0x11, 0x2c, 0x47, 0x62, 0x7d,
			0x98, 0xb3, 0xce, 0xe9, 0x04, 0x1f, 0x3a, 0x55,
			0x70, 0x8b, 0xa6, 0xc1, 0xdc, 0xf7, 0x12, 0x2d,
			0x48, 0x63, 0x7e, 0x99, 0xb4, 0xcf, 0xea, 0x05,
			0x20, 0x3b, 0x56, 0x71, 0x8c, 0xa7, 0xc2, 0xdd,
			0xf8, 0x13, 0x2e, 0x49, 0x64, 0x7f, 0x9a, 0xb5,
			0xd0, 0xeb, 0x06, 0x21, 0x3c, 0x57, 0x72, 0x8d,
			0xa8, 0xc3, 0xde, 0xf9, 0x14, 0x2f, 0x4a, 0x65,
			0x80, 0x9b, 0xb6, 0xd1, 0xec, 0x07, 0x22, 0x3d,
			0x58, 0x73, 0x8e, 0xa9, 0xc4, 0xdf, 0xfa, 0x15,
			0x30, 0x4b, 0x66, 0x81, 0x9c, 0xb7, 0xd2, 0xed,
			0x08, 0x23, 0x3e, 0x59, 0x74, 0x8f, 0xaa, 0xc5,
			0xe0, 0xfb, 0x16, 0x31, 0x4c, 0x67, 0x82, 0x9d,
			0xb8, 0xd3, 0xee, 0x09, 0x24, 0x3f, 0x5a, 0x75,
			0x90, 0xab, 0xc6, 0xe1, 0xfc, 0x17, 0x32, 0x4d,
			0x68, 0x83, 0x9e, 0xb9, 0xd4, 0xef, 0x0a, 0x25,
			0x40, 0x5b, 0x76, 0x91, 0xac, 0xc7, 0xe2, 0xfd,
			0x18, 0x33, 0x4e, 0x69, 0x84, 0x9f, 0xba, 0xd5,
			0xf0, 0x0b, 0x26, 0x41, 0x5c, 0x77, 0x92, 0xad,
			0xc8, 0xe3, 0xfe, 0x19, 0x34, 0x4f, 0x6a, 0x85,
			0xa0, 0xbb, 0xd6, 0xf1, 0x0c, 0x27, 0x42, 0x5d,
			0x78, 0x93, 0xae, 0xc9, 0xe4, 0xff, 0x1a, 0x35,
			0x50, 0x6b, 0x86, 0xa1, 0xbc, 0xd7, 0xf2, 0x0d,
			0x28, 0x43, 0x5e, 0x79, 0x94, 0xaf, 0xca, 0xe5,
			0x00, 0x1d, 0x3a, 0x57, 0x74, 0x91, 0xae, 0xcb,
			0xe8, 0x05, 0x22, 0x3f, 0x5c, 0x79, 0x96, 0xb3,
			0xd0, 0xed, 0x0a, 0x27, 0x44, 0x61, 0x7e, 0x9b,
			0xb8, 0xd5, 0xf2, 0x0f, 0x2c, 0x49, 0x66, 0x83,
			0xa0, 0xbd, 0xda, 0xf7, 0x14, 0x31, 0x4e, 0x6b,
			0x88, 0xa5, 0xc2, 0xdf, 0xfc, 0x19, 0x36, 0x53,
			0x70, 0x8d, 0xaa, 0xc7, 0xe4, 0x01, 0x1e, 0x3b,
			0x58, 0x75, 0x92, 0xaf, 0xcc, 0xe9, 0x06, 0x23,
			0x40, 0x5d, 0x7a, 0x97, 0xb4, 0xd1, 0xee, 0x0b,
			0x28, 0x45, 0x62, 0x7f, 0x9c, 0xb9, 0xd6, 0xf3,
			0x10, 0x2d, 0x4a, 0x67, 0x84, 0xa1, 0xbe, 0xdb,
			0xf8, 0x15, 0x32, 0x4f, 0x6c, 0x89, 0xa6, 0xc3,
			0xe0, 0xfd, 0x1a, 0x37, 0x54, 0x71, 0x8e, 0xab,
			0xc8, 0xe5, 0x02, 0x1f, 0x3c, 0x59, 0x76, 0x93,
			0xb0, 0xcd, 0xea, 0x07, 0x24, 0x41, 0x5e, 0x7b,
			0x98, 0xb5, 0xd2, 0xef, 0x0c, 0x29, 0x46, 0x63,
			0x80, 0x9d, 0xba, 0xd7, 0xf4, 0x11, 0x2e, 0x4b,
			0x68, 0x85, 0xa2, 0xbf, 0xdc, 0xf9, 0x16, 0x33,
			0x50, 0x6d, 0x8a, 0xa7, 0xc4, 0xe1, 0xfe, 0x1b,
			0x38, 0x55, 0x72, 0x8f, 0xac, 0xc9, 0xe6, 0x03,
			0x20, 0x3d, 0x5a, 0x77, 0x94, 0xb1, 0xce, 0xeb,
			0x08, 0x25, 0x42, 0x5f, 0x7c, 0x99, 0xb6, 0xd3,
			0xf0, 0x0d, 0x2a, 0x47, 0x64, 0x81, 0x9e, 0xbb,
			0xd8, 0xf5, 0x12, 0x2f, 0x4c, 0x69, 0x86, 0xa3,
			0xc0, 0xdd, 0xfa, 0x17, 0x34, 0x51, 0x6e, 0x8b,
			0xa8, 0xc5, 0xe2, 0xff, 0x1c, 0x39, 0x56, 0x73,
			0x90, 0xad, 0xca, 0xe7, 0x04, 0x21, 0x3e, 0x5b,
			0x78, 0x95, 0xb2, 0xcf, 0xec, 0x09, 0x26, 0x43,
			0x60, 0x7d, 0x9a, 0xb7, 0xd4, 0xf1, 0x0e, 0x2b,
			0x48, 0x65, 0x82, 0x9f, 0xbc, 0xd9, 0xf6, 0x13,
			0x30, 0x4d, 0x6a, 0x87, 0xa4, 0xc1, 0xde, 0xfb,
			0x18, 0x35, 0x52, 0x6f, 0x8c, 0xa9, 0xc6, 0xe3,
			0x00, 0x1f, 0x3e, 0x5d, 0x7c, 0x9b, 0xba, 0xd9,
			0xf8, 0x17, 0x36, 0x55, 0x74, 0x93, 0xb2, 0xd1,
			0xf0, 0x0f, 0x2e, 0x4d, 0x6c, 0x8b, 0xaa, 0xc9,
			0xe8, 0x07, 0x26, 0x45, 0x64, 0x83, 0xa2, 0xc1,
			0xe0, 0xff, 0x1e, 0x3d, 0x5c, 0x7b, 0x9a, 0xb9,
			0xd8, 0xf7, 0x16, 0x35, 0x54, 0x73, 0x92, 0xb1,
			0xd0, 0xef, 0x0e, 0x2d, 0x4c, 0x6b, 0x8a, 0xa9,
			0xc8, 0xe7, 0x06, 0x25, 0x44, 0x63, 0x82, 0xa1,
			0xc0, 0xdf, 0xfe, 0x1d, 0x3c, 0x5b, 0x7a, 0x99,
			0xb8, 0xd7, 0xf6, 0x15, 0x34, 0x53, 0x72, 0x91,
			0xb0, 0xcf, 0xee, 0x0d, 0x2c, 0x4b, 0x6a, 0x89,
			0xa8, 0xc7, 0xe6, 0x05, 0x24, 0x43, 0x62, 0x81,
			0xa0, 0xbf, 0xde, 0xfd, 0x1c, 0x3b, 0x5a, 0x79,
			0x98, 0xb7, 0xd6, 0xf5, 0x14, 0x33, 0x52, 0x71,
			0x90, 0xaf, 0xce, 0xed, 0x0c, 0x2b, 0x4a, 0x69,
			0x88, 0xa7, 0xc6, 0xe5, 0x04, 0x23, 0x42, 0x61,
			0x80, 0x9f, 0xbe, 0xdd, 0xfc, 0x1b, 0x3a, 0x59,
			0x78, 0x97, 0xb6, 0xd5, 0xf4, 0x13, 0x32, 0x51,
			0x70, 0x8f, 0xae, 0xcd, 0xec, 0x0b, 0x2a, 0x49,
			0x68, 0x87, 0xa6, 0xc5, 0xe4, 0x03, 0x22, 0x41,
			0x60, 0x7f, 0x9e, 0xbd, 0xdc, 0xfb, 0x1a, 0x39,
			0x58, 0x77, 0x96, 0xb5, 0xd4, 0xf3, 0x12, 0x31,
			0x50, 0x6f, 0x8e, 0xad, 0xcc, 0xeb, 0x0a, 0x29,
			0x48, 0x67, 0x86, 0xa5, 0xc4, 0xe3, 0x02, 0x21,
			0x40, 0x5f, 0x7e, 0x9d, 0xbc, 0xdb, 0xfa, 0x19,
			0x38, 0x57, 0x76, 0x95, 0xb4, 0xd3, 0xf2, 0x11,
			0x30, 0x4f, 0x6e, 0x8d, 0xac, 0xcb, 0xea, 0x09,
			0x28, 0x47, 0x66, 0x85, 0xa4, 0xc3, 0xe2, 0x01,
			0x20, 0x3f, 0x5e, 0x7d, 0x9c, 0xbb, 0xda, 0xf9,
			0x18, 0x37, 0x56, 0x75, 0x94, 0xb3, 0xd2, 0xf1,
			0x10, 0x2f, 0x4e, 0x6d, 0x8c, 0xab, 0xca, 0xe9,
			0x08, 0x27, 0x46, 0x65, 0x84, 0xa3, 0xc2, 0xe1,
			0x00, 0x21, 0x42, 0x63,
		},
		.ilen = 4100,
		.result = {
			0xf0, 0x5c, 0x74, 0xad, 0x4e, 0xbc, 0x99, 0xe2,
			0xae, 0xff, 0x91, 0x3a, 0x44, 0xcf, 0x38, 0x32,
			0x1e, 0xad, 0xa7, 0xcd, 0xa1, 0x39, 0x95, 0xaa,
			0x10, 0xb1, 0xb3, 0x2e, 0x04, 0x31, 0x8f, 0x86,
			0xf2, 0x62, 0x74, 0x70, 0x0c, 0xa4, 0x46, 0x08,
			0xa8, 0xb7, 0x99, 0xa8, 0xe9, 0xd2, 0x73, 0x79,
			0x7e, 0x6e, 0xd4, 0x8f, 0x1e, 0xc7, 0x8e, 0x31,
			0x0b, 0xfa, 0x4b, 0xce, 0xfd, 0xf3, 0x57, 0x71,
			0xe9, 0x46, 0x03, 0xa5, 0x3d, 0x34, 0x00, 0xe2,
			0x18, 0xff, 0x75, 0x6d, 0x06, 0x2d, 0x00, 0xab,
			0xb9, 0x3e, 0x6c, 0x59, 0xc5, 0x84, 0x06, 0xb5,
			0x8b, 0xd0, 0x89, 0x9c, 0x4a, 0x79, 0x16, 0xc6,
			0x3d, 0x74, 0x54, 0xfa, 0x44, 0xcd, 0x23, 0x26,
			0x5c, 0xcf, 0x7e, 0x28, 0x92, 0x32, 0xbf, 0xdf,
			0xa7, 0x20, 0x3c, 0x74, 0x58, 0x2a, 0x9a, 0xde,
			0x61, 0x00, 0x1c, 0x4f, 0xff, 0x59, 0xc4, 0x22,
			0xac, 0x3c, 0xd0, 0xe8, 0x6c, 0xf9, 0x97, 0x1b,
			0x58, 0x9b, 0xad, 0x71, 0xe8, 0xa9, 0xb5, 0x0d,
			0xee, 0x2f, 0x04, 0x1f, 0x7f, 0xbc, 0x99, 0xee,
			0x84, 0xff, 0x42, 0x60, 0xdc, 0x3a, 0x18, 0xa5,
			0x81, 0xf9, 0xef, 0xdc, 0x7a, 0x0f, 0x65, 0x41,
			0x2f, 0xa3, 0xd3, 0xf9, 0xc2, 0xcb, 0xc0, 0x4d,
			0x8f, 0xd3, 0x76, 0x96, 0xad, 0x49, 0x6d, 0x38,
			0x3d, 0x39, 0x0b, 0x6c, 0x80, 0xb7, 0x54, 0x69,
			0xf0, 0x2c, 0x90, 0x02, 0x29, 0x0d, 0x1c, 0x12,
			0xad, 0x55, 0xc3, 0x8b, 0x68, 0xd9, 0xcc, 0xb3,
			0xb2, 0x64, 0x33, 0x90, 0x5e, 0xca, 0x4b, 0xe2,
			0xfb, 0x75, 0xdc, 0x63, 0xf7, 0x9f, 0x82, 0x74,
			0xf0, 0xc9, 0xaa, 0x7f, 0xe9, 0x2a, 0x9b, 0x33,
			0xbc, 0x88, 0x00, 0x7f, 0xca, 0xb2, 0x1f, 0x14,
			0xdb, 0xc5, 0x8e, 0x7b, 0x11, 0x3c, 0x3e, 0x08,
			0xf3, 0x83, 0xe8, 0xe0, 0x94, 0x86, 0x2e, 0x92,
			0x78, 0x6b, 0x01, 0xc9, 0xc7, 0x83, 0xba, 0x21,
			0x6a, 0x25, 0x15, 0x33, 0x4e, 0x45, 0x08, 0xec,
			0x35, 0xdb, 0xe0, 0x6e, 0x31, 0x51, 0x79, 0xa9,
			0x42, 0x44, 0x65, 0xc1, 0xa0, 0xf1, 0xf9, 0x2a,
			0x70, 0xd5, 0xb6, 0xc6, 0xc1, 0x8c, 0x39, 0xfc,
			0x25, 0xa6, 0x55, 0xd9, 0xdd, 0x2d, 0x4c, 0xec,
			0x49, 0xc6, 0xeb, 0x0e, 0xa8, 0x25, 0x2a, 0x16,
			0x1b, 0x66, 0x84, 0xda, 0xe2, 0x92, 0xe5, 0xc0,
			0xc8, 0x53, 0x07, 0xaf, 0x80, 0x84, 0xec, 0xfd,
			0xcd, 0xd1, 0x6e, 0xcd, 0x6f, 0x6a, 0xf5, 0x36,
			0xc5, 0x15, 0xe5, 0x25, 0x7d, 0x77, 0xd1, 0x1a,
			0x93, 0x36, 0xa9, 0xcf, 0x7c, 0xa4, 0x54, 0x4a,
			0x06, 0x51, 0x48, 0x4e, 0xf6, 0x59, 0x87, 0xd2,
			0x04, 0x02, 0xef, 0xd3, 0x44, 0xde, 0x76, 0x31,
			0xb3, 0x34, 0x17, 0x1b, 0x9d, 0x66, 0x11, 0x9f,
			0x1e, 0xcc, 0x17, 0xe9, 0xc7, 0x3c, 0x1b, 0xe7,
			0xcb, 0x50, 0x08, 0xfc, 0xdc, 0x2b, 0x24, 0xdb,
			0x65, 0x83, 0xd0, 0x3b, 0xe3, 0x30, 0xea, 0x94,
			0x6c, 0xe7, 0xe8, 0x35, 0x32, 0xc7, 0xdb, 0x64,
			0xb4, 0x01, 0xab, 0x36, 0x2c, 0x77, 0x13, 0xaf,
			0xf8, 0x2b, 0x88, 0x3f, 0x54, 0x39, 0xc4, 0x44,
			0xfe, 0xef, 0x6f, 0x68, 0x34, 0xbe, 0x0f, 0x05,
			0x16, 0x6d, 0xf6, 0x0a, 0x30, 0xe7, 0xe3, 0xed,
			0xc4, 0xde, 0x3c, 0x1b, 0x13, 0xd8, 0xdb, 0xfe,
			0x41, 0x62, 0xe5, 0x28, 0xd4, 0x8d, 0xa3, 0xc7,
			0x93, 0x97, 0xc6, 0x48, 0x45, 0x1d, 0x9f, 0x83,
			0xdf, 0x4b, 0x40, 0x3e, 0x42, 0x25, 0x87, 0x80,
			0x4c, 0x7d, 0xa8, 0xd4, 0x98, 0x23, 0x95, 0x75,
			0x41, 0x8c, 0xda, 0x41, 0x9b, 0xd4, 0xa7, 0x06,
			0xb5, 0xf1, 0x71, 0x09, 0x53, 0xbe, 0xca, 0xbf,
			0x32, 0x03, 0xed, 0xf0, 0x50, 0x1c, 0x56, 0x39,
			0x5b, 0xa4, 0x75, 0x18, 0xf7, 0x9b, 0x58, 0xef,
			0x53, 0xfc, 0x2a, 0x38, 0x23, 0x15, 0x75, 0xcd,
			0x45, 0xe5, 0x5a, 0x82, 0x55, 0xba, 0x21, 0xfa,
			0xd4, 0xbd, 0xc6, 0x94, 0x7c, 0xc5, 0x80, 0x12,
			0xf7, 0x4b, 0x32, 0xc4, 0x9a, 0x82, 0xd8, 0x28,
			0x8f, 0xd9, 0xc2, 0x0f, 0x60, 0x03, 0xbe, 0x5e,
			0x21, 0xd6, 0x5f, 0x58, 0xbf, 0x5c, 0xb1, 0x32,
			0x82, 0x8d, 0xa9, 0xe5, 0xf2, 0x66, 0x1a, 0xc0,
			0xa0, 0xbc, 0x58, 0x2f, 0x71, 0xf5, 0x2f, 0xed,
			0xd1, 0x26, 0xb9, 0xd8, 0x49, 0x5a, 0x07, 0x19,
			0x01, 0x7c, 0x59, 0xb0, 0xf8, 0xa4, 0xb7, 0xd3,
			0x7b, 0x1a, 0x8c, 0x38, 0xf4, 0x50, 0xa4, 0x59,
			0xb0, 0xcc, 0x41, 0x0b, 0x88, 0x7f, 0xe5, 0x31,
			0xb3, 0x42, 0xba, 0xa2, 0x7e, 0xd4, 0x32, 0x71,
			0x45, 0x87, 0x48, 0xa9, 0xc2, 0xf2, 0x89, 0xb3,
			0xe4, 0xa7, 0x7e, 0x52, 0x15, 0x61, 0xfa, 0xfe,
			0xc9, 0xdd, 0x81, 0xeb, 0x13, 0xab, 0xab, 0xc3,
			0x98, 0x59, 0xd8, 0x16, 0x3d, 0x14, 0x7a, 0x1c,
			0x3c, 0x41, 0x9a, 0x16, 0x16, 0x9b, 0xd2, 0xd2,
			0x69, 0x3a, 0x29, 0x23, 0xac, 0x86, 0x32, 0xa5,
			0x48, 0x9c, 0x9e, 0xf3, 0x47, 0x77, 0x81, 0x70,
			0x24, 0xe8, 0x85, 0xd2, 0xf5, 0xb5, 0xfa, 0xff,
			0x59, 0x6a, 0xd3, 0x50, 0x59, 0x43, 0x59, 0xde,
			0xd9, 0xf1, 0x55, 0xa5, 0x0c, 0xc3, 0x1a, 0x1a,
			0x18, 0x34, 0x0d, 0x1a, 0x63, 0x33, 0xed, 0x10,
			0xe0, 0x1d, 0x2a, 0x18, 0xd2, 0xc0, 0x54, 0xa8,
			0xca, 0xb5, 0x9a, 0xd3, 0xdd, 0xca, 0x45, 0x84,
			0x50, 0xe7, 0x0f, 0xfe, 0xa4, 0x99, 0x5a, 0xbe,
			0x43, 0x2d, 0x9a, 0xcb, 0x92, 0x3f, 0x5a, 0x1d,
			0x85, 0xd8, 0xc9, 0xdf, 0x68, 0xc9, 0x12, 0x80,
			0x56, 0x0c, 0xdc, 0x00, 0xdc, 0x3a, 0x7d, 0x9d,
			0xa3, 0xa2, 0xe8, 0x4d, 0xbf, 0xf9, 0x70, 0xa0,
			0xa4, 0x13, 0x4f, 0x6b, 0xaf, 0x0a, 0x89, 0x7f,
			0xda, 0xf0, 0xbf, 0x9b, 0xc8, 0x1d, 0xe5, 0xf8,
			0x2e, 0x8b, 0x07, 0xb5, 0x73, 0x1b, 0xcc, 0xa2,
			0xa6, 0xad, 0x30, 0xbc, 0x78, 0x3c, 0x5b, 0x10,
			0xfa, 0x5e, 0x62, 0x2d, 0x9e, 0x64, 0xb3, 0x33,
			0xce, 0xf9, 0x1f, 0x86, 0xe7, 0x8b, 0xa2, 0xb8,
			0xe8, 0x99, 0x57, 0x8c, 0x11, 0xed, 0x66, 0xd9,
			0x3c, 0x72, 0xb9, 0xc3, 0xe6, 0x4e, 0x17, 0x3a,
			0x6a, 0xcb, 0x42, 0x24, 0x06, 0xed, 0x3e, 0x4e,
			0xa3, 0xe8, 0x6a, 0x94, 0xda, 0x0d, 0x4e, 0xd5,
			0x14, 0x19, 0xcf, 0xb6, 0x26, 0xd8, 0x2e, 0xcc,
			0x64, 0x76, 0x38, 0x49, 0x4d, 0xfe, 0x30, 0x6d,
			0xe4, 0xc8, 0x8c, 0x7b, 0xc4, 0xe0, 0x35, 0xba,
			0x22, 0x6e, 0x76, 0xe1, 0x1a, 0xf2, 0x53, 0xc3,
			0x28, 0xa2, 0x82, 0x1f, 0x61, 0x69, 0xad, 0xc1,
			0x7b, 0x28, 0x4b, 0x1e, 0x6c, 0x85, 0x95, 0x9b,
			0x51, 0xb5, 0x17, 0x7f, 0x12, 0x69, 0x8c, 0x24,
			0xd5, 0xc7, 0x5a, 0x5a, 0x11, 0x54, 0xff, 0x5a,
			0xf7, 0x16, 0xc3, 0x91, 0xa6, 0xf0, 0xdc, 0x0a,
			0xb6, 0xa7, 0x4a, 0x0d, 0x7a, 0x58, 0xfe, 0xa5,
			0xf5, 0xcb, 0x8f, 0x7b, 0x0e, 0xea, 0x57, 0xe7,
			0xbd, 0x79, 0xd6, 0x1c, 0x88, 0x23, 0x6c, 0xf2,
			0x4d, 0x29, 0x77, 0x53, 0x35, 0x6a, 0x00, 0x8d,
			0xcd, 0xa3, 0x58, 0xbe, 0x77, 0x99, 0x18, 0xf8,
			0xe6, 0xe1, 0x8f, 0xe9, 0x37, 0x8f, 0xe3, 0xe2,
			0x5a, 0x8a, 0x93, 0x25, 0xaf, 0xf3, 0x78, 0x80,
			0xbe, 0xa6, 0x1b, 0xc6, 0xac, 0x8b, 0x1c, 0x91,
			0x58, 0xe1, 0x9f, 0x89, 0x35, 0x9d, 0x1d, 0x21,
			0x29, 0x9f, 0xf4, 0x99, 0x02, 0x27, 0x0f, 0xa8,
			0x4f, 0x79, 0x94, 0x2b, 0x33, 0x2c, 0xda, 0xa2,
			0x26, 0x39, 0x83, 0x94, 0xef, 0x27, 0xd8, 0x53,
			0x8f, 0x66, 0x0d, 0xe4, 0x41, 0x7d, 0x34, 0xcd,
			0x43, 0x7c, 0x95, 0x0a, 0x53, 0xef, 0x66, 0xda,
			0x7e, 0x9b, 0xf3, 0x93, 0xaf, 0xd0, 0x73, 0x71,
			0xba, 0x40, 0x9b, 0x74, 0xf8, 0xd7, 0xd7, 0x41,
			0x6d, 0xaf, 0x72, 0x9c, 0x8d, 0x21, 0x87, 0x3c,
			0xfd, 0x0a, 0x90, 0xa9, 0x47, 0x96, 0x9e, 0xd3,
			0x88, 0xee, 0x73, 0xcf, 0x66, 0x2f, 0x52, 0x56,
			0x6d, 0xa9, 0x80, 0x4c, 0xe2, 0x6f, 0x62, 0x88,
			0x3f, 0x0e, 0x54, 0x17, 0x48, 0x80, 0x5d, 0xd3,
			0xc3, 0xda, 0x25, 0x3d, 0xa1, 0xc8, 0xcb, 0x9f,
			0x9b, 0x70, 0xb3, 0xa1, 0xeb, 0x04, 0x52, 0xa1,
			0xf2, 0x22, 0x0f, 0xfc, 0xc8, 0x18, 0xfa, 0xf9,
			0x85, 0x9c, 0xf1, 0xac, 0xeb, 0x0c, 0x02, 0x46,
			0x75, 0xd2, 0xf5, 0x2c, 0xe3, 0xd2, 0x59, 0x94,
			0x12, 0xf3, 0x3c, 0xfc, 0xd7, 0x92, 0xfa, 0x36,
			0xba, 0x61, 0x34, 0x38, 0x7c, 0xda, 0x48, 0x3e,
			0x08, 0xc9, 0x39, 0x23, 0x5e, 0x02, 0x2c, 0x1a,
			0x18, 0x7e, 0xb4, 0xd9, 0xfd, 0x9e, 0x40, 0x02,
			0xb1, 0x33, 0x37, 0x32, 0xe7, 0xde, 0xd6, 0xd0,
			0x7c, 0x58, 0x65, 0x4b, 0xf8, 0x34, 0x27, 0x9c,
			0x44, 0xb4, 0xbd, 0xe9, 0xe9, 0x4c, 0x78, 0x7d,
			0x4b, 0x9f, 0xce, 0xb1, 0xcd, 0x47, 0xa5, 0x37,
			0xe5, 0x6d, 0xbd, 0xb9, 0x43, 0x94, 0x0a, 0xd4,
			0xd6, 0xf9, 0x04, 0x5f, 0xb5, 0x66, 0x6c, 0x1a,
			0x35, 0x12, 0xe3, 0x36, 0x28, 0x27, 0x36, 0x58,
			0x01, 0x2b, 0x79, 0xe4, 0xba, 0x6d, 0x10, 0x7d,
			0x65, 0xdf, 0x84, 0x95, 0xf4, 0xd5, 0xb6, 0x8f,
			0x2b, 0x9f, 0x96, 0x00, 0x86, 0x60, 0xf0, 0x21,
			0x76, 0xa8, 0x6a, 0x8c, 0x28, 0x1c, 0xb3, 0x6b,
			0x97, 0xd7, 0xb6, 0x53, 0x2a, 0xcc, 0xab, 0x40,
			0x9d, 0x62, 0x79, 0x58, 0x52, 0xe6, 0x65, 0xb7,
			0xab, 0x55, 0x67, 0x9c, 0x89, 0x7c, 0x03, 0xb0,
			0x73, 0x59, 0xc5, 0x81, 0xf5, 0x18, 0x17, 0x5c,
			0x89, 0xf3, 0x78, 0x35, 0x44, 0x62, 0x78, 0x72,
			0xd0, 0x96, 0xeb, 0x31, 0xe7, 0x87, 0x77, 0x14,
			0x99, 0x51, 0xf2, 0x59, 0x26, 0x9e, 0xb5, 0xa6,
			0x45, 0xfe, 0x6e, 0xbd, 0x07, 0x4c, 0x94, 0x5a,
			0xa5, 0x7d, 0xfc, 0xf1, 0x2b, 0x77, 0xe2, 0xfe,
			0x17, 0xd4, 0x84, 0xa0, 0xac, 0xb5, 0xc7, 0xda,
			0xa9, 0x1a, 0xb6, 0xf3, 0x74, 0x11, 0xb4, 0x9d,
			0xfb, 0x79, 0x2e, 0x04, 0x2d, 0x50, 0x28, 0x83,
			0xbf, 0xc6, 0x52, 0xd3, 0x34, 0xd6, 0xe8, 0x7a,
			0xb6, 0xea, 0xe7, 0xa8, 0x6c, 0x15, 0x1e, 0x2c,
			0x57, 0xbc, 0x48, 0x4e, 0x5f, 0x5c, 0xb6, 0x92,
			0xd2, 0x49, 0x77, 0x81, 0x6d, 0x90, 0x70, 0xae,
			0x98, 0xa1, 0x03, 0x0d, 0x6b, 0xb9, 0x77, 0x14,
			0xf1, 0x4e, 0x23, 0xd3, 0xf8, 0x68, 0xbd, 0xc2,
			0xfe, 0x04, 0xb7, 0x5c, 0xc5, 0x17, 0x60, 0x8f,
			0x65, 0x54, 0xa4, 0x7a, 0x42, 0xdc, 0x18, 0x0d,
			0xb5, 0xcf, 0x0f, 0xd3, 0xc7, 0x91, 0x66, 0x1b,
			0x45, 0x42, 0x27, 0x75, 0x50, 0xe5, 0xee, 0xb8,
			0x7f, 0x33, 0x2c, 0xba, 0x4a, 0x92, 0x4d, 0x2c,
			0x3c, 0xe3, 0x0d, 0x80, 0x01, 0xba, 0x0d, 0x29,
			0xd8, 0x3c, 0xe9, 0x13, 0x16, 0x57, 0xe6, 0xea,
			0x94, 0x52, 0xe7, 0x00, 0x4d, 0x30, 0xb0, 0x0f,
			0x35, 0xb8, 0xb8, 0xa7, 0xb1, 0xb5, 0x3b, 0x44,
			0xe1, 0x2f, 0xfd, 0x88, 0xed, 0x43, 0xe7, 0x52,
			0x10, 0x93, 0xb3, 0x8a, 0x30, 0x6b, 0x0a, 0xf7,
			0x23, 0xc6, 0x50, 0x9d, 0x4a, 0xb0, 0xde, 0xc3,
			0xdc, 0x9b, 0x2f, 0x01, 0x56, 0x36, 0x09, 0xc5,
			0x2f, 0x6b, 0xfe, 0xf1, 0xd8, 0x27, 0x45, 0x03,
			0x30, 0x5e, 0x5c, 0x5b, 0xb4, 0x62, 0x0e, 0x1a,
			0xa9, 0x21, 0x2b, 0x92, 0x94, 0x87, 0x62, 0x57,
			0x4c, 0x10, 0x74, 0x1a, 0xf1, 0x0a, 0xc5, 0x84,
			0x3b, 0x9e, 0x72, 0x02, 0xd7, 0xcc, 0x09, 0x56,
			0xbd, 0x54, 0xc1, 0xf0, 0xc3, 0xe3, 0xb3, 0xf8,
			0xd2, 0x0d, 0x61, 0xcb, 0xef, 0xce, 0x0d, 0x05,
			0xb0, 0x98, 0xd9, 0x8e, 0x4f, 0xf9, 0xbc, 0x93,
			0xa6, 0xea, 0xc8, 0xcf, 0x10, 0x53, 0x4b, 0xf1,
			0xec, 0xfc, 0x89, 0xf9, 0x64, 0xb0, 0x22, 0xbf,
			0x9e, 0x55, 0x46, 0x9f, 0x7c, 0x50, 0x8e, 0x84,
			0x54, 0x20, 0x98, 0xd7, 0x6c, 0x40, 0x1e, 0xdb,
			0x69, 0x34, 0x78, 0x61, 0x24, 0x21, 0x9c, 0x8a,
			0xb3, 0x62, 0x31, 0x8b, 0x6e, 0xf5, 0x2a, 0x35,
			0x86, 0x13, 0xb1, 0x6c, 0x64, 0x2e, 0x41, 0xa5,
			0x05, 0xf2, 0x42, 0xba, 0xd2, 0x3a, 0x0d, 0x8e,
			0x8a, 0x59, 0x94, 0x3c, 0xcf, 0x36, 0x27, 0x82,
			0xc2, 0x45, 0xee, 0x58, 0xcd, 0x88, 0xb4, 0xec,
			0xde, 0xb2, 0x96, 0x0a, 0xaf, 0x38, 0x6f, 0x88,
			0xd7, 0xd8, 0xe1, 0xdf, 0xb9, 0x96, 0xa9, 0x0a,
			0xb1, 0x95, 0x28, 0x86, 0x20, 0xe9, 0x17, 0x49,
			0xa2, 0x29, 0x38, 0xaa, 0xa5, 0xe9, 0x6e, 0xf1,
			0x19, 0x27, 0xc0, 0xd5, 0x2a, 0x22, 0xc3, 0x0b,
			0xdb, 0x7c, 0x73, 0x10, 0xb9, 0xba, 0x89, 0x76,
			0x54, 0xae, 0x7d, 0x71, 0xb3, 0x93, 0xf6, 0x32,
			0xe6, 0x47, 0x43, 0x55, 0xac, 0xa0, 0x0d, 0xc2,
			0x93, 0x27, 0x4a, 0x8e, 0x0e, 0x74, 0x15, 0xc7,
			0x0b, 0x85, 0xd9, 0x0c, 0xa9, 0x30, 0x7a, 0x3e,
			0xea, 0x8f, 0x85, 0x6d, 0x3a, 0x12, 0x4f, 0x72,
			0x69, 0x58, 0x7a, 0x80, 0xbb, 0xb5, 0x97, 0xf3,
			0xcf, 0x70, 0xd2, 0x5d, 0xdd, 0x4d, 0x21, 0x79,
			0x54, 0x4d, 0xe4, 0x05, 0xe8, 0xbd, 0xc2, 0x62,
			0xb1, 0x3b, 0x77, 0x1c, 0xd6, 0x5c, 0xf3, 0xa0,
			0x79, 0x00, 0xa8, 0x6c, 0x29, 0xd9, 0x18, 0x24,
			0x36, 0xa2, 0x46, 0xc0, 0x96, 0x65, 0x7f, 0xbd,
			0x2a, 0xed, 0x36, 0x16, 0x0c, 0xaa, 0x9f, 0xf4,
			0xc5, 0xb4, 0xe2, 0x12, 0xed, 0x69, 0xed, 0x4f,
			0x26, 0x2c, 0x39, 0x52, 0x89, 0x98, 0xe7, 0x2c,
			0x99, 0xa4, 0x9e, 0xa3, 0x9b, 0x99, 0x46, 0x7a,
			0x3a, 0xdc, 0xa8, 0x59, 0xa3, 0xdb, 0xc3, 0x3b,
			0x95, 0x0d, 0x3b, 0x09, 0x6e, 0xee, 0x83, 0x5d,
			0x32, 0x4d, 0xed, 0xab, 0xfa, 0x98, 0x14, 0x4e,
			0xc3, 0x15, 0x45, 0x53, 0x61, 0xc4, 0x93, 0xbd,
			0x90, 0xf4, 0x99, 0x95, 0x4c, 0xe6, 0x76, 0x92,
			0x29, 0x90, 0x46, 0x30, 0x92, 0x69, 0x7d, 0x13,
			0xf2, 0xa5, 0xcd, 0x69, 0x49, 0x44, 0xb2, 0x0f,
			0x63, 0x40, 0x36, 0x5f, 0x09, 0xe2, 0x78, 0xf8,
			0x91, 0xe3, 0xe2, 0xfa, 0x10, 0xf7, 0xc8, 0x24,
			0xa8, 0x89, 0x32, 0x5c, 0x37, 0x25, 0x1d, 0xb2,
			0xea, 0x17, 0x8a, 0x0a, 0xa9, 0x64, 0xc3, 0x7c,
			0x3c, 0x7c, 0xbd, 0xc6, 0x79, 0x34, 0xe7, 0xe2,
			0x85, 0x8e, 0xbf, 0xf8, 0xde, 0x92, 0xa0, 0xae,
			0x20, 0xc4, 0xf6, 0xbb, 0x1f, 0x38, 0x19, 0x0e,
			0xe8, 0x79, 0x9c, 0xa1, 0x23, 0xe9, 0x54, 0x7e,
			0x37, 0x2f, 0xe2, 0x94, 0x32, 0xaf, 0xa0, 0x23,
			0x49, 0xe4, 0xc0, 0xb3, 0xac, 0x00, 0x8f, 0x36,
			0x05, 0xc4, 0xa6, 0x96, 0xec, 0x05, 0x98, 0x4f,
			0x96, 0x67, 0x57, 0x1f, 0x20, 0x86, 0x1b, 0x2d,
			0x69, 0xe4, 0x29, 0x93, 0x66, 0x5f, 0xaf, 0x6b,
			0x88, 0x26, 0x2c, 0x67, 0x02, 0x4b, 0x52, 0xd0,
			0x83, 0x7a, 0x43, 0x1f, 0xc0, 0x71, 0x15, 0x25,
			0x77, 0x65, 0x08, 0x60, 0x11, 0x76, 0x4c, 0x8d,
			0xed, 0xa9, 0x27, 0xc6, 0xb1, 0x2a, 0x2c, 0x6a,
			0x4a, 0x97, 0xf5, 0xc6, 0xb7, 0x70, 0x42, 0xd3,
			0x03, 0xd1, 0x24, 0x95, 0xec, 0x6d, 0xab, 0x38,
			0x72, 0xce, 0xe2, 0x8b, 0x33, 0xd7, 0x51, 0x09,
			0xdc, 0x45, 0xe0, 0x09, 0x96, 0x32, 0xf3, 0xc4,
			0x84, 0xdc, 0x73, 0x73, 0x2d, 0x1b, 0x11, 0x98,
			0xc5, 0x0e, 0x69, 0x28, 0x94, 0xc7, 0xb5, 0x4d,
			0xc8, 0x8a, 0xd0, 0xaa, 0x13, 0x2e, 0x18, 0x74,
			0xdd, 0xd1, 0x1e, 0xf3, 0x90, 0xe8, 0xfc, 0x9a,
			0x72, 0x4a, 0x0e, 0xd1, 0xe4, 0xfb, 0x0d, 0x96,
			0xd1, 0x0c, 0x79, 0x85, 0x1b, 0x1c, 0xfe, 0xe1,
			0x62, 0x8f, 0x7a, 0x73, 0x32, 0xab, 0xc8, 0x18,
			0x69, 0xe3, 0x34, 0x30, 0xdf, 0x13, 0xa6, 0xe5,
			0xe8, 0x0e, 0x67, 0x7f, 0x81, 0x11, 0xb4, 0x60,
			0xc7, 0xbd, 0x79, 0x65, 0x50, 0xdc, 0xc4, 0x5b,
			0xde, 0x39, 0xa4, 0x01, 0x72, 0x63, 0xf3, 0xd1,
			0x64, 0x4e, 0xdf, 0xfc, 0x27, 0x92, 0x37, 0x0d,
			0x57, 0xcd, 0x11, 0x4f, 0x11, 0x04, 0x8e, 0x1d,
			0x16, 0xf7, 0xcd, 0x92, 0x9a, 0x99, 0x30, 0x14,
			0xf1, 0x7c, 0x67, 0x1b, 0x1f, 0x41, 0x0b, 0xe8,
			0x32, 0xe8, 0xb8, 0xc1, 0x4f, 0x54, 0x86, 0x4f,
			0xe5, 0x79, 0x81, 0x73, 0xcd, 0x43, 0x59, 0x68,
			0x73, 0x02, 0x3b, 0x78, 0x21, 0x72, 0x43, 0x00,
			0x49, 0x17, 0xf7, 0x00, 0xaf, 0x68, 0x24, 0x53,
			0x05, 0x0a, 0xc3, 0x33, 0xe0, 0x33, 0x3f, 0x69,
			0xd2, 0x84, 0x2f, 0x0b, 0xed, 0xde, 0x04, 0xf4,
			0x11, 0x94, 0x13, 0x69, 0x51, 0x09, 0x28, 0xde,
			0x57, 0x5c, 0xef, 0xdc, 0x9a, 0x49, 0x1c, 0x17,
			0x97, 0xf3, 0x96, 0xc1, 0x7f, 0x5d, 0x2e, 0x7d,
			0x55, 0xb8, 0xb3, 0x02, 0x09, 0xb3, 0x1f, 0xe7,
			0xc9, 0x8d, 0xa3, 0x36, 0x34, 0x8a, 0x77, 0x13,
			0x30, 0x63, 0x4c, 0xa5, 0xcd, 0xc3, 0xe0, 0x7e,
			0x05, 0xa1, 0x7b, 0x0c, 0xcb, 0x74, 0x47, 0x31,
			0x62, 0x03, 0x43, 0xf1, 0x87, 0xb4, 0xb0, 0x85,
			0x87, 0x8e, 0x4b, 0x25, 0xc7, 0xcf, 0xae, 0x4b,
			0x36, 0x46, 0x3e, 0x62, 0xbc, 0x6f, 0xeb, 0x5f,
			0x73, 0xac, 0xe6, 0x07, 0xee, 0xc1, 0xa1, 0xd6,
			0xc4, 0xab, 0xc9, 0xd6, 0x89, 0x45, 0xe1, 0xf1,
			0x04, 0x4e, 0x1a, 0x6f, 0xbb, 0x4f, 0x3a, 0xa3,
			0xa0, 0xcb, 0xa3, 0x0a, 0xd8, 0x71, 0x35, 0x55,
			0xe4, 0xbc, 0x2e, 0x04, 0x06, 0xe6, 0xff, 0x5b,
			0x1c, 0xc0, 0x11, 0x7c, 0xc5, 0x17, 0xf3, 0x38,
			0xcf, 0xe9, 0xba, 0x0f, 0x0e, 0xef, 0x02, 0xc2,
			0x8d, 0xc6, 0xbc, 0x4b, 0x67, 0x20, 0x95, 0xd7,
			0x2c, 0x45, 0x5b, 0x86, 0x44, 0x8c, 0x6f, 0x2e,
			0x7e, 0x9f, 0x1c, 0x77, 0xba, 0x6b, 0x0e, 0xa3,
			0x69, 0xdc, 0xab, 0x24, 0x57, 0x60, 0x47, 0xc1,
			0xd1, 0xa5, 0x9d, 0x23, 0xe6, 0xb1, 0x37, 0xfe,
			0x93, 0xd2, 0x4c, 0x46, 0xf9, 0x0c, 0xc6, 0xfb,
			0xd6, 0x9d, 0x99, 0x69, 0xab, 0x7a, 0x07, 0x0c,
			0x65, 0xe7, 0xc4, 0x08, 0x96, 0xe2, 0xa5, 0x01,
			0x3f, 0x46, 0x07, 0x05, 0x7e, 0xe8, 0x9a, 0x90,
			0x50, 0xdc, 0xe9, 0x7a, 0xea, 0xa1, 0x39, 0x6e,
			0x66, 0xe4, 0x6f, 0xa5, 0x5f, 0xb2, 0xd9, 0x5b,
			0xf5, 0xdb, 0x2a, 0x32, 0xf0, 0x11, 0x6f, 0x7c,
			0x26, 0x10, 0x8f, 0x3d, 0x80, 0xe9, 0x58, 0xf7,
			0xe0, 0xa8, 0x57, 0xf8, 0xdb, 0x0e, 0xce, 0x99,
			0x63, 0x19, 0x3d, 0xd5, 0xec, 0x1b, 0x77, 0x69,
			0x98, 0xf6, 0xe4, 0x5f, 0x67, 0x17, 0x4b, 0x09,
			0x85, 0x62, 0x82, 0x70, 0x18, 0xe2, 0x9a, 0x78,
			0xe2, 0x62, 0xbd, 0xb4, 0xf1, 0x42, 0xc6, 0xfb,
			0x08, 0xd0, 0xbd, 0xeb, 0x4e, 0x09, 0xf2, 0xc8,
			0x1e, 0xdc, 0x3d, 0x32, 0x21, 0x56, 0x9c, 0x4f,
			0x35, 0xf3, 0x61, 0x06, 0x72, 0x84, 0xc4, 0x32,
			0xf2, 0xf1, 0xfa, 0x0b, 0x2f, 0xc3, 0xdb, 0x02,
			0x04, 0xc2, 0xde, 0x57, 0x64, 0x60, 0x8d, 0xcf,
			0xcb, 0x86, 0x5d, 0x97, 0x3e, 0xb1, 0x9c, 0x01,
			0xd6, 0x28, 0x8f, 0x99, 0xbc, 0x46, 0xeb, 0x05,
			0xaf, 0x7e, 0xb8, 0x21, 0x2a, 0x56, 0x85, 0x1c,
			0xb3, 0x71, 0xa0, 0xde, 0xca, 0x96, 0xf1, 0x78,
			0x49, 0xa2, 0x99, 0x81, 0x80, 0x5c, 0x01, 0xf5,
			0xa0, 0xa2, 0x56, 0x63, 0xe2, 0x70, 0x07, 0xa5,
			0x95, 0xd6, 0x85, 0xeb, 0x36, 0x9e, 0xa9, 0x51,
			0x66, 0x56, 0x5f, 0x1d, 0x02, 0x19, 0xe2, 0xf6,
			0x4f, 0x73, 0x38, 0x09, 0x75, 0x64, 0x48, 0xe0,
			0xf1, 0x7e, 0x0e, 0xe8, 0x9d, 0xf9, 0xed, 0x94,
			0xfe, 0x16, 0x26, 0x62, 0x49, 0x74, 0xf4, 0xb0,
			0xd4, 0xa9, 0x6c, 0xb0, 0xfd, 0x53, 0xe9, 0x81,
			0xe0, 0x7a, 0xbf, 0xcf, 0xb5, 0xc4, 0x01, 0x81,
			0x79, 0x99, 0x77, 0x01, 0x3b, 0xe9, 0xa2, 0xb6,
			0xe6, 0x6a, 0x8a, 0x9e, 0x56, 0x1c, 0x8d, 0x1e,
			0x8f, 0x06, 0x55, 0x2c, 0x6c, 0xdc, 0x92, 0x87,
			0x64, 0x3b, 0x4b, 0x19, 0xa1, 0x13, 0x64, 0x1d,
			0x4a, 0xe9, 0xc0, 0x00, 0xb8, 0x95, 0xef, 0x6b,
			0x1a, 0x86, 0x6d, 0x37, 0x52, 0x02, 0xc2, 0xe0,
			0xc8, 0xbb, 0x42, 0x0c, 0x02, 0x21, 0x4a, 0xc9,
			0xef, 0xa0, 0x54, 0xe4, 0x5e, 0x16, 0x53, 0x81,
			0x70, 0x62, 0x10, 0xaf, 0xde, 0xb8, 0xb5, 0xd3,
			0xe8, 0x5e, 0x6c, 0xc3, 0x8a, 0x3e, 0x18, 0x07,
			0xf2, 0x2f, 0x7d, 0xa7, 0xe1, 0x3d, 0x4e, 0xb4,
			0x26, 0xa7, 0xa3, 0x93, 0x86, 0xb2, 0x04, 0x1e,
			0x53, 0x5d, 0x86, 0xd6, 0xde, 0x65, 0xca, 0xe3,
			0x4e, 0xc1, 0xcf, 0xef, 0xc8, 0x70, 0x1b, 0x83,
			0x13, 0xdd, 0x18, 0x8b, 0x0d, 0x76, 0xd2, 0xf6,
			0x37, 0x7a, 0x93, 0x7a, 0x50, 0x11, 0x9f, 0x96,
			0x86, 0x25, 0xfd, 0xac, 0xdc, 0xbe, 0x18, 0x93,
			0x19, 0x6b, 0xec, 0x58, 0x4f, 0xb9, 0x75, 0xa7,
			0xdd, 0x3f, 0x2f, 0xec, 0xc8, 0x5a, 0x84, 0xab,
			0xd5, 0xe4, 0x8a, 0x07, 0xf6, 0x4d, 0x23, 0xd6,
			0x03, 0xfb, 0x03, 0x6a, 0xea, 0x66, 0xbf, 0xd4,
			0xb1, 0x34, 0xfb, 0x78, 0xe9, 0x55, 0xdc, 0x7c,
			0x3d, 0x9c, 0xe5, 0x9a, 0xac, 0xc3, 0x7a, 0x80,
			0x24, 0x6d, 0xa0, 0xef, 0x25, 0x7c, 0xb7, 0xea,
			0xce, 0x4d, 0x5f, 0x18, 0x60, 0xce, 0x87, 0x22,
			0x66, 0x2f, 0xd5, 0xdd, 0xdd, 0x02, 0x21, 0x75,
			0x82, 0xa0, 0x1f, 0x58, 0xc6, 0xd3, 0x62, 0xf7,
			0x32, 0xd8, 0xaf, 0x1e, 0x07, 0x77, 0x51, 0x96,
			0xd5, 0x6b, 0x1e, 0x7e, 0x80, 0x02, 0xe8, 0x67,
			0xea, 0x17, 0x0b, 0x10, 0xd2, 0x3f, 0x28, 0x25,
			0x4f, 0x05, 0x77, 0x02, 0x14, 0x69, 0xf0, 0x2c,
			0xbe, 0x0c, 0xf1, 0x74, 0x30, 0xd1, 0xb9, 0x9b,
			0xfc, 0x8c, 0xbb, 0x04, 0x16, 0xd9, 0xba, 0xc3,
			0xbc, 0x91, 0x8a, 0xc4, 0x30, 0xa4, 0xb0, 0x12,
			0x4c, 0x21, 0x87, 0xcb, 0xc9, 0x1d, 0x16, 0x96,
			0x07, 0x6f, 0x23, 0x54, 0xb9, 0x6f, 0x79, 0xe5,
			0x64, 0xc0, 0x64, 0xda, 0xb1, 0xae, 0xdd, 0x60,
			0x6c, 0x1a, 0x9d, 0xd3, 0x04, 0x8e, 0x45, 0xb0,
			0x92, 0x61, 0xd0, 0x48, 0x81, 0xed, 0x5e, 0x1d,
			0xa0, 0xc9, 0xa4, 0x33, 0xc7, 0x13, 0x51, 0x5d,
			0x7f, 0x83, 0x73, 0xb6, 0x70, 0x18, 0x65, 0x3e,
			0x2f, 0x0e, 0x7a, 0x12, 0x39, 0x98, 0xab, 0xd8,
			0x7e, 0x6f, 0xa3, 0xd1, 0xba, 0x56, 0xad, 0xbd,
			0xf0, 0x03, 0x01, 0x1c, 0x85, 0x35, 0x9f, 0xeb,
			0x19, 0x63, 0xa1, 0xaf, 0xfe, 0x2d, 0x35, 0x50,
			0x39, 0xa0, 0x65, 0x7c, 0x95, 0x7e, 0x6b, 0xfe,
			0xc1, 0xac, 0x07, 0x7c, 0x98, 0x4f, 0xbe, 0x57,
			0xa7, 0x22, 0xec, 0xe2, 0x7e, 0x29, 0x09, 0x53,
			0xe8, 0xbf, 0xb4, 0x7e, 0x3f, 0x8f, 0xfc, 0x14,
			0xce, 0x54, 0xf9, 0x18, 0x58, 0xb5, 0xff, 0x44,
			0x05, 0x9d, 0xce, 0x1b, 0xb6, 0x82, 0x23, 0xc8,
			0x2e, 0xbc, 0x69, 0xbb, 0x4a, 0x29, 0x0f, 0x65,
			0x94, 0xf0, 0x63, 0x06, 0x0e, 0xef, 0x8c, 0xbd,
			0xff, 0xfd, 0xb0, 0x21, 0x6e, 0x57, 0x05, 0x75,
			0xda, 0xd5, 0xc4, 0xeb, 0x8d, 0x32, 0xf7, 0x50,
			0xd3, 0x6f, 0x22, 0xed, 0x5f, 0x8e, 0xa2, 0x5b,
			0x80, 0x8c, 0xc8, 0x78, 0x40, 0x24, 0x4b, 0x89,
			0x30, 0xce, 0x7a, 0x97, 0x0e, 0xc4, 0xaf, 0xef,
			0x9b, 0xb4, 0xcd, 0x66, 0x74, 0x14, 0x04, 0x2b,
			0xf7, 0xce, 0x0b, 0x1c, 0x6e, 0xc2, 0x78, 0x8c,
			0xca, 0xc5, 0xd0, 0x1c, 0x95, 0x4a, 0x91, 0x2d,
			0xa7, 0x20, 0xeb, 0x86, 0x52, 0xb7, 0x67, 0xd8,
			0x0c, 0xd6, 0x04, 0x14, 0xde, 0x51, 0x74, 0x75,
			0xe7, 0x11, 0xb4, 0x87, 0xa3, 0x3d, 0x2d, 0xad,
			0x4f, 0xef, 0xa0, 0x0f, 0x70, 0x00, 0x6d, 0x13,
			0x19, 0x1d, 0x41, 0x50, 0xe9, 0xd8, 0xf0, 0x32,
			0x71, 0xbc, 0xd3, 0x11, 0xf2, 0xac, 0xbe, 0xaf,
			0x75, 0x46, 0x65, 0x4e, 0x07, 0x34, 0x37, 0xa3,
			0x89, 0xfe, 0x75, 0xd4, 0x70, 0x4c, 0xc6, 0x3f,
			0x69, 0x24, 0x0e, 0x38, 0x67, 0x43, 0x8c, 0xde,
			0x06, 0xb5, 0xb8, 0xe7, 0xc4, 0xf0, 0x41, 0x8f,
			0xf0, 0xbd, 0x2f, 0x0b, 0xb9, 0x18, 0xf8, 0xde,
			0x64, 0xb1, 0xdb, 0xee, 0x00, 0x50, 0x77, 0xe1,
			0xc7, 0xff, 0xa6, 0xfa, 0xdd, 0x70, 0xf4, 0xe3,
			0x93, 0xe9, 0x77, 0x35, 0x3d, 0x4b, 0x2f, 0x2b,
			0x6d, 0x55, 0xf0, 0xfc, 0x88, 0x54, 0x4e, 0x89,
			0xc1, 0x8a, 0x23, 0x31, 0x2d, 0x14, 0x2a, 0xb8,
			0x1b, 0x15, 0xdd, 0x9e, 0x6e, 0x7b, 0xda, 0x05,
			0x91, 0x7d, 0x62, 0x64, 0x96, 0x72, 0xde, 0xfc,
			0xc1, 0xec, 0xf0, 0x23, 0x51, 0x6f, 0xdb, 0x5b,
			0x1d, 0x08, 0x57, 0xce, 0x09, 0xb8, 0xf6, 0xcd,
			0x8d, 0x95, 0xf2, 0x20, 0xbf, 0x0f, 0x20, 0x57,
			0x98, 0x81, 0x84, 0x4f, 0x15, 0x5c, 0x76, 0xe7,
			0x3e, 0x0a, 0x3a, 0x6c, 0xc4, 0x8a, 0xbe, 0x78,
			0x74, 0x77, 0xc3, 0x09, 0x4b, 0x5d, 0x48, 0xe4,
			0xc8, 0xcb, 0x0b, 0xea, 0x17, 0x28, 0xcf, 0xcf,
			0x31, 0x32, 0x44, 0xa4, 0xe5, 0x0e, 0x1a, 0x98,
			0x94, 0xc4, 0xf0, 0xff, 0xae, 0x3e, 0x44, 0xe8,
			0xa5, 0xb3, 0xb5, 0x37, 0x2f, 0xe8, 0xaf, 0x6f,
			0x28, 0xc1, 0x37, 0x5f, 0x31, 0xd2, 0xb9, 0x33,
			0xb1, 0xb2, 0x52, 0x94, 0x75, 0x2c, 0x29, 0x59,
			0x06, 0xc2, 0x25, 0xe8, 0x71, 0x65, 0x4e, 0xed,
			0xc0, 0x9c, 0xb1, 0xbb, 0x25, 0xdc, 0x6c, 0xe7,
			0x4b, 0xa5, 0x7a, 0x54, 0x7a, 0x60, 0xff, 0x7a,
			0xe0, 0x50, 0x40, 0x96, 0x35, 0x63, 0xe4, 0x0b,
			0x76, 0xbd, 0xa4, 0x65, 0x00, 0x1b, 0x57, 0x88,
			0xae, 0xed, 0x39, 0x88, 0x42, 0x11, 0x3c, 0xed,
			0x85, 0x67, 0x7d, 0xb9, 0x68, 0x82, 0xe9, 0x43,
			0x3c, 0x47, 0x53, 0xfa, 0xe8, 0xf8, 0x9f, 0x1f,
			0x9f, 0xef, 0x0f, 0xf7, 0x30, 0xd9, 0x30, 0x0e,
			0xb9, 0x9f, 0x69, 0x18, 0x2f, 0x7e, 0xf8, 0xf8,
			0xf8, 0x8c, 0x0f, 0xd4, 0x02, 0x4d, 0xea, 0xcd,
			0x0a, 0x9c, 0x6f, 0x71, 0x6d, 0x5a, 0x4c, 0x60,
			0xce, 0x20, 0x56, 0x32, 0xc6, 0xc5, 0x99, 0x1f,
			0x09, 0xe6, 0x4e, 0x18, 0x1a, 0x15, 0x13, 0xa8,
			0x7d, 0xb1, 0x6b, 0xc0, 0xb2, 0x6d, 0xf8, 0x26,
			0x66, 0xf8, 0x3d, 0x18, 0x74, 0x70, 0x66, 0x7a,
			0x34, 0x17, 0xde, 0xba, 0x47, 0xf1, 0x06, 0x18,
			0xcb, 0xaf, 0xeb, 0x4a, 0x1e, 0x8f, 0xa7, 0x77,
			0xe0, 0x3b, 0x78, 0x62, 0x66, 0xc9, 0x10, 0xea,
			0x1f, 0xb7, 0x29, 0x0a, 0x45, 0xa1, 0x1d, 0x1e,
			0x1d, 0xe2, 0x65, 0x61, 0x50, 0x9c, 0xd7, 0x05,
			0xf2, 0x0b, 0x5b, 0x12, 0x61, 0x02, 0xc8, 0xe5,
			0x63, 0x4f, 0x20, 0x0c, 0x07, 0x17, 0x33, 0x5e,
			0x03, 0x9a, 0x53, 0x0f, 0x2e, 0x55, 0xfe, 0x50,
			0x43, 0x7d, 0xd0, 0xb6, 0x7e, 0x5a, 0xda, 0xae,
			0x58, 0xef, 0x15, 0xa9, 0x83, 0xd9, 0x46, 0xb1,
			0x42, 0xaa, 0xf5, 0x02, 0x6c, 0xce, 0x92, 0x06,
			0x1b, 0xdb, 0x66, 0x45, 0x91, 0x79, 0xc2, 0x2d,
			0xe6, 0x53, 0xd3, 0x14, 0xfd, 0xbb, 0x44, 0x63,
			0xc6, 0xd7, 0x3d, 0x7a, 0x0c, 0x75, 0x78, 0x9d,
			0x5c, 0xa6, 0x39, 0xb3, 0xe5, 0x63, 0xca, 0x8b,
			0xfe, 0xd3, 0xef, 0x60, 0x83, 0xf6, 0x8e, 0x70,
			0xb6, 0x67, 0xc7, 0x77, 0xed, 0x23, 0xef, 0x4c,
			0xf0, 0xed, 0x2d, 0x07, 0x59, 0x6f, 0xc1, 0x01,
			0x34, 0x37, 0x08, 0xab, 0xd9, 0x1f, 0x09, 0xb1,
			0xce, 0x5b, 0x17, 0xff, 0x74, 0xf8, 0x9c, 0xd5,
			0x2c, 0x56, 0x39, 0x79, 0x0f, 0x69, 0x44, 0x75,
			0x58, 0x27, 0x01, 0xc4, 0xbf, 0xa7, 0xa1, 0x1d,
			0x90, 0x17, 0x77, 0x86, 0x5a, 0x3f, 0xd9, 0xd1,
			0x0e, 0xa0, 0x10, 0xf8, 0xec, 0x1e, 0xa5, 0x7f,
			0x5e, 0x36, 0xd1, 0xe3, 0x04, 0x2c, 0x70, 0xf7,
			0x8e, 0xc0, 0x98, 0x2f, 0x6c, 0x94, 0x2b, 0x41,
			0xb7, 0x60, 0x00, 0xb7, 0x2e, 0xb8, 0x02, 0x8d,
			0xb8, 0xb0, 0xd3, 0x86, 0xba, 0x1d, 0xd7, 0x90,
			0xd6, 0xb6, 0xe1, 0xfc, 0xd7, 0xd8, 0x28, 0x06,
			0x63, 0x9b, 0xce, 0x61, 0x24, 0x79, 0xc0, 0x70,
			0x52, 0xd0, 0xb6, 0xd4, 0x28, 0x95, 0x24, 0x87,
			0x03, 0x1f, 0xb7, 0x9a, 0xda, 0xa3, 0xfb, 0x52,
			0x5b, 0x68, 0xe7, 0x4c, 0x8c, 0x24, 0xe1, 0x42,
			0xf7, 0xd5, 0xfd, 0xad, 0x06, 0x32, 0x9f, 0xba,
			0xc1, 0xfc, 0xdd, 0xc6, 0xfc, 0xfc, 0xb3, 0x38,
			0x74, 0x56, 0x58, 0x40, 0x02, 0x37, 0x52, 0x2c,
			0x55, 0xcc, 0xb3, 0x9e, 0x7a, 0xe9, 0xd4, 0x38,
			0x41, 0x5e, 0x0c, 0x35, 0xe2, 0x11, 0xd1, 0x13,
			0xf8, 0xb7, 0x8d, 0x72, 0x6b, 0x22, 0x2a, 0xb0,
			0xdb, 0x08, 0xba, 0x35, 0xb9, 0x3f, 0xc8, 0xd3,
			0x24, 0x90, 0xec, 0x58, 0xd2, 0x09, 0xc7, 0x2d,
			0xed, 0x38, 0x80, 0x36, 0x72, 0x43, 0x27, 0x49,
			0x4a, 0x80, 0x8a, 0xa2, 0xe8, 0xd3, 0xda, 0x30,
			0x7d, 0xb6, 0x82, 0x37, 0x86, 0x92, 0x86, 0x3e,
			0x08, 0xb2, 0x28, 0x5a, 0x55, 0x44, 0x24, 0x7d,
			0x40, 0x48, 0x8a, 0xb6, 0x89, 0x58, 0x08, 0xa0,
			0xd6, 0x6d, 0x3a, 0x17, 0xbf, 0xf6, 0x54, 0xa2,
			0xf5, 0xd3, 0x8c, 0x0f, 0x78, 0x12, 0x57, 0x8b,
			0xd5, 0xc2, 0xfd, 0x58, 0x5b, 0x7f, 0x38, 0xe3,
			0xcc, 0xb7, 0x7c, 0x48, 0xb3, 0x20, 0xe8, 0x81,
			0x14, 0x32, 0x45, 0x05, 0xe0, 0xdb, 0x9f, 0x75,
			0x85, 0xb4, 0x6a, 0xfc, 0x95, 0xe3, 0x54, 0x22,
			0x12, 0xee, 0x30, 0xfe, 0xd8, 0x30, 0xef, 0x34,
			0x50, 0xab, 0x46, 0x30, 0x98, 0x2f, 0xb7, 0xc0,
			0x15, 0xa2, 0x83, 0xb6, 0xf2, 0x06, 0x21, 0xa2,
			0xc3, 0x26, 0x37, 0x14, 0xd1, 0x4d, 0xb5, 0x10,
			0x52, 0x76, 0x4d, 0x6a, 0xee, 0xb5, 0x2b, 0x15,
			0xb7, 0xf9, 0x51, 0xe8, 0x2a, 0xaf, 0xc7, 0xfa,
			0x77, 0xaf, 0xb0, 0x05, 0x4d, 0xd1, 0x68, 0x8e,
			0x74, 0x05, 0x9f, 0x9d, 0x93, 0xa5, 0x3e, 0x7f,
			0x4e, 0x5f, 0x9d, 0xcb, 0x09, 0xc7, 0x83, 0xe3,
			0x02, 0x9d, 0x27, 0x1f, 0xef, 0x85, 0x05, 0x8d,
			0xec, 0x55, 0x88, 0x0f, 0x0d, 0x7c, 0x4c, 0xe8,
			0xa1, 0x75, 0xa0, 0xd8, 0x06, 0x47, 0x14, 0xef,
			0xaa, 0x61, 0xcf, 0x26, 0x15, 0xad, 0xd8, 0xa3,
			0xaa, 0x75, 0xf2, 0x78, 0x4a, 0x5a, 0x61, 0xdf,
			0x8b, 0xc7, 0x04, 0xbc, 0xb2, 0x32, 0xd2, 0x7e,
			0x42, 0xee, 0xb4, 0x2f, 0x51, 0xff, 0x7b, 0x2e,
			0xd3, 0x02, 0xe8, 0xdc, 0x5d, 0x0d, 0x50, 0xdc,
			0xae, 0xb7, 0x46, 0xf9, 0xa8, 0xe6, 0xd0, 0x16,
			0xcc, 0xe6, 0x2c, 0x81, 0xc7, 0xad, 0xe9, 0xf0,
			0x05, 0x72, 0x6d, 0x3d, 0x0a, 0x7a, 0xa9, 0x02,
			0xac, 0x82, 0x93, 0x6e, 0xb6, 0x1c, 0x28, 0xfc,
			0x44, 0x12, 0xfb, 0x73, 0x77, 0xd4, 0x13, 0x39,
			0x29, 0x88, 0x8a, 0xf3, 0x5c, 0xa6, 0x36, 0xa0,
			0x2a, 0xed, 0x7e, 0xb1, 0x1d, 0xd6, 0x4c, 0x6b,
			0x41, 0x01, 0x18, 0x5d, 0x5d, 0x07, 0x97, 0xa6,
			0x4b, 0xef, 0x31, 0x18, 0xea, 0xac, 0xb1, 0x84,
			0x21, 0xed, 0xda, 0x86,
		},
		.rlen = 4100,
	},
};

static struct cipher_testvec aes_ctr_dec_tv_template[] = {
	{ /* From RFC 3686 */
		.key	= { 0xae, 0x68, 0x52, 0xf8, 0x12, 0x10, 0x67, 0xcc,
			    0x4b, 0xf7, 0xa5, 0x76, 0x55, 0x77, 0xf3, 0x9e,
			    0x00, 0x00, 0x00, 0x30 },
		.klen	= 20,
		.iv 	= { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
		.input	= { 0xe4, 0x09, 0x5d, 0x4f, 0xb7, 0xa7, 0xb3, 0x79,
			    0x2d, 0x61, 0x75, 0xa3, 0x26, 0x13, 0x11, 0xb8 },
		.ilen	= 16,
		.result	= { "Single block msg" },
		.rlen	= 16,
	}, {
		.key	= { 0x7e, 0x24, 0x06, 0x78, 0x17, 0xfa, 0xe0, 0xd7,
			    0x43, 0xd6, 0xce, 0x1f, 0x32, 0x53, 0x91, 0x63,
			    0x00, 0x6c, 0xb6, 0xdb },
		.klen	= 20,
		.iv 	= { 0xc0, 0x54, 0x3b, 0x59, 0xda, 0x48, 0xd9, 0x0b },
		.input	= { 0x51, 0x04, 0xa1, 0x06, 0x16, 0x8a, 0x72, 0xd9,
			    0x79, 0x0d, 0x41, 0xee, 0x8e, 0xda, 0xd3, 0x88,
			    0xeb, 0x2e, 0x1e, 0xfc, 0x46, 0xda, 0x57, 0xc8,
			    0xfc, 0xe6, 0x30, 0xdf, 0x91, 0x41, 0xbe, 0x28 },
		.ilen 	= 32,
		.result	= { 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
			    0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
			    0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
			    0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f },
		.rlen	= 32,
	}, {
		.key 	= { 0x16, 0xaf, 0x5b, 0x14, 0x5f, 0xc9, 0xf5, 0x79,
			    0xc1, 0x75, 0xf9, 0x3e, 0x3b, 0xfb, 0x0e, 0xed,
			    0x86, 0x3d, 0x06, 0xcc, 0xfd, 0xb7, 0x85, 0x15,
			    0x00, 0x00, 0x00, 0x48 },
		.klen 	= 28,
		.iv	= { 0x36, 0x73, 0x3c, 0x14, 0x7d, 0x6d, 0x93, 0xcb },
		.input	= { 0x4b, 0x55, 0x38, 0x4f, 0xe2, 0x59, 0xc9, 0xc8,
			    0x4e, 0x79, 0x35, 0xa0, 0x03, 0xcb, 0xe9, 0x28 },
		.ilen 	= 16,
		.result	= { "Single block msg" },
		.rlen	= 16,
	}, {
		.key	= { 0x7c, 0x5c, 0xb2, 0x40, 0x1b, 0x3d, 0xc3, 0x3c,
			    0x19, 0xe7, 0x34, 0x08, 0x19, 0xe0, 0xf6, 0x9c,
			    0x67, 0x8c, 0x3d, 0xb8, 0xe6, 0xf6, 0xa9, 0x1a,
			    0x00, 0x96, 0xb0, 0x3b },
		.klen	= 28,
		.iv 	= { 0x02, 0x0c, 0x6e, 0xad, 0xc2, 0xcb, 0x50, 0x0d },
		.input	= { 0x45, 0x32, 0x43, 0xfc, 0x60, 0x9b, 0x23, 0x32,
			    0x7e, 0xdf, 0xaa, 0xfa, 0x71, 0x31, 0xcd, 0x9f,
			    0x84, 0x90, 0x70, 0x1c, 0x5a, 0xd4, 0xa7, 0x9c,
			    0xfc, 0x1f, 0xe0, 0xff, 0x42, 0xf4, 0xfb, 0x00 },
		.ilen	= 32,
		.result	= { 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
			    0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
			    0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
			    0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f },
		.rlen 	= 32,
	}, { 
		.key 	= { 0x77, 0x6b, 0xef, 0xf2, 0x85, 0x1d, 0xb0, 0x6f,
			    0x4c, 0x8a, 0x05, 0x42, 0xc8, 0x69, 0x6f, 0x6c,
			    0x6a, 0x81, 0xaf, 0x1e, 0xec, 0x96, 0xb4, 0xd3,
			    0x7f, 0xc1, 0xd6, 0x89, 0xe6, 0xc1, 0xc1, 0x04,
			    0x00, 0x00, 0x00, 0x60 },
		.klen	= 36,
		.iv 	= { 0xdb, 0x56, 0x72, 0xc9, 0x7a, 0xa8, 0xf0, 0xb2 },
		.input	= { 0x14, 0x5a, 0xd0, 0x1d, 0xbf, 0x82, 0x4e, 0xc7,
			    0x56, 0x08, 0x63, 0xdc, 0x71, 0xe3, 0xe0, 0xc0 },
		.ilen	= 16,
		.result	= { "Single block msg" },
		.rlen 	= 16,
	}, {
		.key	= { 0xf6, 0xd6, 0x6d, 0x6b, 0xd5, 0x2d, 0x59, 0xbb,
			    0x07, 0x96, 0x36, 0x58, 0x79, 0xef, 0xf8, 0x86,
			    0xc6, 0x6d, 0xd5, 0x1a, 0x5b, 0x6a, 0x99, 0x74,
			    0x4b, 0x50, 0x59, 0x0c, 0x87, 0xa2, 0x38, 0x84,
			    0x00, 0xfa, 0xac, 0x24 },
		.klen 	= 36,
		.iv	= { 0xc1, 0x58, 0x5e, 0xf1, 0x5a, 0x43, 0xd8, 0x75 },
		.input	= { 0xf0, 0x5e, 0x23, 0x1b, 0x38, 0x94, 0x61, 0x2c,
			    0x49, 0xee, 0x00, 0x0b, 0x80, 0x4e, 0xb2, 0xa9,
			    0xb8, 0x30, 0x6b, 0x50, 0x8f, 0x83, 0x9d, 0x6a,
			    0x55, 0x30, 0x83, 0x1d, 0x93, 0x44, 0xaf, 0x1c },
		.ilen	= 32,
		.result	= { 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
			    0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
			    0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
			    0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f },
		.rlen	= 32,
	},
};

static struct aead_testvec aes_gcm_enc_tv_template[] = {
	{ /* From McGrew & Viega - http://citeseer.ist.psu.edu/656989.html */
		.klen	= 16,
		.result	= { 0x58, 0xe2, 0xfc, 0xce, 0xfa, 0x7e, 0x30, 0x61,
			    0x36, 0x7f, 0x1d, 0x57, 0xa4, 0xe7, 0x45, 0x5a },
		.rlen	= 16,
	}, {
		.klen	= 16,
		.ilen	= 16,
		.result = { 0x03, 0x88, 0xda, 0xce, 0x60, 0xb6, 0xa3, 0x92,
			    0xf3, 0x28, 0xc2, 0xb9, 0x71, 0xb2, 0xfe, 0x78,
			    0xab, 0x6e, 0x47, 0xd4, 0x2c, 0xec, 0x13, 0xbd,
			    0xf5, 0x3a, 0x67, 0xb2, 0x12, 0x57, 0xbd, 0xdf },
		.rlen	= 32,
	}, {
		.key	= { 0xfe, 0xff, 0xe9, 0x92, 0x86, 0x65, 0x73, 0x1c,
			    0x6d, 0x6a, 0x8f, 0x94, 0x67, 0x30, 0x83, 0x08 },
		.klen	= 16,
		.iv	= { 0xca, 0xfe, 0xba, 0xbe, 0xfa, 0xce, 0xdb, 0xad,
			    0xde, 0xca, 0xf8, 0x88 },
		.input	= { 0xd9, 0x31, 0x32, 0x25, 0xf8, 0x84, 0x06, 0xe5,
			    0xa5, 0x59, 0x09, 0xc5, 0xaf, 0xf5, 0x26, 0x9a,
			    0x86, 0xa7, 0xa9, 0x53, 0x15, 0x34, 0xf7, 0xda,
			    0x2e, 0x4c, 0x30, 0x3d, 0x8a, 0x31, 0x8a, 0x72,
			    0x1c, 0x3c, 0x0c, 0x95, 0x95, 0x68, 0x09, 0x53,
			    0x2f, 0xcf, 0x0e, 0x24, 0x49, 0xa6, 0xb5, 0x25,
			    0xb1, 0x6a, 0xed, 0xf5, 0xaa, 0x0d, 0xe6, 0x57,
			    0xba, 0x63, 0x7b, 0x39, 0x1a, 0xaf, 0xd2, 0x55 },
		.ilen	= 64,
		.result = { 0x42, 0x83, 0x1e, 0xc2, 0x21, 0x77, 0x74, 0x24,
			    0x4b, 0x72, 0x21, 0xb7, 0x84, 0xd0, 0xd4, 0x9c,
			    0xe3, 0xaa, 0x21, 0x2f, 0x2c, 0x02, 0xa4, 0xe0,
			    0x35, 0xc1, 0x7e, 0x23, 0x29, 0xac, 0xa1, 0x2e,
			    0x21, 0xd5, 0x14, 0xb2, 0x54, 0x66, 0x93, 0x1c,
			    0x7d, 0x8f, 0x6a, 0x5a, 0xac, 0x84, 0xaa, 0x05,
			    0x1b, 0xa3, 0x0b, 0x39, 0x6a, 0x0a, 0xac, 0x97,
			    0x3d, 0x58, 0xe0, 0x91, 0x47, 0x3f, 0x59, 0x85,
			    0x4d, 0x5c, 0x2a, 0xf3, 0x27, 0xcd, 0x64, 0xa6,
			    0x2c, 0xf3, 0x5a, 0xbd, 0x2b, 0xa6, 0xfa, 0xb4 },
		.rlen	= 80,
	}, {
		.key	= { 0xfe, 0xff, 0xe9, 0x92, 0x86, 0x65, 0x73, 0x1c,
			    0x6d, 0x6a, 0x8f, 0x94, 0x67, 0x30, 0x83, 0x08 },
		.klen	= 16,
		.iv	= { 0xca, 0xfe, 0xba, 0xbe, 0xfa, 0xce, 0xdb, 0xad,
			    0xde, 0xca, 0xf8, 0x88 },
		.input	= { 0xd9, 0x31, 0x32, 0x25, 0xf8, 0x84, 0x06, 0xe5,
			    0xa5, 0x59, 0x09, 0xc5, 0xaf, 0xf5, 0x26, 0x9a,
			    0x86, 0xa7, 0xa9, 0x53, 0x15, 0x34, 0xf7, 0xda,
			    0x2e, 0x4c, 0x30, 0x3d, 0x8a, 0x31, 0x8a, 0x72,
			    0x1c, 0x3c, 0x0c, 0x95, 0x95, 0x68, 0x09, 0x53,
			    0x2f, 0xcf, 0x0e, 0x24, 0x49, 0xa6, 0xb5, 0x25,
			    0xb1, 0x6a, 0xed, 0xf5, 0xaa, 0x0d, 0xe6, 0x57,
			    0xba, 0x63, 0x7b, 0x39 },
		.ilen	= 60,
		.assoc	= { 0xfe, 0xed, 0xfa, 0xce, 0xde, 0xad, 0xbe, 0xef,
			    0xfe, 0xed, 0xfa, 0xce, 0xde, 0xad, 0xbe, 0xef,
			    0xab, 0xad, 0xda, 0xd2 },
		.alen	= 20,
		.result = { 0x42, 0x83, 0x1e, 0xc2, 0x21, 0x77, 0x74, 0x24,
			    0x4b, 0x72, 0x21, 0xb7, 0x84, 0xd0, 0xd4, 0x9c,
			    0xe3, 0xaa, 0x21, 0x2f, 0x2c, 0x02, 0xa4, 0xe0,
			    0x35, 0xc1, 0x7e, 0x23, 0x29, 0xac, 0xa1, 0x2e,
			    0x21, 0xd5, 0x14, 0xb2, 0x54, 0x66, 0x93, 0x1c,
			    0x7d, 0x8f, 0x6a, 0x5a, 0xac, 0x84, 0xaa, 0x05,
			    0x1b, 0xa3, 0x0b, 0x39, 0x6a, 0x0a, 0xac, 0x97,
			    0x3d, 0x58, 0xe0, 0x91,
			    0x5b, 0xc9, 0x4f, 0xbc, 0x32, 0x21, 0xa5, 0xdb,
			    0x94, 0xfa, 0xe9, 0x5a, 0xe7, 0x12, 0x1a, 0x47 },
		.rlen	= 76,
	}, {
		.klen	= 24,
		.result	= { 0xcd, 0x33, 0xb2, 0x8a, 0xc7, 0x73, 0xf7, 0x4b,
			    0xa0, 0x0e, 0xd1, 0xf3, 0x12, 0x57, 0x24, 0x35 },
		.rlen	= 16,
	}, {
		.klen	= 24,
		.ilen	= 16,
		.result = { 0x98, 0xe7, 0x24, 0x7c, 0x07, 0xf0, 0xfe, 0x41,
			    0x1c, 0x26, 0x7e, 0x43, 0x84, 0xb0, 0xf6, 0x00,
			    0x2f, 0xf5, 0x8d, 0x80, 0x03, 0x39, 0x27, 0xab,
			    0x8e, 0xf4, 0xd4, 0x58, 0x75, 0x14, 0xf0, 0xfb },
		.rlen	= 32,
	}, {
		.key	= { 0xfe, 0xff, 0xe9, 0x92, 0x86, 0x65, 0x73, 0x1c,
			    0x6d, 0x6a, 0x8f, 0x94, 0x67, 0x30, 0x83, 0x08,
			    0xfe, 0xff, 0xe9, 0x92, 0x86, 0x65, 0x73, 0x1c },
		.klen	= 24,
		.iv	= { 0xca, 0xfe, 0xba, 0xbe, 0xfa, 0xce, 0xdb, 0xad,
			    0xde, 0xca, 0xf8, 0x88 },
		.input	= { 0xd9, 0x31, 0x32, 0x25, 0xf8, 0x84, 0x06, 0xe5,
			    0xa5, 0x59, 0x09, 0xc5, 0xaf, 0xf5, 0x26, 0x9a,
			    0x86, 0xa7, 0xa9, 0x53, 0x15, 0x34, 0xf7, 0xda,
			    0x2e, 0x4c, 0x30, 0x3d, 0x8a, 0x31, 0x8a, 0x72,
			    0x1c, 0x3c, 0x0c, 0x95, 0x95, 0x68, 0x09, 0x53,
			    0x2f, 0xcf, 0x0e, 0x24, 0x49, 0xa6, 0xb5, 0x25,
			    0xb1, 0x6a, 0xed, 0xf5, 0xaa, 0x0d, 0xe6, 0x57,
			    0xba, 0x63, 0x7b, 0x39, 0x1a, 0xaf, 0xd2, 0x55 },
		.ilen	= 64,
		.result = { 0x39, 0x80, 0xca, 0x0b, 0x3c, 0x00, 0xe8, 0x41,
			    0xeb, 0x06, 0xfa, 0xc4, 0x87, 0x2a, 0x27, 0x57,
			    0x85, 0x9e, 0x1c, 0xea, 0xa6, 0xef, 0xd9, 0x84,
			    0x62, 0x85, 0x93, 0xb4, 0x0c, 0xa1, 0xe1, 0x9c,
			    0x7d, 0x77, 0x3d, 0x00, 0xc1, 0x44, 0xc5, 0x25,
			    0xac, 0x61, 0x9d, 0x18, 0xc8, 0x4a, 0x3f, 0x47,
			    0x18, 0xe2, 0x44, 0x8b, 0x2f, 0xe3, 0x24, 0xd9,
			    0xcc, 0xda, 0x27, 0x10, 0xac, 0xad, 0xe2, 0x56,
			    0x99, 0x24, 0xa7, 0xc8, 0x58, 0x73, 0x36, 0xbf,
			    0xb1, 0x18, 0x02, 0x4d, 0xb8, 0x67, 0x4a, 0x14 },
		.rlen	= 80,
	}, {
		.key	= { 0xfe, 0xff, 0xe9, 0x92, 0x86, 0x65, 0x73, 0x1c,
			    0x6d, 0x6a, 0x8f, 0x94, 0x67, 0x30, 0x83, 0x08,
			    0xfe, 0xff, 0xe9, 0x92, 0x86, 0x65, 0x73, 0x1c },
		.klen	= 24,
		.iv	= { 0xca, 0xfe, 0xba, 0xbe, 0xfa, 0xce, 0xdb, 0xad,
			    0xde, 0xca, 0xf8, 0x88 },
		.input	= { 0xd9, 0x31, 0x32, 0x25, 0xf8, 0x84, 0x06, 0xe5,
			    0xa5, 0x59, 0x09, 0xc5, 0xaf, 0xf5, 0x26, 0x9a,
			    0x86, 0xa7, 0xa9, 0x53, 0x15, 0x34, 0xf7, 0xda,
			    0x2e, 0x4c, 0x30, 0x3d, 0x8a, 0x31, 0x8a, 0x72,
			    0x1c, 0x3c, 0x0c, 0x95, 0x95, 0x68, 0x09, 0x53,
			    0x2f, 0xcf, 0x0e, 0x24, 0x49, 0xa6, 0xb5, 0x25,
			    0xb1, 0x6a, 0xed, 0xf5, 0xaa, 0x0d, 0xe6, 0x57,
			    0xba, 0x63, 0x7b, 0x39 },
		.ilen	= 60,
		.assoc	= { 0xfe, 0xed, 0xfa, 0xce, 0xde, 0xad, 0xbe, 0xef,
			    0xfe, 0xed, 0xfa, 0xce, 0xde, 0xad, 0xbe, 0xef,
			    0xab, 0xad, 0xda, 0xd2 },
		.alen	= 20,
		.result = { 0x39, 0x80, 0xca, 0x0b, 0x3c, 0x00, 0xe8, 0x41,
			    0xeb, 0x06, 0xfa, 0xc4, 0x87, 0x2a, 0x27, 0x57,
			    0x85, 0x9e, 0x1c, 0xea, 0xa6, 0xef, 0xd9, 0x84,
			    0x62, 0x85, 0x93, 0xb4, 0x0c, 0xa1, 0xe1, 0x9c,
			    0x7d, 0x77, 0x3d, 0x00, 0xc1, 0x44, 0xc5, 0x25,
			    0xac, 0x61, 0x9d, 0x18, 0xc8, 0x4a, 0x3f, 0x47,
			    0x18, 0xe2, 0x44, 0x8b, 0x2f, 0xe3, 0x24, 0xd9,
			    0xcc, 0xda, 0x27, 0x10,
			    0x25, 0x19, 0x49, 0x8e, 0x80, 0xf1, 0x47, 0x8f,
			    0x37, 0xba, 0x55, 0xbd, 0x6d, 0x27, 0x61, 0x8c },
		.rlen	= 76,
		.np	= 2,
		.tap	= { 32, 28 },
		.anp	= 2,
		.atap	= { 8, 12 }
	}, {
		.klen	= 32,
		.result	= { 0x53, 0x0f, 0x8a, 0xfb, 0xc7, 0x45, 0x36, 0xb9,
			    0xa9, 0x63, 0xb4, 0xf1, 0xc4, 0xcb, 0x73, 0x8b },
		.rlen	= 16,
	}
};

static struct aead_testvec aes_gcm_dec_tv_template[] = {
	{ /* From McGrew & Viega - http://citeseer.ist.psu.edu/656989.html */
		.klen	= 32,
		.input	= { 0xce, 0xa7, 0x40, 0x3d, 0x4d, 0x60, 0x6b, 0x6e,
			    0x07, 0x4e, 0xc5, 0xd3, 0xba, 0xf3, 0x9d, 0x18,
			    0xd0, 0xd1, 0xc8, 0xa7, 0x99, 0x99, 0x6b, 0xf0,
			    0x26, 0x5b, 0x98, 0xb5, 0xd4, 0x8a, 0xb9, 0x19 },
		.ilen	= 32,
		.rlen	= 16,
	}, {
		.key	= { 0xfe, 0xff, 0xe9, 0x92, 0x86, 0x65, 0x73, 0x1c,
			    0x6d, 0x6a, 0x8f, 0x94, 0x67, 0x30, 0x83, 0x08,
			    0xfe, 0xff, 0xe9, 0x92, 0x86, 0x65, 0x73, 0x1c,
			    0x6d, 0x6a, 0x8f, 0x94, 0x67, 0x30, 0x83, 0x08 },
		.klen	= 32,
		.iv	= { 0xca, 0xfe, 0xba, 0xbe, 0xfa, 0xce, 0xdb, 0xad,
			    0xde, 0xca, 0xf8, 0x88 },
		.input	= { 0x52, 0x2d, 0xc1, 0xf0, 0x99, 0x56, 0x7d, 0x07,
			    0xf4, 0x7f, 0x37, 0xa3, 0x2a, 0x84, 0x42, 0x7d,
			    0x64, 0x3a, 0x8c, 0xdc, 0xbf, 0xe5, 0xc0, 0xc9,
			    0x75, 0x98, 0xa2, 0xbd, 0x25, 0x55, 0xd1, 0xaa,
			    0x8c, 0xb0, 0x8e, 0x48, 0x59, 0x0d, 0xbb, 0x3d,
			    0xa7, 0xb0, 0x8b, 0x10, 0x56, 0x82, 0x88, 0x38,
			    0xc5, 0xf6, 0x1e, 0x63, 0x93, 0xba, 0x7a, 0x0a,
			    0xbc, 0xc9, 0xf6, 0x62, 0x89, 0x80, 0x15, 0xad,
			    0xb0, 0x94, 0xda, 0xc5, 0xd9, 0x34, 0x71, 0xbd,
			    0xec, 0x1a, 0x50, 0x22, 0x70, 0xe3, 0xcc, 0x6c },
		.ilen	= 80,
		.result = { 0xd9, 0x31, 0x32, 0x25, 0xf8, 0x84, 0x06, 0xe5,
			    0xa5, 0x59, 0x09, 0xc5, 0xaf, 0xf5, 0x26, 0x9a,
			    0x86, 0xa7, 0xa9, 0x53, 0x15, 0x34, 0xf7, 0xda,
			    0x2e, 0x4c, 0x30, 0x3d, 0x8a, 0x31, 0x8a, 0x72,
			    0x1c, 0x3c, 0x0c, 0x95, 0x95, 0x68, 0x09, 0x53,
			    0x2f, 0xcf, 0x0e, 0x24, 0x49, 0xa6, 0xb5, 0x25,
			    0xb1, 0x6a, 0xed, 0xf5, 0xaa, 0x0d, 0xe6, 0x57,
			    0xba, 0x63, 0x7b, 0x39, 0x1a, 0xaf, 0xd2, 0x55 },
		.rlen	= 64,
	}, {
		.key	= { 0xfe, 0xff, 0xe9, 0x92, 0x86, 0x65, 0x73, 0x1c,
			    0x6d, 0x6a, 0x8f, 0x94, 0x67, 0x30, 0x83, 0x08,
			    0xfe, 0xff, 0xe9, 0x92, 0x86, 0x65, 0x73, 0x1c,
			    0x6d, 0x6a, 0x8f, 0x94, 0x67, 0x30, 0x83, 0x08 },
		.klen	= 32,
		.iv	= { 0xca, 0xfe, 0xba, 0xbe, 0xfa, 0xce, 0xdb, 0xad,
			    0xde, 0xca, 0xf8, 0x88 },
		.input	= { 0x52, 0x2d, 0xc1, 0xf0, 0x99, 0x56, 0x7d, 0x07,
			    0xf4, 0x7f, 0x37, 0xa3, 0x2a, 0x84, 0x42, 0x7d,
			    0x64, 0x3a, 0x8c, 0xdc, 0xbf, 0xe5, 0xc0, 0xc9,
			    0x75, 0x98, 0xa2, 0xbd, 0x25, 0x55, 0xd1, 0xaa,
			    0x8c, 0xb0, 0x8e, 0x48, 0x59, 0x0d, 0xbb, 0x3d,
			    0xa7, 0xb0, 0x8b, 0x10, 0x56, 0x82, 0x88, 0x38,
			    0xc5, 0xf6, 0x1e, 0x63, 0x93, 0xba, 0x7a, 0x0a,
			    0xbc, 0xc9, 0xf6, 0x62,
			    0x76, 0xfc, 0x6e, 0xce, 0x0f, 0x4e, 0x17, 0x68,
			    0xcd, 0xdf, 0x88, 0x53, 0xbb, 0x2d, 0x55, 0x1b },
		.ilen	= 76,
		.assoc	= { 0xfe, 0xed, 0xfa, 0xce, 0xde, 0xad, 0xbe, 0xef,
			    0xfe, 0xed, 0xfa, 0xce, 0xde, 0xad, 0xbe, 0xef,
			    0xab, 0xad, 0xda, 0xd2 },
		.alen	= 20,
		.result = { 0xd9, 0x31, 0x32, 0x25, 0xf8, 0x84, 0x06, 0xe5,
			    0xa5, 0x59, 0x09, 0xc5, 0xaf, 0xf5, 0x26, 0x9a,
			    0x86, 0xa7, 0xa9, 0x53, 0x15, 0x34, 0xf7, 0xda,
			    0x2e, 0x4c, 0x30, 0x3d, 0x8a, 0x31, 0x8a, 0x72,
			    0x1c, 0x3c, 0x0c, 0x95, 0x95, 0x68, 0x09, 0x53,
			    0x2f, 0xcf, 0x0e, 0x24, 0x49, 0xa6, 0xb5, 0x25,
			    0xb1, 0x6a, 0xed, 0xf5, 0xaa, 0x0d, 0xe6, 0x57,
			    0xba, 0x63, 0x7b, 0x39 },
		.rlen	= 60,
		.np     = 2,
		.tap    = { 48, 28 },
		.anp	= 3,
		.atap	= { 8, 8, 4 }
	}, {
		.key	= { 0xfe, 0xff, 0xe9, 0x92, 0x86, 0x65, 0x73, 0x1c,
			    0x6d, 0x6a, 0x8f, 0x94, 0x67, 0x30, 0x83, 0x08 },
		.klen	= 16,
		.iv	= { 0xca, 0xfe, 0xba, 0xbe, 0xfa, 0xce, 0xdb, 0xad,
			    0xde, 0xca, 0xf8, 0x88 },
		.input	= { 0x42, 0x83, 0x1e, 0xc2, 0x21, 0x77, 0x74, 0x24,
			    0x4b, 0x72, 0x21, 0xb7, 0x84, 0xd0, 0xd4, 0x9c,
			    0xe3, 0xaa, 0x21, 0x2f, 0x2c, 0x02, 0xa4, 0xe0,
			    0x35, 0xc1, 0x7e, 0x23, 0x29, 0xac, 0xa1, 0x2e,
			    0x21, 0xd5, 0x14, 0xb2, 0x54, 0x66, 0x93, 0x1c,
			    0x7d, 0x8f, 0x6a, 0x5a, 0xac, 0x84, 0xaa, 0x05,
			    0x1b, 0xa3, 0x0b, 0x39, 0x6a, 0x0a, 0xac, 0x97,
			    0x3d, 0x58, 0xe0, 0x91, 0x47, 0x3f, 0x59, 0x85,
			    0x4d, 0x5c, 0x2a, 0xf3, 0x27, 0xcd, 0x64, 0xa6,
			    0x2c, 0xf3, 0x5a, 0xbd, 0x2b, 0xa6, 0xfa, 0xb4 },
		.ilen	= 80,
		.result = { 0xd9, 0x31, 0x32, 0x25, 0xf8, 0x84, 0x06, 0xe5,
			    0xa5, 0x59, 0x09, 0xc5, 0xaf, 0xf5, 0x26, 0x9a,
			    0x86, 0xa7, 0xa9, 0x53, 0x15, 0x34, 0xf7, 0xda,
			    0x2e, 0x4c, 0x30, 0x3d, 0x8a, 0x31, 0x8a, 0x72,
			    0x1c, 0x3c, 0x0c, 0x95, 0x95, 0x68, 0x09, 0x53,
			    0x2f, 0xcf, 0x0e, 0x24, 0x49, 0xa6, 0xb5, 0x25,
			    0xb1, 0x6a, 0xed, 0xf5, 0xaa, 0x0d, 0xe6, 0x57,
			    0xba, 0x63, 0x7b, 0x39, 0x1a, 0xaf, 0xd2, 0x55 },
		.rlen	= 64,
	}, {
		.key	= { 0xfe, 0xff, 0xe9, 0x92, 0x86, 0x65, 0x73, 0x1c,
			    0x6d, 0x6a, 0x8f, 0x94, 0x67, 0x30, 0x83, 0x08 },
		.klen	= 16,
		.iv	= { 0xca, 0xfe, 0xba, 0xbe, 0xfa, 0xce, 0xdb, 0xad,
			    0xde, 0xca, 0xf8, 0x88 },
		.input	= { 0x42, 0x83, 0x1e, 0xc2, 0x21, 0x77, 0x74, 0x24,
			    0x4b, 0x72, 0x21, 0xb7, 0x84, 0xd0, 0xd4, 0x9c,
			    0xe3, 0xaa, 0x21, 0x2f, 0x2c, 0x02, 0xa4, 0xe0,
			    0x35, 0xc1, 0x7e, 0x23, 0x29, 0xac, 0xa1, 0x2e,
			    0x21, 0xd5, 0x14, 0xb2, 0x54, 0x66, 0x93, 0x1c,
			    0x7d, 0x8f, 0x6a, 0x5a, 0xac, 0x84, 0xaa, 0x05,
			    0x1b, 0xa3, 0x0b, 0x39, 0x6a, 0x0a, 0xac, 0x97,
			    0x3d, 0x58, 0xe0, 0x91,
			    0x5b, 0xc9, 0x4f, 0xbc, 0x32, 0x21, 0xa5, 0xdb,
			    0x94, 0xfa, 0xe9, 0x5a, 0xe7, 0x12, 0x1a, 0x47 },
		.ilen	= 76,
		.assoc	= { 0xfe, 0xed, 0xfa, 0xce, 0xde, 0xad, 0xbe, 0xef,
			    0xfe, 0xed, 0xfa, 0xce, 0xde, 0xad, 0xbe, 0xef,
			    0xab, 0xad, 0xda, 0xd2 },
		.alen	= 20,
		.result = { 0xd9, 0x31, 0x32, 0x25, 0xf8, 0x84, 0x06, 0xe5,
			    0xa5, 0x59, 0x09, 0xc5, 0xaf, 0xf5, 0x26, 0x9a,
			    0x86, 0xa7, 0xa9, 0x53, 0x15, 0x34, 0xf7, 0xda,
			    0x2e, 0x4c, 0x30, 0x3d, 0x8a, 0x31, 0x8a, 0x72,
			    0x1c, 0x3c, 0x0c, 0x95, 0x95, 0x68, 0x09, 0x53,
			    0x2f, 0xcf, 0x0e, 0x24, 0x49, 0xa6, 0xb5, 0x25,
			    0xb1, 0x6a, 0xed, 0xf5, 0xaa, 0x0d, 0xe6, 0x57,
			    0xba, 0x63, 0x7b, 0x39 },
		.rlen	= 60,
	}, {
		.klen	= 24,
		.input	= { 0x98, 0xe7, 0x24, 0x7c, 0x07, 0xf0, 0xfe, 0x41,
			    0x1c, 0x26, 0x7e, 0x43, 0x84, 0xb0, 0xf6, 0x00,
			    0x2f, 0xf5, 0x8d, 0x80, 0x03, 0x39, 0x27, 0xab,
			    0x8e, 0xf4, 0xd4, 0x58, 0x75, 0x14, 0xf0, 0xfb },
		.ilen	= 32,
		.rlen	= 16,
	}, {
		.key	= { 0xfe, 0xff, 0xe9, 0x92, 0x86, 0x65, 0x73, 0x1c,
			    0x6d, 0x6a, 0x8f, 0x94, 0x67, 0x30, 0x83, 0x08,
			    0xfe, 0xff, 0xe9, 0x92, 0x86, 0x65, 0x73, 0x1c },
		.klen	= 24,
		.iv	= { 0xca, 0xfe, 0xba, 0xbe, 0xfa, 0xce, 0xdb, 0xad,
			    0xde, 0xca, 0xf8, 0x88 },
		.input	= { 0x39, 0x80, 0xca, 0x0b, 0x3c, 0x00, 0xe8, 0x41,
			    0xeb, 0x06, 0xfa, 0xc4, 0x87, 0x2a, 0x27, 0x57,
			    0x85, 0x9e, 0x1c, 0xea, 0xa6, 0xef, 0xd9, 0x84,
			    0x62, 0x85, 0x93, 0xb4, 0x0c, 0xa1, 0xe1, 0x9c,
			    0x7d, 0x77, 0x3d, 0x00, 0xc1, 0x44, 0xc5, 0x25,
			    0xac, 0x61, 0x9d, 0x18, 0xc8, 0x4a, 0x3f, 0x47,
			    0x18, 0xe2, 0x44, 0x8b, 0x2f, 0xe3, 0x24, 0xd9,
			    0xcc, 0xda, 0x27, 0x10, 0xac, 0xad, 0xe2, 0x56,
			    0x99, 0x24, 0xa7, 0xc8, 0x58, 0x73, 0x36, 0xbf,
			    0xb1, 0x18, 0x02, 0x4d, 0xb8, 0x67, 0x4a, 0x14 },
		.ilen	= 80,
		.result = { 0xd9, 0x31, 0x32, 0x25, 0xf8, 0x84, 0x06, 0xe5,
			    0xa5, 0x59, 0x09, 0xc5, 0xaf, 0xf5, 0x26, 0x9a,
			    0x86, 0xa7, 0xa9, 0x53, 0x15, 0x34, 0xf7, 0xda,
			    0x2e, 0x4c, 0x30, 0x3d, 0x8a, 0x31, 0x8a, 0x72,
			    0x1c, 0x3c, 0x0c, 0x95, 0x95, 0x68, 0x09, 0x53,
			    0x2f, 0xcf, 0x0e, 0x24, 0x49, 0xa6, 0xb5, 0x25,
			    0xb1, 0x6a, 0xed, 0xf5, 0xaa, 0x0d, 0xe6, 0x57,
			    0xba, 0x63, 0x7b, 0x39, 0x1a, 0xaf, 0xd2, 0x55 },
		.rlen	= 64,
	}, {
		.key	= { 0xfe, 0xff, 0xe9, 0x92, 0x86, 0x65, 0x73, 0x1c,
			    0x6d, 0x6a, 0x8f, 0x94, 0x67, 0x30, 0x83, 0x08,
			    0xfe, 0xff, 0xe9, 0x92, 0x86, 0x65, 0x73, 0x1c },
		.klen	= 24,
		.iv	= { 0xca, 0xfe, 0xba, 0xbe, 0xfa, 0xce, 0xdb, 0xad,
			    0xde, 0xca, 0xf8, 0x88 },
		.input	= { 0x39, 0x80, 0xca, 0x0b, 0x3c, 0x00, 0xe8, 0x41,
			    0xeb, 0x06, 0xfa, 0xc4, 0x87, 0x2a, 0x27, 0x57,
			    0x85, 0x9e, 0x1c, 0xea, 0xa6, 0xef, 0xd9, 0x84,
			    0x62, 0x85, 0x93, 0xb4, 0x0c, 0xa1, 0xe1, 0x9c,
			    0x7d, 0x77, 0x3d, 0x00, 0xc1, 0x44, 0xc5, 0x25,
			    0xac, 0x61, 0x9d, 0x18, 0xc8, 0x4a, 0x3f, 0x47,
			    0x18, 0xe2, 0x44, 0x8b, 0x2f, 0xe3, 0x24, 0xd9,
			    0xcc, 0xda, 0x27, 0x10,
			    0x25, 0x19, 0x49, 0x8e, 0x80, 0xf1, 0x47, 0x8f,
			    0x37, 0xba, 0x55, 0xbd, 0x6d, 0x27, 0x61, 0x8c },
		.ilen	= 76,
		.assoc	= { 0xfe, 0xed, 0xfa, 0xce, 0xde, 0xad, 0xbe, 0xef,
			    0xfe, 0xed, 0xfa, 0xce, 0xde, 0xad, 0xbe, 0xef,
			    0xab, 0xad, 0xda, 0xd2 },
		.alen	= 20,
		.result = { 0xd9, 0x31, 0x32, 0x25, 0xf8, 0x84, 0x06, 0xe5,
			    0xa5, 0x59, 0x09, 0xc5, 0xaf, 0xf5, 0x26, 0x9a,
			    0x86, 0xa7, 0xa9, 0x53, 0x15, 0x34, 0xf7, 0xda,
			    0x2e, 0x4c, 0x30, 0x3d, 0x8a, 0x31, 0x8a, 0x72,
			    0x1c, 0x3c, 0x0c, 0x95, 0x95, 0x68, 0x09, 0x53,
			    0x2f, 0xcf, 0x0e, 0x24, 0x49, 0xa6, 0xb5, 0x25,
			    0xb1, 0x6a, 0xed, 0xf5, 0xaa, 0x0d, 0xe6, 0x57,
			    0xba, 0x63, 0x7b, 0x39 },
		.rlen	= 60,
	}
};

static struct aead_testvec aes_ccm_enc_tv_template[] = {
	{ /* From RFC 3610 */
		.key	= { 0xc0, 0xc1, 0xc2, 0xc3, 0xc4, 0xc5, 0xc6, 0xc7,
			    0xc8, 0xc9, 0xca, 0xcb, 0xcc, 0xcd, 0xce, 0xcf },
		.klen	= 16,
		.iv	= { 0x01, 0x00, 0x00, 0x00, 0x03, 0x02, 0x01, 0x00,
			    0xa0, 0xa1, 0xa2, 0xa3, 0xa4, 0xa5, 0x00, 0x00 },
		.assoc	= { 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07 },
		.alen	= 8,
		.input	= { 0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
			    0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
			    0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e },
		.ilen	= 23,
		.result	= { 0x58, 0x8c, 0x97, 0x9a, 0x61, 0xc6, 0x63, 0xd2,
			    0xf0, 0x66, 0xd0, 0xc2, 0xc0, 0xf9, 0x89, 0x80,
			    0x6d, 0x5f, 0x6b, 0x61, 0xda, 0xc3, 0x84, 0x17,
			    0xe8, 0xd1, 0x2c, 0xfd, 0xf9, 0x26, 0xe0 },
		.rlen	= 31,
	}, {
		.key	= { 0xc0, 0xc1, 0xc2, 0xc3, 0xc4, 0xc5, 0xc6, 0xc7,
			    0xc8, 0xc9, 0xca, 0xcb, 0xcc, 0xcd, 0xce, 0xcf },
		.klen	= 16,
		.iv	= { 0x01, 0x00, 0x00, 0x00, 0x07, 0x06, 0x05, 0x04,
			    0xa0, 0xa1, 0xa2, 0xa3, 0xa4, 0xa5, 0x00, 0x00 },
		.assoc	= { 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
			    0x08, 0x09, 0x0a, 0x0b },
		.alen	= 12,
		.input	= { 0x0c, 0x0d, 0x0e, 0x0f, 0x10, 0x11, 0x12, 0x13,
			    0x14, 0x15, 0x16, 0x17, 0x18, 0x19, 0x1a, 0x1b,
			    0x1c, 0x1d, 0x1e, 0x1f },
		.ilen	= 20,
		.result	= { 0xdc, 0xf1, 0xfb, 0x7b, 0x5d, 0x9e, 0x23, 0xfb,
			    0x9d, 0x4e, 0x13, 0x12, 0x53, 0x65, 0x8a, 0xd8,
			    0x6e, 0xbd, 0xca, 0x3e, 0x51, 0xe8, 0x3f, 0x07,
			    0x7d, 0x9c, 0x2d, 0x93 },
		.rlen	= 28,
	}, {
		.key	= { 0xc0, 0xc1, 0xc2, 0xc3, 0xc4, 0xc5, 0xc6, 0xc7,
			    0xc8, 0xc9, 0xca, 0xcb, 0xcc, 0xcd, 0xce, 0xcf },
		.klen	= 16,
		.iv	= { 0x01, 0x00, 0x00, 0x00, 0x0b, 0x0a, 0x09, 0x08,
			    0xa0, 0xa1, 0xa2, 0xa3, 0xa4, 0xa5, 0x00, 0x00 },
		.assoc	= { 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07 },
		.alen	= 8,
		.input	= { 0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
			    0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
			    0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,
			    0x20 },
		.ilen	= 25,
		.result	= { 0x82, 0x53, 0x1a, 0x60, 0xcc, 0x24, 0x94, 0x5a,
			    0x4b, 0x82, 0x79, 0x18, 0x1a, 0xb5, 0xc8, 0x4d,
			    0xf2, 0x1c, 0xe7, 0xf9, 0xb7, 0x3f, 0x42, 0xe1,
			    0x97, 0xea, 0x9c, 0x07, 0xe5, 0x6b, 0x5e, 0xb1,
			    0x7e, 0x5f, 0x4e },
		.rlen	= 35,
	}, {
		.key	= { 0xc0, 0xc1, 0xc2, 0xc3, 0xc4, 0xc5, 0xc6, 0xc7,
			    0xc8, 0xc9, 0xca, 0xcb, 0xcc, 0xcd, 0xce, 0xcf },
		.klen	= 16,
		.iv	= { 0x01, 0x00, 0x00, 0x00, 0x0c, 0x0b, 0x0a, 0x09,
			    0xa0, 0xa1, 0xa2, 0xa3, 0xa4, 0xa5, 0x00, 0x00 },
		.assoc	= { 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
			    0x08, 0x09, 0x0a, 0x0b },
		.alen	= 12,
		.input	= { 0x0c, 0x0d, 0x0e, 0x0f, 0x10, 0x11, 0x12, 0x13,
			    0x14, 0x15, 0x16, 0x17, 0x18, 0x19, 0x1a, 0x1b,
			    0x1c, 0x1d, 0x1e },
		.ilen	= 19,
		.result	= { 0x07, 0x34, 0x25, 0x94, 0x15, 0x77, 0x85, 0x15,
			    0x2b, 0x07, 0x40, 0x98, 0x33, 0x0a, 0xbb, 0x14,
			    0x1b, 0x94, 0x7b, 0x56, 0x6a, 0xa9, 0x40, 0x6b,
			    0x4d, 0x99, 0x99, 0x88, 0xdd },
		.rlen	= 29,
	}, {
		.key	= { 0xd7, 0x82, 0x8d, 0x13, 0xb2, 0xb0, 0xbd, 0xc3,
			    0x25, 0xa7, 0x62, 0x36, 0xdf, 0x93, 0xcc, 0x6b },
		.klen	= 16,
		.iv	= { 0x01, 0x00, 0x33, 0x56, 0x8e, 0xf7, 0xb2, 0x63,
			    0x3c, 0x96, 0x96, 0x76, 0x6c, 0xfa, 0x00, 0x00 },
		.assoc	= { 0x63, 0x01, 0x8f, 0x76, 0xdc, 0x8a, 0x1b, 0xcb },
		.alen	= 8,
		.input	= { 0x90, 0x20, 0xea, 0x6f, 0x91, 0xbd, 0xd8, 0x5a,
			    0xfa, 0x00, 0x39, 0xba, 0x4b, 0xaf, 0xf9, 0xbf,
			    0xb7, 0x9c, 0x70, 0x28, 0x94, 0x9c, 0xd0, 0xec },
		.ilen	= 24,
		.result	= { 0x4c, 0xcb, 0x1e, 0x7c, 0xa9, 0x81, 0xbe, 0xfa,
			    0xa0, 0x72, 0x6c, 0x55, 0xd3, 0x78, 0x06, 0x12,
			    0x98, 0xc8, 0x5c, 0x92, 0x81, 0x4a, 0xbc, 0x33,
			    0xc5, 0x2e, 0xe8, 0x1d, 0x7d, 0x77, 0xc0, 0x8a },
		.rlen	= 32,
	}, {
		.key	= { 0xd7, 0x82, 0x8d, 0x13, 0xb2, 0xb0, 0xbd, 0xc3,
			    0x25, 0xa7, 0x62, 0x36, 0xdf, 0x93, 0xcc, 0x6b },
		.klen	= 16,
		.iv	= { 0x01, 0x00, 0xd5, 0x60, 0x91, 0x2d, 0x3f, 0x70,
			    0x3c, 0x96, 0x96, 0x76, 0x6c, 0xfa, 0x00, 0x00 },
		.assoc	= { 0xcd, 0x90, 0x44, 0xd2, 0xb7, 0x1f, 0xdb, 0x81,
			    0x20, 0xea, 0x60, 0xc0 },
		.alen	= 12,
		.input	= { 0x64, 0x35, 0xac, 0xba, 0xfb, 0x11, 0xa8, 0x2e,
			    0x2f, 0x07, 0x1d, 0x7c, 0xa4, 0xa5, 0xeb, 0xd9,
			    0x3a, 0x80, 0x3b, 0xa8, 0x7f },
		.ilen	= 21,
		.result	= { 0x00, 0x97, 0x69, 0xec, 0xab, 0xdf, 0x48, 0x62,
			    0x55, 0x94, 0xc5, 0x92, 0x51, 0xe6, 0x03, 0x57,
			    0x22, 0x67, 0x5e, 0x04, 0xc8, 0x47, 0x09, 0x9e,
			    0x5a, 0xe0, 0x70, 0x45, 0x51 },
		.rlen	= 29,
	}, {
		.key	= { 0xd7, 0x82, 0x8d, 0x13, 0xb2, 0xb0, 0xbd, 0xc3,
			    0x25, 0xa7, 0x62, 0x36, 0xdf, 0x93, 0xcc, 0x6b },
		.klen	= 16,
		.iv	= { 0x01, 0x00, 0x42, 0xff, 0xf8, 0xf1, 0x95, 0x1c,
			    0x3c, 0x96, 0x96, 0x76, 0x6c, 0xfa, 0x00, 0x00 },
		.assoc	= { 0xd8, 0x5b, 0xc7, 0xe6, 0x9f, 0x94, 0x4f, 0xb8 },
		.alen	= 8,
		.input	= { 0x8a, 0x19, 0xb9, 0x50, 0xbc, 0xf7, 0x1a, 0x01,
			    0x8e, 0x5e, 0x67, 0x01, 0xc9, 0x17, 0x87, 0x65,
			    0x98, 0x09, 0xd6, 0x7d, 0xbe, 0xdd, 0x18 },
		.ilen	= 23,
		.result	= { 0xbc, 0x21, 0x8d, 0xaa, 0x94, 0x74, 0x27, 0xb6,
			    0xdb, 0x38, 0x6a, 0x99, 0xac, 0x1a, 0xef, 0x23,
			    0xad, 0xe0, 0xb5, 0x29, 0x39, 0xcb, 0x6a, 0x63,
			    0x7c, 0xf9, 0xbe, 0xc2, 0x40, 0x88, 0x97, 0xc6,
			    0xba },
		.rlen	= 33,
	},
};

static struct aead_testvec aes_ccm_dec_tv_template[] = {
	{ /* From RFC 3610 */
		.key	= { 0xc0, 0xc1, 0xc2, 0xc3, 0xc4, 0xc5, 0xc6, 0xc7,
			    0xc8, 0xc9, 0xca, 0xcb, 0xcc, 0xcd, 0xce, 0xcf },
		.klen	= 16,
		.iv	= { 0x01, 0x00, 0x00, 0x00, 0x03, 0x02, 0x01, 0x00,
			    0xa0, 0xa1, 0xa2, 0xa3, 0xa4, 0xa5, 0x00, 0x00 },
		.assoc	= { 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07 },
		.alen	= 8,
		.input	= { 0x58, 0x8c, 0x97, 0x9a, 0x61, 0xc6, 0x63, 0xd2,
			    0xf0, 0x66, 0xd0, 0xc2, 0xc0, 0xf9, 0x89, 0x80,
			    0x6d, 0x5f, 0x6b, 0x61, 0xda, 0xc3, 0x84, 0x17,
			    0xe8, 0xd1, 0x2c, 0xfd, 0xf9, 0x26, 0xe0 },
		.ilen	= 31,
		.result	= { 0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
			    0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
			    0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e },
		.rlen	= 23,
	}, {
		.key	= { 0xc0, 0xc1, 0xc2, 0xc3, 0xc4, 0xc5, 0xc6, 0xc7,
			    0xc8, 0xc9, 0xca, 0xcb, 0xcc, 0xcd, 0xce, 0xcf },
		.klen	= 16,
		.iv	= { 0x01, 0x00, 0x00, 0x00, 0x07, 0x06, 0x05, 0x04,
			    0xa0, 0xa1, 0xa2, 0xa3, 0xa4, 0xa5, 0x00, 0x00 },
		.assoc	= { 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
			    0x08, 0x09, 0x0a, 0x0b },
		.alen	= 12,
		.input	= { 0xdc, 0xf1, 0xfb, 0x7b, 0x5d, 0x9e, 0x23, 0xfb,
			    0x9d, 0x4e, 0x13, 0x12, 0x53, 0x65, 0x8a, 0xd8,
			    0x6e, 0xbd, 0xca, 0x3e, 0x51, 0xe8, 0x3f, 0x07,
			    0x7d, 0x9c, 0x2d, 0x93 },
		.ilen	= 28,
		.result	= { 0x0c, 0x0d, 0x0e, 0x0f, 0x10, 0x11, 0x12, 0x13,
			    0x14, 0x15, 0x16, 0x17, 0x18, 0x19, 0x1a, 0x1b,
			    0x1c, 0x1d, 0x1e, 0x1f },
		.rlen	= 20,
	}, {
		.key	= { 0xc0, 0xc1, 0xc2, 0xc3, 0xc4, 0xc5, 0xc6, 0xc7,
			    0xc8, 0xc9, 0xca, 0xcb, 0xcc, 0xcd, 0xce, 0xcf },
		.klen	= 16,
		.iv	= { 0x01, 0x00, 0x00, 0x00, 0x0b, 0x0a, 0x09, 0x08,
			    0xa0, 0xa1, 0xa2, 0xa3, 0xa4, 0xa5, 0x00, 0x00 },
		.assoc	= { 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07 },
		.alen	= 8,
		.input	= { 0x82, 0x53, 0x1a, 0x60, 0xcc, 0x24, 0x94, 0x5a,
			    0x4b, 0x82, 0x79, 0x18, 0x1a, 0xb5, 0xc8, 0x4d,
			    0xf2, 0x1c, 0xe7, 0xf9, 0xb7, 0x3f, 0x42, 0xe1,
			    0x97, 0xea, 0x9c, 0x07, 0xe5, 0x6b, 0x5e, 0xb1,
			    0x7e, 0x5f, 0x4e },
		.ilen	= 35,
		.result	= { 0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
			    0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
			    0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,
			    0x20 },
		.rlen	= 25,
	}, {
		.key	= { 0xc0, 0xc1, 0xc2, 0xc3, 0xc4, 0xc5, 0xc6, 0xc7,
			    0xc8, 0xc9, 0xca, 0xcb, 0xcc, 0xcd, 0xce, 0xcf },
		.klen	= 16,
		.iv	= { 0x01, 0x00, 0x00, 0x00, 0x0c, 0x0b, 0x0a, 0x09,
			    0xa0, 0xa1, 0xa2, 0xa3, 0xa4, 0xa5, 0x00, 0x00 },
		.assoc	= { 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
			    0x08, 0x09, 0x0a, 0x0b },
		.alen	= 12,
		.input	= { 0x07, 0x34, 0x25, 0x94, 0x15, 0x77, 0x85, 0x15,
			    0x2b, 0x07, 0x40, 0x98, 0x33, 0x0a, 0xbb, 0x14,
			    0x1b, 0x94, 0x7b, 0x56, 0x6a, 0xa9, 0x40, 0x6b,
			    0x4d, 0x99, 0x99, 0x88, 0xdd },
		.ilen	= 29,
		.result	= { 0x0c, 0x0d, 0x0e, 0x0f, 0x10, 0x11, 0x12, 0x13,
			    0x14, 0x15, 0x16, 0x17, 0x18, 0x19, 0x1a, 0x1b,
			    0x1c, 0x1d, 0x1e },
		.rlen	= 19,
	}, {
		.key	= { 0xd7, 0x82, 0x8d, 0x13, 0xb2, 0xb0, 0xbd, 0xc3,
			    0x25, 0xa7, 0x62, 0x36, 0xdf, 0x93, 0xcc, 0x6b },
		.klen	= 16,
		.iv	= { 0x01, 0x00, 0x33, 0x56, 0x8e, 0xf7, 0xb2, 0x63,
			    0x3c, 0x96, 0x96, 0x76, 0x6c, 0xfa, 0x00, 0x00 },
		.assoc	= { 0x63, 0x01, 0x8f, 0x76, 0xdc, 0x8a, 0x1b, 0xcb },
		.alen	= 8,
		.input	= { 0x4c, 0xcb, 0x1e, 0x7c, 0xa9, 0x81, 0xbe, 0xfa,
			    0xa0, 0x72, 0x6c, 0x55, 0xd3, 0x78, 0x06, 0x12,
			    0x98, 0xc8, 0x5c, 0x92, 0x81, 0x4a, 0xbc, 0x33,
			    0xc5, 0x2e, 0xe8, 0x1d, 0x7d, 0x77, 0xc0, 0x8a },
		.ilen	= 32,
		.result	= { 0x90, 0x20, 0xea, 0x6f, 0x91, 0xbd, 0xd8, 0x5a,
			    0xfa, 0x00, 0x39, 0xba, 0x4b, 0xaf, 0xf9, 0xbf,
			    0xb7, 0x9c, 0x70, 0x28, 0x94, 0x9c, 0xd0, 0xec },
		.rlen	= 24,
	}, {
		.key	= { 0xd7, 0x82, 0x8d, 0x13, 0xb2, 0xb0, 0xbd, 0xc3,
			    0x25, 0xa7, 0x62, 0x36, 0xdf, 0x93, 0xcc, 0x6b },
		.klen	= 16,
		.iv	= { 0x01, 0x00, 0xd5, 0x60, 0x91, 0x2d, 0x3f, 0x70,
			    0x3c, 0x96, 0x96, 0x76, 0x6c, 0xfa, 0x00, 0x00 },
		.assoc	= { 0xcd, 0x90, 0x44, 0xd2, 0xb7, 0x1f, 0xdb, 0x81,
			    0x20, 0xea, 0x60, 0xc0 },
		.alen	= 12,
		.input	= { 0x00, 0x97, 0x69, 0xec, 0xab, 0xdf, 0x48, 0x62,
			    0x55, 0x94, 0xc5, 0x92, 0x51, 0xe6, 0x03, 0x57,
			    0x22, 0x67, 0x5e, 0x04, 0xc8, 0x47, 0x09, 0x9e,
			    0x5a, 0xe0, 0x70, 0x45, 0x51 },
		.ilen	= 29,
		.result	= { 0x64, 0x35, 0xac, 0xba, 0xfb, 0x11, 0xa8, 0x2e,
			    0x2f, 0x07, 0x1d, 0x7c, 0xa4, 0xa5, 0xeb, 0xd9,
			    0x3a, 0x80, 0x3b, 0xa8, 0x7f },
		.rlen	= 21,
	}, {
		.key	= { 0xd7, 0x82, 0x8d, 0x13, 0xb2, 0xb0, 0xbd, 0xc3,
			    0x25, 0xa7, 0x62, 0x36, 0xdf, 0x93, 0xcc, 0x6b },
		.klen	= 16,
		.iv	= { 0x01, 0x00, 0x42, 0xff, 0xf8, 0xf1, 0x95, 0x1c,
			    0x3c, 0x96, 0x96, 0x76, 0x6c, 0xfa, 0x00, 0x00 },
		.assoc	= { 0xd8, 0x5b, 0xc7, 0xe6, 0x9f, 0x94, 0x4f, 0xb8 },
		.alen	= 8,
		.input	= { 0xbc, 0x21, 0x8d, 0xaa, 0x94, 0x74, 0x27, 0xb6,
			    0xdb, 0x38, 0x6a, 0x99, 0xac, 0x1a, 0xef, 0x23,
			    0xad, 0xe0, 0xb5, 0x29, 0x39, 0xcb, 0x6a, 0x63,
			    0x7c, 0xf9, 0xbe, 0xc2, 0x40, 0x88, 0x97, 0xc6,
			    0xba },
		.ilen	= 33,
		.result	= { 0x8a, 0x19, 0xb9, 0x50, 0xbc, 0xf7, 0x1a, 0x01,
			    0x8e, 0x5e, 0x67, 0x01, 0xc9, 0x17, 0x87, 0x65,
			    0x98, 0x09, 0xd6, 0x7d, 0xbe, 0xdd, 0x18 },
		.rlen	= 23,
	},
};

/* Cast5 test vectors from RFC 2144 */
#define CAST5_ENC_TEST_VECTORS	3
#define CAST5_DEC_TEST_VECTORS	3

static struct cipher_testvec cast5_enc_tv_template[] = {
	{
		.key	= { 0x01, 0x23, 0x45, 0x67, 0x12, 0x34, 0x56, 0x78,
			    0x23, 0x45, 0x67, 0x89, 0x34, 0x56, 0x78, 0x9a },
		.klen	= 16,
		.input	= { 0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef },
		.ilen	= 8,
		.result	= { 0x23, 0x8b, 0x4f, 0xe5, 0x84, 0x7e, 0x44, 0xb2 },
		.rlen	= 8,
	}, {
		.key	= { 0x01, 0x23, 0x45, 0x67, 0x12, 0x34, 0x56, 0x78,
			    0x23, 0x45 },
		.klen	= 10,
		.input	= { 0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef },
		.ilen	= 8,
		.result	= { 0xeb, 0x6a, 0x71, 0x1a, 0x2c, 0x02, 0x27, 0x1b },
		.rlen	= 8,
	}, {
		.key	= { 0x01, 0x23, 0x45, 0x67, 0x12 },
		.klen	= 5,
		.input	= { 0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef },
		.ilen	= 8,
		.result	= { 0x7a, 0xc8, 0x16, 0xd1, 0x6e, 0x9b, 0x30, 0x2e },
		.rlen	= 8,
	},
};

static struct cipher_testvec cast5_dec_tv_template[] = {
	{
		.key	= { 0x01, 0x23, 0x45, 0x67, 0x12, 0x34, 0x56, 0x78,
			    0x23, 0x45, 0x67, 0x89, 0x34, 0x56, 0x78, 0x9a },
		.klen	= 16,
		.input	= { 0x23, 0x8b, 0x4f, 0xe5, 0x84, 0x7e, 0x44, 0xb2 },
		.ilen	= 8,
		.result	= { 0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef },
		.rlen	= 8,
	}, {
		.key	= { 0x01, 0x23, 0x45, 0x67, 0x12, 0x34, 0x56, 0x78,
			    0x23, 0x45 },
		.klen	= 10,
		.input	= { 0xeb, 0x6a, 0x71, 0x1a, 0x2c, 0x02, 0x27, 0x1b },
		.ilen	= 8,
		.result	= { 0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef },
		.rlen	= 8,
	}, {
		.key	= { 0x01, 0x23, 0x45, 0x67, 0x12 },
		.klen	= 5,
		.input	= { 0x7a, 0xc8, 0x16, 0xd1, 0x6e, 0x9b, 0x30, 0x2e },
		.ilen	= 8,
		.result	= { 0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef },
		.rlen	= 8,
	},
};

/*
 * ARC4 test vectors from OpenSSL
 */
#define ARC4_ENC_TEST_VECTORS	7
#define ARC4_DEC_TEST_VECTORS	7

static struct cipher_testvec arc4_enc_tv_template[] = {
	{
		.key	= { 0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef },
		.klen	= 8,
		.input	= { 0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef },
		.ilen	= 8,
		.result	= { 0x75, 0xb7, 0x87, 0x80, 0x99, 0xe0, 0xc5, 0x96 },
		.rlen	= 8,
	}, {
		.key	= { 0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef },
		.klen	= 8,
		.input	= { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
		.ilen	= 8,
		.result	= { 0x74, 0x94, 0xc2, 0xe7, 0x10, 0x4b, 0x08, 0x79 },
		.rlen	= 8,
	}, {
		.key	= { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
		.klen	= 8,
		.input	= { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
		.ilen	= 8,
		.result	= { 0xde, 0x18, 0x89, 0x41, 0xa3, 0x37, 0x5d, 0x3a },
		.rlen	= 8,
	}, {
		.key	= { 0xef, 0x01, 0x23, 0x45},
		.klen	= 4,
		.input	= { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			    0x00, 0x00, 0x00, 0x00 },
		.ilen	= 20,
		.result	= { 0xd6, 0xa1, 0x41, 0xa7, 0xec, 0x3c, 0x38, 0xdf,
			    0xbd, 0x61, 0x5a, 0x11, 0x62, 0xe1, 0xc7, 0xba,
			    0x36, 0xb6, 0x78, 0x58 },
		.rlen	= 20,
	}, {
		.key	= { 0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef },
		.klen	= 8,
		.input	= { 0x12, 0x34, 0x56, 0x78, 0x9A, 0xBC, 0xDE, 0xF0,
			    0x12, 0x34, 0x56, 0x78, 0x9A, 0xBC, 0xDE, 0xF0,
			    0x12, 0x34, 0x56, 0x78, 0x9A, 0xBC, 0xDE, 0xF0,
			    0x12, 0x34, 0x56, 0x78 },
		.ilen	= 28,
		.result	= { 0x66, 0xa0, 0x94, 0x9f, 0x8a, 0xf7, 0xd6, 0x89,
			    0x1f, 0x7f, 0x83, 0x2b, 0xa8, 0x33, 0xc0, 0x0c,
			    0x89, 0x2e, 0xbe, 0x30, 0x14, 0x3c, 0xe2, 0x87,
			    0x40, 0x01, 0x1e, 0xcf },
		.rlen	= 28,
	}, {
		.key	= { 0xef, 0x01, 0x23, 0x45 },
		.klen	= 4,
		.input	= { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			    0x00, 0x00 },
		.ilen	= 10,
		.result	= { 0xd6, 0xa1, 0x41, 0xa7, 0xec, 0x3c, 0x38, 0xdf,
			    0xbd, 0x61 },
		.rlen	= 10,
	}, {
		.key	= { 0x01, 0x23, 0x45, 0x67, 0x89, 0xAB, 0xCD, 0xEF,
		            0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
		.klen	= 16,
		.input	= { 0x01, 0x23, 0x45, 0x67, 0x89, 0xAB, 0xCD, 0xEF },
		.ilen	= 8,
		.result	= { 0x69, 0x72, 0x36, 0x59, 0x1B, 0x52, 0x42, 0xB1 },
		.rlen	= 8,
	},
};

static struct cipher_testvec arc4_dec_tv_template[] = {
	{
		.key	= { 0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef },
		.klen	= 8,
		.input	= { 0x75, 0xb7, 0x87, 0x80, 0x99, 0xe0, 0xc5, 0x96 },
		.ilen	= 8,
		.result	= { 0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef },
		.rlen	= 8,
	}, {
		.key	= { 0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef },
		.klen	= 8,
		.input	= { 0x74, 0x94, 0xc2, 0xe7, 0x10, 0x4b, 0x08, 0x79 },
		.ilen	= 8,
		.result	= { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
		.rlen	= 8,
	}, {
		.key	= { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
		.klen	= 8,
		.input	= { 0xde, 0x18, 0x89, 0x41, 0xa3, 0x37, 0x5d, 0x3a },
		.ilen	= 8,
		.result	= { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
		.rlen	= 8,
	}, {
		.key	= { 0xef, 0x01, 0x23, 0x45},
		.klen	= 4,
		.input	= { 0xd6, 0xa1, 0x41, 0xa7, 0xec, 0x3c, 0x38, 0xdf,
			    0xbd, 0x61, 0x5a, 0x11, 0x62, 0xe1, 0xc7, 0xba,
			    0x36, 0xb6, 0x78, 0x58 },
		.ilen	= 20,
		.result	= { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			    0x00, 0x00, 0x00, 0x00 },
		.rlen	= 20,
	}, {
		.key	= { 0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef },
		.klen	= 8,
		.input	= { 0x66, 0xa0, 0x94, 0x9f, 0x8a, 0xf7, 0xd6, 0x89,
			    0x1f, 0x7f, 0x83, 0x2b, 0xa8, 0x33, 0xc0, 0x0c,
			    0x89, 0x2e, 0xbe, 0x30, 0x14, 0x3c, 0xe2, 0x87,
			    0x40, 0x01, 0x1e, 0xcf },
		.ilen	= 28,
		.result	= { 0x12, 0x34, 0x56, 0x78, 0x9A, 0xBC, 0xDE, 0xF0,
			    0x12, 0x34, 0x56, 0x78, 0x9A, 0xBC, 0xDE, 0xF0,
			    0x12, 0x34, 0x56, 0x78, 0x9A, 0xBC, 0xDE, 0xF0,
			    0x12, 0x34, 0x56, 0x78 },
		.rlen	= 28,
	}, {
		.key	= { 0xef, 0x01, 0x23, 0x45 },
		.klen	= 4,
		.input	= { 0xd6, 0xa1, 0x41, 0xa7, 0xec, 0x3c, 0x38, 0xdf,
			    0xbd, 0x61 },
		.ilen	= 10,
		.result	= { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			    0x00, 0x00 },
		.rlen	= 10,
	}, {
		.key	= { 0x01, 0x23, 0x45, 0x67, 0x89, 0xAB, 0xCD, 0xEF,
		            0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
		.klen	= 16,
		.input	= { 0x69, 0x72, 0x36, 0x59, 0x1B, 0x52, 0x42, 0xB1 },
		.ilen	= 8,
		.result	= { 0x01, 0x23, 0x45, 0x67, 0x89, 0xAB, 0xCD, 0xEF },
		.rlen	= 8,
	},
};

/*
 * TEA test vectors
 */
#define TEA_ENC_TEST_VECTORS	4
#define TEA_DEC_TEST_VECTORS	4

static struct cipher_testvec tea_enc_tv_template[] = {
	{
		.key    = { [0 ... 15] = 0x00 },
		.klen	= 16,
		.input  = { [0 ... 8] = 0x00 },
		.ilen	= 8,
		.result	= { 0x0a, 0x3a, 0xea, 0x41, 0x40, 0xa9, 0xba, 0x94 },
		.rlen	= 8,
	}, {
		.key	= { 0x2b, 0x02, 0x05, 0x68, 0x06, 0x14, 0x49, 0x76,
			    0x77, 0x5d, 0x0e, 0x26, 0x6c, 0x28, 0x78, 0x43 },
		.klen	= 16,
		.input	= { 0x74, 0x65, 0x73, 0x74, 0x20, 0x6d, 0x65, 0x2e },
		.ilen	= 8,
		.result	= { 0x77, 0x5d, 0x2a, 0x6a, 0xf6, 0xce, 0x92, 0x09 },
		.rlen	= 8,
	}, {
		.key	= { 0x09, 0x65, 0x43, 0x11, 0x66, 0x44, 0x39, 0x25,
			    0x51, 0x3a, 0x16, 0x10, 0x0a, 0x08, 0x12, 0x6e },
		.klen	= 16,
		.input	= { 0x6c, 0x6f, 0x6e, 0x67, 0x65, 0x72, 0x5f, 0x74,
			    0x65, 0x73, 0x74, 0x5f, 0x76, 0x65, 0x63, 0x74 },
		.ilen	= 16,
		.result	= { 0xbe, 0x7a, 0xbb, 0x81, 0x95, 0x2d, 0x1f, 0x1e,
			    0xdd, 0x89, 0xa1, 0x25, 0x04, 0x21, 0xdf, 0x95 },
		.rlen	= 16,
	}, {
		.key	= { 0x4d, 0x76, 0x32, 0x17, 0x05, 0x3f, 0x75, 0x2c,
			    0x5d, 0x04, 0x16, 0x36, 0x15, 0x72, 0x63, 0x2f },
		.klen	= 16,
		.input	= { 0x54, 0x65, 0x61, 0x20, 0x69, 0x73, 0x20, 0x67,
			    0x6f, 0x6f, 0x64, 0x20, 0x66, 0x6f, 0x72, 0x20,
			    0x79, 0x6f, 0x75, 0x21, 0x21, 0x21, 0x20, 0x72,
			    0x65, 0x61, 0x6c, 0x6c, 0x79, 0x21, 0x21, 0x21 },
		.ilen	= 32,
		.result	= { 0xe0, 0x4d, 0x5d, 0x3c, 0xb7, 0x8c, 0x36, 0x47,
			    0x94, 0x18, 0x95, 0x91, 0xa9, 0xfc, 0x49, 0xf8,
			    0x44, 0xd1, 0x2d, 0xc2, 0x99, 0xb8, 0x08, 0x2a,
			    0x07, 0x89, 0x73, 0xc2, 0x45, 0x92, 0xc6, 0x90 },
		.rlen	= 32,
	}
};

static struct cipher_testvec tea_dec_tv_template[] = {
	{
		.key    = { [0 ... 15] = 0x00 },
		.klen	= 16,
		.input	= { 0x0a, 0x3a, 0xea, 0x41, 0x40, 0xa9, 0xba, 0x94 },
		.ilen	= 8,
		.result = { [0 ... 8] = 0x00 },
		.rlen	= 8,
	}, {
		.key	= { 0x2b, 0x02, 0x05, 0x68, 0x06, 0x14, 0x49, 0x76,
			    0x77, 0x5d, 0x0e, 0x26, 0x6c, 0x28, 0x78, 0x43 },
		.klen	= 16,
		.input	= { 0x77, 0x5d, 0x2a, 0x6a, 0xf6, 0xce, 0x92, 0x09 },
		.ilen	= 8,
		.result	= { 0x74, 0x65, 0x73, 0x74, 0x20, 0x6d, 0x65, 0x2e },
		.rlen	= 8,
	}, {
		.key	= { 0x09, 0x65, 0x43, 0x11, 0x66, 0x44, 0x39, 0x25,
			    0x51, 0x3a, 0x16, 0x10, 0x0a, 0x08, 0x12, 0x6e },
		.klen	= 16,
		.input	= { 0xbe, 0x7a, 0xbb, 0x81, 0x95, 0x2d, 0x1f, 0x1e,
			    0xdd, 0x89, 0xa1, 0x25, 0x04, 0x21, 0xdf, 0x95 },
		.ilen   = 16,
		.result	= { 0x6c, 0x6f, 0x6e, 0x67, 0x65, 0x72, 0x5f, 0x74,
			    0x65, 0x73, 0x74, 0x5f, 0x76, 0x65, 0x63, 0x74 },
		.rlen	= 16,
	}, {
		.key	= { 0x4d, 0x76, 0x32, 0x17, 0x05, 0x3f, 0x75, 0x2c,
			    0x5d, 0x04, 0x16, 0x36, 0x15, 0x72, 0x63, 0x2f },
		.klen	= 16,
		.input	= { 0xe0, 0x4d, 0x5d, 0x3c, 0xb7, 0x8c, 0x36, 0x47,
			    0x94, 0x18, 0x95, 0x91, 0xa9, 0xfc, 0x49, 0xf8,
			    0x44, 0xd1, 0x2d, 0xc2, 0x99, 0xb8, 0x08, 0x2a,
			    0x07, 0x89, 0x73, 0xc2, 0x45, 0x92, 0xc6, 0x90 },
		.ilen	= 32,
		.result	= { 0x54, 0x65, 0x61, 0x20, 0x69, 0x73, 0x20, 0x67,
			    0x6f, 0x6f, 0x64, 0x20, 0x66, 0x6f, 0x72, 0x20,
			    0x79, 0x6f, 0x75, 0x21, 0x21, 0x21, 0x20, 0x72,
			    0x65, 0x61, 0x6c, 0x6c, 0x79, 0x21, 0x21, 0x21 },
		.rlen	= 32,
	}
};

/*
 * XTEA test vectors
 */
#define XTEA_ENC_TEST_VECTORS	4
#define XTEA_DEC_TEST_VECTORS	4

static struct cipher_testvec xtea_enc_tv_template[] = {
	{
		.key    = { [0 ... 15] = 0x00 },
		.klen	= 16,
		.input  = { [0 ... 8] = 0x00 },
		.ilen	= 8,
		.result	= { 0xd8, 0xd4, 0xe9, 0xde, 0xd9, 0x1e, 0x13, 0xf7 },
		.rlen	= 8,
	}, {
		.key	= { 0x2b, 0x02, 0x05, 0x68, 0x06, 0x14, 0x49, 0x76,
			    0x77, 0x5d, 0x0e, 0x26, 0x6c, 0x28, 0x78, 0x43 },
		.klen	= 16,
		.input	= { 0x74, 0x65, 0x73, 0x74, 0x20, 0x6d, 0x65, 0x2e },
		.ilen	= 8,
		.result	= { 0x94, 0xeb, 0xc8, 0x96, 0x84, 0x6a, 0x49, 0xa8 },
		.rlen	= 8,
	}, {
		.key	= { 0x09, 0x65, 0x43, 0x11, 0x66, 0x44, 0x39, 0x25,
			    0x51, 0x3a, 0x16, 0x10, 0x0a, 0x08, 0x12, 0x6e },
		.klen	= 16,
		.input	= { 0x3e, 0xce, 0xae, 0x22, 0x60, 0x56, 0xa8, 0x9d,
			    0x65, 0x73, 0x74, 0x5f, 0x76, 0x65, 0x63, 0x74 },
		.ilen	= 16,
		.result	= { 0xe2, 0x04, 0xdb, 0xf2, 0x89, 0x85, 0x9e, 0xea, 
			    0x61, 0x35, 0xaa, 0xed, 0xb5, 0xcb, 0x71, 0x2c },
		.rlen	= 16,
	}, {
		.key	= { 0x4d, 0x76, 0x32, 0x17, 0x05, 0x3f, 0x75, 0x2c,
			    0x5d, 0x04, 0x16, 0x36, 0x15, 0x72, 0x63, 0x2f },
		.klen	= 16,
		.input	= { 0x54, 0x65, 0x61, 0x20, 0x69, 0x73, 0x20, 0x67, 
			    0x6f, 0x6f, 0x64, 0x20, 0x66, 0x6f, 0x72, 0x20, 
			    0x79, 0x6f, 0x75, 0x21, 0x21, 0x21, 0x20, 0x72, 
			    0x65, 0x61, 0x6c, 0x6c, 0x79, 0x21, 0x21, 0x21 },
		.ilen	= 32,
		.result	= { 0x99, 0x81, 0x9f, 0x5d, 0x6f, 0x4b, 0x31, 0x3a,
			    0x86, 0xff, 0x6f, 0xd0, 0xe3, 0x87, 0x70, 0x07,
			    0x4d, 0xb8, 0xcf, 0xf3, 0x99, 0x50, 0xb3, 0xd4,
			    0x73, 0xa2, 0xfa, 0xc9, 0x16, 0x59, 0x5d, 0x81 },
		.rlen	= 32,
	}
};

static struct cipher_testvec xtea_dec_tv_template[] = {
	{
		.key    = { [0 ... 15] = 0x00 },
		.klen	= 16,
		.input	= { 0xd8, 0xd4, 0xe9, 0xde, 0xd9, 0x1e, 0x13, 0xf7 },
		.ilen	= 8,
		.result = { [0 ... 8] = 0x00 },
		.rlen	= 8,
	}, {
		.key	= { 0x2b, 0x02, 0x05, 0x68, 0x06, 0x14, 0x49, 0x76,
			    0x77, 0x5d, 0x0e, 0x26, 0x6c, 0x28, 0x78, 0x43 },
		.klen	= 16,
		.input	= { 0x94, 0xeb, 0xc8, 0x96, 0x84, 0x6a, 0x49, 0xa8 },
		.ilen	= 8,
		.result	= { 0x74, 0x65, 0x73, 0x74, 0x20, 0x6d, 0x65, 0x2e },
		.rlen	= 8,
	}, {
		.key	= { 0x09, 0x65, 0x43, 0x11, 0x66, 0x44, 0x39, 0x25,
			    0x51, 0x3a, 0x16, 0x10, 0x0a, 0x08, 0x12, 0x6e },
		.klen	= 16,
		.input	= { 0x3e, 0xce, 0xae, 0x22, 0x60, 0x56, 0xa8, 0x9d,
			    0x77, 0x4d, 0xd4, 0xb4, 0x87, 0x24, 0xe3, 0x9a },
		.ilen	= 16,
		.result	= { 0x6c, 0x6f, 0x6e, 0x67, 0x65, 0x72, 0x5f, 0x74, 
			    0x65, 0x73, 0x74, 0x5f, 0x76, 0x65, 0x63, 0x74 },
		.rlen	= 16,
	}, {
		.key	= { 0x4d, 0x76, 0x32, 0x17, 0x05, 0x3f, 0x75, 0x2c,
			    0x5d, 0x04, 0x16, 0x36, 0x15, 0x72, 0x63, 0x2f },
		.klen	= 16,
		.input	= { 0x99, 0x81, 0x9f, 0x5d, 0x6f, 0x4b, 0x31, 0x3a,
			    0x86, 0xff, 0x6f, 0xd0, 0xe3, 0x87, 0x70, 0x07,
			    0x4d, 0xb8, 0xcf, 0xf3, 0x99, 0x50, 0xb3, 0xd4,
			    0x73, 0xa2, 0xfa, 0xc9, 0x16, 0x59, 0x5d, 0x81 },
		.ilen	= 32,
		.result	= { 0x54, 0x65, 0x61, 0x20, 0x69, 0x73, 0x20, 0x67, 
			    0x6f, 0x6f, 0x64, 0x20, 0x66, 0x6f, 0x72, 0x20, 
			    0x79, 0x6f, 0x75, 0x21, 0x21, 0x21, 0x20, 0x72, 
			    0x65, 0x61, 0x6c, 0x6c, 0x79, 0x21, 0x21, 0x21 },
		.rlen	= 32,
	}
};

/*
 * KHAZAD test vectors.
 */
#define KHAZAD_ENC_TEST_VECTORS 5
#define KHAZAD_DEC_TEST_VECTORS 5

static struct cipher_testvec khazad_enc_tv_template[] = {
	{
		.key	= { 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
		.klen	= 16,
		.input	= { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
		.ilen	= 8,
		.result	= { 0x49, 0xa4, 0xce, 0x32, 0xac, 0x19, 0x0e, 0x3f },
		.rlen	= 8,
	}, {
		.key	= { 0x38, 0x38, 0x38, 0x38, 0x38, 0x38, 0x38, 0x38,
			    0x38, 0x38, 0x38, 0x38, 0x38, 0x38, 0x38, 0x38 },
		.klen	= 16,
		.input	= { 0x38, 0x38, 0x38, 0x38, 0x38, 0x38, 0x38, 0x38 },
		.ilen	= 8,
		.result	= { 0x7e, 0x82, 0x12, 0xa1, 0Xd9, 0X5b, 0Xe4, 0Xf9 },
		.rlen	= 8,
	}, {
		.key	= { 0Xa2, 0Xa2, 0Xa2, 0Xa2, 0Xa2, 0Xa2, 0Xa2, 0Xa2,
			    0Xa2, 0Xa2, 0Xa2, 0Xa2, 0Xa2, 0Xa2, 0Xa2, 0Xa2 },
		.klen	= 16,
		.input	= { 0Xa2, 0Xa2, 0Xa2, 0Xa2, 0Xa2, 0Xa2, 0Xa2, 0Xa2 },
		.ilen	= 8,
		.result	= { 0Xaa, 0Xbe, 0Xc1, 0X95, 0Xc5, 0X94, 0X1a, 0X9c },
		.rlen	= 8,
	}, {
		.key	= { 0X2f, 0X2f, 0X2f, 0X2f, 0X2f, 0X2f, 0X2f, 0X2f,
			    0X2f, 0X2f, 0X2f, 0X2f, 0X2f, 0X2f, 0X2f, 0X2f },
		.klen	= 16,
		.input	= { 0X2f, 0X2f, 0X2f, 0X2f, 0X2f, 0X2f, 0X2f, 0X2f },
		.ilen	= 8,
		.result = { 0X04, 0X74, 0Xf5, 0X70, 0X50, 0X16, 0Xd3, 0Xb8 },
		.rlen	= 8,
	}, {
		.key	= { 0X2f, 0X2f, 0X2f, 0X2f, 0X2f, 0X2f, 0X2f, 0X2f,
			    0X2f, 0X2f, 0X2f, 0X2f, 0X2f, 0X2f, 0X2f, 0X2f },
		.klen	= 16,
		.input	= { 0X2f, 0X2f, 0X2f, 0X2f, 0X2f, 0X2f, 0X2f, 0X2f ,
			    0X2f, 0X2f, 0X2f, 0X2f, 0X2f, 0X2f, 0X2f, 0X2f },
		.ilen	= 16,
		.result = { 0X04, 0X74, 0Xf5, 0X70, 0X50, 0X16, 0Xd3, 0Xb8 ,
			    0X04, 0X74, 0Xf5, 0X70, 0X50, 0X16, 0Xd3, 0Xb8 },
		.rlen	= 16,
	},
};

static struct cipher_testvec khazad_dec_tv_template[] = {
	{
		.key	= { 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
		.klen	= 16,
		.input	= { 0X49, 0Xa4, 0Xce, 0X32, 0Xac, 0X19, 0X0e, 0X3f },
		.ilen	= 8,
		.result	= { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
		.rlen	= 8,
	}, {
		.key	= { 0x38, 0x38, 0x38, 0x38, 0x38, 0x38, 0x38, 0x38,
			    0x38, 0x38, 0x38, 0x38, 0x38, 0x38, 0x38, 0x38 },
		.klen	= 16,
		.input	= { 0X7e, 0X82, 0X12, 0Xa1, 0Xd9, 0X5b, 0Xe4, 0Xf9 },
		.ilen	= 8,
		.result	= { 0x38, 0x38, 0x38, 0x38, 0x38, 0x38, 0x38, 0x38 },
		.rlen	= 8,
	}, {
		.key	= { 0Xa2, 0Xa2, 0Xa2, 0Xa2, 0Xa2, 0Xa2, 0Xa2, 0Xa2,
			    0Xa2, 0Xa2, 0Xa2, 0Xa2, 0Xa2, 0Xa2, 0Xa2, 0Xa2 },
		.klen	= 16,
		.input	= { 0Xaa, 0Xbe, 0Xc1, 0X95, 0Xc5, 0X94, 0X1a, 0X9c },
		.ilen	= 8,
		.result	= { 0Xa2, 0Xa2, 0Xa2, 0Xa2, 0Xa2, 0Xa2, 0Xa2, 0Xa2 },
		.rlen	= 8,
	}, {
		.key	= { 0x2f, 0X2f, 0X2f, 0X2f, 0X2f, 0X2f, 0X2f, 0X2f,
			    0X2f, 0X2f, 0X2f, 0X2f, 0X2f, 0X2f, 0X2f, 0X2f },
		.klen	= 16,
		.input = { 0X04, 0X74, 0Xf5, 0X70, 0X50, 0X16, 0Xd3, 0Xb8 },
		.ilen	= 8,
		.result	= { 0X2f, 0X2f, 0X2f, 0X2f, 0X2f, 0X2f, 0X2f, 0X2f },
		.rlen	= 8,
	}, {
		.key	= { 0X2f, 0X2f, 0X2f, 0X2f, 0X2f, 0X2f, 0X2f, 0X2f,
			    0X2f, 0X2f, 0X2f, 0X2f, 0X2f, 0X2f, 0X2f, 0X2f },
		.klen	= 16,
		.input = { 0X04, 0X74, 0Xf5, 0X70, 0X50, 0X16, 0Xd3, 0Xb8 ,
			    0X04, 0X74, 0Xf5, 0X70, 0X50, 0X16, 0Xd3, 0Xb8 },
		.ilen	= 16,
		.result	= { 0X2f, 0X2f, 0X2f, 0X2f, 0X2f, 0X2f, 0X2f, 0X2f ,
			    0X2f, 0X2f, 0X2f, 0X2f, 0X2f, 0X2f, 0X2f, 0X2f },
		.rlen	= 16,
	},
};

/*
 * Anubis test vectors.
 */

#define ANUBIS_ENC_TEST_VECTORS			5
#define ANUBIS_DEC_TEST_VECTORS			5
#define ANUBIS_CBC_ENC_TEST_VECTORS		2
#define ANUBIS_CBC_DEC_TEST_VECTORS		2

static struct cipher_testvec anubis_enc_tv_template[] = {
	{
		.key	= { 0xfe, 0xfe, 0xfe, 0xfe, 0xfe, 0xfe, 0xfe, 0xfe,
			    0xfe, 0xfe, 0xfe, 0xfe, 0xfe, 0xfe, 0xfe, 0xfe },
		.klen	= 16,
		.input	= { 0xfe, 0xfe, 0xfe, 0xfe, 0xfe, 0xfe, 0xfe, 0xfe,
			    0xfe, 0xfe, 0xfe, 0xfe, 0xfe, 0xfe, 0xfe, 0xfe },
		.ilen	= 16,
		.result	= { 0x6d, 0xc5, 0xda, 0xa2, 0x26, 0x7d, 0x62, 0x6f,
			    0x08, 0xb7, 0x52, 0x8e, 0x6e, 0x6e, 0x86, 0x90 },
		.rlen	= 16,
	}, {

		.key	= { 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03,
			    0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03,
			    0x03, 0x03, 0x03, 0x03 },
		.klen	= 20,
		.input	= { 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03,
			    0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03 },
		.ilen	= 16,
		.result	= { 0xdb, 0xf1, 0x42, 0xf4, 0xd1, 0x8a, 0xc7, 0x49,
			    0x87, 0x41, 0x6f, 0x82, 0x0a, 0x98, 0x64, 0xae },
		.rlen	= 16,
	}, {
		.key	= { 0x24, 0x24, 0x24, 0x24, 0x24, 0x24, 0x24, 0x24,
			    0x24, 0x24, 0x24, 0x24, 0x24, 0x24, 0x24, 0x24,
			    0x24, 0x24, 0x24, 0x24, 0x24, 0x24, 0x24, 0x24,
			    0x24, 0x24, 0x24, 0x24 },
		.klen	= 28,
		.input	= { 0x24, 0x24, 0x24, 0x24, 0x24, 0x24, 0x24, 0x24,
			    0x24, 0x24, 0x24, 0x24, 0x24, 0x24, 0x24, 0x24 },
		.ilen	= 16,
		.result	= { 0xfd, 0x1b, 0x4a, 0xe3, 0xbf, 0xf0, 0xad, 0x3d,
			    0x06, 0xd3, 0x61, 0x27, 0xfd, 0x13, 0x9e, 0xde },
		.rlen	= 16,
	}, {
		.key	= { 0x25, 0x25, 0x25, 0x25, 0x25, 0x25, 0x25, 0x25,
			    0x25, 0x25, 0x25, 0x25, 0x25, 0x25, 0x25, 0x25,
			    0x25, 0x25, 0x25, 0x25, 0x25, 0x25, 0x25, 0x25,
			    0x25, 0x25, 0x25, 0x25, 0x25, 0x25, 0x25, 0x25 },
		.klen	= 32,
		.input	= { 0x25, 0x25, 0x25, 0x25, 0x25, 0x25, 0x25, 0x25,
			    0x25, 0x25, 0x25, 0x25, 0x25, 0x25, 0x25, 0x25 },
		.ilen	= 16,
		.result	= { 0x1a, 0x91, 0xfb, 0x2b, 0xb7, 0x78, 0x6b, 0xc4,
		            0x17, 0xd9, 0xff, 0x40, 0x3b, 0x0e, 0xe5, 0xfe },
		.rlen	= 16,
	}, {
		.key	= { 0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 0x35,
			    0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 0x35,
			    0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 0x35,
			    0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 0x35,
			    0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 0x35 },
		.klen	= 40,
		.input	= { 0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 0x35,
			    0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 0x35 },
		.ilen	= 16,
		.result = { 0xa5, 0x2c, 0x85, 0x6f, 0x9c, 0xba, 0xa0, 0x97,
			    0x9e, 0xc6, 0x84, 0x0f, 0x17, 0x21, 0x07, 0xee },
		.rlen	= 16,
	},
};

static struct cipher_testvec anubis_dec_tv_template[] = {
	{
		.key	= { 0xfe, 0xfe, 0xfe, 0xfe, 0xfe, 0xfe, 0xfe, 0xfe,
			    0xfe, 0xfe, 0xfe, 0xfe, 0xfe, 0xfe, 0xfe, 0xfe },
		.klen	= 16,
		.input	= { 0x6d, 0xc5, 0xda, 0xa2, 0x26, 0x7d, 0x62, 0x6f,
			    0x08, 0xb7, 0x52, 0x8e, 0x6e, 0x6e, 0x86, 0x90 },
		.ilen	= 16,
		.result	= { 0xfe, 0xfe, 0xfe, 0xfe, 0xfe, 0xfe, 0xfe, 0xfe,
			    0xfe, 0xfe, 0xfe, 0xfe, 0xfe, 0xfe, 0xfe, 0xfe },
		.rlen	= 16,
	}, {

		.key	= { 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03,
			    0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03,
			    0x03, 0x03, 0x03, 0x03 },
		.klen	= 20,
		.input	= { 0xdb, 0xf1, 0x42, 0xf4, 0xd1, 0x8a, 0xc7, 0x49,
			    0x87, 0x41, 0x6f, 0x82, 0x0a, 0x98, 0x64, 0xae },
		.ilen	= 16,
		.result	= { 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03,
			    0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03 },
		.rlen	= 16,
	}, {
		.key	= { 0x24, 0x24, 0x24, 0x24, 0x24, 0x24, 0x24, 0x24,
			    0x24, 0x24, 0x24, 0x24, 0x24, 0x24, 0x24, 0x24,
			    0x24, 0x24, 0x24, 0x24, 0x24, 0x24, 0x24, 0x24,
			    0x24, 0x24, 0x24, 0x24 },
		.klen	= 28,
		.input	= { 0xfd, 0x1b, 0x4a, 0xe3, 0xbf, 0xf0, 0xad, 0x3d,
			    0x06, 0xd3, 0x61, 0x27, 0xfd, 0x13, 0x9e, 0xde },
		.ilen	= 16,
		.result	= { 0x24, 0x24, 0x24, 0x24, 0x24, 0x24, 0x24, 0x24,
			    0x24, 0x24, 0x24, 0x24, 0x24, 0x24, 0x24, 0x24 },
		.rlen	= 16,
	}, {
		.key	= { 0x25, 0x25, 0x25, 0x25, 0x25, 0x25, 0x25, 0x25,
			    0x25, 0x25, 0x25, 0x25, 0x25, 0x25, 0x25, 0x25,
			    0x25, 0x25, 0x25, 0x25, 0x25, 0x25, 0x25, 0x25,
			    0x25, 0x25, 0x25, 0x25, 0x25, 0x25, 0x25, 0x25 },
		.klen	= 32,
		.input	= { 0x1a, 0x91, 0xfb, 0x2b, 0xb7, 0x78, 0x6b, 0xc4,
		            0x17, 0xd9, 0xff, 0x40, 0x3b, 0x0e, 0xe5, 0xfe },
		.ilen	= 16,
		.result	= { 0x25, 0x25, 0x25, 0x25, 0x25, 0x25, 0x25, 0x25,
			    0x25, 0x25, 0x25, 0x25, 0x25, 0x25, 0x25, 0x25 },
		.rlen	= 16,
	}, {
		.key	= { 0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 0x35,
			    0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 0x35,
			    0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 0x35,
			    0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 0x35,
			    0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 0x35 },
		.input = {  0xa5, 0x2c, 0x85, 0x6f, 0x9c, 0xba, 0xa0, 0x97,
			    0x9e, 0xc6, 0x84, 0x0f, 0x17, 0x21, 0x07, 0xee },
		.klen	= 40,
		.ilen	= 16,
		.result	= { 0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 0x35,
			    0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 0x35 },
		.rlen	= 16,
	},
};

static struct cipher_testvec anubis_cbc_enc_tv_template[] = {
	{
		.key	= { 0xfe, 0xfe, 0xfe, 0xfe, 0xfe, 0xfe, 0xfe, 0xfe,
			    0xfe, 0xfe, 0xfe, 0xfe, 0xfe, 0xfe, 0xfe, 0xfe },
		.klen	= 16,
		.input	= { 0xfe, 0xfe, 0xfe, 0xfe, 0xfe, 0xfe, 0xfe, 0xfe,
			    0xfe, 0xfe, 0xfe, 0xfe, 0xfe, 0xfe, 0xfe, 0xfe,
			    0xfe, 0xfe, 0xfe, 0xfe, 0xfe, 0xfe, 0xfe, 0xfe,
			    0xfe, 0xfe, 0xfe, 0xfe, 0xfe, 0xfe, 0xfe, 0xfe },
		.ilen	= 32,
		.result	= { 0x6d, 0xc5, 0xda, 0xa2, 0x26, 0x7d, 0x62, 0x6f,
			    0x08, 0xb7, 0x52, 0x8e, 0x6e, 0x6e, 0x86, 0x90,
			    0x86, 0xd8, 0xb5, 0x6f, 0x98, 0x5e, 0x8a, 0x66,
			    0x4f, 0x1f, 0x78, 0xa1, 0xbb, 0x37, 0xf1, 0xbe },
		.rlen	= 32,
	}, {
		.key	= { 0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 0x35,
			    0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 0x35,
			    0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 0x35,
			    0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 0x35,
			    0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 0x35 },
		.klen	= 40,
		.input	= { 0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 0x35,
			    0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 0x35,
			    0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 0x35,
			    0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 0x35 },
		.ilen	= 32,
		.result = { 0xa5, 0x2c, 0x85, 0x6f, 0x9c, 0xba, 0xa0, 0x97,
			    0x9e, 0xc6, 0x84, 0x0f, 0x17, 0x21, 0x07, 0xee,
			    0xa2, 0xbc, 0x06, 0x98, 0xc6, 0x4b, 0xda, 0x75,
			    0x2e, 0xaa, 0xbe, 0x58, 0xce, 0x01, 0x5b, 0xc7 },
		.rlen	= 32,
	},
};

static struct cipher_testvec anubis_cbc_dec_tv_template[] = {
	{
		.key	= { 0xfe, 0xfe, 0xfe, 0xfe, 0xfe, 0xfe, 0xfe, 0xfe,
			    0xfe, 0xfe, 0xfe, 0xfe, 0xfe, 0xfe, 0xfe, 0xfe },
		.klen	= 16,
		.input	= { 0x6d, 0xc5, 0xda, 0xa2, 0x26, 0x7d, 0x62, 0x6f,
			    0x08, 0xb7, 0x52, 0x8e, 0x6e, 0x6e, 0x86, 0x90,
			    0x86, 0xd8, 0xb5, 0x6f, 0x98, 0x5e, 0x8a, 0x66,
			    0x4f, 0x1f, 0x78, 0xa1, 0xbb, 0x37, 0xf1, 0xbe },
		.ilen	= 32,
		.result	= { 0xfe, 0xfe, 0xfe, 0xfe, 0xfe, 0xfe, 0xfe, 0xfe,
			    0xfe, 0xfe, 0xfe, 0xfe, 0xfe, 0xfe, 0xfe, 0xfe,
			    0xfe, 0xfe, 0xfe, 0xfe, 0xfe, 0xfe, 0xfe, 0xfe,
			    0xfe, 0xfe, 0xfe, 0xfe, 0xfe, 0xfe, 0xfe, 0xfe },
		.rlen	= 32,
	}, {
		.key	= { 0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 0x35,
			    0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 0x35,
			    0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 0x35,
			    0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 0x35,
			    0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 0x35 },
		.klen	= 40,
		.input = { 0xa5, 0x2c, 0x85, 0x6f, 0x9c, 0xba, 0xa0, 0x97,
			    0x9e, 0xc6, 0x84, 0x0f, 0x17, 0x21, 0x07, 0xee,
			    0xa2, 0xbc, 0x06, 0x98, 0xc6, 0x4b, 0xda, 0x75,
			    0x2e, 0xaa, 0xbe, 0x58, 0xce, 0x01, 0x5b, 0xc7 },
		.ilen	= 32,
		.result	= { 0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 0x35,
			    0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 0x35,
			    0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 0x35,
			    0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 0x35, 0x35 },
		.rlen	= 32,
	},
};

/* 
 * XETA test vectors 
 */
#define XETA_ENC_TEST_VECTORS	4
#define XETA_DEC_TEST_VECTORS	4

static struct cipher_testvec xeta_enc_tv_template[] = {
	{
		.key    = { [0 ... 15] = 0x00 },
		.klen	= 16,
		.input  = { [0 ... 8] = 0x00 },
		.ilen	= 8,
		.result	= { 0xaa, 0x22, 0x96, 0xe5, 0x6c, 0x61, 0xf3, 0x45 },
		.rlen	= 8,
	}, {
		.key	= { 0x2b, 0x02, 0x05, 0x68, 0x06, 0x14, 0x49, 0x76,
			    0x77, 0x5d, 0x0e, 0x26, 0x6c, 0x28, 0x78, 0x43 },
		.klen	= 16,
		.input	= { 0x74, 0x65, 0x73, 0x74, 0x20, 0x6d, 0x65, 0x2e },
		.ilen	= 8,
		.result	= { 0x82, 0x3e, 0xeb, 0x35, 0xdc, 0xdd, 0xd9, 0xc3 },
		.rlen	= 8,
	}, {
		.key	= { 0x09, 0x65, 0x43, 0x11, 0x66, 0x44, 0x39, 0x25,
			    0x51, 0x3a, 0x16, 0x10, 0x0a, 0x08, 0x12, 0x6e },
		.klen	= 16,
		.input	= { 0x6c, 0x6f, 0x6e, 0x67, 0x65, 0x72, 0x5f, 0x74, 
			    0x65, 0x73, 0x74, 0x5f, 0x76, 0x65, 0x63, 0x74 },
		.ilen	= 16,
		.result	= { 0xe2, 0x04, 0xdb, 0xf2, 0x89, 0x85, 0x9e, 0xea, 
			    0x61, 0x35, 0xaa, 0xed, 0xb5, 0xcb, 0x71, 0x2c },
		.rlen	= 16,
	}, {
		.key	= { 0x4d, 0x76, 0x32, 0x17, 0x05, 0x3f, 0x75, 0x2c,
			    0x5d, 0x04, 0x16, 0x36, 0x15, 0x72, 0x63, 0x2f },
		.klen	= 16,
		.input	= { 0x54, 0x65, 0x61, 0x20, 0x69, 0x73, 0x20, 0x67, 
			    0x6f, 0x6f, 0x64, 0x20, 0x66, 0x6f, 0x72, 0x20, 
			    0x79, 0x6f, 0x75, 0x21, 0x21, 0x21, 0x20, 0x72, 
			    0x65, 0x61, 0x6c, 0x6c, 0x79, 0x21, 0x21, 0x21 },
		.ilen	= 32,
		.result	= { 0x0b, 0x03, 0xcd, 0x8a, 0xbe, 0x95, 0xfd, 0xb1, 
			    0xc1, 0x44, 0x91, 0x0b, 0xa5, 0xc9, 0x1b, 0xb4, 
			    0xa9, 0xda, 0x1e, 0x9e, 0xb1, 0x3e, 0x2a, 0x8f, 
			    0xea, 0xa5, 0x6a, 0x85, 0xd1, 0xf4, 0xa8, 0xa5 },
		.rlen	= 32,
	}
};

static struct cipher_testvec xeta_dec_tv_template[] = {
	{
		.key    = { [0 ... 15] = 0x00 },
		.klen	= 16,
		.input	= { 0xaa, 0x22, 0x96, 0xe5, 0x6c, 0x61, 0xf3, 0x45 },
		.ilen	= 8,
		.result = { [0 ... 8] = 0x00 },
		.rlen	= 8,
	}, {
		.key	= { 0x2b, 0x02, 0x05, 0x68, 0x06, 0x14, 0x49, 0x76,
			    0x77, 0x5d, 0x0e, 0x26, 0x6c, 0x28, 0x78, 0x43 },
		.klen	= 16,
		.input	= { 0x82, 0x3e, 0xeb, 0x35, 0xdc, 0xdd, 0xd9, 0xc3 },
		.ilen	= 8,
		.result	= { 0x74, 0x65, 0x73, 0x74, 0x20, 0x6d, 0x65, 0x2e },
		.rlen	= 8,
	}, {
		.key	= { 0x09, 0x65, 0x43, 0x11, 0x66, 0x44, 0x39, 0x25,
			    0x51, 0x3a, 0x16, 0x10, 0x0a, 0x08, 0x12, 0x6e },
		.klen	= 16,
		.input	= { 0xe2, 0x04, 0xdb, 0xf2, 0x89, 0x85, 0x9e, 0xea, 
			    0x61, 0x35, 0xaa, 0xed, 0xb5, 0xcb, 0x71, 0x2c },
		.ilen	= 16,
		.result	= { 0x6c, 0x6f, 0x6e, 0x67, 0x65, 0x72, 0x5f, 0x74, 
			    0x65, 0x73, 0x74, 0x5f, 0x76, 0x65, 0x63, 0x74 },
		.rlen	= 16,
	}, {
		.key	= { 0x4d, 0x76, 0x32, 0x17, 0x05, 0x3f, 0x75, 0x2c,
			    0x5d, 0x04, 0x16, 0x36, 0x15, 0x72, 0x63, 0x2f },
		.klen	= 16,
		.input	= { 0x0b, 0x03, 0xcd, 0x8a, 0xbe, 0x95, 0xfd, 0xb1, 
			    0xc1, 0x44, 0x91, 0x0b, 0xa5, 0xc9, 0x1b, 0xb4, 
			    0xa9, 0xda, 0x1e, 0x9e, 0xb1, 0x3e, 0x2a, 0x8f, 
			    0xea, 0xa5, 0x6a, 0x85, 0xd1, 0xf4, 0xa8, 0xa5 },
		.ilen	= 32,
		.result	= { 0x54, 0x65, 0x61, 0x20, 0x69, 0x73, 0x20, 0x67, 
			    0x6f, 0x6f, 0x64, 0x20, 0x66, 0x6f, 0x72, 0x20, 
			    0x79, 0x6f, 0x75, 0x21, 0x21, 0x21, 0x20, 0x72, 
			    0x65, 0x61, 0x6c, 0x6c, 0x79, 0x21, 0x21, 0x21 },
		.rlen	= 32,
	}
};

/* 
 * FCrypt test vectors 
 */
#define FCRYPT_ENC_TEST_VECTORS	ARRAY_SIZE(fcrypt_pcbc_enc_tv_template)
#define FCRYPT_DEC_TEST_VECTORS	ARRAY_SIZE(fcrypt_pcbc_dec_tv_template)

static struct cipher_testvec fcrypt_pcbc_enc_tv_template[] = {
	{ /* http://www.openafs.org/pipermail/openafs-devel/2000-December/005320.html */
		.key	= { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
		.klen	= 8,
		.iv	= { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
		.input	= { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
		.ilen	= 8,
		.result	= { 0x0E, 0x09, 0x00, 0xC7, 0x3E, 0xF7, 0xED, 0x41 },
		.rlen	= 8,
	}, {
		.key	= { 0x11, 0x44, 0x77, 0xAA, 0xDD, 0x00, 0x33, 0x66 },
		.klen	= 8,
		.iv	= { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
		.input	= { 0x12, 0x34, 0x56, 0x78, 0x9A, 0xBC, 0xDE, 0xF0 },
		.ilen	= 8,
		.result	= { 0xD8, 0xED, 0x78, 0x74, 0x77, 0xEC, 0x06, 0x80 },
		.rlen	= 8,
	}, { /* From Arla */
		.key	= { 0xf0, 0xe1, 0xd2, 0xc3, 0xb4, 0xa5, 0x96, 0x87 },
		.klen	= 8,
		.iv	= { 0xfe, 0xdc, 0xba, 0x98, 0x76, 0x54, 0x32, 0x10 },
		.input	= "The quick brown fox jumps over the lazy dogs.\0\0",
		.ilen	= 48,
		.result	= { 0x00, 0xf0, 0xe,  0x11, 0x75, 0xe6, 0x23, 0x82,
			    0xee, 0xac, 0x98, 0x62, 0x44, 0x51, 0xe4, 0x84,
			    0xc3, 0x59, 0xd8, 0xaa, 0x64, 0x60, 0xae, 0xf7,
			    0xd2, 0xd9, 0x13, 0x79, 0x72, 0xa3, 0x45, 0x03,
			    0x23, 0xb5, 0x62, 0xd7, 0x0c, 0xf5, 0x27, 0xd1,
			    0xf8, 0x91, 0x3c, 0xac, 0x44, 0x22, 0x92, 0xef },
		.rlen	= 48,
	}, {
		.key	= { 0xfe, 0xdc, 0xba, 0x98, 0x76, 0x54, 0x32, 0x10 },
		.klen	= 8,
		.iv	= { 0xf0, 0xe1, 0xd2, 0xc3, 0xb4, 0xa5, 0x96, 0x87 },
		.input	= "The quick brown fox jumps over the lazy dogs.\0\0",
		.ilen	= 48,
		.result	= { 0xca, 0x90, 0xf5, 0x9d, 0xcb, 0xd4, 0xd2, 0x3c,
			    0x01, 0x88, 0x7f, 0x3e, 0x31, 0x6e, 0x62, 0x9d,
			    0xd8, 0xe0, 0x57, 0xa3, 0x06, 0x3a, 0x42, 0x58,
			    0x2a, 0x28, 0xfe, 0x72, 0x52, 0x2f, 0xdd, 0xe0,
			    0x19, 0x89, 0x09, 0x1c, 0x2a, 0x8e, 0x8c, 0x94,
			    0xfc, 0xc7, 0x68, 0xe4, 0x88, 0xaa, 0xde, 0x0f },
		.rlen	= 48,
	}, { /* split-page version */
		.key	= { 0xfe, 0xdc, 0xba, 0x98, 0x76, 0x54, 0x32, 0x10 },
		.klen	= 8,
		.iv	= { 0xf0, 0xe1, 0xd2, 0xc3, 0xb4, 0xa5, 0x96, 0x87 },
		.input	= "The quick brown fox jumps over the lazy dogs.\0\0",
		.ilen	= 48,
		.result	= { 0xca, 0x90, 0xf5, 0x9d, 0xcb, 0xd4, 0xd2, 0x3c,
			    0x01, 0x88, 0x7f, 0x3e, 0x31, 0x6e, 0x62, 0x9d,
			    0xd8, 0xe0, 0x57, 0xa3, 0x06, 0x3a, 0x42, 0x58,
			    0x2a, 0x28, 0xfe, 0x72, 0x52, 0x2f, 0xdd, 0xe0,
			    0x19, 0x89, 0x09, 0x1c, 0x2a, 0x8e, 0x8c, 0x94,
			    0xfc, 0xc7, 0x68, 0xe4, 0x88, 0xaa, 0xde, 0x0f },
		.rlen	= 48,
		.np	= 2,
		.tap	= { 20, 28 },
	}
};

static struct cipher_testvec fcrypt_pcbc_dec_tv_template[] = {
	{ /* http://www.openafs.org/pipermail/openafs-devel/2000-December/005320.html */
		.key	= { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
		.klen	= 8,
		.iv	= { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
		.input	= { 0x0E, 0x09, 0x00, 0xC7, 0x3E, 0xF7, 0xED, 0x41 },
		.ilen	= 8,
		.result	= { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
		.rlen	= 8,
	}, {
		.key	= { 0x11, 0x44, 0x77, 0xAA, 0xDD, 0x00, 0x33, 0x66 },
		.klen	= 8,
		.iv	= { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
		.input	= { 0xD8, 0xED, 0x78, 0x74, 0x77, 0xEC, 0x06, 0x80 },
		.ilen	= 8,
		.result	= { 0x12, 0x34, 0x56, 0x78, 0x9A, 0xBC, 0xDE, 0xF0 },
		.rlen	= 8,
	}, { /* From Arla */
		.key	= { 0xf0, 0xe1, 0xd2, 0xc3, 0xb4, 0xa5, 0x96, 0x87 },
		.klen	= 8,
		.iv	= { 0xfe, 0xdc, 0xba, 0x98, 0x76, 0x54, 0x32, 0x10 },
		.input	= { 0x00, 0xf0, 0xe,  0x11, 0x75, 0xe6, 0x23, 0x82,
			    0xee, 0xac, 0x98, 0x62, 0x44, 0x51, 0xe4, 0x84,
			    0xc3, 0x59, 0xd8, 0xaa, 0x64, 0x60, 0xae, 0xf7,
			    0xd2, 0xd9, 0x13, 0x79, 0x72, 0xa3, 0x45, 0x03,
			    0x23, 0xb5, 0x62, 0xd7, 0x0c, 0xf5, 0x27, 0xd1,
			    0xf8, 0x91, 0x3c, 0xac, 0x44, 0x22, 0x92, 0xef },
		.ilen	= 48,
		.result	= "The quick brown fox jumps over the lazy dogs.\0\0",
		.rlen	= 48,
	}, {
		.key	= { 0xfe, 0xdc, 0xba, 0x98, 0x76, 0x54, 0x32, 0x10 },
		.klen	= 8,
		.iv	= { 0xf0, 0xe1, 0xd2, 0xc3, 0xb4, 0xa5, 0x96, 0x87 },
		.input	= { 0xca, 0x90, 0xf5, 0x9d, 0xcb, 0xd4, 0xd2, 0x3c,
			    0x01, 0x88, 0x7f, 0x3e, 0x31, 0x6e, 0x62, 0x9d,
			    0xd8, 0xe0, 0x57, 0xa3, 0x06, 0x3a, 0x42, 0x58,
			    0x2a, 0x28, 0xfe, 0x72, 0x52, 0x2f, 0xdd, 0xe0,
			    0x19, 0x89, 0x09, 0x1c, 0x2a, 0x8e, 0x8c, 0x94,
			    0xfc, 0xc7, 0x68, 0xe4, 0x88, 0xaa, 0xde, 0x0f },
		.ilen	= 48,
		.result	= "The quick brown fox jumps over the lazy dogs.\0\0",
		.rlen	= 48,
	}, { /* split-page version */
		.key	= { 0xfe, 0xdc, 0xba, 0x98, 0x76, 0x54, 0x32, 0x10 },
		.klen	= 8,
		.iv	= { 0xf0, 0xe1, 0xd2, 0xc3, 0xb4, 0xa5, 0x96, 0x87 },
		.input	= { 0xca, 0x90, 0xf5, 0x9d, 0xcb, 0xd4, 0xd2, 0x3c,
			    0x01, 0x88, 0x7f, 0x3e, 0x31, 0x6e, 0x62, 0x9d,
			    0xd8, 0xe0, 0x57, 0xa3, 0x06, 0x3a, 0x42, 0x58,
			    0x2a, 0x28, 0xfe, 0x72, 0x52, 0x2f, 0xdd, 0xe0,
			    0x19, 0x89, 0x09, 0x1c, 0x2a, 0x8e, 0x8c, 0x94,
			    0xfc, 0xc7, 0x68, 0xe4, 0x88, 0xaa, 0xde, 0x0f },
		.ilen	= 48,
		.result	= "The quick brown fox jumps over the lazy dogs.\0\0",
		.rlen	= 48,
		.np	= 2,
		.tap	= { 20, 28 },
	}
};

/*
 * CAMELLIA test vectors.
 */
#define CAMELLIA_ENC_TEST_VECTORS 3
#define CAMELLIA_DEC_TEST_VECTORS 3
#define CAMELLIA_CBC_ENC_TEST_VECTORS 2
#define CAMELLIA_CBC_DEC_TEST_VECTORS 2

static struct cipher_testvec camellia_enc_tv_template[] = {
	{
		.key	= { 0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef,
			    0xfe, 0xdc, 0xba, 0x98, 0x76, 0x54, 0x32, 0x10 },
		.klen	= 16,
		.input	= { 0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef,
			    0xfe, 0xdc, 0xba, 0x98, 0x76, 0x54, 0x32, 0x10 },
		.ilen	= 16,
		.result	= { 0x67, 0x67, 0x31, 0x38, 0x54, 0x96, 0x69, 0x73,
			    0x08, 0x57, 0x06, 0x56, 0x48, 0xea, 0xbe, 0x43 },
		.rlen	= 16,
	}, {
		.key	= { 0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef,
			    0xfe, 0xdc, 0xba, 0x98, 0x76, 0x54, 0x32, 0x10,
			    0x00, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77 },
		.klen	= 24,
		.input	= { 0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef,
			    0xfe, 0xdc, 0xba, 0x98, 0x76, 0x54, 0x32, 0x10 },
		.ilen	= 16,
		.result	= { 0xb4, 0x99, 0x34, 0x01, 0xb3, 0xe9, 0x96, 0xf8,
			    0x4e, 0xe5, 0xce, 0xe7, 0xd7, 0x9b, 0x09, 0xb9 },
		.rlen	= 16,
	}, {
		.key	= { 0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef,
			    0xfe, 0xdc, 0xba, 0x98, 0x76, 0x54, 0x32, 0x10,
			    0x00, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77,
			    0x88, 0x99, 0xaa, 0xbb, 0xcc, 0xdd, 0xee, 0xff },
		.klen	= 32,
		.input	= { 0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef,
			    0xfe, 0xdc, 0xba, 0x98, 0x76, 0x54, 0x32, 0x10 },
		.ilen	= 16,
		.result	= { 0x9a, 0xcc, 0x23, 0x7d, 0xff, 0x16, 0xd7, 0x6c,
			    0x20, 0xef, 0x7c, 0x91, 0x9e, 0x3a, 0x75, 0x09 },
		.rlen	= 16,
	},
};

static struct cipher_testvec camellia_dec_tv_template[] = {
	{
		.key	= { 0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef,
			    0xfe, 0xdc, 0xba, 0x98, 0x76, 0x54, 0x32, 0x10 },
		.klen	= 16,
		.input	= { 0x67, 0x67, 0x31, 0x38, 0x54, 0x96, 0x69, 0x73,
			    0x08, 0x57, 0x06, 0x56, 0x48, 0xea, 0xbe, 0x43 },
		.ilen	= 16,
		.result	= { 0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef,
			    0xfe, 0xdc, 0xba, 0x98, 0x76, 0x54, 0x32, 0x10 },
		.rlen	= 16,
	}, {
		.key	= { 0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef,
			    0xfe, 0xdc, 0xba, 0x98, 0x76, 0x54, 0x32, 0x10,
			    0x00, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77 },
		.klen	= 24,
		.input	= { 0xb4, 0x99, 0x34, 0x01, 0xb3, 0xe9, 0x96, 0xf8,
			    0x4e, 0xe5, 0xce, 0xe7, 0xd7, 0x9b, 0x09, 0xb9 },
		.ilen	= 16,
		.result	= { 0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef,
			    0xfe, 0xdc, 0xba, 0x98, 0x76, 0x54, 0x32, 0x10 },
		.rlen	= 16,
	}, {
		.key	= { 0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef,
			    0xfe, 0xdc, 0xba, 0x98, 0x76, 0x54, 0x32, 0x10,
			    0x00, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77,
			    0x88, 0x99, 0xaa, 0xbb, 0xcc, 0xdd, 0xee, 0xff },
		.klen	= 32,
		.input	= { 0x9a, 0xcc, 0x23, 0x7d, 0xff, 0x16, 0xd7, 0x6c,
			    0x20, 0xef, 0x7c, 0x91, 0x9e, 0x3a, 0x75, 0x09 },
		.ilen	= 16,
		.result	= { 0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef,
			    0xfe, 0xdc, 0xba, 0x98, 0x76, 0x54, 0x32, 0x10 },
		.rlen	= 16,
	},
};

static struct cipher_testvec camellia_cbc_enc_tv_template[] = {
	{
		.key    = { 0x06, 0xa9, 0x21, 0x40, 0x36, 0xb8, 0xa1, 0x5b,
			    0x51, 0x2e, 0x03, 0xd5, 0x34, 0x12, 0x00, 0x06 },
		.klen   = 16,
		.iv	= { 0x3d, 0xaf, 0xba, 0x42, 0x9d, 0x9e, 0xb4, 0x30,
			    0xb4, 0x22, 0xda, 0x80, 0x2c, 0x9f, 0xac, 0x41 },
		.input	= { "Single block msg" },
		.ilen   = 16,
		.result = { 0xea, 0x32, 0x12, 0x76, 0x3b, 0x50, 0x10, 0xe7,
			    0x18, 0xf6, 0xfd, 0x5d, 0xf6, 0x8f, 0x13, 0x51 },
		.rlen   = 16,
	}, {
		.key    = { 0xc2, 0x86, 0x69, 0x6d, 0x88, 0x7c, 0x9a, 0xa0,
			    0x61, 0x1b, 0xbb, 0x3e, 0x20, 0x25, 0xa4, 0x5a },
		.klen   = 16,
		.iv     = { 0x56, 0x2e, 0x17, 0x99, 0x6d, 0x09, 0x3d, 0x28,
			    0xdd, 0xb3, 0xba, 0x69, 0x5a, 0x2e, 0x6f, 0x58 },
		.input  = { 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
			    0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
			    0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
			    0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f },
		.ilen   = 32,
		.result = { 0xa5, 0xdf, 0x6e, 0x50, 0xda, 0x70, 0x6c, 0x01,
			    0x4a, 0xab, 0xf3, 0xf2, 0xd6, 0xfc, 0x6c, 0xfd,
			    0x19, 0xb4, 0x3e, 0x57, 0x1c, 0x02, 0x5e, 0xa0,
			    0x15, 0x78, 0xe0, 0x5e, 0xf2, 0xcb, 0x87, 0x16 },
		.rlen   = 32,
	},
};

static struct cipher_testvec camellia_cbc_dec_tv_template[] = {
	{
		.key    = { 0x06, 0xa9, 0x21, 0x40, 0x36, 0xb8, 0xa1, 0x5b,
			    0x51, 0x2e, 0x03, 0xd5, 0x34, 0x12, 0x00, 0x06 },
		.klen   = 16,
		.iv	= { 0x3d, 0xaf, 0xba, 0x42, 0x9d, 0x9e, 0xb4, 0x30,
			    0xb4, 0x22, 0xda, 0x80, 0x2c, 0x9f, 0xac, 0x41 },
		.input	= { 0xea, 0x32, 0x12, 0x76, 0x3b, 0x50, 0x10, 0xe7,
			    0x18, 0xf6, 0xfd, 0x5d, 0xf6, 0x8f, 0x13, 0x51 },
		.ilen   = 16,
		.result = { "Single block msg" },
		.rlen   = 16,
	}, {
		.key    = { 0xc2, 0x86, 0x69, 0x6d, 0x88, 0x7c, 0x9a, 0xa0,
			    0x61, 0x1b, 0xbb, 0x3e, 0x20, 0x25, 0xa4, 0x5a },
		.klen   = 16,
		.iv     = { 0x56, 0x2e, 0x17, 0x99, 0x6d, 0x09, 0x3d, 0x28,
			    0xdd, 0xb3, 0xba, 0x69, 0x5a, 0x2e, 0x6f, 0x58 },
		.input = { 0xa5, 0xdf, 0x6e, 0x50, 0xda, 0x70, 0x6c, 0x01,
			    0x4a, 0xab, 0xf3, 0xf2, 0xd6, 0xfc, 0x6c, 0xfd,
			    0x19, 0xb4, 0x3e, 0x57, 0x1c, 0x02, 0x5e, 0xa0,
			    0x15, 0x78, 0xe0, 0x5e, 0xf2, 0xcb, 0x87, 0x16 },
		.ilen   = 32,
		.result = { 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
			    0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
			    0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
			    0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f },
		.rlen   = 32,
	},
};

/*
 * SEED test vectors
 */
#define SEED_ENC_TEST_VECTORS	4
#define SEED_DEC_TEST_VECTORS	4

static struct cipher_testvec seed_enc_tv_template[] = {
	{
		.key    = { [0 ... 15] = 0x00 },
		.klen	= 16,
		.input  = { 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
			    0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f },
		.ilen	= 16,
		.result	= { 0x5e, 0xba, 0xc6, 0xe0, 0x05, 0x4e, 0x16, 0x68,
			    0x19, 0xaf, 0xf1, 0xcc, 0x6d, 0x34, 0x6c, 0xdb },
		.rlen	= 16,
	}, {
		.key	= { 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
			    0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f },
		.klen	= 16,
		.input	= { [0 ... 15] = 0x00 },
		.ilen	= 16,
		.result	= { 0xc1, 0x1f, 0x22, 0xf2, 0x01, 0x40, 0x50, 0x50,
			    0x84, 0x48, 0x35, 0x97, 0xe4, 0x37, 0x0f, 0x43 },
		.rlen	= 16,
	}, {
		.key	= { 0x47, 0x06, 0x48, 0x08, 0x51, 0xe6, 0x1b, 0xe8,
			    0x5d, 0x74, 0xbf, 0xb3, 0xfd, 0x95, 0x61, 0x85 },
		.klen	= 16,
		.input	= { 0x83, 0xa2, 0xf8, 0xa2, 0x88, 0x64, 0x1f, 0xb9,
			    0xa4, 0xe9, 0xa5, 0xcc, 0x2f, 0x13, 0x1c, 0x7d },
		.ilen	= 16,
		.result	= { 0xee, 0x54, 0xd1, 0x3e, 0xbc, 0xae, 0x70, 0x6d,
			    0x22, 0x6b, 0xc3, 0x14, 0x2c, 0xd4, 0x0d, 0x4a },
		.rlen	= 16,
	}, {
		.key	= { 0x28, 0xdb, 0xc3, 0xbc, 0x49, 0xff, 0xd8, 0x7d,
			    0xcf, 0xa5, 0x09, 0xb1, 0x1d, 0x42, 0x2b, 0xe7 },
		.klen	= 16,
		.input	= { 0xb4, 0x1e, 0x6b, 0xe2, 0xeb, 0xa8, 0x4a, 0x14,
			    0x8e, 0x2e, 0xed, 0x84, 0x59, 0x3c, 0x5e, 0xc7 },
		.ilen	= 16,
		.result	= { 0x9b, 0x9b, 0x7b, 0xfc, 0xd1, 0x81, 0x3c, 0xb9,
			    0x5d, 0x0b, 0x36, 0x18, 0xf4, 0x0f, 0x51, 0x22 },
		.rlen	= 16,
	}
};

static struct cipher_testvec seed_dec_tv_template[] = {
	{
		.key    = { [0 ... 15] = 0x00 },
		.klen	= 16,
		.input	= { 0x5e, 0xba, 0xc6, 0xe0, 0x05, 0x4e, 0x16, 0x68,
			    0x19, 0xaf, 0xf1, 0xcc, 0x6d, 0x34, 0x6c, 0xdb },
		.ilen	= 16,
		.result	= { 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
			    0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f },
		.rlen	= 16,
	}, {
		.key	= { 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
			    0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f },
		.klen	= 16,
		.input	= { 0xc1, 0x1f, 0x22, 0xf2, 0x01, 0x40, 0x50, 0x50,
			    0x84, 0x48, 0x35, 0x97, 0xe4, 0x37, 0x0f, 0x43 },
		.ilen	= 16,
		.result	= { [0 ... 15] = 0x00 },
		.rlen	= 16,
	}, {
		.key	= { 0x47, 0x06, 0x48, 0x08, 0x51, 0xe6, 0x1b, 0xe8,
			    0x5d, 0x74, 0xbf, 0xb3, 0xfd, 0x95, 0x61, 0x85 },
		.klen	= 16,
		.input	= { 0xee, 0x54, 0xd1, 0x3e, 0xbc, 0xae, 0x70, 0x6d,
			    0x22, 0x6b, 0xc3, 0x14, 0x2c, 0xd4, 0x0d, 0x4a },
		.ilen	= 16,
		.result	= { 0x83, 0xa2, 0xf8, 0xa2, 0x88, 0x64, 0x1f, 0xb9,
			    0xa4, 0xe9, 0xa5, 0xcc, 0x2f, 0x13, 0x1c, 0x7d },
		.rlen	= 16,
	}, {
		.key	= { 0x28, 0xdb, 0xc3, 0xbc, 0x49, 0xff, 0xd8, 0x7d,
			    0xcf, 0xa5, 0x09, 0xb1, 0x1d, 0x42, 0x2b, 0xe7 },
		.klen	= 16,
		.input	= { 0x9b, 0x9b, 0x7b, 0xfc, 0xd1, 0x81, 0x3c, 0xb9,
			    0x5d, 0x0b, 0x36, 0x18, 0xf4, 0x0f, 0x51, 0x22 },
		.ilen	= 16,
		.result	= { 0xb4, 0x1e, 0x6b, 0xe2, 0xeb, 0xa8, 0x4a, 0x14,
			    0x8e, 0x2e, 0xed, 0x84, 0x59, 0x3c, 0x5e, 0xc7 },
		.rlen	= 16,
	}
};

#define SALSA20_STREAM_ENC_TEST_VECTORS 5
static struct cipher_testvec salsa20_stream_enc_tv_template[] = {
	/*
	* Testvectors from verified.test-vectors submitted to ECRYPT.
	* They are truncated to size 39, 64, 111, 129 to test a variety
	* of input length.
	*/
	{ /* Set 3, vector 0 */
		.key	= {
			    0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
			    0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F
			  },
		.klen	= 16,
		.iv     = { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
		.input	= {
			    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			  },
		.ilen	= 39,
		.result	= {
			    0x2D, 0xD5, 0xC3, 0xF7, 0xBA, 0x2B, 0x20, 0xF7,
                            0x68, 0x02, 0x41, 0x0C, 0x68, 0x86, 0x88, 0x89,
                            0x5A, 0xD8, 0xC1, 0xBD, 0x4E, 0xA6, 0xC9, 0xB1,
                            0x40, 0xFB, 0x9B, 0x90, 0xE2, 0x10, 0x49, 0xBF,
                            0x58, 0x3F, 0x52, 0x79, 0x70, 0xEB, 0xC1,
			},
		.rlen	= 39,
	}, { /* Set 5, vector 0 */
		.key	= {
			    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
			  },
		.klen	= 16,
		.iv     = { 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
		.input	= {
			    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			  },
		.ilen	= 64,
		.result	= {
			    0xB6, 0x6C, 0x1E, 0x44, 0x46, 0xDD, 0x95, 0x57,
                            0xE5, 0x78, 0xE2, 0x23, 0xB0, 0xB7, 0x68, 0x01,
                            0x7B, 0x23, 0xB2, 0x67, 0xBB, 0x02, 0x34, 0xAE,
                            0x46, 0x26, 0xBF, 0x44, 0x3F, 0x21, 0x97, 0x76,
                            0x43, 0x6F, 0xB1, 0x9F, 0xD0, 0xE8, 0x86, 0x6F,
                            0xCD, 0x0D, 0xE9, 0xA9, 0x53, 0x8F, 0x4A, 0x09,
                            0xCA, 0x9A, 0xC0, 0x73, 0x2E, 0x30, 0xBC, 0xF9,
                            0x8E, 0x4F, 0x13, 0xE4, 0xB9, 0xE2, 0x01, 0xD9,
			  },
		.rlen	= 64,
	}, { /* Set 3, vector 27 */
		.key	= {
			    0x1B, 0x1C, 0x1D, 0x1E, 0x1F, 0x20, 0x21, 0x22,
			    0x23, 0x24, 0x25, 0x26, 0x27, 0x28, 0x29, 0x2A,
                            0x2B, 0x2C, 0x2D, 0x2E, 0x2F, 0x30, 0x31, 0x32,
			    0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x3A
			  },
		.klen	= 32,
		.iv     = { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
		.input	= {
			    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,

			    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			  },
		.ilen	= 111,
		.result	= {
			    0xAE, 0x39, 0x50, 0x8E, 0xAC, 0x9A, 0xEC, 0xE7,
                            0xBF, 0x97, 0xBB, 0x20, 0xB9, 0xDE, 0xE4, 0x1F,
                            0x87, 0xD9, 0x47, 0xF8, 0x28, 0x91, 0x35, 0x98,
                            0xDB, 0x72, 0xCC, 0x23, 0x29, 0x48, 0x56, 0x5E,
                            0x83, 0x7E, 0x0B, 0xF3, 0x7D, 0x5D, 0x38, 0x7B,
                            0x2D, 0x71, 0x02, 0xB4, 0x3B, 0xB5, 0xD8, 0x23,
                            0xB0, 0x4A, 0xDF, 0x3C, 0xEC, 0xB6, 0xD9, 0x3B,
                            0x9B, 0xA7, 0x52, 0xBE, 0xC5, 0xD4, 0x50, 0x59,

                            0x15, 0x14, 0xB4, 0x0E, 0x40, 0xE6, 0x53, 0xD1,
                            0x83, 0x9C, 0x5B, 0xA0, 0x92, 0x29, 0x6B, 0x5E,
                            0x96, 0x5B, 0x1E, 0x2F, 0xD3, 0xAC, 0xC1, 0x92,
                            0xB1, 0x41, 0x3F, 0x19, 0x2F, 0xC4, 0x3B, 0xC6,
                            0x95, 0x46, 0x45, 0x54, 0xE9, 0x75, 0x03, 0x08,
                            0x44, 0xAF, 0xE5, 0x8A, 0x81, 0x12, 0x09,
			  },
		.rlen	= 111,

	}, { /* Set 5, vector 27 */
		.key	= {
			    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
			  },
		.klen	= 32,
		.iv     = { 0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00 },
		.input	= {
			    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,

			    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,

			    0x00,
			  },
		.ilen	= 129,
		.result	= {
			    0xD2, 0xDB, 0x1A, 0x5C, 0xF1, 0xC1, 0xAC, 0xDB,
                            0xE8, 0x1A, 0x7A, 0x43, 0x40, 0xEF, 0x53, 0x43,
                            0x5E, 0x7F, 0x4B, 0x1A, 0x50, 0x52, 0x3F, 0x8D,
                            0x28, 0x3D, 0xCF, 0x85, 0x1D, 0x69, 0x6E, 0x60,
                            0xF2, 0xDE, 0x74, 0x56, 0x18, 0x1B, 0x84, 0x10,
                            0xD4, 0x62, 0xBA, 0x60, 0x50, 0xF0, 0x61, 0xF2,
                            0x1C, 0x78, 0x7F, 0xC1, 0x24, 0x34, 0xAF, 0x58,
                            0xBF, 0x2C, 0x59, 0xCA, 0x90, 0x77, 0xF3, 0xB0,

                            0x5B, 0x4A, 0xDF, 0x89, 0xCE, 0x2C, 0x2F, 0xFC,
                            0x67, 0xF0, 0xE3, 0x45, 0xE8, 0xB3, 0xB3, 0x75,
                            0xA0, 0x95, 0x71, 0xA1, 0x29, 0x39, 0x94, 0xCA,
                            0x45, 0x2F, 0xBD, 0xCB, 0x10, 0xB6, 0xBE, 0x9F,
                            0x8E, 0xF9, 0xB2, 0x01, 0x0A, 0x5A, 0x0A, 0xB7,
                            0x6B, 0x9D, 0x70, 0x8E, 0x4B, 0xD6, 0x2F, 0xCD,
                            0x2E, 0x40, 0x48, 0x75, 0xE9, 0xE2, 0x21, 0x45,
                            0x0B, 0xC9, 0xB6, 0xB5, 0x66, 0xBC, 0x9A, 0x59,

                            0x5A,
			  },
		.rlen	= 129,
	}, { /* large test vector generated using Crypto++ */
		.key = {
			0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
			0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
			0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
			0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,
		},
		.klen = 32,
		.iv = {
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		},
		.input = {
			0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
			0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
			0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
			0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,
			0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27,
			0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f,
			0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
			0x38, 0x39, 0x3a, 0x3b, 0x3c, 0x3d, 0x3e, 0x3f,
			0x40, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47,
			0x48, 0x49, 0x4a, 0x4b, 0x4c, 0x4d, 0x4e, 0x4f,
			0x50, 0x51, 0x52, 0x53, 0x54, 0x55, 0x56, 0x57,
			0x58, 0x59, 0x5a, 0x5b, 0x5c, 0x5d, 0x5e, 0x5f,
			0x60, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67,
			0x68, 0x69, 0x6a, 0x6b, 0x6c, 0x6d, 0x6e, 0x6f,
			0x70, 0x71, 0x72, 0x73, 0x74, 0x75, 0x76, 0x77,
			0x78, 0x79, 0x7a, 0x7b, 0x7c, 0x7d, 0x7e, 0x7f,
			0x80, 0x81, 0x82, 0x83, 0x84, 0x85, 0x86, 0x87,
			0x88, 0x89, 0x8a, 0x8b, 0x8c, 0x8d, 0x8e, 0x8f,
			0x90, 0x91, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97,
			0x98, 0x99, 0x9a, 0x9b, 0x9c, 0x9d, 0x9e, 0x9f,
			0xa0, 0xa1, 0xa2, 0xa3, 0xa4, 0xa5, 0xa6, 0xa7,
			0xa8, 0xa9, 0xaa, 0xab, 0xac, 0xad, 0xae, 0xaf,
			0xb0, 0xb1, 0xb2, 0xb3, 0xb4, 0xb5, 0xb6, 0xb7,
			0xb8, 0xb9, 0xba, 0xbb, 0xbc, 0xbd, 0xbe, 0xbf,
			0xc0, 0xc1, 0xc2, 0xc3, 0xc4, 0xc5, 0xc6, 0xc7,
			0xc8, 0xc9, 0xca, 0xcb, 0xcc, 0xcd, 0xce, 0xcf,
			0xd0, 0xd1, 0xd2, 0xd3, 0xd4, 0xd5, 0xd6, 0xd7,
			0xd8, 0xd9, 0xda, 0xdb, 0xdc, 0xdd, 0xde, 0xdf,
			0xe0, 0xe1, 0xe2, 0xe3, 0xe4, 0xe5, 0xe6, 0xe7,
			0xe8, 0xe9, 0xea, 0xeb, 0xec, 0xed, 0xee, 0xef,
			0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7,
			0xf8, 0xf9, 0xfa, 0xfb, 0xfc, 0xfd, 0xfe, 0xff,
			0x00, 0x03, 0x06, 0x09, 0x0c, 0x0f, 0x12, 0x15,
			0x18, 0x1b, 0x1e, 0x21, 0x24, 0x27, 0x2a, 0x2d,
			0x30, 0x33, 0x36, 0x39, 0x3c, 0x3f, 0x42, 0x45,
			0x48, 0x4b, 0x4e, 0x51, 0x54, 0x57, 0x5a, 0x5d,
			0x60, 0x63, 0x66, 0x69, 0x6c, 0x6f, 0x72, 0x75,
			0x78, 0x7b, 0x7e, 0x81, 0x84, 0x87, 0x8a, 0x8d,
			0x90, 0x93, 0x96, 0x99, 0x9c, 0x9f, 0xa2, 0xa5,
			0xa8, 0xab, 0xae, 0xb1, 0xb4, 0xb7, 0xba, 0xbd,
			0xc0, 0xc3, 0xc6, 0xc9, 0xcc, 0xcf, 0xd2, 0xd5,
			0xd8, 0xdb, 0xde, 0xe1, 0xe4, 0xe7, 0xea, 0xed,
			0xf0, 0xf3, 0xf6, 0xf9, 0xfc, 0xff, 0x02, 0x05,
			0x08, 0x0b, 0x0e, 0x11, 0x14, 0x17, 0x1a, 0x1d,
			0x20, 0x23, 0x26, 0x29, 0x2c, 0x2f, 0x32, 0x35,
			0x38, 0x3b, 0x3e, 0x41, 0x44, 0x47, 0x4a, 0x4d,
			0x50, 0x53, 0x56, 0x59, 0x5c, 0x5f, 0x62, 0x65,
			0x68, 0x6b, 0x6e, 0x71, 0x74, 0x77, 0x7a, 0x7d,
			0x80, 0x83, 0x86, 0x89, 0x8c, 0x8f, 0x92, 0x95,
			0x98, 0x9b, 0x9e, 0xa1, 0xa4, 0xa7, 0xaa, 0xad,
			0xb0, 0xb3, 0xb6, 0xb9, 0xbc, 0xbf, 0xc2, 0xc5,
			0xc8, 0xcb, 0xce, 0xd1, 0xd4, 0xd7, 0xda, 0xdd,
			0xe0, 0xe3, 0xe6, 0xe9, 0xec, 0xef, 0xf2, 0xf5,
			0xf8, 0xfb, 0xfe, 0x01, 0x04, 0x07, 0x0a, 0x0d,
			0x10, 0x13, 0x16, 0x19, 0x1c, 0x1f, 0x22, 0x25,
			0x28, 0x2b, 0x2e, 0x31, 0x34, 0x37, 0x3a, 0x3d,
			0x40, 0x43, 0x46, 0x49, 0x4c, 0x4f, 0x52, 0x55,
			0x58, 0x5b, 0x5e, 0x61, 0x64, 0x67, 0x6a, 0x6d,
			0x70, 0x73, 0x76, 0x79, 0x7c, 0x7f, 0x82, 0x85,
			0x88, 0x8b, 0x8e, 0x91, 0x94, 0x97, 0x9a, 0x9d,
			0xa0, 0xa3, 0xa6, 0xa9, 0xac, 0xaf, 0xb2, 0xb5,
			0xb8, 0xbb, 0xbe, 0xc1, 0xc4, 0xc7, 0xca, 0xcd,
			0xd0, 0xd3, 0xd6, 0xd9, 0xdc, 0xdf, 0xe2, 0xe5,
			0xe8, 0xeb, 0xee, 0xf1, 0xf4, 0xf7, 0xfa, 0xfd,
			0x00, 0x05, 0x0a, 0x0f, 0x14, 0x19, 0x1e, 0x23,
			0x28, 0x2d, 0x32, 0x37, 0x3c, 0x41, 0x46, 0x4b,
			0x50, 0x55, 0x5a, 0x5f, 0x64, 0x69, 0x6e, 0x73,
			0x78, 0x7d, 0x82, 0x87, 0x8c, 0x91, 0x96, 0x9b,
			0xa0, 0xa5, 0xaa, 0xaf, 0xb4, 0xb9, 0xbe, 0xc3,
			0xc8, 0xcd, 0xd2, 0xd7, 0xdc, 0xe1, 0xe6, 0xeb,
			0xf0, 0xf5, 0xfa, 0xff, 0x04, 0x09, 0x0e, 0x13,
			0x18, 0x1d, 0x22, 0x27, 0x2c, 0x31, 0x36, 0x3b,
			0x40, 0x45, 0x4a, 0x4f, 0x54, 0x59, 0x5e, 0x63,
			0x68, 0x6d, 0x72, 0x77, 0x7c, 0x81, 0x86, 0x8b,
			0x90, 0x95, 0x9a, 0x9f, 0xa4, 0xa9, 0xae, 0xb3,
			0xb8, 0xbd, 0xc2, 0xc7, 0xcc, 0xd1, 0xd6, 0xdb,
			0xe0, 0xe5, 0xea, 0xef, 0xf4, 0xf9, 0xfe, 0x03,
			0x08, 0x0d, 0x12, 0x17, 0x1c, 0x21, 0x26, 0x2b,
			0x30, 0x35, 0x3a, 0x3f, 0x44, 0x49, 0x4e, 0x53,
			0x58, 0x5d, 0x62, 0x67, 0x6c, 0x71, 0x76, 0x7b,
			0x80, 0x85, 0x8a, 0x8f, 0x94, 0x99, 0x9e, 0xa3,
			0xa8, 0xad, 0xb2, 0xb7, 0xbc, 0xc1, 0xc6, 0xcb,
			0xd0, 0xd5, 0xda, 0xdf, 0xe4, 0xe9, 0xee, 0xf3,
			0xf8, 0xfd, 0x02, 0x07, 0x0c, 0x11, 0x16, 0x1b,
			0x20, 0x25, 0x2a, 0x2f, 0x34, 0x39, 0x3e, 0x43,
			0x48, 0x4d, 0x52, 0x57, 0x5c, 0x61, 0x66, 0x6b,
			0x70, 0x75, 0x7a, 0x7f, 0x84, 0x89, 0x8e, 0x93,
			0x98, 0x9d, 0xa2, 0xa7, 0xac, 0xb1, 0xb6, 0xbb,
			0xc0, 0xc5, 0xca, 0xcf, 0xd4, 0xd9, 0xde, 0xe3,
			0xe8, 0xed, 0xf2, 0xf7, 0xfc, 0x01, 0x06, 0x0b,
			0x10, 0x15, 0x1a, 0x1f, 0x24, 0x29, 0x2e, 0x33,
			0x38, 0x3d, 0x42, 0x47, 0x4c, 0x51, 0x56, 0x5b,
			0x60, 0x65, 0x6a, 0x6f, 0x74, 0x79, 0x7e, 0x83,
			0x88, 0x8d, 0x92, 0x97, 0x9c, 0xa1, 0xa6, 0xab,
			0xb0, 0xb5, 0xba, 0xbf, 0xc4, 0xc9, 0xce, 0xd3,
			0xd8, 0xdd, 0xe2, 0xe7, 0xec, 0xf1, 0xf6, 0xfb,
			0x00, 0x07, 0x0e, 0x15, 0x1c, 0x23, 0x2a, 0x31,
			0x38, 0x3f, 0x46, 0x4d, 0x54, 0x5b, 0x62, 0x69,
			0x70, 0x77, 0x7e, 0x85, 0x8c, 0x93, 0x9a, 0xa1,
			0xa8, 0xaf, 0xb6, 0xbd, 0xc4, 0xcb, 0xd2, 0xd9,
			0xe0, 0xe7, 0xee, 0xf5, 0xfc, 0x03, 0x0a, 0x11,
			0x18, 0x1f, 0x26, 0x2d, 0x34, 0x3b, 0x42, 0x49,
			0x50, 0x57, 0x5e, 0x65, 0x6c, 0x73, 0x7a, 0x81,
			0x88, 0x8f, 0x96, 0x9d, 0xa4, 0xab, 0xb2, 0xb9,
			0xc0, 0xc7, 0xce, 0xd5, 0xdc, 0xe3, 0xea, 0xf1,
			0xf8, 0xff, 0x06, 0x0d, 0x14, 0x1b, 0x22, 0x29,
			0x30, 0x37, 0x3e, 0x45, 0x4c, 0x53, 0x5a, 0x61,
			0x68, 0x6f, 0x76, 0x7d, 0x84, 0x8b, 0x92, 0x99,
			0xa0, 0xa7, 0xae, 0xb5, 0xbc, 0xc3, 0xca, 0xd1,
			0xd8, 0xdf, 0xe6, 0xed, 0xf4, 0xfb, 0x02, 0x09,
			0x10, 0x17, 0x1e, 0x25, 0x2c, 0x33, 0x3a, 0x41,
			0x48, 0x4f, 0x56, 0x5d, 0x64, 0x6b, 0x72, 0x79,
			0x80, 0x87, 0x8e, 0x95, 0x9c, 0xa3, 0xaa, 0xb1,
			0xb8, 0xbf, 0xc6, 0xcd, 0xd4, 0xdb, 0xe2, 0xe9,
			0xf0, 0xf7, 0xfe, 0x05, 0x0c, 0x13, 0x1a, 0x21,
			0x28, 0x2f, 0x36, 0x3d, 0x44, 0x4b, 0x52, 0x59,
			0x60, 0x67, 0x6e, 0x75, 0x7c, 0x83, 0x8a, 0x91,
			0x98, 0x9f, 0xa6, 0xad, 0xb4, 0xbb, 0xc2, 0xc9,
			0xd0, 0xd7, 0xde, 0xe5, 0xec, 0xf3, 0xfa, 0x01,
			0x08, 0x0f, 0x16, 0x1d, 0x24, 0x2b, 0x32, 0x39,
			0x40, 0x47, 0x4e, 0x55, 0x5c, 0x63, 0x6a, 0x71,
			0x78, 0x7f, 0x86, 0x8d, 0x94, 0x9b, 0xa2, 0xa9,
			0xb0, 0xb7, 0xbe, 0xc5, 0xcc, 0xd3, 0xda, 0xe1,
			0xe8, 0xef, 0xf6, 0xfd, 0x04, 0x0b, 0x12, 0x19,
			0x20, 0x27, 0x2e, 0x35, 0x3c, 0x43, 0x4a, 0x51,
			0x58, 0x5f, 0x66, 0x6d, 0x74, 0x7b, 0x82, 0x89,
			0x90, 0x97, 0x9e, 0xa5, 0xac, 0xb3, 0xba, 0xc1,
			0xc8, 0xcf, 0xd6, 0xdd, 0xe4, 0xeb, 0xf2, 0xf9,
			0x00, 0x09, 0x12, 0x1b, 0x24, 0x2d, 0x36, 0x3f,
			0x48, 0x51, 0x5a, 0x63, 0x6c, 0x75, 0x7e, 0x87,
			0x90, 0x99, 0xa2, 0xab, 0xb4, 0xbd, 0xc6, 0xcf,
			0xd8, 0xe1, 0xea, 0xf3, 0xfc, 0x05, 0x0e, 0x17,
			0x20, 0x29, 0x32, 0x3b, 0x44, 0x4d, 0x56, 0x5f,
			0x68, 0x71, 0x7a, 0x83, 0x8c, 0x95, 0x9e, 0xa7,
			0xb0, 0xb9, 0xc2, 0xcb, 0xd4, 0xdd, 0xe6, 0xef,
			0xf8, 0x01, 0x0a, 0x13, 0x1c, 0x25, 0x2e, 0x37,
			0x40, 0x49, 0x52, 0x5b, 0x64, 0x6d, 0x76, 0x7f,
			0x88, 0x91, 0x9a, 0xa3, 0xac, 0xb5, 0xbe, 0xc7,
			0xd0, 0xd9, 0xe2, 0xeb, 0xf4, 0xfd, 0x06, 0x0f,
			0x18, 0x21, 0x2a, 0x33, 0x3c, 0x45, 0x4e, 0x57,
			0x60, 0x69, 0x72, 0x7b, 0x84, 0x8d, 0x96, 0x9f,
			0xa8, 0xb1, 0xba, 0xc3, 0xcc, 0xd5, 0xde, 0xe7,
			0xf0, 0xf9, 0x02, 0x0b, 0x14, 0x1d, 0x26, 0x2f,
			0x38, 0x41, 0x4a, 0x53, 0x5c, 0x65, 0x6e, 0x77,
			0x80, 0x89, 0x92, 0x9b, 0xa4, 0xad, 0xb6, 0xbf,
			0xc8, 0xd1, 0xda, 0xe3, 0xec, 0xf5, 0xfe, 0x07,
			0x10, 0x19, 0x22, 0x2b, 0x34, 0x3d, 0x46, 0x4f,
			0x58, 0x61, 0x6a, 0x73, 0x7c, 0x85, 0x8e, 0x97,
			0xa0, 0xa9, 0xb2, 0xbb, 0xc4, 0xcd, 0xd6, 0xdf,
			0xe8, 0xf1, 0xfa, 0x03, 0x0c, 0x15, 0x1e, 0x27,
			0x30, 0x39, 0x42, 0x4b, 0x54, 0x5d, 0x66, 0x6f,
			0x78, 0x81, 0x8a, 0x93, 0x9c, 0xa5, 0xae, 0xb7,
			0xc0, 0xc9, 0xd2, 0xdb, 0xe4, 0xed, 0xf6, 0xff,
			0x08, 0x11, 0x1a, 0x23, 0x2c, 0x35, 0x3e, 0x47,
			0x50, 0x59, 0x62, 0x6b, 0x74, 0x7d, 0x86, 0x8f,
			0x98, 0xa1, 0xaa, 0xb3, 0xbc, 0xc5, 0xce, 0xd7,
			0xe0, 0xe9, 0xf2, 0xfb, 0x04, 0x0d, 0x16, 0x1f,
			0x28, 0x31, 0x3a, 0x43, 0x4c, 0x55, 0x5e, 0x67,
			0x70, 0x79, 0x82, 0x8b, 0x94, 0x9d, 0xa6, 0xaf,
			0xb8, 0xc1, 0xca, 0xd3, 0xdc, 0xe5, 0xee, 0xf7,
			0x00, 0x0b, 0x16, 0x21, 0x2c, 0x37, 0x42, 0x4d,
			0x58, 0x63, 0x6e, 0x79, 0x84, 0x8f, 0x9a, 0xa5,
			0xb0, 0xbb, 0xc6, 0xd1, 0xdc, 0xe7, 0xf2, 0xfd,
			0x08, 0x13, 0x1e, 0x29, 0x34, 0x3f, 0x4a, 0x55,
			0x60, 0x6b, 0x76, 0x81, 0x8c, 0x97, 0xa2, 0xad,
			0xb8, 0xc3, 0xce, 0xd9, 0xe4, 0xef, 0xfa, 0x05,
			0x10, 0x1b, 0x26, 0x31, 0x3c, 0x47, 0x52, 0x5d,
			0x68, 0x73, 0x7e, 0x89, 0x94, 0x9f, 0xaa, 0xb5,
			0xc0, 0xcb, 0xd6, 0xe1, 0xec, 0xf7, 0x02, 0x0d,
			0x18, 0x23, 0x2e, 0x39, 0x44, 0x4f, 0x5a, 0x65,
			0x70, 0x7b, 0x86, 0x91, 0x9c, 0xa7, 0xb2, 0xbd,
			0xc8, 0xd3, 0xde, 0xe9, 0xf4, 0xff, 0x0a, 0x15,
			0x20, 0x2b, 0x36, 0x41, 0x4c, 0x57, 0x62, 0x6d,
			0x78, 0x83, 0x8e, 0x99, 0xa4, 0xaf, 0xba, 0xc5,
			0xd0, 0xdb, 0xe6, 0xf1, 0xfc, 0x07, 0x12, 0x1d,
			0x28, 0x33, 0x3e, 0x49, 0x54, 0x5f, 0x6a, 0x75,
			0x80, 0x8b, 0x96, 0xa1, 0xac, 0xb7, 0xc2, 0xcd,
			0xd8, 0xe3, 0xee, 0xf9, 0x04, 0x0f, 0x1a, 0x25,
			0x30, 0x3b, 0x46, 0x51, 0x5c, 0x67, 0x72, 0x7d,
			0x88, 0x93, 0x9e, 0xa9, 0xb4, 0xbf, 0xca, 0xd5,
			0xe0, 0xeb, 0xf6, 0x01, 0x0c, 0x17, 0x22, 0x2d,
			0x38, 0x43, 0x4e, 0x59, 0x64, 0x6f, 0x7a, 0x85,
			0x90, 0x9b, 0xa6, 0xb1, 0xbc, 0xc7, 0xd2, 0xdd,
			0xe8, 0xf3, 0xfe, 0x09, 0x14, 0x1f, 0x2a, 0x35,
			0x40, 0x4b, 0x56, 0x61, 0x6c, 0x77, 0x82, 0x8d,
			0x98, 0xa3, 0xae, 0xb9, 0xc4, 0xcf, 0xda, 0xe5,
			0xf0, 0xfb, 0x06, 0x11, 0x1c, 0x27, 0x32, 0x3d,
			0x48, 0x53, 0x5e, 0x69, 0x74, 0x7f, 0x8a, 0x95,
			0xa0, 0xab, 0xb6, 0xc1, 0xcc, 0xd7, 0xe2, 0xed,
			0xf8, 0x03, 0x0e, 0x19, 0x24, 0x2f, 0x3a, 0x45,
			0x50, 0x5b, 0x66, 0x71, 0x7c, 0x87, 0x92, 0x9d,
			0xa8, 0xb3, 0xbe, 0xc9, 0xd4, 0xdf, 0xea, 0xf5,
			0x00, 0x0d, 0x1a, 0x27, 0x34, 0x41, 0x4e, 0x5b,
			0x68, 0x75, 0x82, 0x8f, 0x9c, 0xa9, 0xb6, 0xc3,
			0xd0, 0xdd, 0xea, 0xf7, 0x04, 0x11, 0x1e, 0x2b,
			0x38, 0x45, 0x52, 0x5f, 0x6c, 0x79, 0x86, 0x93,
			0xa0, 0xad, 0xba, 0xc7, 0xd4, 0xe1, 0xee, 0xfb,
			0x08, 0x15, 0x22, 0x2f, 0x3c, 0x49, 0x56, 0x63,
			0x70, 0x7d, 0x8a, 0x97, 0xa4, 0xb1, 0xbe, 0xcb,
			0xd8, 0xe5, 0xf2, 0xff, 0x0c, 0x19, 0x26, 0x33,
			0x40, 0x4d, 0x5a, 0x67, 0x74, 0x81, 0x8e, 0x9b,
			0xa8, 0xb5, 0xc2, 0xcf, 0xdc, 0xe9, 0xf6, 0x03,
			0x10, 0x1d, 0x2a, 0x37, 0x44, 0x51, 0x5e, 0x6b,
			0x78, 0x85, 0x92, 0x9f, 0xac, 0xb9, 0xc6, 0xd3,
			0xe0, 0xed, 0xfa, 0x07, 0x14, 0x21, 0x2e, 0x3b,
			0x48, 0x55, 0x62, 0x6f, 0x7c, 0x89, 0x96, 0xa3,
			0xb0, 0xbd, 0xca, 0xd7, 0xe4, 0xf1, 0xfe, 0x0b,
			0x18, 0x25, 0x32, 0x3f, 0x4c, 0x59, 0x66, 0x73,
			0x80, 0x8d, 0x9a, 0xa7, 0xb4, 0xc1, 0xce, 0xdb,
			0xe8, 0xf5, 0x02, 0x0f, 0x1c, 0x29, 0x36, 0x43,
			0x50, 0x5d, 0x6a, 0x77, 0x84, 0x91, 0x9e, 0xab,
			0xb8, 0xc5, 0xd2, 0xdf, 0xec, 0xf9, 0x06, 0x13,
			0x20, 0x2d, 0x3a, 0x47, 0x54, 0x61, 0x6e, 0x7b,
			0x88, 0x95, 0xa2, 0xaf, 0xbc, 0xc9, 0xd6, 0xe3,
			0xf0, 0xfd, 0x0a, 0x17, 0x24, 0x31, 0x3e, 0x4b,
			0x58, 0x65, 0x72, 0x7f, 0x8c, 0x99, 0xa6, 0xb3,
			0xc0, 0xcd, 0xda, 0xe7, 0xf4, 0x01, 0x0e, 0x1b,
			0x28, 0x35, 0x42, 0x4f, 0x5c, 0x69, 0x76, 0x83,
			0x90, 0x9d, 0xaa, 0xb7, 0xc4, 0xd1, 0xde, 0xeb,
			0xf8, 0x05, 0x12, 0x1f, 0x2c, 0x39, 0x46, 0x53,
			0x60, 0x6d, 0x7a, 0x87, 0x94, 0xa1, 0xae, 0xbb,
			0xc8, 0xd5, 0xe2, 0xef, 0xfc, 0x09, 0x16, 0x23,
			0x30, 0x3d, 0x4a, 0x57, 0x64, 0x71, 0x7e, 0x8b,
			0x98, 0xa5, 0xb2, 0xbf, 0xcc, 0xd9, 0xe6, 0xf3,
			0x00, 0x0f, 0x1e, 0x2d, 0x3c, 0x4b, 0x5a, 0x69,
			0x78, 0x87, 0x96, 0xa5, 0xb4, 0xc3, 0xd2, 0xe1,
			0xf0, 0xff, 0x0e, 0x1d, 0x2c, 0x3b, 0x4a, 0x59,
			0x68, 0x77, 0x86, 0x95, 0xa4, 0xb3, 0xc2, 0xd1,
			0xe0, 0xef, 0xfe, 0x0d, 0x1c, 0x2b, 0x3a, 0x49,
			0x58, 0x67, 0x76, 0x85, 0x94, 0xa3, 0xb2, 0xc1,
			0xd0, 0xdf, 0xee, 0xfd, 0x0c, 0x1b, 0x2a, 0x39,
			0x48, 0x57, 0x66, 0x75, 0x84, 0x93, 0xa2, 0xb1,
			0xc0, 0xcf, 0xde, 0xed, 0xfc, 0x0b, 0x1a, 0x29,
			0x38, 0x47, 0x56, 0x65, 0x74, 0x83, 0x92, 0xa1,
			0xb0, 0xbf, 0xce, 0xdd, 0xec, 0xfb, 0x0a, 0x19,
			0x28, 0x37, 0x46, 0x55, 0x64, 0x73, 0x82, 0x91,
			0xa0, 0xaf, 0xbe, 0xcd, 0xdc, 0xeb, 0xfa, 0x09,
			0x18, 0x27, 0x36, 0x45, 0x54, 0x63, 0x72, 0x81,
			0x90, 0x9f, 0xae, 0xbd, 0xcc, 0xdb, 0xea, 0xf9,
			0x08, 0x17, 0x26, 0x35, 0x44, 0x53, 0x62, 0x71,
			0x80, 0x8f, 0x9e, 0xad, 0xbc, 0xcb, 0xda, 0xe9,
			0xf8, 0x07, 0x16, 0x25, 0x34, 0x43, 0x52, 0x61,
			0x70, 0x7f, 0x8e, 0x9d, 0xac, 0xbb, 0xca, 0xd9,
			0xe8, 0xf7, 0x06, 0x15, 0x24, 0x33, 0x42, 0x51,
			0x60, 0x6f, 0x7e, 0x8d, 0x9c, 0xab, 0xba, 0xc9,
			0xd8, 0xe7, 0xf6, 0x05, 0x14, 0x23, 0x32, 0x41,
			0x50, 0x5f, 0x6e, 0x7d, 0x8c, 0x9b, 0xaa, 0xb9,
			0xc8, 0xd7, 0xe6, 0xf5, 0x04, 0x13, 0x22, 0x31,
			0x40, 0x4f, 0x5e, 0x6d, 0x7c, 0x8b, 0x9a, 0xa9,
			0xb8, 0xc7, 0xd6, 0xe5, 0xf4, 0x03, 0x12, 0x21,
			0x30, 0x3f, 0x4e, 0x5d, 0x6c, 0x7b, 0x8a, 0x99,
			0xa8, 0xb7, 0xc6, 0xd5, 0xe4, 0xf3, 0x02, 0x11,
			0x20, 0x2f, 0x3e, 0x4d, 0x5c, 0x6b, 0x7a, 0x89,
			0x98, 0xa7, 0xb6, 0xc5, 0xd4, 0xe3, 0xf2, 0x01,
			0x10, 0x1f, 0x2e, 0x3d, 0x4c, 0x5b, 0x6a, 0x79,
			0x88, 0x97, 0xa6, 0xb5, 0xc4, 0xd3, 0xe2, 0xf1,
			0x00, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77,
			0x88, 0x99, 0xaa, 0xbb, 0xcc, 0xdd, 0xee, 0xff,
			0x10, 0x21, 0x32, 0x43, 0x54, 0x65, 0x76, 0x87,
			0x98, 0xa9, 0xba, 0xcb, 0xdc, 0xed, 0xfe, 0x0f,
			0x20, 0x31, 0x42, 0x53, 0x64, 0x75, 0x86, 0x97,
			0xa8, 0xb9, 0xca, 0xdb, 0xec, 0xfd, 0x0e, 0x1f,
			0x30, 0x41, 0x52, 0x63, 0x74, 0x85, 0x96, 0xa7,
			0xb8, 0xc9, 0xda, 0xeb, 0xfc, 0x0d, 0x1e, 0x2f,
			0x40, 0x51, 0x62, 0x73, 0x84, 0x95, 0xa6, 0xb7,
			0xc8, 0xd9, 0xea, 0xfb, 0x0c, 0x1d, 0x2e, 0x3f,
			0x50, 0x61, 0x72, 0x83, 0x94, 0xa5, 0xb6, 0xc7,
			0xd8, 0xe9, 0xfa, 0x0b, 0x1c, 0x2d, 0x3e, 0x4f,
			0x60, 0x71, 0x82, 0x93, 0xa4, 0xb5, 0xc6, 0xd7,
			0xe8, 0xf9, 0x0a, 0x1b, 0x2c, 0x3d, 0x4e, 0x5f,
			0x70, 0x81, 0x92, 0xa3, 0xb4, 0xc5, 0xd6, 0xe7,
			0xf8, 0x09, 0x1a, 0x2b, 0x3c, 0x4d, 0x5e, 0x6f,
			0x80, 0x91, 0xa2, 0xb3, 0xc4, 0xd5, 0xe6, 0xf7,
			0x08, 0x19, 0x2a, 0x3b, 0x4c, 0x5d, 0x6e, 0x7f,
			0x90, 0xa1, 0xb2, 0xc3, 0xd4, 0xe5, 0xf6, 0x07,
			0x18, 0x29, 0x3a, 0x4b, 0x5c, 0x6d, 0x7e, 0x8f,
			0xa0, 0xb1, 0xc2, 0xd3, 0xe4, 0xf5, 0x06, 0x17,
			0x28, 0x39, 0x4a, 0x5b, 0x6c, 0x7d, 0x8e, 0x9f,
			0xb0, 0xc1, 0xd2, 0xe3, 0xf4, 0x05, 0x16, 0x27,
			0x38, 0x49, 0x5a, 0x6b, 0x7c, 0x8d, 0x9e, 0xaf,
			0xc0, 0xd1, 0xe2, 0xf3, 0x04, 0x15, 0x26, 0x37,
			0x48, 0x59, 0x6a, 0x7b, 0x8c, 0x9d, 0xae, 0xbf,
			0xd0, 0xe1, 0xf2, 0x03, 0x14, 0x25, 0x36, 0x47,
			0x58, 0x69, 0x7a, 0x8b, 0x9c, 0xad, 0xbe, 0xcf,
			0xe0, 0xf1, 0x02, 0x13, 0x24, 0x35, 0x46, 0x57,
			0x68, 0x79, 0x8a, 0x9b, 0xac, 0xbd, 0xce, 0xdf,
			0xf0, 0x01, 0x12, 0x23, 0x34, 0x45, 0x56, 0x67,
			0x78, 0x89, 0x9a, 0xab, 0xbc, 0xcd, 0xde, 0xef,
			0x00, 0x13, 0x26, 0x39, 0x4c, 0x5f, 0x72, 0x85,
			0x98, 0xab, 0xbe, 0xd1, 0xe4, 0xf7, 0x0a, 0x1d,
			0x30, 0x43, 0x56, 0x69, 0x7c, 0x8f, 0xa2, 0xb5,
			0xc8, 0xdb, 0xee, 0x01, 0x14, 0x27, 0x3a, 0x4d,
			0x60, 0x73, 0x86, 0x99, 0xac, 0xbf, 0xd2, 0xe5,
			0xf8, 0x0b, 0x1e, 0x31, 0x44, 0x57, 0x6a, 0x7d,
			0x90, 0xa3, 0xb6, 0xc9, 0xdc, 0xef, 0x02, 0x15,
			0x28, 0x3b, 0x4e, 0x61, 0x74, 0x87, 0x9a, 0xad,
			0xc0, 0xd3, 0xe6, 0xf9, 0x0c, 0x1f, 0x32, 0x45,
			0x58, 0x6b, 0x7e, 0x91, 0xa4, 0xb7, 0xca, 0xdd,
			0xf0, 0x03, 0x16, 0x29, 0x3c, 0x4f, 0x62, 0x75,
			0x88, 0x9b, 0xae, 0xc1, 0xd4, 0xe7, 0xfa, 0x0d,
			0x20, 0x33, 0x46, 0x59, 0x6c, 0x7f, 0x92, 0xa5,
			0xb8, 0xcb, 0xde, 0xf1, 0x04, 0x17, 0x2a, 0x3d,
			0x50, 0x63, 0x76, 0x89, 0x9c, 0xaf, 0xc2, 0xd5,
			0xe8, 0xfb, 0x0e, 0x21, 0x34, 0x47, 0x5a, 0x6d,
			0x80, 0x93, 0xa6, 0xb9, 0xcc, 0xdf, 0xf2, 0x05,
			0x18, 0x2b, 0x3e, 0x51, 0x64, 0x77, 0x8a, 0x9d,
			0xb0, 0xc3, 0xd6, 0xe9, 0xfc, 0x0f, 0x22, 0x35,
			0x48, 0x5b, 0x6e, 0x81, 0x94, 0xa7, 0xba, 0xcd,
			0xe0, 0xf3, 0x06, 0x19, 0x2c, 0x3f, 0x52, 0x65,
			0x78, 0x8b, 0x9e, 0xb1, 0xc4, 0xd7, 0xea, 0xfd,
			0x10, 0x23, 0x36, 0x49, 0x5c, 0x6f, 0x82, 0x95,
			0xa8, 0xbb, 0xce, 0xe1, 0xf4, 0x07, 0x1a, 0x2d,
			0x40, 0x53, 0x66, 0x79, 0x8c, 0x9f, 0xb2, 0xc5,
			0xd8, 0xeb, 0xfe, 0x11, 0x24, 0x37, 0x4a, 0x5d,
			0x70, 0x83, 0x96, 0xa9, 0xbc, 0xcf, 0xe2, 0xf5,
			0x08, 0x1b, 0x2e, 0x41, 0x54, 0x67, 0x7a, 0x8d,
			0xa0, 0xb3, 0xc6, 0xd9, 0xec, 0xff, 0x12, 0x25,
			0x38, 0x4b, 0x5e, 0x71, 0x84, 0x97, 0xaa, 0xbd,
			0xd0, 0xe3, 0xf6, 0x09, 0x1c, 0x2f, 0x42, 0x55,
			0x68, 0x7b, 0x8e, 0xa1, 0xb4, 0xc7, 0xda, 0xed,
			0x00, 0x15, 0x2a, 0x3f, 0x54, 0x69, 0x7e, 0x93,
			0xa8, 0xbd, 0xd2, 0xe7, 0xfc, 0x11, 0x26, 0x3b,
			0x50, 0x65, 0x7a, 0x8f, 0xa4, 0xb9, 0xce, 0xe3,
			0xf8, 0x0d, 0x22, 0x37, 0x4c, 0x61, 0x76, 0x8b,
			0xa0, 0xb5, 0xca, 0xdf, 0xf4, 0x09, 0x1e, 0x33,
			0x48, 0x5d, 0x72, 0x87, 0x9c, 0xb1, 0xc6, 0xdb,
			0xf0, 0x05, 0x1a, 0x2f, 0x44, 0x59, 0x6e, 0x83,
			0x98, 0xad, 0xc2, 0xd7, 0xec, 0x01, 0x16, 0x2b,
			0x40, 0x55, 0x6a, 0x7f, 0x94, 0xa9, 0xbe, 0xd3,
			0xe8, 0xfd, 0x12, 0x27, 0x3c, 0x51, 0x66, 0x7b,
			0x90, 0xa5, 0xba, 0xcf, 0xe4, 0xf9, 0x0e, 0x23,
			0x38, 0x4d, 0x62, 0x77, 0x8c, 0xa1, 0xb6, 0xcb,
			0xe0, 0xf5, 0x0a, 0x1f, 0x34, 0x49, 0x5e, 0x73,
			0x88, 0x9d, 0xb2, 0xc7, 0xdc, 0xf1, 0x06, 0x1b,
			0x30, 0x45, 0x5a, 0x6f, 0x84, 0x99, 0xae, 0xc3,
			0xd8, 0xed, 0x02, 0x17, 0x2c, 0x41, 0x56, 0x6b,
			0x80, 0x95, 0xaa, 0xbf, 0xd4, 0xe9, 0xfe, 0x13,
			0x28, 0x3d, 0x52, 0x67, 0x7c, 0x91, 0xa6, 0xbb,
			0xd0, 0xe5, 0xfa, 0x0f, 0x24, 0x39, 0x4e, 0x63,
			0x78, 0x8d, 0xa2, 0xb7, 0xcc, 0xe1, 0xf6, 0x0b,
			0x20, 0x35, 0x4a, 0x5f, 0x74, 0x89, 0x9e, 0xb3,
			0xc8, 0xdd, 0xf2, 0x07, 0x1c, 0x31, 0x46, 0x5b,
			0x70, 0x85, 0x9a, 0xaf, 0xc4, 0xd9, 0xee, 0x03,
			0x18, 0x2d, 0x42, 0x57, 0x6c, 0x81, 0x96, 0xab,
			0xc0, 0xd5, 0xea, 0xff, 0x14, 0x29, 0x3e, 0x53,
			0x68, 0x7d, 0x92, 0xa7, 0xbc, 0xd1, 0xe6, 0xfb,
			0x10, 0x25, 0x3a, 0x4f, 0x64, 0x79, 0x8e, 0xa3,
			0xb8, 0xcd, 0xe2, 0xf7, 0x0c, 0x21, 0x36, 0x4b,
			0x60, 0x75, 0x8a, 0x9f, 0xb4, 0xc9, 0xde, 0xf3,
			0x08, 0x1d, 0x32, 0x47, 0x5c, 0x71, 0x86, 0x9b,
			0xb0, 0xc5, 0xda, 0xef, 0x04, 0x19, 0x2e, 0x43,
			0x58, 0x6d, 0x82, 0x97, 0xac, 0xc1, 0xd6, 0xeb,
			0x00, 0x17, 0x2e, 0x45, 0x5c, 0x73, 0x8a, 0xa1,
			0xb8, 0xcf, 0xe6, 0xfd, 0x14, 0x2b, 0x42, 0x59,
			0x70, 0x87, 0x9e, 0xb5, 0xcc, 0xe3, 0xfa, 0x11,
			0x28, 0x3f, 0x56, 0x6d, 0x84, 0x9b, 0xb2, 0xc9,
			0xe0, 0xf7, 0x0e, 0x25, 0x3c, 0x53, 0x6a, 0x81,
			0x98, 0xaf, 0xc6, 0xdd, 0xf4, 0x0b, 0x22, 0x39,
			0x50, 0x67, 0x7e, 0x95, 0xac, 0xc3, 0xda, 0xf1,
			0x08, 0x1f, 0x36, 0x4d, 0x64, 0x7b, 0x92, 0xa9,
			0xc0, 0xd7, 0xee, 0x05, 0x1c, 0x33, 0x4a, 0x61,
			0x78, 0x8f, 0xa6, 0xbd, 0xd4, 0xeb, 0x02, 0x19,
			0x30, 0x47, 0x5e, 0x75, 0x8c, 0xa3, 0xba, 0xd1,
			0xe8, 0xff, 0x16, 0x2d, 0x44, 0x5b, 0x72, 0x89,
			0xa0, 0xb7, 0xce, 0xe5, 0xfc, 0x13, 0x2a, 0x41,
			0x58, 0x6f, 0x86, 0x9d, 0xb4, 0xcb, 0xe2, 0xf9,
			0x10, 0x27, 0x3e, 0x55, 0x6c, 0x83, 0x9a, 0xb1,
			0xc8, 0xdf, 0xf6, 0x0d, 0x24, 0x3b, 0x52, 0x69,
			0x80, 0x97, 0xae, 0xc5, 0xdc, 0xf3, 0x0a, 0x21,
			0x38, 0x4f, 0x66, 0x7d, 0x94, 0xab, 0xc2, 0xd9,
			0xf0, 0x07, 0x1e, 0x35, 0x4c, 0x63, 0x7a, 0x91,
			0xa8, 0xbf, 0xd6, 0xed, 0x04, 0x1b, 0x32, 0x49,
			0x60, 0x77, 0x8e, 0xa5, 0xbc, 0xd3, 0xea, 0x01,
			0x18, 0x2f, 0x46, 0x5d, 0x74, 0x8b, 0xa2, 0xb9,
			0xd0, 0xe7, 0xfe, 0x15, 0x2c, 0x43, 0x5a, 0x71,
			0x88, 0x9f, 0xb6, 0xcd, 0xe4, 0xfb, 0x12, 0x29,
			0x40, 0x57, 0x6e, 0x85, 0x9c, 0xb3, 0xca, 0xe1,
			0xf8, 0x0f, 0x26, 0x3d, 0x54, 0x6b, 0x82, 0x99,
			0xb0, 0xc7, 0xde, 0xf5, 0x0c, 0x23, 0x3a, 0x51,
			0x68, 0x7f, 0x96, 0xad, 0xc4, 0xdb, 0xf2, 0x09,
			0x20, 0x37, 0x4e, 0x65, 0x7c, 0x93, 0xaa, 0xc1,
			0xd8, 0xef, 0x06, 0x1d, 0x34, 0x4b, 0x62, 0x79,
			0x90, 0xa7, 0xbe, 0xd5, 0xec, 0x03, 0x1a, 0x31,
			0x48, 0x5f, 0x76, 0x8d, 0xa4, 0xbb, 0xd2, 0xe9,
			0x00, 0x19, 0x32, 0x4b, 0x64, 0x7d, 0x96, 0xaf,
			0xc8, 0xe1, 0xfa, 0x13, 0x2c, 0x45, 0x5e, 0x77,
			0x90, 0xa9, 0xc2, 0xdb, 0xf4, 0x0d, 0x26, 0x3f,
			0x58, 0x71, 0x8a, 0xa3, 0xbc, 0xd5, 0xee, 0x07,
			0x20, 0x39, 0x52, 0x6b, 0x84, 0x9d, 0xb6, 0xcf,
			0xe8, 0x01, 0x1a, 0x33, 0x4c, 0x65, 0x7e, 0x97,
			0xb0, 0xc9, 0xe2, 0xfb, 0x14, 0x2d, 0x46, 0x5f,
			0x78, 0x91, 0xaa, 0xc3, 0xdc, 0xf5, 0x0e, 0x27,
			0x40, 0x59, 0x72, 0x8b, 0xa4, 0xbd, 0xd6, 0xef,
			0x08, 0x21, 0x3a, 0x53, 0x6c, 0x85, 0x9e, 0xb7,
			0xd0, 0xe9, 0x02, 0x1b, 0x34, 0x4d, 0x66, 0x7f,
			0x98, 0xb1, 0xca, 0xe3, 0xfc, 0x15, 0x2e, 0x47,
			0x60, 0x79, 0x92, 0xab, 0xc4, 0xdd, 0xf6, 0x0f,
			0x28, 0x41, 0x5a, 0x73, 0x8c, 0xa5, 0xbe, 0xd7,
			0xf0, 0x09, 0x22, 0x3b, 0x54, 0x6d, 0x86, 0x9f,
			0xb8, 0xd1, 0xea, 0x03, 0x1c, 0x35, 0x4e, 0x67,
			0x80, 0x99, 0xb2, 0xcb, 0xe4, 0xfd, 0x16, 0x2f,
			0x48, 0x61, 0x7a, 0x93, 0xac, 0xc5, 0xde, 0xf7,
			0x10, 0x29, 0x42, 0x5b, 0x74, 0x8d, 0xa6, 0xbf,
			0xd8, 0xf1, 0x0a, 0x23, 0x3c, 0x55, 0x6e, 0x87,
			0xa0, 0xb9, 0xd2, 0xeb, 0x04, 0x1d, 0x36, 0x4f,
			0x68, 0x81, 0x9a, 0xb3, 0xcc, 0xe5, 0xfe, 0x17,
			0x30, 0x49, 0x62, 0x7b, 0x94, 0xad, 0xc6, 0xdf,
			0xf8, 0x11, 0x2a, 0x43, 0x5c, 0x75, 0x8e, 0xa7,
			0xc0, 0xd9, 0xf2, 0x0b, 0x24, 0x3d, 0x56, 0x6f,
			0x88, 0xa1, 0xba, 0xd3, 0xec, 0x05, 0x1e, 0x37,
			0x50, 0x69, 0x82, 0x9b, 0xb4, 0xcd, 0xe6, 0xff,
			0x18, 0x31, 0x4a, 0x63, 0x7c, 0x95, 0xae, 0xc7,
			0xe0, 0xf9, 0x12, 0x2b, 0x44, 0x5d, 0x76, 0x8f,
			0xa8, 0xc1, 0xda, 0xf3, 0x0c, 0x25, 0x3e, 0x57,
			0x70, 0x89, 0xa2, 0xbb, 0xd4, 0xed, 0x06, 0x1f,
			0x38, 0x51, 0x6a, 0x83, 0x9c, 0xb5, 0xce, 0xe7,
			0x00, 0x1b, 0x36, 0x51, 0x6c, 0x87, 0xa2, 0xbd,
			0xd8, 0xf3, 0x0e, 0x29, 0x44, 0x5f, 0x7a, 0x95,
			0xb0, 0xcb, 0xe6, 0x01, 0x1c, 0x37, 0x52, 0x6d,
			0x88, 0xa3, 0xbe, 0xd9, 0xf4, 0x0f, 0x2a, 0x45,
			0x60, 0x7b, 0x96, 0xb1, 0xcc, 0xe7, 0x02, 0x1d,
			0x38, 0x53, 0x6e, 0x89, 0xa4, 0xbf, 0xda, 0xf5,
			0x10, 0x2b, 0x46, 0x61, 0x7c, 0x97, 0xb2, 0xcd,
			0xe8, 0x03, 0x1e, 0x39, 0x54, 0x6f, 0x8a, 0xa5,
			0xc0, 0xdb, 0xf6, 0x11, 0x2c, 0x47, 0x62, 0x7d,
			0x98, 0xb3, 0xce, 0xe9, 0x04, 0x1f, 0x3a, 0x55,
			0x70, 0x8b, 0xa6, 0xc1, 0xdc, 0xf7, 0x12, 0x2d,
			0x48, 0x63, 0x7e, 0x99, 0xb4, 0xcf, 0xea, 0x05,
			0x20, 0x3b, 0x56, 0x71, 0x8c, 0xa7, 0xc2, 0xdd,
			0xf8, 0x13, 0x2e, 0x49, 0x64, 0x7f, 0x9a, 0xb5,
			0xd0, 0xeb, 0x06, 0x21, 0x3c, 0x57, 0x72, 0x8d,
			0xa8, 0xc3, 0xde, 0xf9, 0x14, 0x2f, 0x4a, 0x65,
			0x80, 0x9b, 0xb6, 0xd1, 0xec, 0x07, 0x22, 0x3d,
			0x58, 0x73, 0x8e, 0xa9, 0xc4, 0xdf, 0xfa, 0x15,
			0x30, 0x4b, 0x66, 0x81, 0x9c, 0xb7, 0xd2, 0xed,
			0x08, 0x23, 0x3e, 0x59, 0x74, 0x8f, 0xaa, 0xc5,
			0xe0, 0xfb, 0x16, 0x31, 0x4c, 0x67, 0x82, 0x9d,
			0xb8, 0xd3, 0xee, 0x09, 0x24, 0x3f, 0x5a, 0x75,
			0x90, 0xab, 0xc6, 0xe1, 0xfc, 0x17, 0x32, 0x4d,
			0x68, 0x83, 0x9e, 0xb9, 0xd4, 0xef, 0x0a, 0x25,
			0x40, 0x5b, 0x76, 0x91, 0xac, 0xc7, 0xe2, 0xfd,
			0x18, 0x33, 0x4e, 0x69, 0x84, 0x9f, 0xba, 0xd5,
			0xf0, 0x0b, 0x26, 0x41, 0x5c, 0x77, 0x92, 0xad,
			0xc8, 0xe3, 0xfe, 0x19, 0x34, 0x4f, 0x6a, 0x85,
			0xa0, 0xbb, 0xd6, 0xf1, 0x0c, 0x27, 0x42, 0x5d,
			0x78, 0x93, 0xae, 0xc9, 0xe4, 0xff, 0x1a, 0x35,
			0x50, 0x6b, 0x86, 0xa1, 0xbc, 0xd7, 0xf2, 0x0d,
			0x28, 0x43, 0x5e, 0x79, 0x94, 0xaf, 0xca, 0xe5,
			0x00, 0x1d, 0x3a, 0x57, 0x74, 0x91, 0xae, 0xcb,
			0xe8, 0x05, 0x22, 0x3f, 0x5c, 0x79, 0x96, 0xb3,
			0xd0, 0xed, 0x0a, 0x27, 0x44, 0x61, 0x7e, 0x9b,
			0xb8, 0xd5, 0xf2, 0x0f, 0x2c, 0x49, 0x66, 0x83,
			0xa0, 0xbd, 0xda, 0xf7, 0x14, 0x31, 0x4e, 0x6b,
			0x88, 0xa5, 0xc2, 0xdf, 0xfc, 0x19, 0x36, 0x53,
			0x70, 0x8d, 0xaa, 0xc7, 0xe4, 0x01, 0x1e, 0x3b,
			0x58, 0x75, 0x92, 0xaf, 0xcc, 0xe9, 0x06, 0x23,
			0x40, 0x5d, 0x7a, 0x97, 0xb4, 0xd1, 0xee, 0x0b,
			0x28, 0x45, 0x62, 0x7f, 0x9c, 0xb9, 0xd6, 0xf3,
			0x10, 0x2d, 0x4a, 0x67, 0x84, 0xa1, 0xbe, 0xdb,
			0xf8, 0x15, 0x32, 0x4f, 0x6c, 0x89, 0xa6, 0xc3,
			0xe0, 0xfd, 0x1a, 0x37, 0x54, 0x71, 0x8e, 0xab,
			0xc8, 0xe5, 0x02, 0x1f, 0x3c, 0x59, 0x76, 0x93,
			0xb0, 0xcd, 0xea, 0x07, 0x24, 0x41, 0x5e, 0x7b,
			0x98, 0xb5, 0xd2, 0xef, 0x0c, 0x29, 0x46, 0x63,
			0x80, 0x9d, 0xba, 0xd7, 0xf4, 0x11, 0x2e, 0x4b,
			0x68, 0x85, 0xa2, 0xbf, 0xdc, 0xf9, 0x16, 0x33,
			0x50, 0x6d, 0x8a, 0xa7, 0xc4, 0xe1, 0xfe, 0x1b,
			0x38, 0x55, 0x72, 0x8f, 0xac, 0xc9, 0xe6, 0x03,
			0x20, 0x3d, 0x5a, 0x77, 0x94, 0xb1, 0xce, 0xeb,
			0x08, 0x25, 0x42, 0x5f, 0x7c, 0x99, 0xb6, 0xd3,
			0xf0, 0x0d, 0x2a, 0x47, 0x64, 0x81, 0x9e, 0xbb,
			0xd8, 0xf5, 0x12, 0x2f, 0x4c, 0x69, 0x86, 0xa3,
			0xc0, 0xdd, 0xfa, 0x17, 0x34, 0x51, 0x6e, 0x8b,
			0xa8, 0xc5, 0xe2, 0xff, 0x1c, 0x39, 0x56, 0x73,
			0x90, 0xad, 0xca, 0xe7, 0x04, 0x21, 0x3e, 0x5b,
			0x78, 0x95, 0xb2, 0xcf, 0xec, 0x09, 0x26, 0x43,
			0x60, 0x7d, 0x9a, 0xb7, 0xd4, 0xf1, 0x0e, 0x2b,
			0x48, 0x65, 0x82, 0x9f, 0xbc, 0xd9, 0xf6, 0x13,
			0x30, 0x4d, 0x6a, 0x87, 0xa4, 0xc1, 0xde, 0xfb,
			0x18, 0x35, 0x52, 0x6f, 0x8c, 0xa9, 0xc6, 0xe3,
			0x00, 0x1f, 0x3e, 0x5d, 0x7c, 0x9b, 0xba, 0xd9,
			0xf8, 0x17, 0x36, 0x55, 0x74, 0x93, 0xb2, 0xd1,
			0xf0, 0x0f, 0x2e, 0x4d, 0x6c, 0x8b, 0xaa, 0xc9,
			0xe8, 0x07, 0x26, 0x45, 0x64, 0x83, 0xa2, 0xc1,
			0xe0, 0xff, 0x1e, 0x3d, 0x5c, 0x7b, 0x9a, 0xb9,
			0xd8, 0xf7, 0x16, 0x35, 0x54, 0x73, 0x92, 0xb1,
			0xd0, 0xef, 0x0e, 0x2d, 0x4c, 0x6b, 0x8a, 0xa9,
			0xc8, 0xe7, 0x06, 0x25, 0x44, 0x63, 0x82, 0xa1,
			0xc0, 0xdf, 0xfe, 0x1d, 0x3c, 0x5b, 0x7a, 0x99,
			0xb8, 0xd7, 0xf6, 0x15, 0x34, 0x53, 0x72, 0x91,
			0xb0, 0xcf, 0xee, 0x0d, 0x2c, 0x4b, 0x6a, 0x89,
			0xa8, 0xc7, 0xe6, 0x05, 0x24, 0x43, 0x62, 0x81,
			0xa0, 0xbf, 0xde, 0xfd, 0x1c, 0x3b, 0x5a, 0x79,
			0x98, 0xb7, 0xd6, 0xf5, 0x14, 0x33, 0x52, 0x71,
			0x90, 0xaf, 0xce, 0xed, 0x0c, 0x2b, 0x4a, 0x69,
			0x88, 0xa7, 0xc6, 0xe5, 0x04, 0x23, 0x42, 0x61,
			0x80, 0x9f, 0xbe, 0xdd, 0xfc, 0x1b, 0x3a, 0x59,
			0x78, 0x97, 0xb6, 0xd5, 0xf4, 0x13, 0x32, 0x51,
			0x70, 0x8f, 0xae, 0xcd, 0xec, 0x0b, 0x2a, 0x49,
			0x68, 0x87, 0xa6, 0xc5, 0xe4, 0x03, 0x22, 0x41,
			0x60, 0x7f, 0x9e, 0xbd, 0xdc, 0xfb, 0x1a, 0x39,
			0x58, 0x77, 0x96, 0xb5, 0xd4, 0xf3, 0x12, 0x31,
			0x50, 0x6f, 0x8e, 0xad, 0xcc, 0xeb, 0x0a, 0x29,
			0x48, 0x67, 0x86, 0xa5, 0xc4, 0xe3, 0x02, 0x21,
			0x40, 0x5f, 0x7e, 0x9d, 0xbc, 0xdb, 0xfa, 0x19,
			0x38, 0x57, 0x76, 0x95, 0xb4, 0xd3, 0xf2, 0x11,
			0x30, 0x4f, 0x6e, 0x8d, 0xac, 0xcb, 0xea, 0x09,
			0x28, 0x47, 0x66, 0x85, 0xa4, 0xc3, 0xe2, 0x01,
			0x20, 0x3f, 0x5e, 0x7d, 0x9c, 0xbb, 0xda, 0xf9,
			0x18, 0x37, 0x56, 0x75, 0x94, 0xb3, 0xd2, 0xf1,
			0x10, 0x2f, 0x4e, 0x6d, 0x8c, 0xab, 0xca, 0xe9,
			0x08, 0x27, 0x46, 0x65, 0x84, 0xa3, 0xc2, 0xe1,
			0x00, 0x21, 0x42, 0x63,
		},
		.ilen = 4100,
		.result = {
			0xb5, 0x81, 0xf5, 0x64, 0x18, 0x73, 0xe3, 0xf0,
			0x4c, 0x13, 0xf2, 0x77, 0x18, 0x60, 0x65, 0x5e,
			0x29, 0x01, 0xce, 0x98, 0x55, 0x53, 0xf9, 0x0c,
			0x2a, 0x08, 0xd5, 0x09, 0xb3, 0x57, 0x55, 0x56,
			0xc5, 0xe9, 0x56, 0x90, 0xcb, 0x6a, 0xa3, 0xc0,
			0xff, 0xc4, 0x79, 0xb4, 0xd2, 0x97, 0x5d, 0xc4,
			0x43, 0xd1, 0xfe, 0x94, 0x7b, 0x88, 0x06, 0x5a,
			0xb2, 0x9e, 0x2c, 0xfc, 0x44, 0x03, 0xb7, 0x90,
			0xa0, 0xc1, 0xba, 0x6a, 0x33, 0xb8, 0xc7, 0xb2,
			0x9d, 0xe1, 0x12, 0x4f, 0xc0, 0x64, 0xd4, 0x01,
			0xfe, 0x8c, 0x7a, 0x66, 0xf7, 0xe6, 0x5a, 0x91,
			0xbb, 0xde, 0x56, 0x86, 0xab, 0x65, 0x21, 0x30,
			0x00, 0x84, 0x65, 0x24, 0xa5, 0x7d, 0x85, 0xb4,
			0xe3, 0x17, 0xed, 0x3a, 0xb7, 0x6f, 0xb4, 0x0b,
			0x0b, 0xaf, 0x15, 0xae, 0x5a, 0x8f, 0xf2, 0x0c,
			0x2f, 0x27, 0xf4, 0x09, 0xd8, 0xd2, 0x96, 0xb7,
			0x71, 0xf2, 0xc5, 0x99, 0x4d, 0x7e, 0x7f, 0x75,
			0x77, 0x89, 0x30, 0x8b, 0x59, 0xdb, 0xa2, 0xb2,
			0xa0, 0xf3, 0x19, 0x39, 0x2b, 0xc5, 0x7e, 0x3f,
			0x4f, 0xd9, 0xd3, 0x56, 0x28, 0x97, 0x44, 0xdc,
			0xc0, 0x8b, 0x77, 0x24, 0xd9, 0x52, 0xe7, 0xc5,
			0xaf, 0xf6, 0x7d, 0x59, 0xb2, 0x44, 0x05, 0x1d,
			0xb1, 0xb0, 0x11, 0xa5, 0x0f, 0xec, 0x33, 0xe1,
			0x6d, 0x1b, 0x4e, 0x1f, 0xff, 0x57, 0x91, 0xb4,
			0x5b, 0x9a, 0x96, 0xc5, 0x53, 0xbc, 0xae, 0x20,
			0x3c, 0xbb, 0x14, 0xe2, 0xe8, 0x22, 0x33, 0xc1,
			0x5e, 0x76, 0x9e, 0x46, 0x99, 0xf6, 0x2a, 0x15,
			0xc6, 0x97, 0x02, 0xa0, 0x66, 0x43, 0xd1, 0xa6,
			0x31, 0xa6, 0x9f, 0xfb, 0xf4, 0xd3, 0x69, 0xe5,
			0xcd, 0x76, 0x95, 0xb8, 0x7a, 0x82, 0x7f, 0x21,
			0x45, 0xff, 0x3f, 0xce, 0x55, 0xf6, 0x95, 0x10,
			0x08, 0x77, 0x10, 0x43, 0xc6, 0xf3, 0x09, 0xe5,
			0x68, 0xe7, 0x3c, 0xad, 0x00, 0x52, 0x45, 0x0d,
			0xfe, 0x2d, 0xc6, 0xc2, 0x94, 0x8c, 0x12, 0x1d,
			0xe6, 0x25, 0xae, 0x98, 0x12, 0x8e, 0x19, 0x9c,
			0x81, 0x68, 0xb1, 0x11, 0xf6, 0x69, 0xda, 0xe3,
			0x62, 0x08, 0x18, 0x7a, 0x25, 0x49, 0x28, 0xac,
			0xba, 0x71, 0x12, 0x0b, 0xe4, 0xa2, 0xe5, 0xc7,
			0x5d, 0x8e, 0xec, 0x49, 0x40, 0x21, 0xbf, 0x5a,
			0x98, 0xf3, 0x02, 0x68, 0x55, 0x03, 0x7f, 0x8a,
			0xe5, 0x94, 0x0c, 0x32, 0x5c, 0x07, 0x82, 0x63,
			0xaf, 0x6f, 0x91, 0x40, 0x84, 0x8e, 0x52, 0x25,
			0xd0, 0xb0, 0x29, 0x53, 0x05, 0xe2, 0x50, 0x7a,
			0x34, 0xeb, 0xc9, 0x46, 0x20, 0xa8, 0x3d, 0xde,
			0x7f, 0x16, 0x5f, 0x36, 0xc5, 0x2e, 0xdc, 0xd1,
			0x15, 0x47, 0xc7, 0x50, 0x40, 0x6d, 0x91, 0xc5,
			0xe7, 0x93, 0x95, 0x1a, 0xd3, 0x57, 0xbc, 0x52,
			0x33, 0xee, 0x14, 0x19, 0x22, 0x52, 0x89, 0xa7,
			0x4a, 0x25, 0x56, 0x77, 0x4b, 0xca, 0xcf, 0x0a,
			0xe1, 0xf5, 0x35, 0x85, 0x30, 0x7e, 0x59, 0x4a,
			0xbd, 0x14, 0x5b, 0xdf, 0xe3, 0x46, 0xcb, 0xac,
			0x1f, 0x6c, 0x96, 0x0e, 0xf4, 0x81, 0xd1, 0x99,
			0xca, 0x88, 0x63, 0x3d, 0x02, 0x58, 0x6b, 0xa9,
			0xe5, 0x9f, 0xb3, 0x00, 0xb2, 0x54, 0xc6, 0x74,
			0x1c, 0xbf, 0x46, 0xab, 0x97, 0xcc, 0xf8, 0x54,
			0x04, 0x07, 0x08, 0x52, 0xe6, 0xc0, 0xda, 0x93,
			0x74, 0x7d, 0x93, 0x99, 0x5d, 0x78, 0x68, 0xa6,
			0x2e, 0x6b, 0xd3, 0x6a, 0x69, 0xcc, 0x12, 0x6b,
			0xd4, 0xc7, 0xa5, 0xc6, 0xe7, 0xf6, 0x03, 0x04,
			0x5d, 0xcd, 0x61, 0x5e, 0x17, 0x40, 0xdc, 0xd1,
			0x5c, 0xf5, 0x08, 0xdf, 0x5c, 0x90, 0x85, 0xa4,
			0xaf, 0xf6, 0x78, 0xbb, 0x0d, 0xf1, 0xf4, 0xa4,
			0x54, 0x26, 0x72, 0x9e, 0x61, 0xfa, 0x86, 0xcf,
			0xe8, 0x9e, 0xa1, 0xe0, 0xc7, 0x48, 0x23, 0xae,
			0x5a, 0x90, 0xae, 0x75, 0x0a, 0x74, 0x18, 0x89,
			0x05, 0xb1, 0x92, 0xb2, 0x7f, 0xd0, 0x1b, 0xa6,
			0x62, 0x07, 0x25, 0x01, 0xc7, 0xc2, 0x4f, 0xf9,
			0xe8, 0xfe, 0x63, 0x95, 0x80, 0x07, 0xb4, 0x26,
			0xcc, 0xd1, 0x26, 0xb6, 0xc4, 0x3f, 0x9e, 0xcb,
			0x8e, 0x3b, 0x2e, 0x44, 0x16, 0xd3, 0x10, 0x9a,
			0x95, 0x08, 0xeb, 0xc8, 0xcb, 0xeb, 0xbf, 0x6f,
			0x0b, 0xcd, 0x1f, 0xc8, 0xca, 0x86, 0xaa, 0xec,
			0x33, 0xe6, 0x69, 0xf4, 0x45, 0x25, 0x86, 0x3a,
			0x22, 0x94, 0x4f, 0x00, 0x23, 0x6a, 0x44, 0xc2,
			0x49, 0x97, 0x33, 0xab, 0x36, 0x14, 0x0a, 0x70,
			0x24, 0xc3, 0xbe, 0x04, 0x3b, 0x79, 0xa0, 0xf9,
			0xb8, 0xe7, 0x76, 0x29, 0x22, 0x83, 0xd7, 0xf2,
			0x94, 0xf4, 0x41, 0x49, 0xba, 0x5f, 0x7b, 0x07,
			0xb5, 0xfb, 0xdb, 0x03, 0x1a, 0x9f, 0xb6, 0x4c,
			0xc2, 0x2e, 0x37, 0x40, 0x49, 0xc3, 0x38, 0x16,
			0xe2, 0x4f, 0x77, 0x82, 0xb0, 0x68, 0x4c, 0x71,
			0x1d, 0x57, 0x61, 0x9c, 0xd9, 0x4e, 0x54, 0x99,
			0x47, 0x13, 0x28, 0x73, 0x3c, 0xbb, 0x00, 0x90,
			0xf3, 0x4d, 0xc9, 0x0e, 0xfd, 0xe7, 0xb1, 0x71,
			0xd3, 0x15, 0x79, 0xbf, 0xcc, 0x26, 0x2f, 0xbd,
			0xad, 0x6c, 0x50, 0x69, 0x6c, 0x3e, 0x6d, 0x80,
			0x9a, 0xea, 0x78, 0xaf, 0x19, 0xb2, 0x0d, 0x4d,
			0xad, 0x04, 0x07, 0xae, 0x22, 0x90, 0x4a, 0x93,
			0x32, 0x0e, 0x36, 0x9b, 0x1b, 0x46, 0xba, 0x3b,
			0xb4, 0xac, 0xc6, 0xd1, 0xa2, 0x31, 0x53, 0x3b,
			0x2a, 0x3d, 0x45, 0xfe, 0x03, 0x61, 0x10, 0x85,
			0x17, 0x69, 0xa6, 0x78, 0xcc, 0x6c, 0x87, 0x49,
			0x53, 0xf9, 0x80, 0x10, 0xde, 0x80, 0xa2, 0x41,
			0x6a, 0xc3, 0x32, 0x02, 0xad, 0x6d, 0x3c, 0x56,
			0x00, 0x71, 0x51, 0x06, 0xa7, 0xbd, 0xfb, 0xef,
			0x3c, 0xb5, 0x9f, 0xfc, 0x48, 0x7d, 0x53, 0x7c,
			0x66, 0xb0, 0x49, 0x23, 0xc4, 0x47, 0x10, 0x0e,
			0xe5, 0x6c, 0x74, 0x13, 0xe6, 0xc5, 0x3f, 0xaa,
			0xde, 0xff, 0x07, 0x44, 0xdd, 0x56, 0x1b, 0xad,
			0x09, 0x77, 0xfb, 0x5b, 0x12, 0xb8, 0x0d, 0x38,
			0x17, 0x37, 0x35, 0x7b, 0x9b, 0xbc, 0xfe, 0xd4,
			0x7e, 0x8b, 0xda, 0x7e, 0x5b, 0x04, 0xa7, 0x22,
			0xa7, 0x31, 0xa1, 0x20, 0x86, 0xc7, 0x1b, 0x99,
			0xdb, 0xd1, 0x89, 0xf4, 0x94, 0xa3, 0x53, 0x69,
			0x8d, 0xe7, 0xe8, 0x74, 0x11, 0x8d, 0x74, 0xd6,
			0x07, 0x37, 0x91, 0x9f, 0xfd, 0x67, 0x50, 0x3a,
			0xc9, 0xe1, 0xf4, 0x36, 0xd5, 0xa0, 0x47, 0xd1,
			0xf9, 0xe5, 0x39, 0xa3, 0x31, 0xac, 0x07, 0x36,
			0x23, 0xf8, 0x66, 0x18, 0x14, 0x28, 0x34, 0x0f,
			0xb8, 0xd0, 0xe7, 0x29, 0xb3, 0x04, 0x4b, 0x55,
			0x01, 0x41, 0xb2, 0x75, 0x8d, 0xcb, 0x96, 0x85,
			0x3a, 0xfb, 0xab, 0x2b, 0x9e, 0xfa, 0x58, 0x20,
			0x44, 0x1f, 0xc0, 0x14, 0x22, 0x75, 0x61, 0xe8,
			0xaa, 0x19, 0xcf, 0xf1, 0x82, 0x56, 0xf4, 0xd7,
			0x78, 0x7b, 0x3d, 0x5f, 0xb3, 0x9e, 0x0b, 0x8a,
			0x57, 0x50, 0xdb, 0x17, 0x41, 0x65, 0x4d, 0xa3,
			0x02, 0xc9, 0x9c, 0x9c, 0x53, 0xfb, 0x39, 0x39,
			0x9b, 0x1d, 0x72, 0x24, 0xda, 0xb7, 0x39, 0xbe,
			0x13, 0x3b, 0xfa, 0x29, 0xda, 0x9e, 0x54, 0x64,
			0x6e, 0xba, 0xd8, 0xa1, 0xcb, 0xb3, 0x36, 0xfa,
			0xcb, 0x47, 0x85, 0xe9, 0x61, 0x38, 0xbc, 0xbe,
			0xc5, 0x00, 0x38, 0x2a, 0x54, 0xf7, 0xc4, 0xb9,
			0xb3, 0xd3, 0x7b, 0xa0, 0xa0, 0xf8, 0x72, 0x7f,
			0x8c, 0x8e, 0x82, 0x0e, 0xc6, 0x1c, 0x75, 0x9d,
			0xca, 0x8e, 0x61, 0x87, 0xde, 0xad, 0x80, 0xd2,
			0xf5, 0xf9, 0x80, 0xef, 0x15, 0x75, 0xaf, 0xf5,
			0x80, 0xfb, 0xff, 0x6d, 0x1e, 0x25, 0xb7, 0x40,
			0x61, 0x6a, 0x39, 0x5a, 0x6a, 0xb5, 0x31, 0xab,
			0x97, 0x8a, 0x19, 0x89, 0x44, 0x40, 0xc0, 0xa6,
			0xb4, 0x4e, 0x30, 0x32, 0x7b, 0x13, 0xe7, 0x67,
			0xa9, 0x8b, 0x57, 0x04, 0xc2, 0x01, 0xa6, 0xf4,
			0x28, 0x99, 0xad, 0x2c, 0x76, 0xa3, 0x78, 0xc2,
			0x4a, 0xe6, 0xca, 0x5c, 0x50, 0x6a, 0xc1, 0xb0,
			0x62, 0x4b, 0x10, 0x8e, 0x7c, 0x17, 0x43, 0xb3,
			0x17, 0x66, 0x1c, 0x3e, 0x8d, 0x69, 0xf0, 0x5a,
			0x71, 0xf5, 0x97, 0xdc, 0xd1, 0x45, 0xdd, 0x28,
			0xf3, 0x5d, 0xdf, 0x53, 0x7b, 0x11, 0xe5, 0xbc,
			0x4c, 0xdb, 0x1b, 0x51, 0x6b, 0xe9, 0xfb, 0x3d,
			0xc1, 0xc3, 0x2c, 0xb9, 0x71, 0xf5, 0xb6, 0xb2,
			0x13, 0x36, 0x79, 0x80, 0x53, 0xe8, 0xd3, 0xa6,
			0x0a, 0xaf, 0xfd, 0x56, 0x97, 0xf7, 0x40, 0x8e,
			0x45, 0xce, 0xf8, 0xb0, 0x9e, 0x5c, 0x33, 0x82,
			0xb0, 0x44, 0x56, 0xfc, 0x05, 0x09, 0xe9, 0x2a,
			0xac, 0x26, 0x80, 0x14, 0x1d, 0xc8, 0x3a, 0x35,
			0x4c, 0x82, 0x97, 0xfd, 0x76, 0xb7, 0xa9, 0x0a,
			0x35, 0x58, 0x79, 0x8e, 0x0f, 0x66, 0xea, 0xaf,
			0x51, 0x6c, 0x09, 0xa9, 0x6e, 0x9b, 0xcb, 0x9a,
			0x31, 0x47, 0xa0, 0x2f, 0x7c, 0x71, 0xb4, 0x4a,
			0x11, 0xaa, 0x8c, 0x66, 0xc5, 0x64, 0xe6, 0x3a,
			0x54, 0xda, 0x24, 0x6a, 0xc4, 0x41, 0x65, 0x46,
			0x82, 0xa0, 0x0a, 0x0f, 0x5f, 0xfb, 0x25, 0xd0,
			0x2c, 0x91, 0xa7, 0xee, 0xc4, 0x81, 0x07, 0x86,
			0x75, 0x5e, 0x33, 0x69, 0x97, 0xe4, 0x2c, 0xa8,
			0x9d, 0x9f, 0x0b, 0x6a, 0xbe, 0xad, 0x98, 0xda,
			0x6d, 0x94, 0x41, 0xda, 0x2c, 0x1e, 0x89, 0xc4,
			0xc2, 0xaf, 0x1e, 0x00, 0x05, 0x0b, 0x83, 0x60,
			0xbd, 0x43, 0xea, 0x15, 0x23, 0x7f, 0xb9, 0xac,
			0xee, 0x4f, 0x2c, 0xaf, 0x2a, 0xf3, 0xdf, 0xd0,
			0xf3, 0x19, 0x31, 0xbb, 0x4a, 0x74, 0x84, 0x17,
			0x52, 0x32, 0x2c, 0x7d, 0x61, 0xe4, 0xcb, 0xeb,
			0x80, 0x38, 0x15, 0x52, 0xcb, 0x6f, 0xea, 0xe5,
			0x73, 0x9c, 0xd9, 0x24, 0x69, 0xc6, 0x95, 0x32,
			0x21, 0xc8, 0x11, 0xe4, 0xdc, 0x36, 0xd7, 0x93,
			0x38, 0x66, 0xfb, 0xb2, 0x7f, 0x3a, 0xb9, 0xaf,
			0x31, 0xdd, 0x93, 0x75, 0x78, 0x8a, 0x2c, 0x94,
			0x87, 0x1a, 0x58, 0xec, 0x9e, 0x7d, 0x4d, 0xba,
			0xe1, 0xe5, 0x4d, 0xfc, 0xbc, 0xa4, 0x2a, 0x14,
			0xef, 0xcc, 0xa7, 0xec, 0xab, 0x43, 0x09, 0x18,
			0xd3, 0xab, 0x68, 0xd1, 0x07, 0x99, 0x44, 0x47,
			0xd6, 0x83, 0x85, 0x3b, 0x30, 0xea, 0xa9, 0x6b,
			0x63, 0xea, 0xc4, 0x07, 0xfb, 0x43, 0x2f, 0xa4,
			0xaa, 0xb0, 0xab, 0x03, 0x89, 0xce, 0x3f, 0x8c,
			0x02, 0x7c, 0x86, 0x54, 0xbc, 0x88, 0xaf, 0x75,
			0xd2, 0xdc, 0x63, 0x17, 0xd3, 0x26, 0xf6, 0x96,
			0xa9, 0x3c, 0xf1, 0x61, 0x8c, 0x11, 0x18, 0xcc,
			0xd6, 0xea, 0x5b, 0xe2, 0xcd, 0xf0, 0xf1, 0xb2,
			0xe5, 0x35, 0x90, 0x1f, 0x85, 0x4c, 0x76, 0x5b,
			0x66, 0xce, 0x44, 0xa4, 0x32, 0x9f, 0xe6, 0x7b,
			0x71, 0x6e, 0x9f, 0x58, 0x15, 0x67, 0x72, 0x87,
			0x64, 0x8e, 0x3a, 0x44, 0x45, 0xd4, 0x76, 0xfa,
			0xc2, 0xf6, 0xef, 0x85, 0x05, 0x18, 0x7a, 0x9b,
			0xba, 0x41, 0x54, 0xac, 0xf0, 0xfc, 0x59, 0x12,
			0x3f, 0xdf, 0xa0, 0xe5, 0x8a, 0x65, 0xfd, 0x3a,
			0x62, 0x8d, 0x83, 0x2c, 0x03, 0xbe, 0x05, 0x76,
			0x2e, 0x53, 0x49, 0x97, 0x94, 0x33, 0xae, 0x40,
			0x81, 0x15, 0xdb, 0x6e, 0xad, 0xaa, 0xf5, 0x4b,
			0xe3, 0x98, 0x70, 0xdf, 0xe0, 0x7c, 0xcd, 0xdb,
			0x02, 0xd4, 0x7d, 0x2f, 0xc1, 0xe6, 0xb4, 0xf3,
			0xd7, 0x0d, 0x7a, 0xd9, 0x23, 0x9e, 0x87, 0x2d,
			0xce, 0x87, 0xad, 0xcc, 0x72, 0x05, 0x00, 0x29,
			0xdc, 0x73, 0x7f, 0x64, 0xc1, 0x15, 0x0e, 0xc2,
			0xdf, 0xa7, 0x5f, 0xeb, 0x41, 0xa1, 0xcd, 0xef,
			0x5c, 0x50, 0x79, 0x2a, 0x56, 0x56, 0x71, 0x8c,
			0xac, 0xc0, 0x79, 0x50, 0x69, 0xca, 0x59, 0x32,
			0x65, 0xf2, 0x54, 0xe4, 0x52, 0x38, 0x76, 0xd1,
			0x5e, 0xde, 0x26, 0x9e, 0xfb, 0x75, 0x2e, 0x11,
			0xb5, 0x10, 0xf4, 0x17, 0x73, 0xf5, 0x89, 0xc7,
			0x4f, 0x43, 0x5c, 0x8e, 0x7c, 0xb9, 0x05, 0x52,
			0x24, 0x40, 0x99, 0xfe, 0x9b, 0x85, 0x0b, 0x6c,
			0x22, 0x3e, 0x8b, 0xae, 0x86, 0xa1, 0xd2, 0x79,
			0x05, 0x68, 0x6b, 0xab, 0xe3, 0x41, 0x49, 0xed,
			0x15, 0xa1, 0x8d, 0x40, 0x2d, 0x61, 0xdf, 0x1a,
			0x59, 0xc9, 0x26, 0x8b, 0xef, 0x30, 0x4c, 0x88,
			0x4b, 0x10, 0xf8, 0x8d, 0xa6, 0x92, 0x9f, 0x4b,
			0xf3, 0xc4, 0x53, 0x0b, 0x89, 0x5d, 0x28, 0x92,
			0xcf, 0x78, 0xb2, 0xc0, 0x5d, 0xed, 0x7e, 0xfc,
			0xc0, 0x12, 0x23, 0x5f, 0x5a, 0x78, 0x86, 0x43,
			0x6e, 0x27, 0xf7, 0x5a, 0xa7, 0x6a, 0xed, 0x19,
			0x04, 0xf0, 0xb3, 0x12, 0xd1, 0xbd, 0x0e, 0x89,
			0x6e, 0xbc, 0x96, 0xa8, 0xd8, 0x49, 0x39, 0x9f,
			0x7e, 0x67, 0xf0, 0x2e, 0x3e, 0x01, 0xa9, 0xba,
			0xec, 0x8b, 0x62, 0x8e, 0xcb, 0x4a, 0x70, 0x43,
			0xc7, 0xc2, 0xc4, 0xca, 0x82, 0x03, 0x73, 0xe9,
			0x11, 0xdf, 0xcf, 0x54, 0xea, 0xc9, 0xb0, 0x95,
			0x51, 0xc0, 0x13, 0x3d, 0x92, 0x05, 0xfa, 0xf4,
			0xa9, 0x34, 0xc8, 0xce, 0x6c, 0x3d, 0x54, 0xcc,
			0xc4, 0xaf, 0xf1, 0xdc, 0x11, 0x44, 0x26, 0xa2,
			0xaf, 0xf1, 0x85, 0x75, 0x7d, 0x03, 0x61, 0x68,
			0x4e, 0x78, 0xc6, 0x92, 0x7d, 0x86, 0x7d, 0x77,
			0xdc, 0x71, 0x72, 0xdb, 0xc6, 0xae, 0xa1, 0xcb,
			0x70, 0x9a, 0x0b, 0x19, 0xbe, 0x4a, 0x6c, 0x2a,
			0xe2, 0xba, 0x6c, 0x64, 0x9a, 0x13, 0x28, 0xdf,
			0x85, 0x75, 0xe6, 0x43, 0xf6, 0x87, 0x08, 0x68,
			0x6e, 0xba, 0x6e, 0x79, 0x9f, 0x04, 0xbc, 0x23,
			0x50, 0xf6, 0x33, 0x5c, 0x1f, 0x24, 0x25, 0xbe,
			0x33, 0x47, 0x80, 0x45, 0x56, 0xa3, 0xa7, 0xd7,
			0x7a, 0xb1, 0x34, 0x0b, 0x90, 0x3c, 0x9c, 0xad,
			0x44, 0x5f, 0x9e, 0x0e, 0x9d, 0xd4, 0xbd, 0x93,
			0x5e, 0xfa, 0x3c, 0xe0, 0xb0, 0xd9, 0xed, 0xf3,
			0xd6, 0x2e, 0xff, 0x24, 0xd8, 0x71, 0x6c, 0xed,
			0xaf, 0x55, 0xeb, 0x22, 0xac, 0x93, 0x68, 0x32,
			0x05, 0x5b, 0x47, 0xdd, 0xc6, 0x4a, 0xcb, 0xc7,
			0x10, 0xe1, 0x3c, 0x92, 0x1a, 0xf3, 0x23, 0x78,
			0x2b, 0xa1, 0xd2, 0x80, 0xf4, 0x12, 0xb1, 0x20,
			0x8f, 0xff, 0x26, 0x35, 0xdd, 0xfb, 0xc7, 0x4e,
			0x78, 0xf1, 0x2d, 0x50, 0x12, 0x77, 0xa8, 0x60,
			0x7c, 0x0f, 0xf5, 0x16, 0x2f, 0x63, 0x70, 0x2a,
			0xc0, 0x96, 0x80, 0x4e, 0x0a, 0xb4, 0x93, 0x35,
			0x5d, 0x1d, 0x3f, 0x56, 0xf7, 0x2f, 0xbb, 0x90,
			0x11, 0x16, 0x8f, 0xa2, 0xec, 0x47, 0xbe, 0xac,
			0x56, 0x01, 0x26, 0x56, 0xb1, 0x8c, 0xb2, 0x10,
			0xf9, 0x1a, 0xca, 0xf5, 0xd1, 0xb7, 0x39, 0x20,
			0x63, 0xf1, 0x69, 0x20, 0x4f, 0x13, 0x12, 0x1f,
			0x5b, 0x65, 0xfc, 0x98, 0xf7, 0xc4, 0x7a, 0xbe,
			0xf7, 0x26, 0x4d, 0x2b, 0x84, 0x7b, 0x42, 0xad,
			0xd8, 0x7a, 0x0a, 0xb4, 0xd8, 0x74, 0xbf, 0xc1,
			0xf0, 0x6e, 0xb4, 0x29, 0xa3, 0xbb, 0xca, 0x46,
			0x67, 0x70, 0x6a, 0x2d, 0xce, 0x0e, 0xa2, 0x8a,
			0xa9, 0x87, 0xbf, 0x05, 0xc4, 0xc1, 0x04, 0xa3,
			0xab, 0xd4, 0x45, 0x43, 0x8c, 0xb6, 0x02, 0xb0,
			0x41, 0xc8, 0xfc, 0x44, 0x3d, 0x59, 0xaa, 0x2e,
			0x44, 0x21, 0x2a, 0x8d, 0x88, 0x9d, 0x57, 0xf4,
			0xa0, 0x02, 0x77, 0xb8, 0xa6, 0xa0, 0xe6, 0x75,
			0x5c, 0x82, 0x65, 0x3e, 0x03, 0x5c, 0x29, 0x8f,
			0x38, 0x55, 0xab, 0x33, 0x26, 0xef, 0x9f, 0x43,
			0x52, 0xfd, 0x68, 0xaf, 0x36, 0xb4, 0xbb, 0x9a,
			0x58, 0x09, 0x09, 0x1b, 0xc3, 0x65, 0x46, 0x46,
			0x1d, 0xa7, 0x94, 0x18, 0x23, 0x50, 0x2c, 0xca,
			0x2c, 0x55, 0x19, 0x97, 0x01, 0x9d, 0x93, 0x3b,
			0x63, 0x86, 0xf2, 0x03, 0x67, 0x45, 0xd2, 0x72,
			0x28, 0x52, 0x6c, 0xf4, 0xe3, 0x1c, 0xb5, 0x11,
			0x13, 0xf1, 0xeb, 0x21, 0xc7, 0xd9, 0x56, 0x82,
			0x2b, 0x82, 0x39, 0xbd, 0x69, 0x54, 0xed, 0x62,
			0xc3, 0xe2, 0xde, 0x73, 0xd4, 0x6a, 0x12, 0xae,
			0x13, 0x21, 0x7f, 0x4b, 0x5b, 0xfc, 0xbf, 0xe8,
			0x2b, 0xbe, 0x56, 0xba, 0x68, 0x8b, 0x9a, 0xb1,
			0x6e, 0xfa, 0xbf, 0x7e, 0x5a, 0x4b, 0xf1, 0xac,
			0x98, 0x65, 0x85, 0xd1, 0x93, 0x53, 0xd3, 0x7b,
			0x09, 0xdd, 0x4b, 0x10, 0x6d, 0x84, 0xb0, 0x13,
			0x65, 0xbd, 0xcf, 0x52, 0x09, 0xc4, 0x85, 0xe2,
			0x84, 0x74, 0x15, 0x65, 0xb7, 0xf7, 0x51, 0xaf,
			0x55, 0xad, 0xa4, 0xd1, 0x22, 0x54, 0x70, 0x94,
			0xa0, 0x1c, 0x90, 0x41, 0xfd, 0x99, 0xd7, 0x5a,
			0x31, 0xef, 0xaa, 0x25, 0xd0, 0x7f, 0x4f, 0xea,
			0x1d, 0x55, 0x42, 0xe5, 0x49, 0xb0, 0xd0, 0x46,
			0x62, 0x36, 0x43, 0xb2, 0x82, 0x15, 0x75, 0x50,
			0xa4, 0x72, 0xeb, 0x54, 0x27, 0x1f, 0x8a, 0xe4,
			0x7d, 0xe9, 0x66, 0xc5, 0xf1, 0x53, 0xa4, 0xd1,
			0x0c, 0xeb, 0xb8, 0xf8, 0xbc, 0xd4, 0xe2, 0xe7,
			0xe1, 0xf8, 0x4b, 0xcb, 0xa9, 0xa1, 0xaf, 0x15,
			0x83, 0xcb, 0x72, 0xd0, 0x33, 0x79, 0x00, 0x2d,
			0x9f, 0xd7, 0xf1, 0x2e, 0x1e, 0x10, 0xe4, 0x45,
			0xc0, 0x75, 0x3a, 0x39, 0xea, 0x68, 0xf7, 0x5d,
			0x1b, 0x73, 0x8f, 0xe9, 0x8e, 0x0f, 0x72, 0x47,
			0xae, 0x35, 0x0a, 0x31, 0x7a, 0x14, 0x4d, 0x4a,
			0x6f, 0x47, 0xf7, 0x7e, 0x91, 0x6e, 0x74, 0x8b,
			0x26, 0x47, 0xf9, 0xc3, 0xf9, 0xde, 0x70, 0xf5,
			0x61, 0xab, 0xa9, 0x27, 0x9f, 0x82, 0xe4, 0x9c,
			0x89, 0x91, 0x3f, 0x2e, 0x6a, 0xfd, 0xb5, 0x49,
			0xe9, 0xfd, 0x59, 0x14, 0x36, 0x49, 0x40, 0x6d,
			0x32, 0xd8, 0x85, 0x42, 0xf3, 0xa5, 0xdf, 0x0c,
			0xa8, 0x27, 0xd7, 0x54, 0xe2, 0x63, 0x2f, 0xf2,
			0x7e, 0x8b, 0x8b, 0xe7, 0xf1, 0x9a, 0x95, 0x35,
			0x43, 0xdc, 0x3a, 0xe4, 0xb6, 0xf4, 0xd0, 0xdf,
			0x9c, 0xcb, 0x94, 0xf3, 0x21, 0xa0, 0x77, 0x50,
			0xe2, 0xc6, 0xc4, 0xc6, 0x5f, 0x09, 0x64, 0x5b,
			0x92, 0x90, 0xd8, 0xe1, 0xd1, 0xed, 0x4b, 0x42,
			0xd7, 0x37, 0xaf, 0x65, 0x3d, 0x11, 0x39, 0xb6,
			0x24, 0x8a, 0x60, 0xae, 0xd6, 0x1e, 0xbf, 0x0e,
			0x0d, 0xd7, 0xdc, 0x96, 0x0e, 0x65, 0x75, 0x4e,
			0x29, 0x06, 0x9d, 0xa4, 0x51, 0x3a, 0x10, 0x63,
			0x8f, 0x17, 0x07, 0xd5, 0x8e, 0x3c, 0xf4, 0x28,
			0x00, 0x5a, 0x5b, 0x05, 0x19, 0xd8, 0xc0, 0x6c,
			0xe5, 0x15, 0xe4, 0x9c, 0x9d, 0x71, 0x9d, 0x5e,
			0x94, 0x29, 0x1a, 0xa7, 0x80, 0xfa, 0x0e, 0x33,
			0x03, 0xdd, 0xb7, 0x3e, 0x9a, 0xa9, 0x26, 0x18,
			0x37, 0xa9, 0x64, 0x08, 0x4d, 0x94, 0x5a, 0x88,
			0xca, 0x35, 0xce, 0x81, 0x02, 0xe3, 0x1f, 0x1b,
			0x89, 0x1a, 0x77, 0x85, 0xe3, 0x41, 0x6d, 0x32,
			0x42, 0x19, 0x23, 0x7d, 0xc8, 0x73, 0xee, 0x25,
			0x85, 0x0d, 0xf8, 0x31, 0x25, 0x79, 0x1b, 0x6f,
			0x79, 0x25, 0xd2, 0xd8, 0xd4, 0x23, 0xfd, 0xf7,
			0x82, 0x36, 0x6a, 0x0c, 0x46, 0x22, 0x15, 0xe9,
			0xff, 0x72, 0x41, 0x91, 0x91, 0x7d, 0x3a, 0xb7,
			0xdd, 0x65, 0x99, 0x70, 0xf6, 0x8d, 0x84, 0xf8,
			0x67, 0x15, 0x20, 0x11, 0xd6, 0xb2, 0x55, 0x7b,
			0xdb, 0x87, 0xee, 0xef, 0x55, 0x89, 0x2a, 0x59,
			0x2b, 0x07, 0x8f, 0x43, 0x8a, 0x59, 0x3c, 0x01,
			0x8b, 0x65, 0x54, 0xa1, 0x66, 0xd5, 0x38, 0xbd,
			0xc6, 0x30, 0xa9, 0xcc, 0x49, 0xb6, 0xa8, 0x1b,
			0xb8, 0xc0, 0x0e, 0xe3, 0x45, 0x28, 0xe2, 0xff,
			0x41, 0x9f, 0x7e, 0x7c, 0xd1, 0xae, 0x9e, 0x25,
			0x3f, 0x4c, 0x7c, 0x7c, 0xf4, 0xa8, 0x26, 0x4d,
			0x5c, 0xfd, 0x4b, 0x27, 0x18, 0xf9, 0x61, 0x76,
			0x48, 0xba, 0x0c, 0x6b, 0xa9, 0x4d, 0xfc, 0xf5,
			0x3b, 0x35, 0x7e, 0x2f, 0x4a, 0xa9, 0xc2, 0x9a,
			0xae, 0xab, 0x86, 0x09, 0x89, 0xc9, 0xc2, 0x40,
			0x39, 0x2c, 0x81, 0xb3, 0xb8, 0x17, 0x67, 0xc2,
			0x0d, 0x32, 0x4a, 0x3a, 0x67, 0x81, 0xd7, 0x1a,
			0x34, 0x52, 0xc5, 0xdb, 0x0a, 0xf5, 0x63, 0x39,
			0xea, 0x1f, 0xe1, 0x7c, 0xa1, 0x9e, 0xc1, 0x35,
			0xe3, 0xb1, 0x18, 0x45, 0x67, 0xf9, 0x22, 0x38,
			0x95, 0xd9, 0x34, 0x34, 0x86, 0xc6, 0x41, 0x94,
			0x15, 0xf9, 0x5b, 0x41, 0xa6, 0x87, 0x8b, 0xf8,
			0xd5, 0xe1, 0x1b, 0xe2, 0x5b, 0xf3, 0x86, 0x10,
			0xff, 0xe6, 0xae, 0x69, 0x76, 0xbc, 0x0d, 0xb4,
			0x09, 0x90, 0x0c, 0xa2, 0x65, 0x0c, 0xad, 0x74,
			0xf5, 0xd7, 0xff, 0xda, 0xc1, 0xce, 0x85, 0xbe,
			0x00, 0xa7, 0xff, 0x4d, 0x2f, 0x65, 0xd3, 0x8c,
			0x86, 0x2d, 0x05, 0xe8, 0xed, 0x3e, 0x6b, 0x8b,
			0x0f, 0x3d, 0x83, 0x8c, 0xf1, 0x1d, 0x5b, 0x96,
			0x2e, 0xb1, 0x9c, 0xc2, 0x98, 0xe1, 0x70, 0xb9,
			0xba, 0x5c, 0x8a, 0x43, 0xd6, 0x34, 0xa7, 0x2d,
			0xc9, 0x92, 0xae, 0xf2, 0xa5, 0x7b, 0x05, 0x49,
			0xa7, 0x33, 0x34, 0x86, 0xca, 0xe4, 0x96, 0x23,
			0x76, 0x5b, 0xf2, 0xc6, 0xf1, 0x51, 0x28, 0x42,
			0x7b, 0xcc, 0x76, 0x8f, 0xfa, 0xa2, 0xad, 0x31,
			0xd4, 0xd6, 0x7a, 0x6d, 0x25, 0x25, 0x54, 0xe4,
			0x3f, 0x50, 0x59, 0xe1, 0x5c, 0x05, 0xb7, 0x27,
			0x48, 0xbf, 0x07, 0xec, 0x1b, 0x13, 0xbe, 0x2b,
			0xa1, 0x57, 0x2b, 0xd5, 0xab, 0xd7, 0xd0, 0x4c,
			0x1e, 0xcb, 0x71, 0x9b, 0xc5, 0x90, 0x85, 0xd3,
			0xde, 0x59, 0xec, 0x71, 0xeb, 0x89, 0xbb, 0xd0,
			0x09, 0x50, 0xe1, 0x16, 0x3f, 0xfd, 0x1c, 0x34,
			0xc3, 0x1c, 0xa1, 0x10, 0x77, 0x53, 0x98, 0xef,
			0xf2, 0xfd, 0xa5, 0x01, 0x59, 0xc2, 0x9b, 0x26,
			0xc7, 0x42, 0xd9, 0x49, 0xda, 0x58, 0x2b, 0x6e,
			0x9f, 0x53, 0x19, 0x76, 0x7e, 0xd9, 0xc9, 0x0e,
			0x68, 0xc8, 0x7f, 0x51, 0x22, 0x42, 0xef, 0x49,
			0xa4, 0x55, 0xb6, 0x36, 0xac, 0x09, 0xc7, 0x31,
			0x88, 0x15, 0x4b, 0x2e, 0x8f, 0x3a, 0x08, 0xf7,
			0xd8, 0xf7, 0xa8, 0xc5, 0xa9, 0x33, 0xa6, 0x45,
			0xe4, 0xc4, 0x94, 0x76, 0xf3, 0x0d, 0x8f, 0x7e,
			0xc8, 0xf6, 0xbc, 0x23, 0x0a, 0xb6, 0x4c, 0xd3,
			0x6a, 0xcd, 0x36, 0xc2, 0x90, 0x5c, 0x5c, 0x3c,
			0x65, 0x7b, 0xc2, 0xd6, 0xcc, 0xe6, 0x0d, 0x87,
			0x73, 0x2e, 0x71, 0x79, 0x16, 0x06, 0x63, 0x28,
			0x09, 0x15, 0xd8, 0x89, 0x38, 0x38, 0x3d, 0xb5,
			0x42, 0x1c, 0x08, 0x24, 0xf7, 0x2a, 0xd2, 0x9d,
			0xc8, 0xca, 0xef, 0xf9, 0x27, 0xd8, 0x07, 0x86,
			0xf7, 0x43, 0x0b, 0x55, 0x15, 0x3f, 0x9f, 0x83,
			0xef, 0xdc, 0x49, 0x9d, 0x2a, 0xc1, 0x54, 0x62,
			0xbd, 0x9b, 0x66, 0x55, 0x9f, 0xb7, 0x12, 0xf3,
			0x1b, 0x4d, 0x9d, 0x2a, 0x5c, 0xed, 0x87, 0x75,
			0x87, 0x26, 0xec, 0x61, 0x2c, 0xb4, 0x0f, 0x89,
			0xb0, 0xfb, 0x2e, 0x68, 0x5d, 0x15, 0xc7, 0x8d,
			0x2e, 0xc0, 0xd9, 0xec, 0xaf, 0x4f, 0xd2, 0x25,
			0x29, 0xe8, 0xd2, 0x26, 0x2b, 0x67, 0xe9, 0xfc,
			0x2b, 0xa8, 0x67, 0x96, 0x12, 0x1f, 0x5b, 0x96,
			0xc6, 0x14, 0x53, 0xaf, 0x44, 0xea, 0xd6, 0xe2,
			0x94, 0x98, 0xe4, 0x12, 0x93, 0x4c, 0x92, 0xe0,
			0x18, 0xa5, 0x8d, 0x2d, 0xe4, 0x71, 0x3c, 0x47,
			0x4c, 0xf7, 0xe6, 0x47, 0x9e, 0xc0, 0x68, 0xdf,
			0xd4, 0xf5, 0x5a, 0x74, 0xb1, 0x2b, 0x29, 0x03,
			0x19, 0x07, 0xaf, 0x90, 0x62, 0x5c, 0x68, 0x98,
			0x48, 0x16, 0x11, 0x02, 0x9d, 0xee, 0xb4, 0x9b,
			0xe5, 0x42, 0x7f, 0x08, 0xfd, 0x16, 0x32, 0x0b,
			0xd0, 0xb3, 0xfa, 0x2b, 0xb7, 0x99, 0xf9, 0x29,
			0xcd, 0x20, 0x45, 0x9f, 0xb3, 0x1a, 0x5d, 0xa2,
			0xaf, 0x4d, 0xe0, 0xbd, 0x42, 0x0d, 0xbc, 0x74,
			0x99, 0x9c, 0x8e, 0x53, 0x1a, 0xb4, 0x3e, 0xbd,
			0xa2, 0x9a, 0x2d, 0xf7, 0xf8, 0x39, 0x0f, 0x67,
			0x63, 0xfc, 0x6b, 0xc0, 0xaf, 0xb3, 0x4b, 0x4f,
			0x55, 0xc4, 0xcf, 0xa7, 0xc8, 0x04, 0x11, 0x3e,
			0x14, 0x32, 0xbb, 0x1b, 0x38, 0x77, 0xd6, 0x7f,
			0x54, 0x4c, 0xdf, 0x75, 0xf3, 0x07, 0x2d, 0x33,
			0x9b, 0xa8, 0x20, 0xe1, 0x7b, 0x12, 0xb5, 0xf3,
			0xef, 0x2f, 0xce, 0x72, 0xe5, 0x24, 0x60, 0xc1,
			0x30, 0xe2, 0xab, 0xa1, 0x8e, 0x11, 0x09, 0xa8,
			0x21, 0x33, 0x44, 0xfe, 0x7f, 0x35, 0x32, 0x93,
			0x39, 0xa7, 0xad, 0x8b, 0x79, 0x06, 0xb2, 0xcb,
			0x4e, 0xa9, 0x5f, 0xc7, 0xba, 0x74, 0x29, 0xec,
			0x93, 0xa0, 0x4e, 0x54, 0x93, 0xc0, 0xbc, 0x55,
			0x64, 0xf0, 0x48, 0xe5, 0x57, 0x99, 0xee, 0x75,
			0xd6, 0x79, 0x0f, 0x66, 0xb7, 0xc6, 0x57, 0x76,
			0xf7, 0xb7, 0xf3, 0x9c, 0xc5, 0x60, 0xe8, 0x7f,
			0x83, 0x76, 0xd6, 0x0e, 0xaa, 0xe6, 0x90, 0x39,
			0x1d, 0xa6, 0x32, 0x6a, 0x34, 0xe3, 0x55, 0xf8,
			0x58, 0xa0, 0x58, 0x7d, 0x33, 0xe0, 0x22, 0x39,
			0x44, 0x64, 0x87, 0x86, 0x5a, 0x2f, 0xa7, 0x7e,
			0x0f, 0x38, 0xea, 0xb0, 0x30, 0xcc, 0x61, 0xa5,
			0x6a, 0x32, 0xae, 0x1e, 0xf7, 0xe9, 0xd0, 0xa9,
			0x0c, 0x32, 0x4b, 0xb5, 0x49, 0x28, 0xab, 0x85,
			0x2f, 0x8e, 0x01, 0x36, 0x38, 0x52, 0xd0, 0xba,
			0xd6, 0x02, 0x78, 0xf8, 0x0e, 0x3e, 0x9c, 0x8b,
			0x6b, 0x45, 0x99, 0x3f, 0x5c, 0xfe, 0x58, 0xf1,
			0x5c, 0x94, 0x04, 0xe1, 0xf5, 0x18, 0x6d, 0x51,
			0xb2, 0x5d, 0x18, 0x20, 0xb6, 0xc2, 0x9a, 0x42,
			0x1d, 0xb3, 0xab, 0x3c, 0xb6, 0x3a, 0x13, 0x03,
			0xb2, 0x46, 0x82, 0x4f, 0xfc, 0x64, 0xbc, 0x4f,
			0xca, 0xfa, 0x9c, 0xc0, 0xd5, 0xa7, 0xbd, 0x11,
			0xb7, 0xe4, 0x5a, 0xf6, 0x6f, 0x4d, 0x4d, 0x54,
			0xea, 0xa4, 0x98, 0x66, 0xd4, 0x22, 0x3b, 0xd3,
			0x8f, 0x34, 0x47, 0xd9, 0x7c, 0xf4, 0x72, 0x3b,
			0x4d, 0x02, 0x77, 0xf6, 0xd6, 0xdd, 0x08, 0x0a,
			0x81, 0xe1, 0x86, 0x89, 0x3e, 0x56, 0x10, 0x3c,
			0xba, 0xd7, 0x81, 0x8c, 0x08, 0xbc, 0x8b, 0xe2,
			0x53, 0xec, 0xa7, 0x89, 0xee, 0xc8, 0x56, 0xb5,
			0x36, 0x2c, 0xb2, 0x03, 0xba, 0x99, 0xdd, 0x7c,
			0x48, 0xa0, 0xb0, 0xbc, 0x91, 0x33, 0xe9, 0xa8,
			0xcb, 0xcd, 0xcf, 0x59, 0x5f, 0x1f, 0x15, 0xe2,
			0x56, 0xf5, 0x4e, 0x01, 0x35, 0x27, 0x45, 0x77,
			0x47, 0xc8, 0xbc, 0xcb, 0x7e, 0x39, 0xc1, 0x97,
			0x28, 0xd3, 0x84, 0xfc, 0x2c, 0x3e, 0xc8, 0xad,
			0x9c, 0xf8, 0x8a, 0x61, 0x9c, 0x28, 0xaa, 0xc5,
			0x99, 0x20, 0x43, 0x85, 0x9d, 0xa5, 0xe2, 0x8b,
			0xb8, 0xae, 0xeb, 0xd0, 0x32, 0x0d, 0x52, 0x78,
			0x09, 0x56, 0x3f, 0xc7, 0xd8, 0x7e, 0x26, 0xfc,
			0x37, 0xfb, 0x6f, 0x04, 0xfc, 0xfa, 0x92, 0x10,
			0xac, 0xf8, 0x3e, 0x21, 0xdc, 0x8c, 0x21, 0x16,
			0x7d, 0x67, 0x6e, 0xf6, 0xcd, 0xda, 0xb6, 0x98,
			0x23, 0xab, 0x23, 0x3c, 0xb2, 0x10, 0xa0, 0x53,
			0x5a, 0x56, 0x9f, 0xc5, 0xd0, 0xff, 0xbb, 0xe4,
			0x98, 0x3c, 0x69, 0x1e, 0xdb, 0x38, 0x8f, 0x7e,
			0x0f, 0xd2, 0x98, 0x88, 0x81, 0x8b, 0x45, 0x67,
			0xea, 0x33, 0xf1, 0xeb, 0xe9, 0x97, 0x55, 0x2e,
			0xd9, 0xaa, 0xeb, 0x5a, 0xec, 0xda, 0xe1, 0x68,
			0xa8, 0x9d, 0x3c, 0x84, 0x7c, 0x05, 0x3d, 0x62,
			0x87, 0x8f, 0x03, 0x21, 0x28, 0x95, 0x0c, 0x89,
			0x25, 0x22, 0x4a, 0xb0, 0x93, 0xa9, 0x50, 0xa2,
			0x2f, 0x57, 0x6e, 0x18, 0x42, 0x19, 0x54, 0x0c,
			0x55, 0x67, 0xc6, 0x11, 0x49, 0xf4, 0x5c, 0xd2,
			0xe9, 0x3d, 0xdd, 0x8b, 0x48, 0x71, 0x21, 0x00,
			0xc3, 0x9a, 0x6c, 0x85, 0x74, 0x28, 0x83, 0x4a,
			0x1b, 0x31, 0x05, 0xe1, 0x06, 0x92, 0xe7, 0xda,
			0x85, 0x73, 0x78, 0x45, 0x20, 0x7f, 0xae, 0x13,
			0x7c, 0x33, 0x06, 0x22, 0xf4, 0x83, 0xf9, 0x35,
			0x3f, 0x6c, 0x71, 0xa8, 0x4e, 0x48, 0xbe, 0x9b,
			0xce, 0x8a, 0xba, 0xda, 0xbe, 0x28, 0x08, 0xf7,
			0xe2, 0x14, 0x8c, 0x71, 0xea, 0x72, 0xf9, 0x33,
			0xf2, 0x88, 0x3f, 0xd7, 0xbb, 0x69, 0x6c, 0x29,
			0x19, 0xdc, 0x84, 0xce, 0x1f, 0x12, 0x4f, 0xc8,
			0xaf, 0xa5, 0x04, 0xba, 0x5a, 0xab, 0xb0, 0xd9,
			0x14, 0x1f, 0x6c, 0x68, 0x98, 0x39, 0x89, 0x7a,
			0xd9, 0xd8, 0x2f, 0xdf, 0xa8, 0x47, 0x4a, 0x25,
			0xe2, 0xfb, 0x33, 0xf4, 0x59, 0x78, 0xe1, 0x68,
			0x85, 0xcf, 0xfe, 0x59, 0x20, 0xd4, 0x05, 0x1d,
			0x80, 0x99, 0xae, 0xbc, 0xca, 0xae, 0x0f, 0x2f,
			0x65, 0x43, 0x34, 0x8e, 0x7e, 0xac, 0xd3, 0x93,
			0x2f, 0xac, 0x6d, 0x14, 0x3d, 0x02, 0x07, 0x70,
			0x9d, 0xa4, 0xf3, 0x1b, 0x5c, 0x36, 0xfc, 0x01,
			0x73, 0x34, 0x85, 0x0c, 0x6c, 0xd6, 0xf1, 0xbd,
			0x3f, 0xdf, 0xee, 0xf5, 0xd9, 0xba, 0x56, 0xef,
			0xf4, 0x9b, 0x6b, 0xee, 0x9f, 0x5a, 0x78, 0x6d,
			0x32, 0x19, 0xf4, 0xf7, 0xf8, 0x4c, 0x69, 0x0b,
			0x4b, 0xbc, 0xbb, 0xb7, 0xf2, 0x85, 0xaf, 0x70,
			0x75, 0x24, 0x6c, 0x54, 0xa7, 0x0e, 0x4d, 0x1d,
			0x01, 0xbf, 0x08, 0xac, 0xcf, 0x7f, 0x2c, 0xe3,
			0x14, 0x89, 0x5e, 0x70, 0x5a, 0x99, 0x92, 0xcd,
			0x01, 0x84, 0xc8, 0xd2, 0xab, 0xe5, 0x4f, 0x58,
			0xe7, 0x0f, 0x2f, 0x0e, 0xff, 0x68, 0xea, 0xfd,
			0x15, 0xb3, 0x17, 0xe6, 0xb0, 0xe7, 0x85, 0xd8,
			0x23, 0x2e, 0x05, 0xc7, 0xc9, 0xc4, 0x46, 0x1f,
			0xe1, 0x9e, 0x49, 0x20, 0x23, 0x24, 0x4d, 0x7e,
			0x29, 0x65, 0xff, 0xf4, 0xb6, 0xfd, 0x1a, 0x85,
			0xc4, 0x16, 0xec, 0xfc, 0xea, 0x7b, 0xd6, 0x2c,
			0x43, 0xf8, 0xb7, 0xbf, 0x79, 0xc0, 0x85, 0xcd,
			0xef, 0xe1, 0x98, 0xd3, 0xa5, 0xf7, 0x90, 0x8c,
			0xe9, 0x7f, 0x80, 0x6b, 0xd2, 0xac, 0x4c, 0x30,
			0xa7, 0xc6, 0x61, 0x6c, 0xd2, 0xf9, 0x2c, 0xff,
			0x30, 0xbc, 0x22, 0x81, 0x7d, 0x93, 0x12, 0xe4,
			0x0a, 0xcd, 0xaf, 0xdd, 0xe8, 0xab, 0x0a, 0x1e,
			0x13, 0xa4, 0x27, 0xc3, 0x5f, 0xf7, 0x4b, 0xbb,
			0x37, 0x09, 0x4b, 0x91, 0x6f, 0x92, 0x4f, 0xaf,
			0x52, 0xee, 0xdf, 0xef, 0x09, 0x6f, 0xf7, 0x5c,
			0x6e, 0x12, 0x17, 0x72, 0x63, 0x57, 0xc7, 0xba,
			0x3b, 0x6b, 0x38, 0x32, 0x73, 0x1b, 0x9c, 0x80,
			0xc1, 0x7a, 0xc6, 0xcf, 0xcd, 0x35, 0xc0, 0x6b,
			0x31, 0x1a, 0x6b, 0xe9, 0xd8, 0x2c, 0x29, 0x3f,
			0x96, 0xfb, 0xb6, 0xcd, 0x13, 0x91, 0x3b, 0xc2,
			0xd2, 0xa3, 0x31, 0x8d, 0xa4, 0xcd, 0x57, 0xcd,
			0x13, 0x3d, 0x64, 0xfd, 0x06, 0xce, 0xe6, 0xdc,
			0x0c, 0x24, 0x43, 0x31, 0x40, 0x57, 0xf1, 0x72,
			0x17, 0xe3, 0x3a, 0x63, 0x6d, 0x35, 0xcf, 0x5d,
			0x97, 0x40, 0x59, 0xdd, 0xf7, 0x3c, 0x02, 0xf7,
			0x1c, 0x7e, 0x05, 0xbb, 0xa9, 0x0d, 0x01, 0xb1,
			0x8e, 0xc0, 0x30, 0xa9, 0x53, 0x24, 0xc9, 0x89,
			0x84, 0x6d, 0xaa, 0xd0, 0xcd, 0x91, 0xc2, 0x4d,
			0x91, 0xb0, 0x89, 0xe2, 0xbf, 0x83, 0x44, 0xaa,
			0x28, 0x72, 0x23, 0xa0, 0xc2, 0xad, 0xad, 0x1c,
			0xfc, 0x3f, 0x09, 0x7a, 0x0b, 0xdc, 0xc5, 0x1b,
			0x87, 0x13, 0xc6, 0x5b, 0x59, 0x8d, 0xf2, 0xc8,
			0xaf, 0xdf, 0x11, 0x95,
		},
		.rlen = 4100,
	},
};

/*
 * Compression stuff.
 */
#define COMP_BUF_SIZE           512

struct comp_testvec {
	int inlen, outlen;
	char input[COMP_BUF_SIZE];
	char output[COMP_BUF_SIZE];
};

/*
 * Deflate test vectors (null-terminated strings).
 * Params: winbits=11, Z_DEFAULT_COMPRESSION, MAX_MEM_LEVEL.
 */
#define DEFLATE_COMP_TEST_VECTORS 2
#define DEFLATE_DECOMP_TEST_VECTORS 2

static struct comp_testvec deflate_comp_tv_template[] = {
	{
		.inlen	= 70,
		.outlen	= 38,
	  	.input	= "Join us now and share the software "
			  "Join us now and share the software ",
		.output	= { 0xf3, 0xca, 0xcf, 0xcc, 0x53, 0x28, 0x2d, 0x56,
			    0xc8, 0xcb, 0x2f, 0x57, 0x48, 0xcc, 0x4b, 0x51,
			    0x28, 0xce, 0x48, 0x2c, 0x4a, 0x55, 0x28, 0xc9,
			    0x48, 0x55, 0x28, 0xce, 0x4f, 0x2b, 0x29, 0x07,
			    0x71, 0xbc, 0x08, 0x2b, 0x01, 0x00 },
	}, {
		.inlen	= 191,
		.outlen	= 122,
		.input	= "This document describes a compression method based on the DEFLATE"
			  "compression algorithm.  This document defines the application of "
			  "the DEFLATE algorithm to the IP Payload Compression Protocol.",
		.output	= { 0x5d, 0x8d, 0x31, 0x0e, 0xc2, 0x30, 0x10, 0x04,
			    0xbf, 0xb2, 0x2f, 0xc8, 0x1f, 0x10, 0x04, 0x09,
			    0x89, 0xc2, 0x85, 0x3f, 0x70, 0xb1, 0x2f, 0xf8,
			    0x24, 0xdb, 0x67, 0xd9, 0x47, 0xc1, 0xef, 0x49,
			    0x68, 0x12, 0x51, 0xae, 0x76, 0x67, 0xd6, 0x27,
			    0x19, 0x88, 0x1a, 0xde, 0x85, 0xab, 0x21, 0xf2,
			    0x08, 0x5d, 0x16, 0x1e, 0x20, 0x04, 0x2d, 0xad,
			    0xf3, 0x18, 0xa2, 0x15, 0x85, 0x2d, 0x69, 0xc4,
			    0x42, 0x83, 0x23, 0xb6, 0x6c, 0x89, 0x71, 0x9b,
			    0xef, 0xcf, 0x8b, 0x9f, 0xcf, 0x33, 0xca, 0x2f,
			    0xed, 0x62, 0xa9, 0x4c, 0x80, 0xff, 0x13, 0xaf,
			    0x52, 0x37, 0xed, 0x0e, 0x52, 0x6b, 0x59, 0x02,
			    0xd9, 0x4e, 0xe8, 0x7a, 0x76, 0x1d, 0x02, 0x98,
			    0xfe, 0x8a, 0x87, 0x83, 0xa3, 0x4f, 0x56, 0x8a,
			    0xb8, 0x9e, 0x8e, 0x5c, 0x57, 0xd3, 0xa0, 0x79,
			    0xfa, 0x02 },
	},
};

static struct comp_testvec deflate_decomp_tv_template[] = {
	{
		.inlen	= 122,
		.outlen	= 191,
		.input	= { 0x5d, 0x8d, 0x31, 0x0e, 0xc2, 0x30, 0x10, 0x04,
			    0xbf, 0xb2, 0x2f, 0xc8, 0x1f, 0x10, 0x04, 0x09,
			    0x89, 0xc2, 0x85, 0x3f, 0x70, 0xb1, 0x2f, 0xf8,
			    0x24, 0xdb, 0x67, 0xd9, 0x47, 0xc1, 0xef, 0x49,
			    0x68, 0x12, 0x51, 0xae, 0x76, 0x67, 0xd6, 0x27,
			    0x19, 0x88, 0x1a, 0xde, 0x85, 0xab, 0x21, 0xf2,
			    0x08, 0x5d, 0x16, 0x1e, 0x20, 0x04, 0x2d, 0xad,
			    0xf3, 0x18, 0xa2, 0x15, 0x85, 0x2d, 0x69, 0xc4,
			    0x42, 0x83, 0x23, 0xb6, 0x6c, 0x89, 0x71, 0x9b,
			    0xef, 0xcf, 0x8b, 0x9f, 0xcf, 0x33, 0xca, 0x2f,
			    0xed, 0x62, 0xa9, 0x4c, 0x80, 0xff, 0x13, 0xaf,
			    0x52, 0x37, 0xed, 0x0e, 0x52, 0x6b, 0x59, 0x02,
			    0xd9, 0x4e, 0xe8, 0x7a, 0x76, 0x1d, 0x02, 0x98,
			    0xfe, 0x8a, 0x87, 0x83, 0xa3, 0x4f, 0x56, 0x8a,
			    0xb8, 0x9e, 0x8e, 0x5c, 0x57, 0xd3, 0xa0, 0x79,
			    0xfa, 0x02 },
		.output	= "This document describes a compression method based on the DEFLATE"
			  "compression algorithm.  This document defines the application of "
			  "the DEFLATE algorithm to the IP Payload Compression Protocol.",
	}, {
		.inlen	= 38,
		.outlen	= 70,
		.input	= { 0xf3, 0xca, 0xcf, 0xcc, 0x53, 0x28, 0x2d, 0x56,
			    0xc8, 0xcb, 0x2f, 0x57, 0x48, 0xcc, 0x4b, 0x51,
			    0x28, 0xce, 0x48, 0x2c, 0x4a, 0x55, 0x28, 0xc9,
			    0x48, 0x55, 0x28, 0xce, 0x4f, 0x2b, 0x29, 0x07,
			    0x71, 0xbc, 0x08, 0x2b, 0x01, 0x00 },
		.output	= "Join us now and share the software "
			  "Join us now and share the software ",
	},
};

/*
 * LZO test vectors (null-terminated strings).
 */
#define LZO_COMP_TEST_VECTORS 2
#define LZO_DECOMP_TEST_VECTORS 2

static struct comp_testvec lzo_comp_tv_template[] = {
	{
		.inlen	= 70,
		.outlen	= 46,
		.input	= "Join us now and share the software "
			  "Join us now and share the software ",
		.output	= {  0x00, 0x0d, 0x4a, 0x6f, 0x69, 0x6e, 0x20, 0x75,
			     0x73, 0x20, 0x6e, 0x6f, 0x77, 0x20, 0x61, 0x6e,
			     0x64, 0x20, 0x73, 0x68, 0x61, 0x72, 0x65, 0x20,
			     0x74, 0x68, 0x65, 0x20, 0x73, 0x6f, 0x66, 0x74,
			     0x77, 0x70, 0x01, 0x01, 0x4a, 0x6f, 0x69, 0x6e,
			     0x3d, 0x88, 0x00, 0x11, 0x00, 0x00 },
	}, {
		.inlen	= 159,
		.outlen	= 133,
		.input	= "This document describes a compression method based on the LZO "
			  "compression algorithm.  This document defines the application of "
			  "the LZO algorithm used in UBIFS.",
		.output	= { 0x00, 0x2b, 0x54, 0x68, 0x69, 0x73, 0x20, 0x64,
			    0x6f, 0x63, 0x75, 0x6d, 0x65, 0x6e, 0x74, 0x20,
			    0x64, 0x65, 0x73, 0x63, 0x72, 0x69, 0x62, 0x65,
			    0x73, 0x20, 0x61, 0x20, 0x63, 0x6f, 0x6d, 0x70,
			    0x72, 0x65, 0x73, 0x73, 0x69, 0x6f, 0x6e, 0x20,
			    0x6d, 0x65, 0x74, 0x68, 0x6f, 0x64, 0x20, 0x62,
			    0x61, 0x73, 0x65, 0x64, 0x20, 0x6f, 0x6e, 0x20,
			    0x74, 0x68, 0x65, 0x20, 0x4c, 0x5a, 0x4f, 0x2b,
			    0x8c, 0x00, 0x0d, 0x61, 0x6c, 0x67, 0x6f, 0x72,
			    0x69, 0x74, 0x68, 0x6d, 0x2e, 0x20, 0x20, 0x54,
			    0x68, 0x69, 0x73, 0x2a, 0x54, 0x01, 0x02, 0x66,
			    0x69, 0x6e, 0x65, 0x73, 0x94, 0x06, 0x05, 0x61,
			    0x70, 0x70, 0x6c, 0x69, 0x63, 0x61, 0x74, 0x76,
			    0x0a, 0x6f, 0x66, 0x88, 0x02, 0x60, 0x09, 0x27,
			    0xf0, 0x00, 0x0c, 0x20, 0x75, 0x73, 0x65, 0x64,
			    0x20, 0x69, 0x6e, 0x20, 0x55, 0x42, 0x49, 0x46,
			    0x53, 0x2e, 0x11, 0x00, 0x00 },
	},
};

static struct comp_testvec lzo_decomp_tv_template[] = {
	{
		.inlen	= 133,
		.outlen	= 159,
		.input	= { 0x00, 0x2b, 0x54, 0x68, 0x69, 0x73, 0x20, 0x64,
			    0x6f, 0x63, 0x75, 0x6d, 0x65, 0x6e, 0x74, 0x20,
			    0x64, 0x65, 0x73, 0x63, 0x72, 0x69, 0x62, 0x65,
			    0x73, 0x20, 0x61, 0x20, 0x63, 0x6f, 0x6d, 0x70,
			    0x72, 0x65, 0x73, 0x73, 0x69, 0x6f, 0x6e, 0x20,
			    0x6d, 0x65, 0x74, 0x68, 0x6f, 0x64, 0x20, 0x62,
			    0x61, 0x73, 0x65, 0x64, 0x20, 0x6f, 0x6e, 0x20,
			    0x74, 0x68, 0x65, 0x20, 0x4c, 0x5a, 0x4f, 0x2b,
			    0x8c, 0x00, 0x0d, 0x61, 0x6c, 0x67, 0x6f, 0x72,
			    0x69, 0x74, 0x68, 0x6d, 0x2e, 0x20, 0x20, 0x54,
			    0x68, 0x69, 0x73, 0x2a, 0x54, 0x01, 0x02, 0x66,
			    0x69, 0x6e, 0x65, 0x73, 0x94, 0x06, 0x05, 0x61,
			    0x70, 0x70, 0x6c, 0x69, 0x63, 0x61, 0x74, 0x76,
			    0x0a, 0x6f, 0x66, 0x88, 0x02, 0x60, 0x09, 0x27,
			    0xf0, 0x00, 0x0c, 0x20, 0x75, 0x73, 0x65, 0x64,
			    0x20, 0x69, 0x6e, 0x20, 0x55, 0x42, 0x49, 0x46,
			    0x53, 0x2e, 0x11, 0x00, 0x00 },
		.output	= "This document describes a compression method based on the LZO "
			  "compression algorithm.  This document defines the application of "
			  "the LZO algorithm used in UBIFS.",
	}, {
		.inlen	= 46,
		.outlen	= 70,
		.input	= { 0x00, 0x0d, 0x4a, 0x6f, 0x69, 0x6e, 0x20, 0x75,
			    0x73, 0x20, 0x6e, 0x6f, 0x77, 0x20, 0x61, 0x6e,
			    0x64, 0x20, 0x73, 0x68, 0x61, 0x72, 0x65, 0x20,
			    0x74, 0x68, 0x65, 0x20, 0x73, 0x6f, 0x66, 0x74,
			    0x77, 0x70, 0x01, 0x01, 0x4a, 0x6f, 0x69, 0x6e,
			    0x3d, 0x88, 0x00, 0x11, 0x00, 0x00 },
		.output	= "Join us now and share the software "
			  "Join us now and share the software ",
	},
};

/*
 * Michael MIC test vectors from IEEE 802.11i
 */
#define MICHAEL_MIC_TEST_VECTORS 6

static struct hash_testvec michael_mic_tv_template[] = {
	{
		.key = { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
		.ksize = 8,
		.plaintext = { },
		.psize = 0,
		.digest = { 0x82, 0x92, 0x5c, 0x1c, 0xa1, 0xd1, 0x30, 0xb8 }
	},
	{
		.key = { 0x82, 0x92, 0x5c, 0x1c, 0xa1, 0xd1, 0x30, 0xb8 },
		.ksize = 8,
		.plaintext = { 'M' },
		.psize = 1,
		.digest = { 0x43, 0x47, 0x21, 0xca, 0x40, 0x63, 0x9b, 0x3f }
	},
	{
		.key = { 0x43, 0x47, 0x21, 0xca, 0x40, 0x63, 0x9b, 0x3f },
		.ksize = 8,
		.plaintext = { 'M', 'i' },
		.psize = 2,
		.digest = { 0xe8, 0xf9, 0xbe, 0xca, 0xe9, 0x7e, 0x5d, 0x29 }
	},
	{
		.key = { 0xe8, 0xf9, 0xbe, 0xca, 0xe9, 0x7e, 0x5d, 0x29 },
		.ksize = 8,
		.plaintext = { 'M', 'i', 'c' },
		.psize = 3,
		.digest = { 0x90, 0x03, 0x8f, 0xc6, 0xcf, 0x13, 0xc1, 0xdb }
	},
	{
		.key = { 0x90, 0x03, 0x8f, 0xc6, 0xcf, 0x13, 0xc1, 0xdb },
		.ksize = 8,
		.plaintext = { 'M', 'i', 'c', 'h' },
		.psize = 4,
		.digest = { 0xd5, 0x5e, 0x10, 0x05, 0x10, 0x12, 0x89, 0x86 }
	},
	{
		.key = { 0xd5, 0x5e, 0x10, 0x05, 0x10, 0x12, 0x89, 0x86 },
		.ksize = 8,
		.plaintext = { 'M', 'i', 'c', 'h', 'a', 'e', 'l' },
		.psize = 7,
		.digest = { 0x0a, 0x94, 0x2b, 0x12, 0x4e, 0xca, 0xa5, 0x46 },
	}
};

/*
 * CRC32C test vectors
 */
#define CRC32C_TEST_VECTORS 14

static struct hash_testvec crc32c_tv_template[] = {
	{
		.psize = 0,
		.digest = { 0x00, 0x00, 0x00, 0x00 }
	},
	{
		.key = { 0x87, 0xa9, 0xcb, 0xed },
		.ksize = 4,
		.psize = 0,
		.digest = { 0x78, 0x56, 0x34, 0x12 },
	},
	{
		.key = { 0xff, 0xff, 0xff, 0xff },
		.ksize = 4,
		.plaintext = { 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08,
			       0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f, 0x10,
			       0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18,
			       0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f, 0x20,
			       0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27, 0x28 },
		.psize = 40,
		.digest = { 0x7f, 0x15, 0x2c, 0x0e }
	},
	{
		.key = { 0xff, 0xff, 0xff, 0xff },
		.ksize = 4,
		.plaintext = { 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f, 0x30,
			       0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38,
			       0x39, 0x3a, 0x3b, 0x3c, 0x3d, 0x3e, 0x3f, 0x40,
			       0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47, 0x48,
			       0x49, 0x4a, 0x4b, 0x4c, 0x4d, 0x4e, 0x4f, 0x50 },
		.psize = 40,
		.digest = { 0xf6, 0xeb, 0x80, 0xe9 }
	},
	{
		.key = { 0xff, 0xff, 0xff, 0xff },
		.ksize = 4,
		.plaintext = { 0x51, 0x52, 0x53, 0x54, 0x55, 0x56, 0x57, 0x58,
			       0x59, 0x5a, 0x5b, 0x5c, 0x5d, 0x5e, 0x5f, 0x60,
			       0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0x68,
			       0x69, 0x6a, 0x6b, 0x6c, 0x6d, 0x6e, 0x6f, 0x70,
			       0x71, 0x72, 0x73, 0x74, 0x75, 0x76, 0x77, 0x78 },
		.psize = 40,
		.digest = { 0xed, 0xbd, 0x74, 0xde }
	},
	{
		.key = { 0xff, 0xff, 0xff, 0xff },
		.ksize = 4,
		.plaintext = { 0x79, 0x7a, 0x7b, 0x7c, 0x7d, 0x7e, 0x7f, 0x80,
			       0x81, 0x82, 0x83, 0x84, 0x85, 0x86, 0x87, 0x88,
			       0x89, 0x8a, 0x8b, 0x8c, 0x8d, 0x8e, 0x8f, 0x90,
			       0x91, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97, 0x98,
			       0x99, 0x9a, 0x9b, 0x9c, 0x9d, 0x9e, 0x9f, 0xa0 },
		.psize = 40,
		.digest = { 0x62, 0xc8, 0x79, 0xd5 }
	},
	{
		.key = { 0xff, 0xff, 0xff, 0xff },
		.ksize = 4,
		.plaintext = { 0xa1, 0xa2, 0xa3, 0xa4, 0xa5, 0xa6, 0xa7, 0xa8,
			       0xa9, 0xaa, 0xab, 0xac, 0xad, 0xae, 0xaf, 0xb0,
			       0xb1, 0xb2, 0xb3, 0xb4, 0xb5, 0xb6, 0xb7, 0xb8,
			       0xb9, 0xba, 0xbb, 0xbc, 0xbd, 0xbe, 0xbf, 0xc0,
			       0xc1, 0xc2, 0xc3, 0xc4, 0xc5, 0xc6, 0xc7, 0xc8 },
		.psize = 40,
		.digest = { 0xd0, 0x9a, 0x97, 0xba }
	},
	{
		.key = { 0xff, 0xff, 0xff, 0xff },
		.ksize = 4,
		.plaintext = { 0xc9, 0xca, 0xcb, 0xcc, 0xcd, 0xce, 0xcf, 0xd0,
			       0xd1, 0xd2, 0xd3, 0xd4, 0xd5, 0xd6, 0xd7, 0xd8,
			       0xd9, 0xda, 0xdb, 0xdc, 0xdd, 0xde, 0xdf, 0xe0,
			       0xe1, 0xe2, 0xe3, 0xe4, 0xe5, 0xe6, 0xe7, 0xe8,
			       0xe9, 0xea, 0xeb, 0xec, 0xed, 0xee, 0xef, 0xf0 },
		.psize = 40,
		.digest = { 0x13, 0xd9, 0x29, 0x2b }
	},
	{
		.key = { 0x80, 0xea, 0xd3, 0xf1 },
		.ksize = 4,
		.plaintext = { 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f, 0x30,
			       0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38,
			       0x39, 0x3a, 0x3b, 0x3c, 0x3d, 0x3e, 0x3f, 0x40,
			       0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47, 0x48,
			       0x49, 0x4a, 0x4b, 0x4c, 0x4d, 0x4e, 0x4f, 0x50 },
		.psize = 40,
		.digest = { 0x0c, 0xb5, 0xe2, 0xa2 }
	},
	{
		.key = { 0xf3, 0x4a, 0x1d, 0x5d },
		.ksize = 4,
		.plaintext = { 0x51, 0x52, 0x53, 0x54, 0x55, 0x56, 0x57, 0x58,
			       0x59, 0x5a, 0x5b, 0x5c, 0x5d, 0x5e, 0x5f, 0x60,
			       0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0x68,
			       0x69, 0x6a, 0x6b, 0x6c, 0x6d, 0x6e, 0x6f, 0x70,
			       0x71, 0x72, 0x73, 0x74, 0x75, 0x76, 0x77, 0x78 },
		.psize = 40,
		.digest = { 0xd1, 0x7f, 0xfb, 0xa6 }
	},
	{
		.key = { 0x2e, 0x80, 0x04, 0x59 },
		.ksize = 4,
		.plaintext = { 0x79, 0x7a, 0x7b, 0x7c, 0x7d, 0x7e, 0x7f, 0x80,
			       0x81, 0x82, 0x83, 0x84, 0x85, 0x86, 0x87, 0x88,
			       0x89, 0x8a, 0x8b, 0x8c, 0x8d, 0x8e, 0x8f, 0x90,
			       0x91, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97, 0x98,
			       0x99, 0x9a, 0x9b, 0x9c, 0x9d, 0x9e, 0x9f, 0xa0 },
		.psize = 40,
		.digest = { 0x59, 0x33, 0xe6, 0x7a }
	},
	{
		.key = { 0xa6, 0xcc, 0x19, 0x85 },
		.ksize = 4,
		.plaintext = { 0xa1, 0xa2, 0xa3, 0xa4, 0xa5, 0xa6, 0xa7, 0xa8,
			       0xa9, 0xaa, 0xab, 0xac, 0xad, 0xae, 0xaf, 0xb0,
			       0xb1, 0xb2, 0xb3, 0xb4, 0xb5, 0xb6, 0xb7, 0xb8,
			       0xb9, 0xba, 0xbb, 0xbc, 0xbd, 0xbe, 0xbf, 0xc0,
			       0xc1, 0xc2, 0xc3, 0xc4, 0xc5, 0xc6, 0xc7, 0xc8 },
		.psize = 40,
		.digest = { 0xbe, 0x03, 0x01, 0xd2 }
	},
	{
		.key = { 0x41, 0xfc, 0xfe, 0x2d },
		.ksize = 4,
		.plaintext = { 0xc9, 0xca, 0xcb, 0xcc, 0xcd, 0xce, 0xcf, 0xd0,
			       0xd1, 0xd2, 0xd3, 0xd4, 0xd5, 0xd6, 0xd7, 0xd8,
			       0xd9, 0xda, 0xdb, 0xdc, 0xdd, 0xde, 0xdf, 0xe0,
			       0xe1, 0xe2, 0xe3, 0xe4, 0xe5, 0xe6, 0xe7, 0xe8,
			       0xe9, 0xea, 0xeb, 0xec, 0xed, 0xee, 0xef, 0xf0 },
		.psize = 40,
		.digest = { 0x75, 0xd3, 0xc5, 0x24 }
	},
	{
		.key = { 0xff, 0xff, 0xff, 0xff },
		.ksize = 4,
		.plaintext = { 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08,
			       0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f, 0x10,
			       0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18,
			       0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f, 0x20,
			       0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27, 0x28,
			       0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f, 0x30,
			       0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38,
			       0x39, 0x3a, 0x3b, 0x3c, 0x3d, 0x3e, 0x3f, 0x40,
			       0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47, 0x48,
			       0x49, 0x4a, 0x4b, 0x4c, 0x4d, 0x4e, 0x4f, 0x50,
			       0x51, 0x52, 0x53, 0x54, 0x55, 0x56, 0x57, 0x58,
			       0x59, 0x5a, 0x5b, 0x5c, 0x5d, 0x5e, 0x5f, 0x60,
			       0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0x68,
			       0x69, 0x6a, 0x6b, 0x6c, 0x6d, 0x6e, 0x6f, 0x70,
			       0x71, 0x72, 0x73, 0x74, 0x75, 0x76, 0x77, 0x78,
			       0x79, 0x7a, 0x7b, 0x7c, 0x7d, 0x7e, 0x7f, 0x80,
			       0x81, 0x82, 0x83, 0x84, 0x85, 0x86, 0x87, 0x88,
			       0x89, 0x8a, 0x8b, 0x8c, 0x8d, 0x8e, 0x8f, 0x90,
			       0x91, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97, 0x98,
			       0x99, 0x9a, 0x9b, 0x9c, 0x9d, 0x9e, 0x9f, 0xa0,
			       0xa1, 0xa2, 0xa3, 0xa4, 0xa5, 0xa6, 0xa7, 0xa8,
			       0xa9, 0xaa, 0xab, 0xac, 0xad, 0xae, 0xaf, 0xb0,
			       0xb1, 0xb2, 0xb3, 0xb4, 0xb5, 0xb6, 0xb7, 0xb8,
			       0xb9, 0xba, 0xbb, 0xbc, 0xbd, 0xbe, 0xbf, 0xc0,
			       0xc1, 0xc2, 0xc3, 0xc4, 0xc5, 0xc6, 0xc7, 0xc8,
			       0xc9, 0xca, 0xcb, 0xcc, 0xcd, 0xce, 0xcf, 0xd0,
			       0xd1, 0xd2, 0xd3, 0xd4, 0xd5, 0xd6, 0xd7, 0xd8,
			       0xd9, 0xda, 0xdb, 0xdc, 0xdd, 0xde, 0xdf, 0xe0,
			       0xe1, 0xe2, 0xe3, 0xe4, 0xe5, 0xe6, 0xe7, 0xe8,
			       0xe9, 0xea, 0xeb, 0xec, 0xed, 0xee, 0xef, 0xf0 },
		.psize = 240,
		.digest = { 0x75, 0xd3, 0xc5, 0x24 },
		.np = 2,
		.tap = { 31, 209 }
	},
};

/*
 * Cipher speed tests
 */
static struct cipher_speed aes_speed_template[] = {
	{ .klen = 16, .blen = 16, },
	{ .klen = 16, .blen = 64, },
	{ .klen = 16, .blen = 256, },
	{ .klen = 16, .blen = 1024, },
	{ .klen = 16, .blen = 8192, },
	{ .klen = 24, .blen = 16, },
	{ .klen = 24, .blen = 64, },
	{ .klen = 24, .blen = 256, },
	{ .klen = 24, .blen = 1024, },
	{ .klen = 24, .blen = 8192, },
	{ .klen = 32, .blen = 16, },
	{ .klen = 32, .blen = 64, },
	{ .klen = 32, .blen = 256, },
	{ .klen = 32, .blen = 1024, },
	{ .klen = 32, .blen = 8192, },

	/* End marker */
	{  .klen = 0, .blen = 0, }
};

static struct cipher_speed aes_lrw_speed_template[] = {
	{ .klen = 32, .blen = 16, },
	{ .klen = 32, .blen = 64, },
	{ .klen = 32, .blen = 256, },
	{ .klen = 32, .blen = 1024, },
	{ .klen = 32, .blen = 8192, },
	{ .klen = 40, .blen = 16, },
	{ .klen = 40, .blen = 64, },
	{ .klen = 40, .blen = 256, },
	{ .klen = 40, .blen = 1024, },
	{ .klen = 40, .blen = 8192, },
	{ .klen = 48, .blen = 16, },
	{ .klen = 48, .blen = 64, },
	{ .klen = 48, .blen = 256, },
	{ .klen = 48, .blen = 1024, },
	{ .klen = 48, .blen = 8192, },

	/* End marker */
	{  .klen = 0, .blen = 0, }
};

static struct cipher_speed aes_xts_speed_template[] = {
	{ .klen = 32, .blen = 16, },
	{ .klen = 32, .blen = 64, },
	{ .klen = 32, .blen = 256, },
	{ .klen = 32, .blen = 1024, },
	{ .klen = 32, .blen = 8192, },
	{ .klen = 48, .blen = 16, },
	{ .klen = 48, .blen = 64, },
	{ .klen = 48, .blen = 256, },
	{ .klen = 48, .blen = 1024, },
	{ .klen = 48, .blen = 8192, },
	{ .klen = 64, .blen = 16, },
	{ .klen = 64, .blen = 64, },
	{ .klen = 64, .blen = 256, },
	{ .klen = 64, .blen = 1024, },
	{ .klen = 64, .blen = 8192, },

	/* End marker */
	{  .klen = 0, .blen = 0, }
};

static struct cipher_speed des3_ede_speed_template[] = {
	{ .klen = 24, .blen = 16, },
	{ .klen = 24, .blen = 64, },
	{ .klen = 24, .blen = 256, },
	{ .klen = 24, .blen = 1024, },
	{ .klen = 24, .blen = 8192, },

	/* End marker */
	{  .klen = 0, .blen = 0, }
};

static struct cipher_speed twofish_speed_template[] = {
	{ .klen = 16, .blen = 16, },
	{ .klen = 16, .blen = 64, },
	{ .klen = 16, .blen = 256, },
	{ .klen = 16, .blen = 1024, },
	{ .klen = 16, .blen = 8192, },
	{ .klen = 24, .blen = 16, },
	{ .klen = 24, .blen = 64, },
	{ .klen = 24, .blen = 256, },
	{ .klen = 24, .blen = 1024, },
	{ .klen = 24, .blen = 8192, },
	{ .klen = 32, .blen = 16, },
	{ .klen = 32, .blen = 64, },
	{ .klen = 32, .blen = 256, },
	{ .klen = 32, .blen = 1024, },
	{ .klen = 32, .blen = 8192, },

	/* End marker */
	{  .klen = 0, .blen = 0, }
};

static struct cipher_speed blowfish_speed_template[] = {
	/* Don't support blowfish keys > 256 bit in this test */
	{ .klen = 8, .blen = 16, },
	{ .klen = 8, .blen = 64, },
	{ .klen = 8, .blen = 256, },
	{ .klen = 8, .blen = 1024, },
	{ .klen = 8, .blen = 8192, },
	{ .klen = 32, .blen = 16, },
	{ .klen = 32, .blen = 64, },
	{ .klen = 32, .blen = 256, },
	{ .klen = 32, .blen = 1024, },
	{ .klen = 32, .blen = 8192, },

	/* End marker */
	{  .klen = 0, .blen = 0, }
};

static struct cipher_speed des_speed_template[] = {
	{ .klen = 8, .blen = 16, },
	{ .klen = 8, .blen = 64, },
	{ .klen = 8, .blen = 256, },
	{ .klen = 8, .blen = 1024, },
	{ .klen = 8, .blen = 8192, },

	/* End marker */
	{  .klen = 0, .blen = 0, }
};

/*
 * Digest speed tests
 */
static struct hash_speed generic_hash_speed_template[] = {
	{ .blen = 16, 	.plen = 16, },
	{ .blen = 64,	.plen = 16, },
	{ .blen = 64,	.plen = 64, },
	{ .blen = 256,	.plen = 16, },
	{ .blen = 256,	.plen = 64, },
	{ .blen = 256,	.plen = 256, },
	{ .blen = 1024,	.plen = 16, },
	{ .blen = 1024,	.plen = 256, },
	{ .blen = 1024,	.plen = 1024, },
	{ .blen = 2048,	.plen = 16, },
	{ .blen = 2048,	.plen = 256, },
	{ .blen = 2048,	.plen = 1024, },
	{ .blen = 2048,	.plen = 2048, },
	{ .blen = 4096,	.plen = 16, },
	{ .blen = 4096,	.plen = 256, },
	{ .blen = 4096,	.plen = 1024, },
	{ .blen = 4096,	.plen = 4096, },
	{ .blen = 8192,	.plen = 16, },
	{ .blen = 8192,	.plen = 256, },
	{ .blen = 8192,	.plen = 1024, },
	{ .blen = 8192,	.plen = 4096, },
	{ .blen = 8192,	.plen = 8192, },

	/* End marker */
	{  .blen = 0,	.plen = 0, }
};

static struct cipher_speed camellia_speed_template[] = {
      { .klen = 16, .blen = 16, },
      { .klen = 16, .blen = 64, },
      { .klen = 16, .blen = 256, },
      { .klen = 16, .blen = 1024, },
      { .klen = 16, .blen = 8192, },
      { .klen = 24, .blen = 16, },
      { .klen = 24, .blen = 64, },
      { .klen = 24, .blen = 256, },
      { .klen = 24, .blen = 1024, },
      { .klen = 24, .blen = 8192, },
      { .klen = 32, .blen = 16, },
      { .klen = 32, .blen = 64, },
      { .klen = 32, .blen = 256, },
      { .klen = 32, .blen = 1024, },
      { .klen = 32, .blen = 8192, },

      /* End marker */
      {  .klen = 0, .blen = 0, }
};

static struct cipher_speed salsa20_speed_template[] = {
      { .klen = 16, .blen = 16, },
      { .klen = 16, .blen = 64, },
      { .klen = 16, .blen = 256, },
      { .klen = 16, .blen = 1024, },
      { .klen = 16, .blen = 8192, },
      { .klen = 32, .blen = 16, },
      { .klen = 32, .blen = 64, },
      { .klen = 32, .blen = 256, },
      { .klen = 32, .blen = 1024, },
      { .klen = 32, .blen = 8192, },

      /* End marker */
      {  .klen = 0, .blen = 0, }
};

#endif	/* _CRYPTO_TCRYPT_H */
