/*
 * Copyright (c) 2008 Travis Geiselbrecht
 * Copyright (c) 2012-2013, NVIDIA CORPORATION. All rights reserved
 *
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files
 * (the "Software"), to deal in the Software without restriction,
 * including without limitation the rights to use, copy, modify, merge,
 * publish, distribute, sublicense, and/or sell copies of the Software,
 * and to permit persons to whom the Software is furnished to do so,
 * subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 * IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY
 * CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,
 * TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
 * SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

#include <asm.h>

.text

/* void arch_enable_ints(void); */
FUNCTION(arch_enable_ints)
	mrs	r0, cpsr
	bic	r0, r0, #(1<<7)		/* clear the I bit */
	msr	cpsr_c, r0
	bx	lr

/* void arch_disable_ints(void); */
FUNCTION(arch_disable_ints)
	mrs	r0, cpsr
	orr	r0, r0, #(1<<7)
	msr	cpsr_c, r0
	bx	lr

/* int atomic_swap(int *ptr, int val); */
FUNCTION(atomic_swap)
	swp	r2, r1, [r0]
	mov	r0, r2
	bx	lr

/* int atomic_add(int *ptr, int val); */
FUNCTION(atomic_add)
#if ARM_ISA_ARMV6 || ARM_ISA_ARMV7
	/* use load/store exclusive */
.L_loop_add:
	ldrex 	r12, [r0]
	add		r2, r12, r1
	strex 	r3, r2, [r0]
	cmp		r3, #0
	bne 	.L_loop_add

	/* save old value */
	mov		r0, r12
	bx		lr
#else
	/* disable interrupts, do the add, and reenable */
	mrs	r2, cpsr
	mov	r12, r2
	orr	r2, r2, #(3<<6)
	msr	cpsr_c, r2

	/* ints disabled, old cpsr state in r12 */

	/* do the add, leave the previous value in r0 */
	mov	r3, r0
	ldr	r0, [r3]
	add	r2, r0, r1
	str	r2, [r3]

	/* restore interrupts and exit */
	msr	cpsr_c, r12
	bx	lr
#endif

/* int atomic_and(int *ptr, int val); */
FUNCTION(atomic_and)
#if ARM_ISA_ARMV6 || ARM_ISA_ARMV7
	/* use load/store exclusive */
.L_loop_and:
	ldrex 	r12, [r0]
	and		r2, r12, r1
	strex 	r3, r2, [r0]
	cmp		r3, #0
	bne 	.L_loop_and

	/* save old value */
	mov		r0, r12
	bx		lr
#else
	/* disable interrupts, do the and, and reenable */
	mrs	r2, cpsr
	mov	r12, r2
	orr	r2, r2, #(3<<6)
	msr	cpsr_c, r2

	/* ints disabled, old cpsr state in r12 */

	/* do the and, leave the previous value in r0 */
	mov	r3, r0
	ldr	r0, [r3]
	and	r2, r0, r1
	str	r2, [r3]

	/* restore interrupts and exit */
	msr	cpsr_c, r12
	bx	lr
#endif

/* int atomic_or(int *ptr, int val); */
FUNCTION(atomic_or)
#if ARM_ISA_ARMV6 || ARM_ISA_ARMV7
	/* use load/store exclusive */
.L_loop_or:
	ldrex 	r12, [r0]
	orr		r2, r12, r1
	strex 	r3, r2, [r0]
	cmp		r3, #0
	bne 	.L_loop_or

	/* save old value */
	mov		r0, r12
	bx		lr
#else
	/* disable interrupts, do the or, and reenable */
	mrs	r2, cpsr
	mov	r12, r2
	orr	r2, r2, #(3<<6)
	msr	cpsr_c, r2

	/* ints disabled, old cpsr state in r12 */

	/* do the or, leave the previous value in r0 */
	mov	r3, r0
	ldr	r0, [r3]
	orr	r2, r0, r1
	str	r2, [r3]

	/* restore interrupts and exit */
	msr	cpsr_c, r12
	bx	lr
#endif

/* void arch_idle(); */
FUNCTION(arch_idle)
#if ARM_CPU_CORTEX_A8 || ARM_CPU_CORTEX_A9 || ARM_CPU_CORTEX_A15
	wfi
#elif PLATFORM_MSM7K
	/* TODO: safely handle wfi */
#elif ARM_CPU_ARM1136 || ARM_CPU_ARM926
	mov	r0, #0
	mcr	p15, 0, r0, c7, c0, #4
#elif ARM_CPU_ARM7
	/* nothing to do here */
#else
#error unknown cpu
#endif
	bx	lr

/* uint32_t arm_read_sctlr(void) */
FUNCTION(arm_read_sctlr)
	mrc	p15, 0, r0, c1, c0, 0
	bx	lr

/* void arm_write_sctlr(uint32_t val) */
FUNCTION(arm_write_sctlr)
	mcr	p15, 0, r0, c1, c0, 0
	bx	lr

/* uint32_t arm_read_actlr(void) */
FUNCTION(arm_read_actlr)
	mrc	p15, 0, r0, c1, c0, 1
	bx	lr

/* void arm_write_actlr(uint32_t val) */
FUNCTION(arm_write_actlr)
	mcr	p15, 0, r0, c1, c0, 1
	bx	lr

/* void arm_invalidate_tlb(void) */
FUNCTION(arm_invalidate_tlb)
	mov	r0, #0
	mcr	p15, 0, r0, c8, c7, 0
	bx	lr

/* void arm_invalidate_tlb_byaddr(vaddr) */
FUNCTION(arm_invalidate_tlb_byaddr)
	lsr	r0, r0, #12
	lsl	r0, r0, #12		@ page align addr
	dsb				@ sync pgtbl writes
	mcr	p15, 0, r0, c8, c7, 3	@ MVA and all ASIDs
	bx	lr

/* void arm_invalidate_tlb_byaddr_asid(vaddr, asid) */
FUNCTION(arm_invalidate_tlb_byaddr_asid)
	lsr	r0, r0, #12
	lsl	r0, r0, #12		@ page align addr
	and	r1, r1, #0xFF		@ asid is 8bits
	orr	r0, r0, r1
	dsb				@ sync pgtbl writes
	mcr	p15, 0, r0, c8, c7, 1	@ MVA and this ASID
	bx	lr

/* void arm_write_v2p(uint32_t vaddr, uint32_t type) */
FUNCTION(arm_write_v2p)
	cmp	r1, #0x0
	mcreq	p15, 0, r0, c7, c8, 0	@ V2PCWPR
	bxeq	lr
	cmp	r1, #0x1
	mcreq	p15, 0, r0, c7, c8, 1	@ V2PCWPW
	bxeq	lr
	cmp	r1, #0x2
	mcreq	p15, 0, r0, c7, c8, 2	@ V2PCWUR
	bxeq	lr
	cmp	r1, #0x3
	mcreq	p15, 0, r0, c7, c8, 3	@ V2PCWUW
	bxeq	lr
	cmp	r1, #0x4
	mcreq	p15, 0, r0, c7, c8, 4	@ V2POWPR
	bxeq	lr
	cmp	r1, #0x5
	mcreq	p15, 0, r0, c7, c8, 5	@ V2POWPW
	bxeq	lr
	cmp	r1, #0x6
	mcreq	p15, 0, r0, c7, c8, 6	@ V2POWUR
	bxeq	lr
	cmp	r1, #0x7
	mcreq	p15, 0, r0, c7, c8, 7	@ V2POWUW
	bx	lr

/* uint64_t arm_read_par(void) */
FUNCTION(arm_read_par)
	mrc	p15, 0, r0, c7, c4, 0
	tst	r0, #(1 << 11)		@ LPAE / long desc format
	moveq	r1, #0
	mrrcne	p15, 0, r0, r1, c7
	bx	lr

/* void arch_switch_stacks_and_call(addr_t call, addr_t stack) */
FUNCTION(arch_switch_stacks_and_call)
	mov	sp, r1
	bx	r0

/* uint32_t arch_cycle_count(void); */
FUNCTION(arch_cycle_count)
#if defined(ARM_CPU_CORTEX_A8) || defined(ARM_CPU_CORTEX_A9) || defined(ARM_CPU_CORTEX_A15)
	mrc	p15, 0, r0, c9, c13, 0
#else
	mov	r0, #0
#endif
	bx	lr

/*void dsb(void) */
FUNCTION(dsb)
#if defined(ARM_CPU_CORTEX_A8) || defined(ARM_CPU_CORTEX_A9) || defined(ARM_CPU_CORTEX_A15)
	dsb		sy
#elif ARM_CPU_ARM1136
	mov	r0, #0
	mcr	p15, 0, r0, c7, c10, 4
#endif
	bx	lr

/* uint32_t arm_get_tls(void) */
FUNCTION(arm_get_tls)
	mrc	p15, 0, r0, c13, c0, 3
	bx	lr

/* void arm_set_tls(uint32_t) */
FUNCTION(arm_set_tls)
	mcr	p15, 0, r0, c13, c0, 3
	bx	lr

/* uint32_t arm_read_vbar(void) */
FUNCTION(arm_read_vbar)
	mrc	p15, 0, r0, c12, c0, 0
	bx	lr

/* void arm_write_vbar(uint32_t) */
FUNCTION(arm_write_vbar)
	mcr	p15, 0, r0, c12, c0, 0
	bx	lr
