/*
 * Copyright (c) 2014, NVIDIA CORPORATION. All rights reserved
 *
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files
 * (the "Software"), to deal in the Software without restriction,
 * including without limitation the rights to use, copy, modify, merge,
 * publish, distribute, sublicense, and/or sell copies of the Software,
 * and to permit persons to whom the Software is furnished to do so,
 * subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 * IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY
 * CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,
 * TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
 * SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

#include <stdlib.h>
#include <err.h>
#include <debug.h>
#include <assert.h>
#include <platform.h>
#include <platform/memmap.h>
#include <reg.h>
#include <string.h>
#include <psci.h>
#include <target/debugconfig.h>
#include <platform/platform_monitor.h>
#include <lib/monitor/monitor_vector.h>

/* referenced APBDEV_PMC_SECURE registers */
#define MC_SECURITY_CFG1_0		0x74

#define PMC_SECURE_DISABLE2		0x2c4
#define PMC_SECURE_DISABLE2_WRITE22_ON	(1 << 28)

#define PMC_SECURE_DISABLE3		0x2d8
#define PMC_SECURE_DISABLE3_WRITE34_ON	(1 << 20)
#define PMC_SECURE_DISABLE3_WRITE35_ON	(1 << 22)

#define PMC_SECURE_SCRATCH22		0x338
#define PMC_SECURE_SCRATCH34		0x368
#define PMC_SECURE_SCRATCH35		0x36c

#define EVP_CPU_RESET_VECTOR		0x100

#define SB_CSR				0x0
#define SB_CSR_NS_RST_VEC_WR_DIS	(1 << 1)

/* CPU reset vector */
#define SB_AA64_RESET_LOW		0x30	// width = 31:0
#define SB_AA64_RESET_HI		0x34	// width = 11:0

volatile uint32_t cpus_started;
static const uint32_t cpus_expected = ((1 << MONCPUS) - 1);

/* sets of MMIO ranges setup */
#define MMIO_RANGE_0_ADDR	0x50000000
#define MMIO_RANGE_1_ADDR	0x60000000
#define MMIO_RANGE_2_ADDR	0x70000000
#define MMIO_RANGE_SIZE		0x200000

#if defined(MONTARGET_DENVER)
/*
 * For T132, CPUs reset to AARCH32, so the reset vector is first
 * armv8_trampoline and which does a warm reset to AARCH64 and
 * starts execution at the address in SCRATCH34/SCRATCH35.
 */
static uint32_t armv8_trampoline[] __ALIGNED(8) = {
	0xE3A00003,		// mov	r0, #3
	0xEE0C0F50,		// mcr	p15, 0, r0, c12, c0, 2
	0xEAFFFFFE,		// b	.
};
#endif

static void psci_lock_reset_registers(void)
{
	uint32_t reg;

	/* ensure SECURE_SCRATCH34/35 are write locked */
	reg  = readl(TEGRA_PMC_BASE + PMC_SECURE_DISABLE3);
	reg |= (PMC_SECURE_DISABLE3_WRITE34_ON |
		PMC_SECURE_DISABLE3_WRITE35_ON);
	writel(reg, TEGRA_PMC_BASE + PMC_SECURE_DISABLE3);	/* lock */

	/* set secure boot control (read to flush) */
	reg  = readl(TEGRA_SB_BASE + SB_CSR);
	reg |= SB_CSR_NS_RST_VEC_WR_DIS;
	writel(reg, TEGRA_SB_BASE + SB_CSR);
	readl(TEGRA_SB_BASE + SB_CSR);

#if !defined(WITH_AA64_CPU_RESET_VECTORS)
	/* ensure SECURE_SCRATCH22 is write locked */
	reg  = readl(TEGRA_PMC_BASE + PMC_SECURE_DISABLE2);
	reg |= PMC_SECURE_DISABLE2_WRITE22_ON;
	writel(reg, TEGRA_PMC_BASE + PMC_SECURE_DISABLE2);	/* lock */
#endif
}

static void psci_program_reset_vectors(int cpu)
{
	uint64_t phys_cpu_reset;

#if WITH_AA64_CPU_RESET_VECTORS
	phys_cpu_reset = mon_virt_to_phys(&__mon_cpu_reset_vector);

	/* write lower 32 bits first, then the upper 11 bits */
	writel((phys_cpu_reset & 0xFFFFFFFF) | 1, TEGRA_SB_BASE + SB_AA64_RESET_LOW);
	phys_cpu_reset >>= 32;
	writel(phys_cpu_reset & 0x7FF, TEGRA_SB_BASE + SB_AA64_RESET_HI);

	psci_lock_reset_registers();
#else
	phys_cpu_reset = mon_virt_to_phys(armv8_trampoline);

	/* set exception vector (read to flush) */
	writel(phys_cpu_reset, TEGRA_EXCEPTION_VECTORS_BASE + EVP_CPU_RESET_VECTOR);
	readl(TEGRA_EXCEPTION_VECTORS_BASE + EVP_CPU_RESET_VECTOR);

	psci_lock_reset_registers();
#endif
}

static void psci_init_reset_vector(uint32_t cpu)
{
	uint64_t phys_cpu_reset;
	uint32_t reg;

#if !defined(WITH_AA64_CPU_RESET_VECTORS)
	/* SECURE_SCRATCH22 should be writable */
	reg  = readl(TEGRA_PMC_BASE + PMC_SECURE_DISABLE2);
	ASSERT(!(reg & PMC_SECURE_DISABLE2_WRITE22_ON));

	/* initial AARCH32 reset address */
	phys_cpu_reset = mon_virt_to_phys(armv8_trampoline);
	writel(phys_cpu_reset, TEGRA_PMC_BASE + PMC_SECURE_SCRATCH22);

	/* make sure all CPUs have reset before reprogramming vector */
	while (cpus_started != cpus_expected)
		;
#endif

	/* both SECURE_SCRATCH34/SCRATCH35 should be writable */
	reg  = readl(TEGRA_PMC_BASE + PMC_SECURE_DISABLE3);
	reg &= (PMC_SECURE_DISABLE3_WRITE34_ON | PMC_SECURE_DISABLE3_WRITE35_ON);
	ASSERT(!reg);

	/* set exception vector to be used to resume from suspend */
	phys_cpu_reset = mon_virt_to_phys(&__mon_cpu_reset_vector);
	writel(phys_cpu_reset & 0xFFFFFFFF, TEGRA_PMC_BASE + PMC_SECURE_SCRATCH34);
	phys_cpu_reset >>= 32;
	writel(phys_cpu_reset & 0x7FF, TEGRA_PMC_BASE + PMC_SECURE_SCRATCH35);

	psci_program_reset_vectors(cpu);
}

/*
 * One-time init called during cold boot from primary CPU
 */
void platform_psci_init(uint32_t cpu)
{
	/* identity map MMIO ranges for register access */
	mon_mmu_map_mmio(MMIO_RANGE_0_ADDR, MMIO_RANGE_0_ADDR, MMIO_RANGE_SIZE);
	mon_mmu_map_mmio(MMIO_RANGE_1_ADDR, MMIO_RANGE_1_ADDR, MMIO_RANGE_SIZE);
	mon_mmu_map_mmio(MMIO_RANGE_2_ADDR, MMIO_RANGE_2_ADDR, MMIO_RANGE_SIZE);

	platform_init_debug_port(DEFAULT_DEBUG_PORT);

	mon_atomic_or(&cpus_started, 1 << cpu);

	psci_init_reset_vector(cpu);

	platform_monitor_init_cpu();
	platform_init_memory(__mon_phys_base, __mon_phys_size);
	platform_config_interrupts();
	platform_psci_start(cpu);
}

/*
 * One time init call during end of cold boot from primary CPU
 */
void platform_psci_coldboot_epilog(void)
{
	/* mark entire TLK carveout as secure in the MC */
	platform_secure_dram_aperture();
}

/*
 * Routine is called when a CPU goes through reset, either a secondary
 * CPU during cold boot, or all CPUs during system suspend.
 */
void platform_psci_cpu_has_reset(uint32_t cpu)
{
	uint32_t reg;

	mon_atomic_or(&cpus_started, 1 << cpu);

#if defined(MONTARGET_ARM64)
	platform_psci_cpu_resume(cpu);
	platform_monitor_init_cpu();
	platform_psci_start(cpu);
#else
	/*
	 * Opportunity to do an per-CPU setup after reset. For now, only
	 * restores global state, so just run on the primary (CPU0).
	 */
	if (cpu != 0)
		return;
#endif

	/*
	 * Avoid relying on having seen an LP0 enter SMC.
	 *
	 * If MC_SECURITY_CFG1 has gone back zero (its POR value) then LP0
	 * has occurred (as it's not part of BL's warmboot restore) and system
	 * registers need to be reloaded.
	 */
	reg = readl(TEGRA_MC_BASE + MC_SECURITY_CFG1_0);
	if (reg == (__mon_phys_size >> 20))
		return;

	psci_program_reset_vectors(cpu);
	platform_restore_memory();
	platform_config_interrupts();
}

void platform_psci_handler(struct fastcall_frame *frame)
{
	uint64_t ret = PSCI_RETURN_SUCCESS;
	int curr_cpu, target_cpu;

	__asm__ volatile (
		"mrs	%0, mpidr_el1\n"
		"and	%0, %0, #0xf\n"
	       : "=r" (curr_cpu)
	);

	switch (frame->r[0]) {
	/*
	 * Runs on the actual CPU which is being suspended.
	 */
	case PSCI_FUNC_ID_CPU_SUSPEND_LEGACY:
	case PSCI_FUNC_ID_CPU_SUSPEND:
		/* save NS entry point */
		((uint64_t *)&__mon_cpu_return_addr)[curr_cpu] = frame->r[2];
#if !defined(MONTARGET_DENVER)
		ret = platform_psci_cpu_suspend(curr_cpu, frame->r[1]);
#endif
		break;

#if defined(MONTARGET_ARM64)
	/*
	 * Runs on a CPU other than the one we want to get online.
	 */
	case PSCI_FUNC_ID_CPU_ON:
		target_cpu = frame->r[1];
		((uint64_t *)&__mon_cpu_return_addr)[target_cpu] = frame->r[2];
		ret = platform_psci_cpu_on(target_cpu);
		break;

	/*
	 * Runs on the actual CPU which is being powered off.
	 */
	case PSCI_FUNC_ID_CPU_OFF:
		mon_atomic_and(&cpus_started, ~(1 << curr_cpu));
		platform_psci_cpu_off(curr_cpu,
			PSCI_POWER_STATE_TYPE_POWER_DOWN);
		break;
#endif

	default:
		ret = PSCI_RETURN_NOT_SUPPORTED;
	}

	frame->r[0] = ret;
}
