/*
 * Copyright (c) 2013-2014, NVIDIA CORPORATION. All rights reserved
 *
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files
 * (the "Software"), to deal in the Software without restriction,
 * including without limitation the rights to use, copy, modify, merge,
 * publish, distribute, sublicense, and/or sell copies of the Software,
 * and to permit persons to whom the Software is furnished to do so,
 * subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 * IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY
 * CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,
 * TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
 * SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */
#include <config.h>
#include <asm.h>
#include <arch/arm.h>
#include <arm64/asm.h>
#include <arm64/monitor_macros.h>

/* vector entry points */
.macro  vec_entry, label
	.align  7
	b       \label
.endm
.macro  vec_entry_null
	.align  7
	b       .
.endm

.section ".text.boot"
.align 11

/*
 * Vector offset 0x000 - 0x180 (from Current EL, using SP_EL0)
 */
FUNCTION(_start)
FUNCTION(_vector_el3)

	/* entry point */
	vec_entry	_reset

	/* EL3 exception to EL3 (SP_EL0) unexpected */
	vec_entry_null		/* IRQ exc SP_EL0 */
	vec_entry_null		/* FIQ exc SP_EL0 */
	vec_entry_null		/* SERR exc SP_EL0 */

/*
 * Vector offset 0x200 - 0x380 (from Current EL, using SP_ELx)
 */
.align 9
	/* EL3 exception to EL3 (SP_ELx) unexpected */
	vec_entry_null		/* sync exc SP_ELx */
	vec_entry_null		/* IRQ exc SP_ELx */
	vec_entry_null		/* FIQ exc SP_ELx */
	vec_entry_null		/* SERR exc SP_ELx */

/*
 * Vector offset 0x400 - 0x580 (from Lower EL, using AArch64)
 */
.align 9
	/* EL1 exception to EL3 (AArch64) */
	vec_entry	handle_exc_aarch64
	vec_entry_null			/* IRQ exc aarch64  - unexpected in EL3 */
	vec_entry	handle_fiq_aarch64	/* FIQ exc aarch64 */
	vec_entry_null			/* SERR exc aarch64 - unexpected in EL3 */

/*
 * Vector offset 0x600 - 0x680 (from Lower EL, using AArch32)
 */
.align 9
	/* EL1 exception to EL3 (AArch32) */
	vec_entry	handle_exc_aarch32
	vec_entry_null		/* IRQ exc aarch32 */
	vec_entry_null		/* FIQ exc aarch32 */
	vec_entry_null		/* SERR exc aarch32 */

/* void mon_clear_bss(void) */
FUNCTION(mon_clear_bss)
	adr	x0, __bss_start
	adr	x1, __bss_end
	subs	x1, x1, x0		/* size in bytes */
	b.eq	2f
1:
	strb	wzr, [x0], #1
	sub	x1, x1, #1
	cbnz	x1, 1b
2:
	ret

/*
 * Entry from the bootloader.
 *
 * Assuming for now the BL will jump to our _start rather than exception
 * to us (though the latter is doable, given our entry point is where
 * VBAR_EL3 would be setup, so could SMC to us).
 *
 * Setup state so we can return from the EL3 "exception" by pushing
 * the needed state onto the stack (popped during go_nonsecure).
 */
_reset:
	/* derive/save the phys_base addr */
	adr	x9, (. + 0x4)
	sub	x10, x9, #(. - _start)
	adr	x3, __mon_phys_base
	str	x10, [x3]		// physbase

	/* save mon_phys_offset (v -> p conversions) */
	adr	x4, __mon_phys_offset
	ldr	x5, =MONBASE
	sub	x11, x5, x10
	str	x11, [x4]		// physoff

	/* save per-cpu return addr (from LR) */
	cpuidx	x4
	adr	x5, __mon_cpu_return_addr
	str	lr, [x5, x4, lsl #3]

	/*
	 * All secondaries, init their CPU state and then return,
	 * while only the primary continues on to init the OS. For
	 * now, this means only CPU0 can run within the secureos.
	 */
	cbnz	x4, boot_secondary

	/* save carveout size and bootarg ptr */
	adr	x3, __mon_phys_size
	str	x0, [x3]
	mov	x2, x0			// make copy
	adr	x3, __mon_bootarg_addr
	str	x1, [x3]

	/* clear early while serialized */
	bl	mon_clear_bss

	bl	mon_init_cpu

	/* setup MMU pagetables, (args: base, off, size) */
	mov	x0, x10
	mov	x1, x11
	bl	mon_setup_pagetable

	/* enable MMU */
	bl	mon_enable_mmu

	/* reload vbar with virtual address */
	adr	x0, _vector_el3
	msr	vbar_el3, x0
	isb

	cpuidx	x0
	bl	platform_psci_init

#if defined(WITH_EL3_MONITOR_ONLY)
	bl	platform_secure_dram_aperture
	b	mon_return_aarch64_ns
#else
	b	mon_start_tlk_el1
#endif

.ltorg
.align 3
/*
 * Keep location of where the image was loaded, and how much
 * contiguous memory has been made available.
 */
.global __mon_phys_base
__mon_phys_base:
	.quad 0

.global __mon_phys_size
__mon_phys_size:
	.quad 0

.global __mon_bootarg_addr
__mon_bootarg_addr:
	.quad 0

.data
.align 4
/* stack must be 16 byte aligned */
monitor_stack:
	.skip 4096
.global monitor_stack_top
monitor_stack_top:
