/*
 * Copyright (c) 2012-2015, NVIDIA CORPORATION. All rights reserved
 *
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files
 * (the "Software"), to deal in the Software without restriction,
 * including without limitation the rights to use, copy, modify, merge,
 * publish, distribute, sublicense, and/or sell copies of the Software,
 * and to permit persons to whom the Software is furnished to do so,
 * subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 * IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY
 * CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,
 * TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
 * SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

#include <arch/arm/monitor_vectors.h>
#include <lib/monitor/monitor_vector.h>
#include <platform/platform_tos.h>

#define L1_CACHE_ALIGN		5

/* loads a 32-bit value into a register */
.macro mov32, reg, val
	movw	\reg, #:lower16:\val
	movt	\reg, #:upper16:\val
.endm

.macro RESTORE_NS_STATE
	/* restore monitor frame from arg */
	adr	r0, mon_stdcall_frame_addr
	ldr	r0, [r0]
	RESTORE_MON_FRAME_FROM_ARG r0
.endm

/*****************************************************************************/
/* The monitor entry table stuff                                             */
/*****************************************************************************/
.p2align 5
.globl monitor_vector_base
monitor_vector_base:
	nop				/* RESET	*/
	b	mon_undef_entry		/* UNDEF	*/
	b	mon_swi_entry		/* SWI		*/
	b	mon_inst_abort_entry	/* IABORT	*/
	b	mon_data_abort_entry	/* DABORT	*/
	nop				/* reserved	*/
	b	mon_irq_entry		/* IRQ		*/
	b	mon_fiq_entry		/* FIQ		*/

mon_undef_entry:
	b	.

mon_swi_entry:
	b	handle_exception

mon_inst_abort_entry:
	b	.

mon_data_abort_entry:
	b	.

mon_irq_entry:
	b	.

mon_fiq_entry:
	b	.

/*
 * Go to secure world, handle the exception and then switch back to non-secure
 * world.
 *
 */
handle_exception:
	stmfd	sp!, {r1-r2}		@ create scratch regs

	SWITCH_SCR_TO_SECURE r1

	/* check if is mmu is currently enabled */
	mrc	p15, 0, r1, c1, c0, 0
	tst	r1, #0x1
	bne	cont_handle_exc		@ enabled, just continue

	/* it won't be on exit from LP1, so handle it now */
	adr	r1, mon_p2v_offset
	ldr	r2, [r1]
	add	sp, sp, r2		@ convert sp to virt

	ldr	r2, =monitor_vector_base
	mcr	p15, 0, r2, c12, c0, 1	@ reload virt mvbar

	mrc	p15, 0, r2, c1, c0, 0
	orr	r2, r2, #1
	mcr	p15, 0, r2, c1, c0, 0	@ enable mmu

	ldr	r1, =cont_handle_exc
	mov	pc, r1			@ convert pc to virt

cont_handle_exc:
	tst	r0, #(1 << 31)		@ Check if fastcall
	bne	handle_fastcall

	/* continue for handling standard calls */
	SAVE_NONSECURE_STATE r1, r2	@ save NS state
	RESTORE_SECURE_STATE r1, r2	@ restore S state

	/* handle restart SMC */
	ldr	r1, =SMC_TOS_RESTART_LEGACY
	cmp	r0, r1			@ legacy restart SMC?
	beq	go_restart
	ldr	r1, =SMC_TOS_RESTART
	cmp	r0, r1			@ restart SMC?
	beq	go_restart

	/* if returning from FS req, skip CPU save (r1 is still scratch) */
	mov32	r1, 0x32000009
	cmp	r0, r1			@ fs completion?
	beq	go_restart

	ldmia	sp!, {r1-r2}		@ restore scratch regs

	/* save tz_monitor_frame to r0 */
	SAVE_MON_FRAME_TO_ARG mon_stdcall_frame_addr
	adr	r0, mon_stdcall_frame_addr
	ldr	r0, [r0]

return_go_nonsecure:
	/* frame in r0 (register not part of restored CPU state) */
	adr	r1, secure_exit_mode
	ldr	r2, [r1]		@ get previous exit mode
	msr	cpsr_cfsx, r2		@ switch to it
	ldmia	sp!, { r4-r12, pc }	@ restore CPU state from stack (and return)

go_restart:
	/* store new r14/spsr to use after RESTART SMC handling */
	ldmia	sp!, {r1-r2}		@ restore scratch regs
	adr	r0, mon_stdcall_frame_addr
	ldr	r0, [r0]
	str	r14, [r0, #0x68]
	mrs	r14, spsr
	str	r14, [r0, #0x70]
	mov	r0, #0			@ no incoming SMC
	b	return_go_nonsecure

/*
* Handle fast calls - keep interrupts disabled, do not save/restore secure
* world context, handle the SMC and return to non-secure world
*/
handle_fastcall:
	/* restore scratch regs */
	ldmia	sp!, {r1-r2}

	/* save monitor frame */
	SAVE_MON_FRAME_TO_ARG mon_fastcall_frame_addr

	/* handle the SMC, r0 = smc_frame */
	adr	r0, mon_fastcall_frame_addr
	ldr	r0, [r0]
	bl	tz_fastcall_handler

	SWITCH_SCR_TO_NONSECURE r0

	/* restore monitor frame */
	adr	r0, mon_fastcall_frame_addr
	ldr	r0, [r0]
	RESTORE_MON_FRAME_FROM_ARG r0
	movs	pc, r14

/*
 * Go to nonsecure world
 *
 */
.globl go_nonsecure
go_nonsecure:
	/* r0 = smc_type, r1 = frame ptr arg */
	stmfd	sp!, { r4-r12, lr }	@ save CPU state to stack

	adr	r2, secure_exit_mode
	mrs	r3, cpsr
	str	r3, [r2]		@ save current mode
	cpsid	iaf			@ disable intrs

	SAVE_SECURE_STATE r2, r3
	RESTORE_NONSECURE_STATE r2, r3
	SWITCH_SCR_TO_NONSECURE r2

	/* completions always restore from the stdcall stack */
	mov32	r2, SMC_TOS_COMPLETION
	cmp	r0, r2
	beq	restore_stdcall_state

	mov32	r2, SMC_TOS_PREEMPTED
	cmp	r0, r2
	beq	return_preempted

	/* for INITIAL_NS_RETURN restore from arg */
	RESTORE_MON_FRAME_FROM_ARG r1
	movs	pc, r14

restore_stdcall_state:
	RESTORE_NS_STATE
	movs	pc, r14

return_preempted:
	/* return the preempted-by code */
	ldr	r0, [r1]	/* load frame->r[0] */
	push	{ r0 }
	RESTORE_NS_STATE
	pop	{ r0 }
	movs	pc, r14

.globl mon_fastcall_frame_addr
mon_fastcall_frame_addr:
	.long	0

.globl mon_stdcall_frame_addr
mon_stdcall_frame_addr:
	.long	0

.align L1_CACHE_ALIGN

/* save area for NS and S worlds */
secure_state:
	.rept	SAVED_STATE_WORDS
	.long	0
	.endr

nonsecure_state:
	.rept	SAVED_STATE_WORDS
	.long	0
	.endr

/* cpsr mode from last secure world exit */
secure_exit_mode:
	.long	0

.globl	mon_stack_top
mon_stack_top:
	.long	0

.globl	mon_p2v_offset
mon_p2v_offset:
	.long	0
