/*
 * Copyright (c) 2008 Travis Geiselbrecht
 * Copyright (c) 2012, NVIDIA CORPORATION. All rights reserved
 *
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files
 * (the "Software"), to deal in the Software without restriction,
 * including without limitation the rights to use, copy, modify, merge,
 * publish, distribute, sublicense, and/or sell copies of the Software,
 * and to permit persons to whom the Software is furnished to do so,
 * subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 * IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY
 * CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,
 * TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
 * SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */
#include <asm.h>
#include <arch/arm.h>
#include <arch/arch_thread.h>
#include <kernel/task.h>

#if ARM_WITH_LPAE
#include <arch/arm/mmu_ldesc_macros.h>
#else
#include <arch/arm/mmu_sdesc_macros.h>
#endif

/* arm_make_thread_current(thread_t *thread) */
FUNCTION(arm_make_thread_current)
	ldr	r14, [r0, #THREAD_ARCH_TASK]
	cmp	r14, #0
	beq	page_table_current

	/* switch user page tables */
	ldr	r1, [r14, #TASK_PAGETABLE]
	ldr	r2, [r14, #TASK_CONTEXTID]
	mmu_desc_switch_pt r1, r2, r3, r4
	isb

#if ARM_WITH_NEON
	/* reload thread's FPEXC state */
	ldr	r1, [r0, #THREAD_ARCH_FPCTX]
	ldr	r2, [r1, #FPCTX_FPEXC]
	mcr	p10, 7, r2, c8, c0, 0
#endif

page_table_current:
	/* thread state (frame) kept on kernel stack */
	ldr	r14, [r0, #THREAD_ARCH_KERN_STACK]

	/* restore psr (and thread init into user mode) */
	ldr	r1, [r14, #CSF_OFFSET_PSR]
	and	r2, r1, #MODE_MASK
	cmp	r2, #MODE_USR
	beq	return_to_user

continue_in_svc:
	/* safe to set cpsr directly (same mode) */
	msr	cpsr_fsxc, r1

	ldmia	r14, { r0-r13 }
	add	r14, #CSF_OFFSET_LR
	ldmia	r14, { lr, pc }

return_to_user:
	/* setup next mode in shadow spsr */
	msr	spsr_fsxc, r1

	/* restores sp_usr/lr_usr from lr_svc (still context frame ptr) */
	ldmia	r14, { r0-r13, r14 }^

	/* set sp_svc also (syscalls use this stack) and return */
	add	r13, r14, #CSF_SIZE
	ldr	r14, [r14, #CSF_OFFSET_PC]
	movs	pc, r14


/* arm_context_switch(thread_t *old_thread, thread_t *new_thread) */
FUNCTION(arm_context_switch)
	/* save context frame ptr */
	sub	r3, sp, #CSF_SIZE
	str	r3, [r0, #THREAD_ARCH_KERN_STACK]

	/* fill in context frame */
	stmia	r3, { r0-r13, r14 }
	str	lr, [r3, #CSF_OFFSET_PC]
	mrs	r4, cpsr
	str	r4, [r3, #CSF_OFFSET_PSR]

	clrex
	mov	r0, r1
	b	arm_make_thread_current
.ltorg


FUNCTION(arm_save_mode_regs)
	mrs		r1, cpsr

#if ARM_ISA_ARMV6 || ARM_ISA_ARMV7
	cps		#0x11			/* fiq */
	str		r13, [r0], #4
	str		r14, [r0], #4
	cps		#0x12			/* irq */
	str		r13, [r0], #4
	str		r14, [r0], #4
	cps		#0x13			/* svc */
	str		r13, [r0], #4
	str		r14, [r0], #4
	cps		#0x17			/* abt */
	str		r13, [r0], #4
	str		r14, [r0], #4
	cps		#0x1b			/* und */
	str		r13, [r0], #4
	str		r14, [r0], #4
	cps		#0x1f			/* sys */
	str		r13, [r0], #4
	str		r14, [r0], #4
#else
	// XXX implement
	b		.
#endif
	msr		cpsr_c, r1
	bx		lr

