/*
 * Copyright (c) 2014, NVIDIA CORPORATION. All rights reserved
 *
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files
 * (the "Software"), to deal in the Software without restriction,
 * including without limitation the rights to use, copy, modify, merge,
 * publish, distribute, sublicense, and/or sell copies of the Software,
 * and to permit persons to whom the Software is furnished to do so,
 * subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 * IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY
 * CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,
 * TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
 * SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */
#include <assert.h>
#include <debug.h>
#include <platform.h>
#include <reg.h>

#include <platform/memmap.h>

/* fuse register information */
#define FUSECTRL_0			(0x0)
#define FUSEADDR_0			(0x4)
#define FUSERDATA_0			(0x8)
#define FUSEWDATA_0			(0xC)

#define FUSECTRL_STATE_MASK		(0x1F << 16)
#define FUSECTRL_STATE_IDLE		(0x4 << 16)

#define FUSECTRL_CMD_MASK		(0x3 << 0)
#define FUSECTRL_CMD_READ		(0x1)
#define FUSECTRL_CMD_WRITE		(0x2)
#define FUSECTRL_CMD_SENSE		(0x3)

#define FUSE_WRITE32(o,v) 		writel((v), TEGRA_FUSE_BASE + (o))
#define FUSE_READ32(o) 			readl(TEGRA_FUSE_BASE + (o))

/* car register information */
#define CLK_MASK_ARM_0			(0x48)
#define CLK_MASK_ARM_CFG_ALL_VISIBLE	(0x10000000)

#define RST_DEVICES_H_0			(0x8)
#define RST_DEVICES_H_SWR_FUSE_RST	(0x80)

#define CLK_OUT_ENB_H_0			(0x14)
#define CLK_OUT_ENB_H_CLK_ENB_FUSE	(0x80)

#define CAR_WRITE32(o,v) 		writel((v), TEGRA_CLK_RESET_BASE + (o))
#define CAR_READ32(o) 			readl(TEGRA_CLK_RESET_BASE + (o))

static void fuse_wait_for_idle(void)
{
	uint32_t data;

	do {
		spin(1);
		data = FUSE_READ32(FUSECTRL_0);
	} while ((data & FUSECTRL_STATE_MASK) != FUSECTRL_STATE_IDLE);
}

static void fuse_cmd_sense(void)
{
	uint32_t data;

	fuse_wait_for_idle();

	data = FUSE_READ32(FUSECTRL_0);
	data &= ~FUSECTRL_CMD_MASK;
	data |= FUSECTRL_CMD_SENSE;
	FUSE_WRITE32(FUSECTRL_0, data);

	fuse_wait_for_idle();
}

bool fuse_make_visible(void)
{
	uint32_t clk_mask_arm_data = 0;

	/* for now assume fuse block is out of reset with clock enabled */
	ASSERT(!(CAR_READ32(RST_DEVICES_H_0) & RST_DEVICES_H_SWR_FUSE_RST));
	ASSERT(CAR_READ32(CLK_OUT_ENB_H_0) & CLK_OUT_ENB_H_CLK_ENB_FUSE);

	/* make all fuse registers visible */
	clk_mask_arm_data = CAR_READ32(CLK_MASK_ARM_0);
	if ((clk_mask_arm_data & CLK_MASK_ARM_CFG_ALL_VISIBLE) == 0) {
		CAR_WRITE32(CLK_MASK_ARM_0,
			clk_mask_arm_data | CLK_MASK_ARM_CFG_ALL_VISIBLE);
		return true;
	}

	return false;
}

void fuse_make_invisible(void)
{
	uint32_t clk_mask_arm_data = 0;

	/* hide fuse registers */
	clk_mask_arm_data = CAR_READ32(CLK_MASK_ARM_0);
	clk_mask_arm_data &= ~CLK_MASK_ARM_CFG_ALL_VISIBLE;
	CAR_WRITE32(CLK_MASK_ARM_0, clk_mask_arm_data);

	return;
}

uint32_t fuse_read(uint32_t addr)
{
	uint32_t data;

	fuse_cmd_sense();

	fuse_wait_for_idle();

	FUSE_WRITE32(FUSEADDR_0, addr);
	data = FUSE_READ32(FUSECTRL_0);
	data &= ~FUSECTRL_CMD_MASK;
	data |= FUSECTRL_CMD_READ;
	FUSE_WRITE32(FUSECTRL_0, data);

	fuse_wait_for_idle();

	data = FUSE_READ32(FUSERDATA_0);

	return data;
}

