/*
 * Copyright (c) 2014, NVIDIA CORPORATION. All rights reserved
 *
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files
 * (the "Software"), to deal in the Software without restriction,
 * including without limitation the rights to use, copy, modify, merge,
 * publish, distribute, sublicense, and/or sell copies of the Software,
 * and to permit persons to whom the Software is furnished to do so,
 * subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 * IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY
 * CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,
 * TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
 * SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */
#ifndef __FUSE_EXT_H
#define __FUSE_EXT_H

/*
 * Fuse location information.
 */
#if defined(TARGET_T124) || defined(TARGET_T132)

#define ODM_PROD_ADDR			(0x0)
#define ODM_PROD_START_BIT		(11)

#define ODM_RESERVED0_ADDR		(0x2E)
#define ODM_RESERVED0_START_BIT		(5)

#elif defined(TARGET_T210) || defined(TARGET_T186)

#define ODM_PROD_ADDR			(0x0)
#define ODM_PROD_START_BIT		(11)

#define ODM_RESERVED0_ADDR		(0x2E)
#define ODM_RESERVED0_START_BIT		(17)

#else

#error unknown architecture

#endif

/*
 * The rollback protection fuses are in the ODM reserved fuse range.
 * ODM reserved fuses are 256bits wide.  To calculate the location of
 * a given fuse we need to determine its offset from ODM_RESERVED0's
 * address/start bit.
 *
 * Note that ODM reserved addresses increment by 2 (so if ODM_RESERVED0
 * is 0x2e then ODM_RESERVED1 is 0x30, etc).
 */

/* calculate ODM_RESERVED1[13:12] bit offset from start of reserved block */
#define RB_ENABLE_BIT			(ODM_RESERVED0_START_BIT + 32 + 12)
#define RB_KEY_PROGRAMMED_BIT		(ODM_RESERVED0_START_BIT + 32 + 13)

/* given bit offset from start of reserved block get address */
#define GET_ADDR(b)			(ODM_RESERVED0_ADDR + (((b) / 32) * 2))

/* given bit offset from start of reserved block get addr-relative start bit */
#define GET_START_BIT(b)		((b) % 32)

/* get fuse-specific address/start bit */
#define RB_ENABLE_ADDR			GET_ADDR(RB_ENABLE_BIT)
#define RB_ENABLE_START_BIT		GET_START_BIT(RB_ENABLE_BIT)

#define RB_KEY_PROGRAMMED_ADDR		GET_ADDR(RB_KEY_PROGRAMMED_BIT)
#define RB_KEY_PROGRAMMED_START_BIT	GET_START_BIT(RB_KEY_PROGRAMMED_BIT)

/* platform-dependent routines */
bool fuse_make_visible(void);
void fuse_make_invisible(void);
uint32_t fuse_read(uint32_t addr);

#endif
