/*
 * Copyright (c) 2012-2014, NVIDIA CORPORATION. All rights reserved
 *
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files
 * (the "Software"), to deal in the Software without restriction,
 * including without limitation the rights to use, copy, modify, merge,
 * publish, distribute, sublicense, and/or sell copies of the Software,
 * and to permit persons to whom the Software is furnished to do so,
 * subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 * IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY
 * CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,
 * TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
 * SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

#include <err.h>
#include <errno.h>
#include <debug.h>
#include <assert.h>
#include <malloc.h>
#include <string.h>
#include <platform.h>
#include <arch.h>
#include <stdlib.h>
#include <lib/heap.h>
#include <arch/outercache.h>
#include <platform/memmap.h>
#include <platform/tzrammap.h>
#include <platform/irqs.h>
#include <arch/arm.h>
#include <arch/arm/mmu.h>
#include <platform/platform_p.h>
#include <platform/platform_monitor.h>
#include <platform/platform_cpu.h>
#include <platform/platform_sip.h>
#include <ote_intf.h>

#define MONITOR_MODE_STACK_SZ	4096

/* referenced APBDEV_PMC_SECURE registers */
#define PMC_SECURE_DISABLE2	0x2c4
#define PMC_SECURE_SCRATCH22	0x338

/*
 * The PSCI spec from ARM states the following for power mgmt:
 * SMC         (r0) - CPU_SUSPEND  = 0x84000001
 * power_state (r1) - Bits [0:15]  = StateID
 *                    Bit  [16]    = StateType <0=Stdby, 1=PwrDn>
 *                    Bits [17:23] = MBZ
 *                    Bits [24:25] = AffinityLevel <0=cpu, 1=cluster>
 *                    Bits [26:31] = MBZ
 * entry_addr  (r2) - CPU wake up addr
 * context_id  (r3) - args in r0 when cpu wakes up from pwrdn state and enters
 *                    exception level of caller
 * returns     (r0) - SUCCESS/INVALID PARAM
 */
#define LP0				((1 << 16) | (1 << 24) | 1)
#define LP1				((1 << 16) | 2)
#define LP1_MC_CLK_STOP			((1 << 16) | 3)
#define LP2_CLUSTER_PWR_DN		((1 << 16) | (1 << 24) | 4)
#define LP2_CLUSTER_PWR_DN_LEGACY	((1 << 16) | 4)
#define LP2_NO_FLUSH_LEGACY		((1 << 16) | 5)

extern unsigned int monitor_vector_base;
extern unsigned long mon_stack_top;
extern unsigned long _ns_resume_addr;
extern unsigned long boot_secondary_cpu_addr;

/* location in NS to boot cpus */
unsigned long _ns_addr_secondary_cpus = 0;

unsigned int cpu_power_down_mode = 0;

/* tracks if we need to load resume handlers into tzram */
static bool load_tzram_lp1_resume_handler = true;

static void pm_set_monitor_stack(void)
{
	void *stack_top_mon;
	int stack_size = MONITOR_MODE_STACK_SZ;

	stack_top_mon = heap_alloc(stack_size, 0);
	if (stack_top_mon)
	{
		__asm__ volatile (
			"mrs	r2, cpsr	\n" // save current mode
			"cps	#0x16		\n" // change to monitor mode
			"mov	sp, %0		\n" // set mon_sp
			"add	sp, sp, %1	\n" // set mon_sp
			"msr	cpsr, r2	\n" // restore previous mode
			: : "r" (stack_top_mon),
			    "r" (stack_size / 2) : "memory"
		);
	} else {
		panic("no memory available for monitor stack");
	}

	memset(stack_top_mon, 0, stack_size);
	mon_stack_top = (unsigned long)(stack_top_mon + (stack_size / 2));
}

static void pm_set_mvbar(unsigned int val)
{
	__asm__ volatile (
		"mcr	p15, 0, %0, c12, c0, 1		\n"
		: : "r" (val)
	);
}

static void pm_set_reset_vector(unsigned long vector_addr)
{
	uint32_t evp_cpu_reset = TEGRA_EXCEPTION_VECTORS_BASE + 0x100;
	uint32_t sb_ctrl = TEGRA_SB_BASE;
	uint32_t reg;

	/* set new reset vector */
	*(volatile uint32_t *)evp_cpu_reset = (uint32_t)vector_addr;
	__asm__ volatile ("dmb" : : : "memory");

	/* dummy read to ensure successful write */
	reg = *(volatile uint32_t *)evp_cpu_reset;

	/* lock reset vector */
	reg = *(volatile uint32_t *)sb_ctrl;
	reg |= 2;
	*(volatile uint32_t *)sb_ctrl = reg;
	__asm__ volatile ("dmb" : : : "memory");
}

static void pm_set_nsacr(void)
{
	/* let normal world enable SMP, lock TLB, access CP10/11 */
	__asm__ volatile (
		"mrc	p15, 0, r0, c1, c1, 2		@ NSACR \n"
		"orr	r0, r0, #(0x00000C00)			\n"
		"orr	r0, r0, #(0x00060000)			\n"
		"mcr	p15, 0, r0, c1, c1, 2		@ NSACR \n"
		::: "r0"
	);
}

void pm_early_init(void)
{
	/* set monitor vector base address (use vaddr, since MMU's enabled) */
	pm_set_mvbar((unsigned int)&monitor_vector_base);

	/* populate the reset vector to boot all the secondary cpus */
	pm_set_reset_vector(virtual_to_physical(boot_secondary_cpu_addr));
}

/* LP1 resume code */
extern uint32_t _lp1_resume;
extern uint32_t _lp1_resume_end;

#define LP1_RESUME_HANDLER_SIZE \
	((uint32_t)&_lp1_resume_end - (uint32_t)&_lp1_resume)

static void pm_load_tzram_lp1_resume_handler(void)
{
	ASSERT(LP1_RESUME_HANDLER_SIZE < TEGRA_TZRAM_SIZE);

	memcpy((void *)(TZRAM_LP1_RESUME_HANDLER),
	       (void *)&_lp1_resume,
	       LP1_RESUME_HANDLER_SIZE);
}

static void pm_save_lp1_context(void)
{
	/* store any state needed for lp1 resume to tzram */
	TZRAM_STORE(TZRAM_BOOT_SECONDARY_CPU_ADDR,
		virtual_to_physical(boot_secondary_cpu_addr));
	TZRAM_STORE(TZRAM_MON_STACK_TOP, virtual_to_physical(mon_stack_top));
	TZRAM_STORE(TZRAM_NS_RESUME_ADDR, _ns_resume_addr);
	TZRAM_STORE(TZRAM_MVBAR, virtual_to_physical(&monitor_vector_base));

	cpu_copy_context((void *)TZRAM_CPU_CONTEXT);
}

void pm_init(void)
{
	extern unsigned long mon_p2v_offset;
	extern uint32_t _boot_secondary_phys_base;
	extern uint32_t __load_phys_size;
	uint32_t reg;

	pm_early_init();

	/* set monitor vector stack */
	pm_set_monitor_stack();

	/* set normal world access in NSACR */
	pm_set_nsacr();

	/* save mmu setup used to bring up secondary cpus */
	cpu_save_context();

	/* save off values to help with v-to-p operations */
	_boot_secondary_phys_base = __load_phys_base;
	mon_p2v_offset = (VMEMBASE - __load_phys_base);

	/* install the cpu resume handler to PMC_SEC_SCRATCH22 */
	reg = readl(TEGRA_PMC_BASE + PMC_SECURE_DISABLE2);
	writel(reg & ~(1 << 28), TEGRA_PMC_BASE + PMC_SECURE_DISABLE2);	/* unlock */

	writel(virtual_to_physical(boot_secondary_cpu_addr), TEGRA_PMC_BASE + PMC_SECURE_SCRATCH22);

	reg = readl(TEGRA_PMC_BASE + PMC_SECURE_DISABLE2);
	writel(reg | 1 << 28, TEGRA_PMC_BASE + PMC_SECURE_DISABLE2);	/* lock */

	platform_init_memory(__load_phys_base, __load_phys_size);
	platform_config_interrupts();
}

/*
 * Stubs for routines not used by all platforms
 */
__WEAK void pm_handle_smc_l2(unsigned int smc)
{
	dprintf(CRITICAL, "stubbed L2 SMC handler (shouldn't have been issued)\n");
	return;
}

__WEAK void pm_handle_smc_deep_sleep(void)
{
	return;
}

/*
 * Suspend-related SMCs.
 */
static void pm_handle_lp0_suspend_smc(struct tz_monitor_frame *frame)
{
	cpu_power_down_mode = CPU_IN_LP0;

	/* store passed in non-secure resume handler address */
	_ns_resume_addr = frame->r[2];

	/* set our LP0 reset vector */
	pm_set_reset_vector(virtual_to_physical(boot_secondary_cpu_addr));

	/* save off current state */
	cpu_save_context();

	/* need to reload LP1 handler into tzram before next LP1 suspend */
	load_tzram_lp1_resume_handler = true;

	/* handle any chip-specific steps */
	pm_handle_smc_deep_sleep();

	platform_disable_debug_intf();

	/* flush/disable dcache last */
	arch_disable_cache(DCACHE);
}

static void pm_handle_lp1_suspend_smc(struct tz_monitor_frame *frame)
{
	cpu_power_down_mode = CPU_IN_LP1;

	/* store passed in non-secure resume handler address */
	_ns_resume_addr = frame->r[2];

	/* set our LP1 reset vector */
	pm_set_reset_vector(TZRAM_LP1_RESUME_HANDLER);

	/* save off current state */
	cpu_save_context();

	/* save off state needed by LP1 resume handler */
	TZRAM_STORE(TZRAM_CPU_AVOID_CLKM_SWITCH,
		frame->r[1] == LP1_MC_CLK_STOP);
	TZRAM_STORE(TZRAM_BOOT_SECONDARY_CPU_ADDR,
		virtual_to_physical(boot_secondary_cpu_addr));
	pm_save_lp1_context();

	/* load LP1 resume handler to TZRAM if necessary */
	if (load_tzram_lp1_resume_handler) {
		pm_load_tzram_lp1_resume_handler();
		load_tzram_lp1_resume_handler = false;
	}

	/* flush/disable dcache last */
	arch_disable_cache(DCACHE);
}

static void pm_handle_lp2_suspend_smc(struct tz_monitor_frame *frame)
{
	cpu_power_down_mode = CPU_IN_LP2;

	/* set our LP2 reset vector */
	pm_set_reset_vector(virtual_to_physical(boot_secondary_cpu_addr));

	/* save off current state */
	cpu_save_context();

	/* flush/disable dcache last */
	arch_disable_cache(DCACHE);
}

/*
 * System related SMCs handled on the current idle stack.
 * These should be simple operations that can't block.
 */
void pm_handle_platform_smc(struct tz_monitor_frame *frame)
{
	int error = 0;

	switch (frame->r[0]) {
		case SMC_SIP_L2_MANAGEMENT:
			pm_handle_smc_l2(frame->r[1]);
			break;

		case SMC_SIP_CPU_RESET_VECTOR_LEGACY:
		case SMC_SIP_CPU_RESET_VECTOR:
			_ns_addr_secondary_cpus = frame->r[1];
			if (frame->r[1] == 0)
				_ns_addr_secondary_cpus = frame->r[2];
#if defined(ARM_USE_CPU_CACHING)
			platform_clean_invalidate_cache_range(
				(vaddr_t)&_ns_addr_secondary_cpus,
				sizeof(_ns_addr_secondary_cpus));
#endif
			platform_enable_debug_intf();
			break;

		case SMC_SIP_DEVICE_SUSPEND:
			switch (frame->r[1]) {
			case LP2_CLUSTER_PWR_DN:
			case LP2_CLUSTER_PWR_DN_LEGACY:
			case LP2_NO_FLUSH_LEGACY:
				pm_handle_lp2_suspend_smc(frame);
				break;
			case LP1:
			case LP1_MC_CLK_STOP:
				pm_handle_lp1_suspend_smc(frame);
				break;
			case LP0:
				pm_handle_lp0_suspend_smc(frame);
				break;
			default:
				error = -EINVAL;
				break;
			}
			break;
	}
	frame->r[0] = error;
}
