/*
 * Copyright (c) 2014, NVIDIA CORPORATION. All rights reserved
 *
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files
 * (the "Software"), to deal in the Software without restriction,
 * including without limitation the rights to use, copy, modify, merge,
 * publish, distribute, sublicense, and/or sell copies of the Software,
 * and to permit persons to whom the Software is furnished to do so,
 * subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 * IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY
 * CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,
 * TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
 * SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */
#include <assert.h>
#include <debug.h>
#include <string.h>
#include <platform.h>

#include <platform/platform_fuse.h>

/* fuse values table */
static uint32_t fuse_values[FUSE_ID_MAX];
static bool fuse_init_done;

void platform_fuse_init(void)
{
	uint32_t fuse_data;
	bool made_visible;

	/* expose fuse registers for access */
	made_visible = fuse_make_visible();

	/* clear fuse values */
	memset(fuse_values, 0, sizeof(fuse_values));

	/* save odm production fuse value */
	fuse_data = fuse_read(ODM_PROD_ADDR);
	if (fuse_data & (1 << ODM_PROD_START_BIT))
		fuse_values[FUSE_ID_ODM_PROD] = true;

	/* save rollback protection enable fuse value */
	fuse_data = fuse_read(RB_ENABLE_ADDR);
	if (fuse_data & (1 << RB_ENABLE_START_BIT))
		fuse_values[FUSE_ID_RB_ENABLE] = true;

	/* save rollback protection key programmed fuse value */
	fuse_data = fuse_read(RB_KEY_PROGRAMMED_ADDR);
	if (fuse_data & (1 << RB_KEY_PROGRAMMED_START_BIT))
		fuse_values[FUSE_ID_RB_KEY_PROGRAMMED] = true;

	dprintf(INFO, "%s: odm_prod %d rpb_en %d rpb_key_prgrm %d\n",
		__func__,
		fuse_values[FUSE_ID_ODM_PROD],
		fuse_values[FUSE_ID_RB_ENABLE],
		fuse_values[FUSE_ID_RB_KEY_PROGRAMMED]);

	/* move fuses back to invisible if necessary */
	if (made_visible)
		fuse_make_invisible();

	fuse_init_done = true;

	return;
}

uint32_t platform_fuse_get(enum fuse_ids fuse_id)
{
	ASSERT(fuse_init_done);

	if (fuse_id >= FUSE_ID_MAX)
		return false;

	return fuse_values[fuse_id];
}
