/*
 * Copyright (c) 2013-2014, NVIDIA CORPORATION. All rights reserved
 *
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files
 * (the "Software"), to deal in the Software without restriction,
 * including without limitation the rights to use, copy, modify, merge,
 * publish, distribute, sublicense, and/or sell copies of the Software,
 * and to permit persons to whom the Software is furnished to do so,
 * subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 * IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY
 * CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,
 * TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
 * SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */
#include <asm.h>
#include <arm64/asm.h>
#include <arm64/monitor_macros.h>
#include <lib/monitor/monitor_vector.h>

#define EC_BIT_POS		26
#define EC_WIDTH		6

#define	EC_SMC_AARCH32		0x13
#define	EC_SMC_AARCH64		0x17

#define FIQ_STATE_ELR_SPSR	0
#define FIQ_STATE_SP_EL0_SP_EL1	16
#define FIQ_STATE_PC_CPSR	32

.macro mon_handle_aarch_switch, scr, scr2
	ldr	\scr, =SMC_SIP_AARCH_SWITCH
	cmp	\scr, x0
	b.ne	1f	// continue, not SMC_SIP_AARCH_SWITCH

	/* save address/mode and use during return */
	cpuidx  \scr
	adr	\scr2, __mon_cpu_return_addr
	str	x1, [\scr2, \scr, lsl #3]
	adr	\scr2, __mon_cpu_return_mode
	str	x2, [\scr2, \scr, lsl #3]

	ldp	\scr, \scr2, [sp], #16	// restore scratch
	mov	x0, xzr			// return success

	cbz	x2, mon_return_aarch64_ns
	b	mon_return_aarch32_ns
1:
.endm

.macro mon_handle_get_fiq_regs, scr, scr2
	ldr	\scr, =SMC_SIP_GET_FIQ_REGS
	cmp	\scr, x0
	b.ne	1f	// continue, not SMC_SIP_GET_FIQ_REGS

	/* regs saved during a fiq exception */
	adr	\scr, __fiq_state
	ldp	x0, x1, [\scr, #FIQ_STATE_PC_CPSR]
	ldp	x2, x3, [\scr, #FIQ_STATE_SP_EL0_SP_EL1]

	/* switch non-secure for return */
	mon_scr_non_secure_64 \scr

	ldp	\scr, \scr2, [sp], #16	// restore scratch
	eret
1:
.endm

/*
 * Occurs from either non-secure/secure EL1.
 *
 * If coming from the non-secure world, we may have to save more state
 * depending on if we're going to transition to EL1 in secure mode. This
 * is needed if passing along a standard call SMC, which launches a
 * separate thread.
 *
 * If coming from S-EL1, detectable by NS=0 already, then just need to
 * save the GP regs (all the other NS-EL1 state has been saved already).
 *
 * Args are passed in x0-x6 and x18-x30 are callee saved.
 */
FUNCTION(handle_exc_aarch32)
	stp	x9, x10, [sp, #-16]!	// create scratch

	/* get exception code */
	mrs	x9, esr_el3
	ubfx	x9, x9, #EC_BIT_POS, #EC_WIDTH

	cmp	x9, #EC_SMC_AARCH32
	b.ne	.		// fail

	/* check for 32bit Trusted OS (TOS) calls */
	and	x9, x0, #(SMC_OWNER_MASK << SMC_OWNER_SHIFT)
	ldr	x10, =SMC_TOS_CALL
	cmp	x9, x10
	b.eq	handle_trusted_os_call

	/* service NS SIP call to switch aarch */
	mon_handle_aarch_switch x9, x10
	b	.		// unrecognized aarch32 SMC

.weak handle_trusted_os_call
handle_trusted_os_call:
fail:
	mvn	x0, xzr		// return -1
	eret

/*
 * FIQ exception handler for the monitor mode.
 *
 * We service the FIQ exceptions in the secure world and then jump to the NS
 * world. The NS world has already registered its handler with us during
 * boot. We store the necessary FIQ context before returning to the NS world
 * handler. The saved FIQ context is requested by the NS world as a response to
 * SMC_SIP_GET_FIQ_REGS function ID.
 */
FUNCTION(handle_fiq_aarch64)
	stp	x9, x10, [sp, #-16]!	// create scratch
	stp	x11, x12, [sp, #-16]!	// create scratch

	adr	x11, __fiq_state
	mrs	x9, elr_el1
	mrs	x10, spsr_el1
	stp	x9, x10, [x11, #FIQ_STATE_ELR_SPSR]
	mrs	x9, sp_el0
	mrs	x10, sp_el1
	stp	x9, x10, [x11, #FIQ_STATE_SP_EL0_SP_EL1]
	mrs	x9, elr_el3
	mrs	x10, spsr_el3
	stp	x9, x10, [x11, #FIQ_STATE_PC_CPSR]

	adr	x9, __mon_cpu_fiq_glue
	ldr	x9, [x9]
	cbz	x9, .			// NS world's handler not present
	msr	elr_el3, x9
	mov	x9, #(MON_SCR_RESV1 | MON_SCR_64BIT | MON_SCR_NS_MODE)
	msr	scr_el3, x9

	ldp	x11, x12, [sp], #16	// restore scratch
	ldp	x9, x10, [sp], #16	// restore scratch
	eret

/*
 * SMCs from 64bit non-secure world.
 *
 * This would be the path for PSCI calls and Trusted OS SMCs interfacing
 * with secure TAs. For PSCI, they'd be serviced here, but Trusted OS
 * SMCs need to transition to the secureos in EL1.
 */
FUNCTION(handle_exc_aarch64)
	stp	x9, x10, [sp, #-16]!	// create scratch

	/* get exception code */
	mrs	x9, esr_el3
	ubfx	x10, x9, #EC_BIT_POS, #EC_WIDTH

	cmp	x10, #EC_SMC_AARCH64
	b.ne	.			// fail

	/* for now, only expect non-secure EL1 */
	mrs	x9, scr_el3
	tst	x9, #1
	b.eq	secure_el1_smc64	// fail
	b	non_secure_el1_smc64

	/*
	 * SMCs from 64bit secure world.
	 * Currently these are unexpected as secureos runs in a 32bit mode.
	 */
secure_el1_smc64:
	b	.
	ldp	x9, x10, [sp], #16	// restore scratch
	movn	x0, #0
	eret

	/*
	 * SMCs from 64bit non-secure world.
	 *
	 * This is the path for both monitor fastcalls (i.e. those serviced
	 * entirely within the monitor) and Trusted OS SMCs interfacing with
	 * secure TAs (which call the secureos in EL1 for handling).
	 */
non_secure_el1_smc64:
	/* service NS SIP call to get regs saved during FIQ excptn */
	mon_handle_get_fiq_regs x9, x10

	/* service NS SIP call to switch aarch */
	mon_handle_aarch_switch x9, x10

	/* handle fastcall SMCs */
	tst	x0, #SMC_FASTCALL
	b.ne	monitor_fastcall

	b non_secure_stdcall

.weak non_secure_stdcall
non_secure_stdcall:
	mvn	x0, xzr		// return -1
	eret

.align 3
.global __mon_cpu_fiq_glue
__mon_cpu_fiq_glue:
	.quad 0

.global __mon_cpu_fiq_ns_stack
__mon_cpu_fiq_ns_stack:
	.quad 0

__fiq_state:
	.quad 0, 0, 0, 0, 0, 0
