/*
 * Copyright (c) 2014, NVIDIA CORPORATION. All rights reserved
 *
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files
 * (the "Software"), to deal in the Software without restriction,
 * including without limitation the rights to use, copy, modify, merge,
 * publish, distribute, sublicense, and/or sell copies of the Software,
 * and to permit persons to whom the Software is furnished to do so,
 * subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 * IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY
 * CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,
 * TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
 * SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

#include <stdlib.h>
#include <err.h>
#include <debug.h>
#include <platform.h>
#include <platform/platform_monitor.h>
#include <platform/platform_sip.h>
#include <lib/monitor/monitor_vector.h>

#define CORTEX_A57	0xD07
#define CORTEX_A53	0xD03

#define L2ACTLR		(1 << 6)
#define L2ECTLR		(1 << 5)
#define L2CTLR		(1 << 4)
#define CPUECTLR	(1 << 1)
#define CPUACTLR	(1 << 0)
#define ACTLR_MODE	(L2ACTLR | L2ECTLR | L2CTLR | CPUECTLR | CPUACTLR)

extern uint64_t __mon_cpu_fiq_glue, __mon_cpu_fiq_ns_stack;

void platform_sip_handler(struct fastcall_frame *frame, uint32_t cpu)
{
	switch (frame->r[0]) {
	case SMC_SIP_PROGRAM_VPR:
		frame->r[0] = platform_program_vpr(frame->r[1], frame->r[2]);
		break;

	case SMC_SIP_REGISTER_FIQ_GLUE:
		__mon_cpu_fiq_glue = frame->r[1];
		__mon_cpu_fiq_ns_stack = frame->r[2];
		frame->r[0] = NO_ERROR;
		break;

	default:
		frame->r[0] = ERR_NOT_SUPPORTED;
	}
}

/* Implementation specific CPU init */
void platform_monitor_init_cpu(void)
{
	uint32_t val = ACTLR_MODE;
	int cpu = mon_get_cpu_id();

	/* enable L2 and CPU ECTLR RW access from non-secure world */
	if (cpu == CORTEX_A57 || cpu == CORTEX_A53) {
		__asm__ volatile (
			"msr	actlr_el3, %0	\n"
			"msr	actlr_el2, %0	\n"
			:: "r" (val)
		);
	}
}
