/*
 * Copyright (c) 2014, NVIDIA CORPORATION. All rights reserved
 *
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files
 * (the "Software"), to deal in the Software without restriction,
 * including without limitation the rights to use, copy, modify, merge,
 * publish, distribute, sublicense, and/or sell copies of the Software,
 * and to permit persons to whom the Software is furnished to do so,
 * subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 * IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY
 * CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,
 * TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
 * SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

#include <stdlib.h>
#include <err.h>
#include <debug.h>
#include <platform.h>
#include <platform/memmap.h>
#include <reg.h>
#include <string.h>
#include <psci.h>
#include <target/debugconfig.h>
#include <platform/platform_p.h>
#include <lib/monitor/monitor_vector.h>

/* referenced APBDEV_PMC_SECURE registers */
#define MC_SECURITY_CFG1_0		0x74

#define PMC_SECURE_DISABLE2		0x2c4
#define PMC_SECURE_DISABLE2_WRITE22_ON	(1 << 28)
#define PMC_SECURE_SCRATCH22		0x338

#define EVP_CPU_RESET_VECTOR		0x100

#define SB_CSR				0x0
#define SB_CSR_NS_RST_VEC_WR_DIS	(1 << 1)

static volatile uint32_t cpus_started;
static const uint32_t cpus_expected = ((1 << MONCPUS) - 1);

/* sets of MMIO ranges setup */
#define MMIO_RANGE_0_ADDR	0x50000000
#define MMIO_RANGE_1_ADDR	0x60000000
#define MMIO_RANGE_2_ADDR	0x70000000
#define MMIO_RANGE_SIZE		0x200000

static void psci_program_reset_vectors()
{
	uint64_t phys_cpu_reset;
	uint32_t reg;

	phys_cpu_reset = mon_virt_to_phys(&__mon_cpu_reset_vector);

	/* set exception vector (read to flush) */
	writel(phys_cpu_reset, TEGRA_EXCEPTION_VECTORS_BASE + EVP_CPU_RESET_VECTOR);
	(void)readl(TEGRA_EXCEPTION_VECTORS_BASE + EVP_CPU_RESET_VECTOR);

	/* ensure SECURE_SCRATCH22 is write locked */
	reg  = readl(TEGRA_PMC_BASE + PMC_SECURE_DISABLE2);
	reg |= PMC_SECURE_DISABLE2_WRITE22_ON;
	writel(reg, TEGRA_PMC_BASE + PMC_SECURE_DISABLE2);	/* lock */

	/* set secure boot control (read to flush) */
	reg  = readl(TEGRA_SB_BASE + SB_CSR);
	reg |= SB_CSR_NS_RST_VEC_WR_DIS;
	writel(reg, TEGRA_SB_BASE + SB_CSR);
	(void)readl(TEGRA_SB_BASE + SB_CSR);
}

/*
 * One-time init called during cold boot from primary CPU
 */
void platform_psci_init(uint32_t cpu)
{
	uint64_t phys_cpu_reset;
	uint32_t reg;

	/* identity map MMIO ranges for register access */
	mon_mmu_map_mmio(MMIO_RANGE_0_ADDR, MMIO_RANGE_0_ADDR, MMIO_RANGE_SIZE);
	mon_mmu_map_mmio(MMIO_RANGE_1_ADDR, MMIO_RANGE_1_ADDR, MMIO_RANGE_SIZE);
	mon_mmu_map_mmio(MMIO_RANGE_2_ADDR, MMIO_RANGE_2_ADDR, MMIO_RANGE_SIZE);

	platform_init_debug_port(DEFAULT_DEBUG_PORT);

	reg  = readl(TEGRA_PMC_BASE + PMC_SECURE_DISABLE2);
	reg &= ~PMC_SECURE_DISABLE2_WRITE22_ON;
	writel(reg, TEGRA_PMC_BASE + PMC_SECURE_DISABLE2);	/* unlock */

	phys_cpu_reset = mon_virt_to_phys(&__mon_cpu_reset_vector);
	writel(phys_cpu_reset, TEGRA_PMC_BASE + PMC_SECURE_SCRATCH22);

	reg  = readl(TEGRA_PMC_BASE + PMC_SECURE_DISABLE2);
	reg |= PMC_SECURE_DISABLE2_WRITE22_ON;
	writel(reg, TEGRA_PMC_BASE + PMC_SECURE_DISABLE2);	/* lock */

	mon_atomic_or(&cpus_started, 1 << cpu);

	platform_monitor_init_cpu();
	platform_init_memory(__mon_phys_base, __mon_phys_size);
	platform_config_interrupts();
}

/*
 * One time init call during end of cold boot from primary CPU
 */
void platform_psci_coldboot_epilog()
{
	/* mark entire TLK carveout as secure in the MC */
	platform_secure_dram_aperture();

#if !defined(WITH_ARM_PSCI_SUPPORT)
	/* make sure all CPUs have reset before reprogramming vector */
	while (cpus_started != cpus_expected)
		;
#endif

	psci_program_reset_vectors();
}

/*
 * Routine is called when a CPU goes through reset, either a secondary
 * CPU during cold boot, or all CPUs during system suspend.
 */
void platform_psci_cpu_has_reset(uint32_t cpu)
{
	uint32_t reg;

	mon_atomic_or(&cpus_started, 1 << cpu);

	/*
	 * Opportunity to do an per-CPU setup after reset. For now, only
	 * restores global state, so just run on the primary (CPU0).
	 */
	if (cpu != 0)
		return;

	/*
	 * Avoid relying on having seen an LP0 enter SMC.
	 *
	 * If MC_SECURITY_CFG1 has gone back zero (its POR value) then LP0
	 * has occurred (as it's not part of BL's warmboot restore) and system
	 * registers need to be reloaded.
	 */
	reg = readl(TEGRA_MC_BASE + MC_SECURITY_CFG1_0);
	if (reg == (__mon_phys_size >> 20))
		return;

#if !defined(WITH_ARM_PSCI_SUPPORT)
	/* make sure all CPUs have reset before reprogramming vector */
	while (cpus_started != cpus_expected)
		;
#endif

	psci_program_reset_vectors();
	platform_monitor_init_cpu();
	platform_restore_memory();
	platform_config_interrupts();
}

void platform_psci_handler(struct fastcall_frame *frame, uint32_t cpu)
{
	/* currently, only handling CPU_SUSPEND */
	if (frame->r[0] != PSCI_FUNC_ID_CPU_SUSPEND_LEGACY &&
	    frame->r[0] != PSCI_FUNC_ID_CPU_SUSPEND) {
		frame->r[0] = PSCI_RETURN_NOT_SUPPORTED;
		return;
	}

	/* save NS entry point */
	((uint64_t *)&__mon_cpu_return_addr)[cpu] = frame->r[2];

	cpus_started = 0;

	frame->r[0] = PSCI_RETURN_SUCCESS;
}
