/*
 * Copyright (c) 2014, NVIDIA CORPORATION. All rights reserved
 *
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files
 * (the "Software"), to deal in the Software without restriction,
 * including without limitation the rights to use, copy, modify, merge,
 * publish, distribute, sublicense, and/or sell copies of the Software,
 * and to permit persons to whom the Software is furnished to do so,
 * subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 * IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY
 * CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,
 * TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
 * SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */
#include <sys/types.h>
#include <assert.h>
#include <debug.h>
#include <err.h>
#include <reg.h>
#include <string.h>
#include <kernel/thread.h>
#include <platform/interrupts.h>
#include <arch/ops.h>
#include <arch/arm.h>
#include <platform/memmap.h>
#include <platform/irqs.h>
#include <platform/platform_p.h>
#include <lib/monitor/monitor_vector.h>

#define GIC_DIST_CTR	0x004
#define GIC_CPU_ICCIAR	0x00c

#define ICTLR_CPU_IEP_VFIQ	0x08
#define ICTLR_CPU_IEP_FIR	0x14
#define ICTLR_CPU_IEP_FIR_SET	0x18
#define ICTLR_CPU_IEP_FIR_CLR	0x1c

#define ICTLR_CPU_IER		0x20
#define ICTLR_CPU_IER_SET	0x24
#define ICTLR_CPU_IER_CLR	0x28
#define ICTLR_CPU_IEP_CLASS	0x2C

#define ICTLR_COP_IER		0x30
#define ICTLR_COP_IER_SET	0x34
#define ICTLR_COP_IER_CLR	0x38
#define ICTLR_COP_IEP_CLASS	0x3c

static uint32_t num_ictrlrs;

static uintptr_t ictlr_reg_base[] = {
	TEGRA_PRIMARY_ICTLR_BASE,
	TEGRA_SECONDARY_ICTLR_BASE,
	TEGRA_TERTIARY_ICTLR_BASE,
	TEGRA_QUATERNARY_ICTLR_BASE,
	TEGRA_QUINARY_ICTLR_BASE,
};

void platform_init_interrupts(void)
{
	uint32_t i;

	num_ictrlrs = *REG32(TEGRA_ARM_INT_DIST_BASE + GIC_DIST_CTR);
	num_ictrlrs &= 0x1f;

	for (i = 0; i < num_ictrlrs; i++) {
		uintptr_t ictrl = ictlr_reg_base[i];
		*REG32(ictrl + ICTLR_CPU_IER_CLR) = ~0;
		*REG32(ictrl + ICTLR_CPU_IEP_CLASS) = 0;
		*REG32(ictrl + ICTLR_CPU_IEP_FIR_CLR) = ~0;
		*REG32(ictrl + ICTLR_CPU_IER_SET) = ~0;
		*REG32(ictrl + ICTLR_COP_IER_CLR) = ~0;
	}
}

void platform_config_interrupts(void)
{
	int i;
	uintptr_t eic_dist_gicd_ctlr = TEGRA_ARM_PERIF_BASE + 0x1000;
	uintptr_t eic_dist_gicd_groupr_base = TEGRA_ARM_PERIF_BASE + 0x1080;
	uintptr_t eic_dist_gicd_isenabler = TEGRA_ARM_PERIF_BASE + 0x1110;
	uintptr_t eic_dist_gicd_prio_base = TEGRA_ARM_PERIF_BASE + 0x1400;
	uintptr_t eic_dist_gicd_targetsr_base = TEGRA_ARM_PERIF_BASE + 0x1800;
	uintptr_t eic_dist_gicd_intconfig_base = TEGRA_ARM_PERIF_BASE + 0x1C00;
	uintptr_t eic_proc_if_gicc_ctlr = TEGRA_ARM_PERIF_BASE + 0x2000;
	uintptr_t eic_proc_if_gicc_pmr = TEGRA_ARM_PERIF_BASE + 0x2004;
	uint32_t val = 0, wdt_cpu = (4 * 32) + 27, wdt_cop = (4 * 32) + 28;

	uintptr_t base = eic_dist_gicd_groupr_base;
	for (i = 0; i < NR_IRQS; i += 32)
		*(volatile uint32_t *)(base + i * 4 / 32) = ~0;

	/* enable fiqs, group 0/1, irq/fiq bypass fields */
	val = (1 << 0 | 1 << 1 | 1 << 3 | 0xF << 4);
	*(volatile uint32_t *)(eic_proc_if_gicc_ctlr) = val;

	/* priority mask max, accept ints at all priority levels */
	*(volatile uint32_t *)(eic_proc_if_gicc_pmr) = 0xFF;

	/* enable group 0/1 in the distributor */
	*(volatile uint32_t *)(eic_dist_gicd_ctlr) = (1 << 1) | (1 << 0);

	/* enable path for the 2 WDTs in GIC: CPU = 27, COP = 28 */
	val = *(volatile uint32_t *)(eic_dist_gicd_isenabler);
	val |= (1 << 27) | (1 << 28);
	*(volatile uint32_t *)(eic_dist_gicd_isenabler) = val;

	/* map WDT_CPU to CPU 0 */
	*(volatile uint32_t *)(eic_dist_gicd_targetsr_base +
		((wdt_cpu / 4) * 4)) |= ~(0xFF << ((wdt_cpu % 4) * 8));
	*(volatile uint32_t *)(eic_dist_gicd_targetsr_base +
		((wdt_cpu / 4) * 4)) |= (1 << ((wdt_cpu % 4) * 8));

	/* map WDT_AVP to CPUs 0-3 */
	*(volatile uint32_t *)(eic_dist_gicd_targetsr_base +
		((wdt_cop / 4) * 4)) |= ~(0xFF << ((wdt_cop % 4) * 8));
	*(volatile uint32_t *)(eic_dist_gicd_targetsr_base +
		((wdt_cop / 4) * 4)) |= (0xF << ((wdt_cop % 4) * 8));

	/* configure both WDTs as level-triggered interrupts */
	val = *(volatile uint32_t *)(eic_dist_gicd_intconfig_base +
		((wdt_cpu / 16) * 4));
	val &= ~(3 << ((wdt_cpu % 16) * 2));
	*(volatile uint32_t *)(eic_dist_gicd_intconfig_base +
		((wdt_cpu / 16) * 4)) = val;
	val = *(volatile uint32_t *)(eic_dist_gicd_intconfig_base +
		((wdt_cop / 16) * 4));
	val &= ~(3 << ((wdt_cop % 16) * 2));
	*(volatile uint32_t *)(eic_dist_gicd_intconfig_base +
		((wdt_cop / 16) * 4)) = val;

	/* configure WDT0 as IRQ (grp 1), WDT4 as FIQ (grp 0) */
	val = *(volatile uint32_t *)(eic_dist_gicd_groupr_base +
		((wdt_cpu / 32) * 4));
	val |= (1 << (wdt_cpu % 32));
	val &= ~(1 << (wdt_cop % 32));
	*(volatile uint32_t *)(eic_dist_gicd_groupr_base +
		((wdt_cpu / 32) * 4)) = val;

	/* WDT0 is low priority */
	val = *(volatile uint32_t *)(eic_dist_gicd_prio_base +
		((wdt_cpu / 4) * 4));
	val &= ~(0xFF << ((wdt_cop % 4) * 8));
	*(volatile uint32_t *)(eic_dist_gicd_prio_base +
		((wdt_cpu / 4) * 4)) = val;
}
