/*
 * Copyright (c) 2012-2014, NVIDIA CORPORATION. All rights reserved
 *
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files
 * (the "Software"), to deal in the Software without restriction,
 * including without limitation the rights to use, copy, modify, merge,
 * publish, distribute, sublicense, and/or sell copies of the Software,
 * and to permit persons to whom the Software is furnished to do so,
 * subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 * IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY
 * CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,
 * TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
 * SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

#include <assert.h>
#include <errno.h>
#include <err.h>
#include <debug.h>
#include <rand.h>
#include <string.h>
#include <stdlib.h>
#include <lib/heap.h>
#include <arch/arm/mmu.h>
#include <arch/ops.h>
#include <arch/arm.h>
#include <platform.h>
#include <platform/memmap.h>
#include <platform/irqs.h>
#include <kernel/task.h>
#include <target/debugconfig.h>
#include <lib/monitor/monitor_vector.h>
#if ARM_WITH_OUTER_CACHE
#include <arch/outercache.h>
#endif
#include <platform/platform_p.h>

#define	MB		(1024 * 1024)

extern unsigned long boot_secondary_cpu_addr;
extern unsigned int coldboot_normal_os;
extern unsigned int normal_os_coldboot_fn;
extern uint32_t device_uid[4];

#if !defined(WITH_MONITOR_BIN)
extern uint32_t __save_boot_regs[9];
extern uint32_t __save_boot_cpsr;
extern uint32_t __jumpback_addr;
#endif

uint32_t debug_uart_id = DEFAULT_DEBUG_PORT;

static te_ss_op_t *ss_op_shmem;
static te_ss_op_t *ns_ss_op_shmem;

/* track available kernel VA space */
vaddr_t platform_vaspace_ptr;
vaddr_t platform_vaspace_end;

extern unsigned long cbstruct_addr;
extern unsigned long cbuf_addr;

static addr_t user_vector_page;

void platform_early_init(void)
{
	platform_init_debug_port(debug_uart_id);
}

void platform_idle(void)
{
	struct tz_monitor_frame frame, *smc_frame;
#if ARM_CPU_CORTEX_A9
	uint32_t val;
#endif

#if !defined(WITH_MONITOR_BIN)
	/* mark the entire TLK carveout as secure in the MC */
	platform_secure_dram_aperture();
#endif

#if !defined(WITH_MONITOR_BIN)
	memset(&frame, 0, sizeof(frame));

	ASSERT(__jumpback_addr);
	frame.pc = __jumpback_addr;
	frame.spsr = __save_boot_cpsr;	/* interrupts disabled, in svc */
	memcpy(&frame.r[4], __save_boot_regs, sizeof(__save_boot_regs));
#endif

#if ARM_CPU_CORTEX_A9
	/*
	 * Before going to the NS world for the first time, set ACTLR.FW.
	 * The NSACR.NS_SMP setting granted it the capability to set ACTLR.SMP,
	 * but it doesn't cover NS writing ACTLR.FW.
	 */
	val = arm_read_actlr();
	val |= 0x1;
	arm_write_actlr(val);
#endif

	dputs(CRITICAL, "TLK initialization complete. Jumping to non-secure world\n");

	smc_frame = tz_switch_to_ns(SMC_TOS_INITIAL_NS_RETURN, &frame);
	while (smc_frame) {
		tz_stdcall_handler(smc_frame);
		smc_frame = tz_switch_to_ns(SMC_TOS_COMPLETION, smc_frame);
	}
}

void platform_init(void)
{
	task_map_t mptr;
	arm_phys_attrs_t attrs;
	uint32_t reg = 0;

	platform_init_cpu();

	/* map read-only user vector page in kernel */
	arm_mmu_desc_set_default_kernel(&attrs);

	mptr.map_attrs = &attrs;
	mptr.size = PAGE_SIZE;
	mptr.flags = TM_UR;

	arm_mmu_map_kpage(0xFFFF0000, virtual_to_physical(user_vector_page), &mptr);
	arm_invalidate_tlb();

	memset((void *)user_vector_page, 0, mptr.size);

	/* load user vectors (used by libc to get tls) */
	memcpy((char *)user_vector_page + 0xFE0, arm_get_tls, 0x8);
	memcpy((char *)user_vector_page + 0xFE8, arm_get_tls, 0x8);

	platform_clean_invalidate_cache_range(user_vector_page, mptr.size);

	platform_setup_keys();

	/*
	 * Set SE_TZRAM_SECURITY sticky bit to respect secure TZRAM accesses.
	 * Note: No need to reprogram it after LP0 exit as it's part of SE's
	 * sticky bits HW LP0 context, so will be restored by the BR.
	 */
	reg = *(volatile uint32_t *)(TEGRA_SE_BASE + 0x4);
	reg &= ~(0x1);
	*(volatile uint32_t *)(TEGRA_SE_BASE + 0x4) = reg;
}

void platform_init_mmu_mappings(void)
{
	extern int _heap_end;
	extern uint32_t __load_phys_size, __early_heap_allocs;

	/*
	 * End of the kernel's heap is the carveout size, reduced by
	 * any early allocations (e.g. pages used for pagetables).
	 */
	_heap_end = (VMEMBASE + __load_phys_size) - __early_heap_allocs;
	_heap_end &= ~PAGE_MASK;

	/* reserve user vector page (before heap is initialized) */
	_heap_end -= PAGE_SIZE;
	user_vector_page = _heap_end;

	/* setup available vaspace (starts at end of carveout memory) */
	platform_vaspace_ptr = VMEMBASE + __load_phys_size;
	platform_vaspace_end = platform_vaspace_ptr + (VMEMSIZE - __load_phys_size);
}

uint64_t platform_translate_nsaddr(nsaddr_t vaddr, uint32_t type)
{
#if defined(WITH_MONITOR_BIN)
	struct tz_monitor_frame frame;
	frame.r[0] = vaddr;
	frame.r[1] = type;
	monitor_send_receive(SMC_TOS_ADDR_TRANSLATE, &frame);
	return frame.r[0];
#else
	arm_write_v2p(vaddr, type);
	return arm_read_par();
#endif
}

uint32_t platform_get_rand32(void)
{
	return rand();
}

uint32_t platform_get_time_us(void)
{
	return *(volatile uint32_t *)(TEGRA_TMRUS_BASE);
}

status_t platform_ss_register_handler(struct tz_monitor_frame *frame)
{
	/* r[1] -> address of shared fs operation buffer */
	ns_ss_op_shmem = (te_ss_op_t *)(uintptr_t)frame->r[1];

	ss_op_shmem = (te_ss_op_t *)tz_map_shared_mem((nsaddr_t)frame->r[1],
						sizeof(*ss_op_shmem));
	if (!ss_op_shmem)
		return ERR_GENERIC;

	return NO_ERROR;
}

/*
 * Calculate the physical address of the shared buffer that we have got for
 * logging from the linux kernel. All references to the shared buffer from
 * within tlk are made directly to the physical address.
 */
status_t set_log_phy_addr(nsaddr_t _ns_cb_struct_addr)
{
	struct circular_buffer *cbstruct;
	nsaddr_t cbuf;

	cbstruct = (struct circular_buffer *)
			tz_map_shared_mem(_ns_cb_struct_addr, PAGE_SIZE);
	if (cbstruct == NULL) {
		dprintf(CRITICAL, "%s: failed to map cbstruct\n", __func__);
		return ERR_NO_MEMORY;
	}

	cbuf = tz_map_shared_mem(cbstruct->buf, cbstruct->size);
	if (cbuf == NULL) {
		dprintf(CRITICAL, "%s: failed to map cbuf\n", __func__);
		return ERR_NO_MEMORY;
	}

	cbstruct_addr = (unsigned long)cbstruct;
	cbuf_addr = (unsigned long)cbuf;

	return NO_ERROR;
}

static int validate_rpmb_frame_arg(te_storage_param_t *arg)
{
	/* validate frame length */
	if (arg->mem.len != RPMB_FRAME_SIZE)
		return -EINVAL;

	/* validate frame memory range */
	if (!task_valid_address((vaddr_t)arg->mem.base, arg->mem.len))
		return -EFAULT;

	return 0;
}

int platform_ss_request_handler(te_storage_request_t *req)
{
	struct tz_monitor_frame frame, *smc_frame;
	int result;

	ss_op_shmem->type = req->type;
	ss_op_shmem->result = 0;

	dprintf(INFO, "%s: type 0x%x\n", __func__, req->type);

	switch (ss_op_shmem->type) {
	case OTE_FILE_REQ_TYPE_CREATE:
		/* input args: dirname, filename, flags */
		strncpy(ss_op_shmem->params.f_create.dname,
			req->args[0].mem.base,
			sizeof(ss_op_shmem->params.f_create.dname));
		strncpy(ss_op_shmem->params.f_create.fname,
			req->args[1].mem.base,
			sizeof(ss_op_shmem->params.f_create.fname));
		ss_op_shmem->params.f_create.flags = req->args[2].val.a;

		ss_op_shmem->params_size =
			sizeof(ss_op_shmem->params.f_create);
		break;
	case OTE_FILE_REQ_TYPE_DELETE:
		/* input args: dirname, filename */
		strncpy((char *)ss_op_shmem->params.f_delete.dname,
			req->args[0].mem.base,
			sizeof(ss_op_shmem->params.f_delete.dname));
		strncpy((char *)ss_op_shmem->params.f_delete.fname,
			req->args[1].mem.base,
			sizeof(ss_op_shmem->params.f_delete.fname));

		ss_op_shmem->params_size =
			sizeof(ss_op_shmem->params.f_delete);
		break;
	case OTE_FILE_REQ_TYPE_OPEN:
		/* input args: dirname, filename, flags */
		strncpy((char *)ss_op_shmem->params.f_open.dname,
			req->args[0].mem.base,
			sizeof(ss_op_shmem->params.f_open.dname));
		strncpy((char *)ss_op_shmem->params.f_open.fname,
			req->args[1].mem.base,
			sizeof(ss_op_shmem->params.f_open.fname));
		ss_op_shmem->params.f_open.flags = req->args[2].val.a;

		ss_op_shmem->params_size = sizeof(ss_op_shmem->params.f_open);
		break;
	case OTE_FILE_REQ_TYPE_CLOSE:
		/* input args: file handle */
		ss_op_shmem->params.f_close.handle = req->args[0].val.a;

		ss_op_shmem->params_size = sizeof(ss_op_shmem->params.f_close);
		break;
	case OTE_FILE_REQ_TYPE_READ:
		/* validate read buffer */
		if (!task_valid_address((vaddr_t)req->args[1].mem.base,
					req->args[1].mem.len)) {
			return -EFAULT;
		}

		/* validate read buffer size */
		if (req->args[1].mem.len >
			sizeof(ss_op_shmem->params.f_read.data)) {
			return -EINVAL;
		}

		/* input args: file_handle, read buffer */
		ss_op_shmem->params.f_close.handle = req->args[0].val.a;
		ss_op_shmem->params.f_read.data_size = req->args[1].mem.len;

		ss_op_shmem->params_size = sizeof(ss_op_shmem->params.f_read);
		break;
	case OTE_FILE_REQ_TYPE_WRITE:
		/* validate write buffer */
		if (!task_valid_address((vaddr_t)req->args[1].mem.base,
					req->args[1].mem.len)) {
			return -EFAULT;
		}

		/* validate write buffer size */
		if (req->args[1].mem.len >
			sizeof(ss_op_shmem->params.f_write.data)) {
			return -EINVAL;
		}

		/* input args: file_handle, write buffer */
		ss_op_shmem->params.f_write.handle = req->args[0].val.a;
		ss_op_shmem->params.f_write.data_size = req->args[1].mem.len;
		memcpy((void*)ss_op_shmem->params.f_write.data,
			req->args[1].mem.base,
			ss_op_shmem->params.f_write.data_size);

		ss_op_shmem->params_size = sizeof(ss_op_shmem->params.f_write);
		break;
	case OTE_FILE_REQ_TYPE_GET_SIZE:
		/* input arg: file_handle */
		ss_op_shmem->params.f_getsize.handle = req->args[0].val.a;

		ss_op_shmem->params_size =
			sizeof(ss_op_shmem->params.f_getsize);
		break;
	case OTE_FILE_REQ_TYPE_SEEK:
		/* input args: file_handle, offset, whence */
		ss_op_shmem->params.f_seek.handle = req->args[0].val.a;
		ss_op_shmem->params.f_seek.offset = req->args[1].val.a;
		ss_op_shmem->params.f_seek.whence = req->args[2].val.a;

		ss_op_shmem->params_size = sizeof(ss_op_shmem->params.f_seek);
		break;
	case OTE_FILE_REQ_TYPE_TRUNC:
		/* input args: file_handle, length */
		ss_op_shmem->params.f_trunc.handle = req->args[0].val.a;
		ss_op_shmem->params.f_trunc.length = req->args[1].val.a;

		ss_op_shmem->params_size = sizeof(ss_op_shmem->params.f_trunc);
		break;
	case OTE_FILE_REQ_TYPE_RPMB_WRITE:
		/* validate request frame */
		result = validate_rpmb_frame_arg(&req->args[0]);
		if (result) {
			dprintf(CRITICAL, "%s: write req frame invalid\n",
				__func__);
			return result;
		}

		/* validate request-response frame */
		result = validate_rpmb_frame_arg(&req->args[1]);
		if (result) {
			dprintf(CRITICAL, "%s: write req-resp frame invalid\n",
				__func__);
			return result;
		}

		/* validate response frame */
		result = validate_rpmb_frame_arg(&req->args[2]);
		if (result) {
			dprintf(CRITICAL, "%s: write resp frame invalid\n",
				__func__);
			return result;
		}

		/* input args: request frame, request-response frame */
		memcpy((void*)ss_op_shmem->params.f_rpmb_write.req_frame,
			req->args[0].mem.base, req->args[0].mem.len);
		memcpy((void*)ss_op_shmem->params.f_rpmb_write.req_resp_frame,
			req->args[1].mem.base, req->args[1].mem.len);

		ss_op_shmem->params_size =
			sizeof(ss_op_shmem->params.f_rpmb_write);
		break;
	case OTE_FILE_REQ_TYPE_RPMB_READ:
		/* validate request frame */
		result = validate_rpmb_frame_arg(&req->args[0]);
		if (result) {
			dprintf(CRITICAL, "%s: read req frame invalid\n",
				__func__);
			return result;
		}

		/* validate response frame */
		result = validate_rpmb_frame_arg(&req->args[1]);
		if (result) {
			dprintf(CRITICAL, "%s: read resp frame invalid\n",
				__func__);
			return result;
		}

		/* input arg: request frame */
		memcpy((void*)ss_op_shmem->params.f_rpmb_read.req_frame,
			req->args[0].mem.base, req->args[0].mem.len);

		ss_op_shmem->params_size =
			sizeof(ss_op_shmem->params.f_rpmb_read);
		break;
	default:
		return -EINVAL;
	}

	/* adjust size down to only include required parameters */
	ss_op_shmem->req_size = (sizeof(te_ss_op_t) - sizeof(uint32_t) -
		sizeof(te_ss_req_params_t)) + ss_op_shmem->params_size;

#if defined(WITH_MONITOR_BIN)
	memset(&frame, 0, sizeof(struct tz_monitor_frame));
	frame.r[0] = SMC_ERR_PREEMPT_BY_FS;

	smc_frame = tz_switch_to_ns(SMC_TOS_COMPLETION, &frame);
	while (smc_frame->r[0] != SMC_TOS_RESTART) {
		frame.r[0] = SMC_ERR_PREEMPT_BY_IRQ;
		smc_frame = tz_switch_to_ns(SMC_TOS_COMPLETION, &frame);
	}
#else
	smc_frame = tz_switch_to_ns(SMC_TOS_PREEMPT_BY_FS, NULL);
	while (smc_frame)
		smc_frame = tz_switch_to_ns(SMC_TOS_PREEMPT_BY_IRQ, NULL);
#endif

	req->result = ss_op_shmem->result;
	if (req->result != 0) {
		dprintf(CRITICAL, "%s: call to non-secure world failed 0x%x\n",
			__func__, req->result);
		return req->result;
	}

	/* move any expected return data into request structure */
	switch (ss_op_shmem->type) {
	case OTE_FILE_REQ_TYPE_OPEN:
		/* output arg: file_handle */
		req->args[3].val.a = ss_op_shmem->params.f_open.handle;
		break;
	case OTE_FILE_REQ_TYPE_READ:
		/* output args: amount of data read, data */
		req->args[1].mem.len = ss_op_shmem->params.f_read.data_size;
		memcpy(req->args[1].mem.base,
			(void *)ss_op_shmem->params.f_read.data,
			req->args[1].mem.len);
		break;
	case OTE_FILE_REQ_TYPE_GET_SIZE:
		/* output arg: file size */
		req->args[1].val.a = ss_op_shmem->params.f_getsize.size;
		break;
	case OTE_FILE_REQ_TYPE_RPMB_WRITE:
		/* output arg: response frame */
		memcpy(req->args[2].mem.base,
			(void *)ss_op_shmem->params.f_rpmb_write.resp_frame,
			req->args[2].mem.len);
		break;
	case OTE_FILE_REQ_TYPE_RPMB_READ:
		/* output arg: response frame */
		memcpy(req->args[1].mem.base,
			(void *)ss_op_shmem->params.f_rpmb_read.resp_frame,
			req->args[1].mem.len);
		break;
	default:
		break;
	}

	return 0;
}

void platform_get_device_id(te_device_id_args_t *out)
{
	if (out)
		memcpy(out, device_uid, sizeof(te_device_id_args_t));
}

void platform_clean_invalidate_cache_range(vaddr_t range, uint32_t length)
{
#if defined(ARM_USE_CPU_CACHING)
	arch_clean_invalidate_cache_range(range, length);

#if ARM_WITH_OUTER_CACHE
	outer_clean_range(virtual_to_physical(range), length);
	outer_inv_range(virtual_to_physical(range), length);
#endif
#endif
}
