/*
 * Copyright (c) 2013-2014, NVIDIA CORPORATION. All rights reserved
 *
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files
 * (the "Software"), to deal in the Software without restriction,
 * including without limitation the rights to use, copy, modify, merge,
 * publish, distribute, sublicense, and/or sell copies of the Software,
 * and to permit persons to whom the Software is furnished to do so,
 * subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 * IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY
 * CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,
 * TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
 * SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */
#include <config.h>
#include <asm.h>
#include <arch/arm.h>
#include <arm64/asm.h>
#include <arm64/monitor_macros.h>
#include <lib/monitor/monitor_vector.h>

/*
 * Set up initial registers and transition to EL1 to start the
 * secureos (no args are passed to this routine and it does not
 * return).
 */
FUNCTION(mon_start_tlk_el1)
        /*
	 * Save EL1 entry state before secureos init (which modifies
	 * the same EL1 arch state), so on return to the BL, it can
	 * be restored.
	 */
	adr	x3, el1_non_secure_context
	mon_save_el1_regs x3

	/* entry into EL1 is at _end of monitor binary */
	ldr	x6, __mon_phys_offset
	adr	x4, _end
	sub     x5, x4, x6	// phys addr of _end

	/* adjust carveout (reduced by what monitor's used) */
	adr	x3, __mon_phys_size
	ldr	w0, [x3]
	adr	x4, _start
	sub     x7, x4, x6	// phys addr of _start
	sub	x3, x5, x7	// amount used by monitor
	sub	w0, w0, w3	// reduced size

	/* size and bootarg passed as 32bit args */
	adr	x3, __mon_bootarg_addr
	ldr	w1, [x3]

	mon_scr_secure_32 x3

	mov	x3, #(MON_SPSR_EXC_MASKED | MODE_SVC)
	msr	spsr_el3, x3
	msr	elr_el3, x5
	eret

.global handle_trusted_os_call
handle_trusted_os_call:
	/* SMC expected from secureos in EL1(S) */
	mrs	x9, scr_el3
	tst	x9, #MON_SCR_NS_MODE
	b.ne	.		// not in secure mode

	and	x0, x0, #SMC_TOS_FUNC_ID_MASK
	cmp	x0, #SMC_TOS_MAX_FUNC_IDX
	b.gt	.		// too large an index

	/* call function at tos_table[idx] */
	adr	x9, tos_table
	ldr	x10, [x9, x0, lsl #3]
	br	x10

tos_completion:
	ldp	x9, x10, [sp], #16	// restore scratch

	/* save secure EL1 state */
	adr	x3, el1_secure_context
	mon_save_el1_regs x3

	/* restore NS EL1 state (loads spsr_el3/elr_el3) */
	adr	x3, el1_non_secure_context
	mon_restore_el1_regs x3

	/* switch non-secure for return */
	mon_scr_non_secure_64 x3

	/* load SMC results into registers */
	ldr	x3, el1_smc_args_results
	ldp	x0, x1, [x3], #16
	ldp	x2, x3, [x3], #16

	eret

// Is this the first SMC we receive (from the bootloader) on boot?
tos_initial_ns_return:
	ldp	x9, x10, [sp], #16	// restore scratch

	/* save S EL1 state */
	adr	x3, el1_secure_context
	mon_save_el1_regs x3

	/* restore NS EL1 state (from monitor entry point) */
	adr	x3, el1_non_secure_context
	mon_restore_el1_regs x3

	bl	platform_psci_coldboot_epilog
	b	mon_return_aarch64_ns

tos_init_shared_addr:
	ldp	x9, x10, [sp], #16	// restore scratch

	/* save shared mem address (in x1) and return */
	adr	x3, el1_smc_args_results
	str	x1, [x3]

	eret

/*
 * This TOS call is serviced entirely within the monitor and returns
 * to secure EL1. If the virt->phys translation is for the normal world,
 * then make that MMU state current.
 */
tos_addr_translation:
	/* get virt address and type */
	ldr	x9, el1_smc_args_results
	ldp	x0, x1, [x9]

	/* translate in current or non-secure world */
	tst	x1, 0x4
	b.eq	write_vreg	// current: no need to change state */

	/* make non-secure MMU state current (corrupts x0, x1) */
	adr	x3, el1_secure_context
	mon_save_el1_mmu x3
	adr	x3, el1_non_secure_context
	mon_restore_el1_mmu x3

	/* switch non-secure for translation */
	mon_scr_non_secure_64 x3

	/* reload args */
	ldp	x0, x1, [x9]

write_vreg:
	mov	x10, x1
	and	x10, x10, #3
	adr	x2, v2p_table
	add	x2, x2, x10, lsl #3	// each type is 2 instrs
	br	x2

v2p_table:
	at	s12e1r, x0	// S12 priv read
	b	read_par
	at	s12e1w, x0	// S12 priv write
	b	read_par
	at	s12e0r, x0	// S12 user read
	b	read_par
	at	s12e0w, x0	// S12 user write

read_par:
	mrs	x0, par_el1
	str	x0, [x9]	// return par in x0

	/* translate in current or non-secure world */
	tst	x1, 0x4
	b.eq	do_return	// current: no need to restore state */

	/* restore secure MMU state */
	adr	x3, el1_secure_context
	mon_restore_el1_mmu x3

	/* go back to secure mode */
	mon_scr_secure_32 x3

do_return:
	ldp	x9, x10, [sp], #16	// restore scratch
	eret

/*
 * Callbacks from secure->non-secure (not ready yet)
 */
tos_callback:
	b	.

.global non_secure_stdcall
non_secure_stdcall:
	/* save incoming SMC args from registers */
	ldr	x9, el1_smc_args_results
	stp	x0, x1, [x9], #16
	stp	x2, x3, [x9], #16
	stp	x4, x5, [x9], #16
	stp	x6, x7, [x9], #16

	ldp	x9, x10, [sp], #16	// restore scratch

	/*
	 * Some SMCs (besides PSCI) don't transition to the secureos.
	 * For instance, the registration of IRQ and FS callback handlers.
	 *
	 * It does mean we're not supporting this functionality at the
	 * moment, but the intent is to change the implementation, to not
	 * require returning to a different PC than where we were called.
	 */
	ldr	w3, =0x32000004		// SMC_TOS_NS_IRQ_PENDING_VECTOR
	cmp	w0, w3
	b.ne	call_secureos

.global call_secureos
call_secureos:
	/* save NS EL1 state */
	adr	x3, el1_non_secure_context
	mon_save_el1_regs x3

	/* restore S EL1 state (including spsr_el3/elr_el3) */
	adr	x3, el1_secure_context
	mon_restore_el1_regs x3

	/* set scr to secure32 */
	mon_scr_secure_32 x3
	eret

.align 3
el1_secure_context:
	.rept   NUM_CTX_REGS
	.quad   0
	.endr

;.global el1_non_secure_context // Not needed anymore?
el1_non_secure_context:
	.rept   NUM_CTX_REGS
	.quad   0
	.endr

/* SMCs issued from the Trusted OS */
tos_table:
	.quad	-1
	.quad	tos_completion
	.quad	tos_completion		/* IRQ */
	.quad	tos_completion		/* filesystem */
	.quad	tos_initial_ns_return
	.quad	tos_addr_translation
	.quad	tos_init_shared_addr

/* address of shared mem for passing SMC args/results */
el1_smc_args_results:
	.quad   0
